/*
Copyright (c) 2005, Jiye Zeng (http://www.zegraph.com/)

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

[1] Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer. 

[2] Neither the name of the organization nor the names of its contributors
may be used to endorse or promote products derived from this software
without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "config.h"
#include "parse.h"
#include "token.h"
#include "scan.h"
#include "compiler.h"
#include <cmath>
#include <cstring>
#include <iostream>

static char zs_esc(const char* p, int &k)
{
	if (p[k] != '\\') {
		k++;
		return p[k-1];
	}

	k++;

	int code;

	switch (p[k]) {
	case 'a':
		k++;
		return '\a';
	case 'b':
		k++;
		return '\b';
	case 'f':
		k++;
		return '\f';
	case 'n':
		k++;
		return '\n';
	case 'r':
		k++;
		return '\r';
	case 't':
		k++;
		return '\t';
	case 'v':
		k++;
		return '\v';
	case '?':
		k++;
		return '\?';
	case '`':
		k++;
		return '`';
	case '\'':
		k++;
		return '\'';
	case '\"':
		k++;
		return '\"';
	case '\\':
		k++;
		return '\\';
	case 'x':
		switch(p[k+1]) {
		case '0':
			code = 0;
			break;
		case '1':
			code = 1 * 16;
			break;
		case '2':
			code = 2 * 16;
			break;
		case '3':
			code = 3 * 16;
			break;
		case '4':
			code = 4 * 16;
			break;
		case '5':
			code = 5 * 16;
			break;
		case '6':
			code = 6 * 16;
			break;
		case '7':
			code = 7 * 16;
			break;
		case '8':
			code = 8 * 16;
			break;
		case '9':
			code = 9 * 16;
			break;
		case 'a':
		case 'A':
			code = 10 * 16;
			break;
		case 'b':
		case 'B':
			code = 11 * 16;
			break;
		case 'c':
		case 'C':
			code = 12 * 16;
			break;
		case 'd':
		case 'D':
			code = 13 * 16;
			break;
		case 'e':
		case 'E':
			code = 14 * 16;
			break;
		case 'f':
		case 'F':
			code = 15 * 16;
			break;
		default:
			return p[k-1];
		}
		switch(p[k+2]) {
		case '0':
			break;
		case '1':
			code += 1;
			break;
		case '2':
			code += 2;
			break;
		case '3':
			code += 3;
			break;
		case '4':
			code += 4;
			break;
		case '5':
			code += 5;
			break;
		case '6':
			code += 6;
			break;
		case '7':
			code += 7;
			break;
		case '8':
			code += 8;
			break;
		case '9':
			code += 9;
			break;
		case 'a':
		case 'A':
			code += 10;
			break;
		case 'b':
		case 'B':
			code += 11;
			break;
		case 'c':
		case 'C':
			code += 12;
			break;
		case 'd':
		case 'D':
			code += 13;
			break;
		case 'e':
		case 'E':
			code += 14;
			break;
		case 'f':
		case 'F':
			code += 15;
			break;
		default:
			return p[k-1];
		}
		k += 3;
		return code;
	}
	return p[k-1];
}

void zsTokens::error(int idx, const char *msg, const char *msg2) const
{
	std::string error(_fname);
	if (idx < 0) idx = 0;
	if (idx >= _id.size()) idx = _id.size()-1;
	char buf[64];
	sprintf(buf, " on line %d", line(idx));
	if (!msg) {
		error += " -- syntax error prior to ";
		error += _str[idx];
	}
	else {
		error += ": ";
		error += _str[idx];
		error += " -- ";
		error += msg;
		if (msg2) {
			error += " ";
			error += msg2;
		}
	}
	error += buf;
	throw error;
}

void zsTokens::add(int id, const std::string& str, int line)
{
	_id.push_back(id);
	_str.push_back(str);
	_line.push_back(line);
}

void zsTokens::scan(const char *pname, const char *fname, bool codestring)
{
	char buf[32];
	sprintf(buf, "M%p", fname);
	add(TOK_ID, buf, 0);
	add('{', "{", 0);

    Scanner lex;
    memset((char*) &lex, 0, sizeof(lex));
	lex.line = 1;

	if (codestring) {
		_fname = buf;
		lex.bot = (uchar*)fname;
		lex.tok = lex.bot;
		lex.ptr = lex.bot;
		lex.cur = lex.bot;
		lex.pos = lex.bot;
		lex.top = lex.bot + strlen(fname);
		lex.lim = lex.top;
	}
	else {
		_fname = fname;
		lex.fd = fopen(pname, "r");
	}

	int id;
	std::stack<int> pair, lnum;

	while ((id = zs_scan(&lex)) != TOK_EOI && id != TOK_ANY) {
		switch (id) {
		case TOK_STRING: {
			const char* p = (const char*)lex.tok + 1;
			int len = lex.cur - lex.tok - 2;
			std::string s;
			s.reserve(len+2);
			int k = 0;
			while (k < len) s += zs_esc(p, k);
			add(id, s, lex.line);
			}
			break;
		case '(':
			add(id, "(", lex.line);
			pair.push(id);
			lnum.push(_id.size()-1);
			break;
		case ')':
			add(id, ")", lex.line);
			if (pair.empty() || pair.top()!='(') error(_id.size()-1, "missing ( for the token");
			pair.pop();
			lnum.pop();
			break;
		case '[':
			add(id, "[", lex.line);
			pair.push(id);
			lnum.push(_id.size()-1);
			break;
		case ']':
			add(id, "]", lex.line);
			if (pair.empty() || pair.top()!='[') error(_id.size()-1, "missing [ for the token");
			pair.pop();
			lnum.pop();
			break;
		case '{':
			add(id, "{", lex.line);
			pair.push(id);
			lnum.push(_id.size()-1);
			break;
		case '}':
			add(id, "}", lex.line);
			if (pair.empty() || pair.top()!='{') error(_id.size()-1, "missing { for the token");
			pair.pop();
			lnum.pop();
			break;
		case TOK_NIL_COL:
			add(TOK_NIL, "*", lex.line);
			add(':', ":", lex.line);
			break;
		case TOK_COL_NIL:
			add(':', ":", lex.line);
			add(TOK_NIL, "*", lex.line);
			break;
		case TOK_NIL_BRA:
			add(TOK_NIL, "*", lex.line);
			add(']', "]", lex.line);
			if (pair.empty() || pair.top()!='[') error(_id.size()-1, "missing [ for the token");
			pair.pop();
			lnum.pop();
			break;
		case TOK_BRA_NIL:
			add('[', "[", lex.line);
			add(TOK_NIL, "*", lex.line);
			pair.push('[');
			lnum.push(_id.size()-1);
			break;
		case TOK_NIL_COM:
			add(TOK_NIL, "*", lex.line);
			add(',', ",", lex.line);
			break;
		case TOK_COM_NIL:
			add(',', ",", lex.line);
			add(TOK_NIL, "*", lex.line);
			break;
		case '-':
			if (_id.back() == '(' ||
				_id.back() == '[' ||
				_id.back() == ',' ||
				_id.back() == TOK_RETURN ||
				_id.back() == ':' ||
				_id.back() == '?' ||
				zs_precedence(_id.back()) > 3) id = TOK_NEGATE;	// unary -
			add(id, "-", lex.line);
			break;
		default:
			add(id, std::string((char*)lex.tok, lex.cur-lex.tok), lex.line);
		}
	}

	if (codestring) {
		if (lex.top >= lex.cur) error(_id.size()-1);
	}
	else {
		if (id == TOK_ANY) error(_id.size()-1);
	    fclose(lex.fd);
		free(lex.bot);
	}

	if (!pair.empty()) {
		switch (pair.top()) {
		case '(':
			error(lnum.top(), "missing ) for the token");
			break;
		case '[':
			error(lnum.top(), "missing ] for the token");
			break;
		case '{':
			error(lnum.top(), "missing } for the token");
			break;
		default:
			error(lnum.top(), "missing counterpart for the token");
			break;
		}
	}

	lex.line++;
	add('}', "}", lex.line);
}

