/*
Copyright (c) 2005, Jiye Zeng (http://www.zegraph.com/)

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

[1] Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer. 

[2] Neither the name of the organization nor the names of its contributors
may be used to endorse or promote products derived from this software
without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "config.h"
#include "deelx.h"
#include "buffer.h"
#include "compiler.h"
#include "thread.h"
#include "sfl.h"
#include "syslib.h"
#include <ctime>
#include <cmath>
#include <algorithm>
#ifndef _WIN32
#include <utime.h>
#include <iconv.h>
#endif


#define FILE_TYPE	'FP'		// file pointer object type

#define SYSTEM_EXCEPTION "default exception"

//////////////////////////////// UTLs ///////////////////////////////

static void input_error(zsContext *ctx)
{
	ctx->expr->error("improper number/type of parameter input in function call");
}

static integer_t get_integer(zsContext *ctx, zoObject *o)
{
	if (o->type() != ZO_INTEGER) ctx->expr->error("integer type expected");
	return ((zoNumber*)o)->getInteger();
}

static real_t get_real(zsContext *ctx, zoObject *o)
{
	if (o->type() != ZO_REAL) ctx->expr->error("real type expected");
	return ((zoNumber*)o)->getReal();
}

static real_t get_number(zsContext *ctx, zoObject *o)
{
	if (o->type() == ZO_INTEGER) return ((zoNumber*)o)->getInteger();
	if (o->type() == ZO_REAL) return ((zoNumber*)o)->getReal();
	ctx->expr->error("number expected");
	return 0;
}

static const std::string& get_string(zsContext *ctx, zoObject *o)
{
	if (o->type() != ZO_STRING) ctx->expr->error("string expected");
	return ((zoString*)o)->get();
}

static zoArray* get_array(zsContext *ctx, zoObject *o)
{
	if (o->type() != ZO_ARRAY) ctx->expr->error( "array type expected");
	return (zoArray*)o;
}

static void* get_ptr(zsContext *ctx, zoObject *o)
{
	if (o->type() < ZO_USER) ctx->expr->error( "user type expected");
	return ((zoUser*)o)->get();
}

static void* get_user(zsContext *ctx, zoObject *o, int type)
{
	if (o->type() != ZO_USER+type) {
		std::string s("not a ");
		int endian = 1;
		char *c = (char*)(&endian);
		if (c[0] == 0) {
			c = (char*)(&type);
			if (c[3] != 0) s += c[3];
			if (c[2] != 0) s += c[2];
			if (c[1] != 0) s += c[1];
			if (c[0] != 0) s += c[0];
		}
		else {
			c = (char*)(&type);
			if (c[0] != 0) s += c[0];
			if (c[1] != 0) s += c[1];
			if (c[2] != 0) s += c[2];
			if (c[3] != 0) s += c[3];
		}
		s += "type";
		ctx->expr->error(s.c_str());
	}
	return ((zoUser*)o)->get();
}

static zoObject* use_primitive(zsContext *ctx, int nargs, zoObject **args, const char* name)
{
	zsPrimitiveFunc prim = zs_get_primitive(name, args[0]->type(), ctx->expr);
	return prim(ctx, nargs, args);
}

static zoObject* use_primitive2(zsContext *ctx, int nargs, zoObject **args, const char* name)
{
	zsPrimitiveFunc prim = zs_get_primitive(name, args[1]->type(), ctx->expr);
	return prim(ctx, nargs, args);
}
/*
static int cal2jul(int yy, int mm, int dd)
{
	yy -= mm < 3;
	mm += 12 * (mm < 3);
	int a = 365.25 * (yy + 4716),
		b = 30.6001 * (mm + 1);
	return a + b + dd - 2416556;
}

static int jul2tim(int *hh, int *mn, int *sc, real_t jul)
{
	const real_t sd = 1.0e-10; // this constant may be OS or compiler dependent
	if (jul == 0.0) {
		*hh = 0; *mn = 0; *sc = 0;
		return 0;
	}
	if (jul + sd < 1.0) jul += sd; 
	jul *= 24.0;
	*hh = jul;
	jul = (jul - *hh) * 60.0;
	*mn = jul;
	jul = (jul - *mn) * 60.0;
	*sc = jul;
	return (jul - *sc) * 1000;
}

static real_t jul2cal(int *yy, int *mm, int *dd, real_t jul)
{
	int a = jul + 2415032,
		b = a + 1524,
		c = (double(b) - 122.1) / 365.25,
		d = 365.25 * c,
		e = double(b - d) / 30.6001,
		f = 30.6001 * e;

	*dd = b - d - f;
	*mm = e - 1 - 12 * (e > 13);
	*yy = c - 4715 - (*mm > 2);

	return jul - floor(jul);
}
*/
static int cal2jul(int yy, int mm, int dd)
{
	// 2012.10
	// http://en.wikipedia.org/wiki/Julian_day
	int a = (14 - mm) / 12;
	yy += 4800 - a;
	mm += 12*a - 3;
	dd += (153*mm + 2)/5 + 365*yy + yy/4 - yy/100 + yy/400 - 32045;
	// match excel
	dd -= 2415019;
	return dd;
}

static void jul2cal(int *yy, int *mm, int *dd, real_t jul)
{
	jul = floor(jul);
	// match excel by adding 2415019
	// excel has problem for treating 1900 as a leap year
	// so the match show one day difference before 1900.3.1
	jul += 2415019.0;
	// http://en.wikipedia.org/wiki/Julian_day
	int j = int(jul + 0.5) + 32044;
	int g  = j/146097;
	int dg = j%146097;
	int c  = (dg/36524 + 1)*3/4;
	int dc = dg - c*36524;
	int b  = dc/1461;
	int db = dc%1461;
	int a  = (db/365 + 1)*3/4;
	int da = db - a*365;
	int y  = g*400 + c*100 + b*4 + a;
	int m  = (da*5 + 308)/153 - 2;
	int d  = da - (m + 4)*153/5 + 122;
	*yy = y - 4800 + (m + 2)/12;
	*mm = (m + 2)%12 + 1;
	*dd = d + 1;
}

static int jul2tim(int *hh, int *mn, int *sc, real_t jul)
{
	// this constant is for solving double precision problem (may be OS or compiler dependent)
	const real_t sd = 1.0e-10;
	if (jul >= 1.0) jul -= floor(jul); 
	if (jul <= 0.0) {
		*hh = 0; *mn = 0; *sc = 0;
		return 0;
	}
	if (jul + sd < 1.0) jul += sd; 
	jul *= 24.0;
	*hh = jul;
	jul = (jul - *hh) * 60.0;
	*mn = jul;
	jul = (jul - *mn) * 60.0;
	*sc = jul;
	return (jul - *sc) * 1000;
}

static real_t jultime(int hh, int mm, int ss)
{
	return hh/24.0 + mm/1440.0 + ss/86400.0;
}

////////////////////////// Generic ///////////////////////////////////////

static zoObject* sys_malloc(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	integer_t size = get_integer(ctx, args[0]);
	if (size <= 0) input_error(ctx);
	void *ptr = malloc(size);
	if (!ptr) ctx->expr->error("failed to allocate memory");
	return zs_create_user(ctx, ptr, 0, free, 0);
}

static zoObject* sys_sendmail(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 2) input_error(ctx);
	SMTP smtp;
	memset(&smtp, 0, sizeof(smtp));
	smtp.strSmtpServer = "localhost";
	smtp.connect_retry_cnt = 3;
	for (int i = 1; i < nargs; i+=2) {
		const std::string& s1 = get_string(ctx, args[i-1]);
		const std::string& s2 = get_string(ctx, args[i]);
		if (s1 == "server") {
			smtp.strSmtpServer = (char*)s2.c_str();
		}
		else if (s1 == "message") {
			smtp.strMessageBody = (char*)s2.c_str();
		}
		else if (s1 == "subject") {
			smtp.strSubject = (char*)s2.c_str();
		}
		else if (s1 == "sender") {
			smtp.strSenderUserId = (char*)s2.c_str();
		}
		else if (s1 == "receptors") {
			smtp.strDestUserIds = (char*)s2.c_str();
		}
		else if (s1 == "ccs") {
			smtp.strCcUserIds = (char*)s2.c_str();
		}
		else if (s1 == "bccs") {
			smtp.strBccUserIds = (char*)s2.c_str();
		}
		else if (s1 == "return") {
			smtp.strRrcpUserId = (char*)s2.c_str();
		}
		else if (s1 == "charset") {
			smtp.strCharSet = (char*)s2.c_str();
		}
		else if (s1 == "files") {
			smtp.strBinFiles = (char*)s2.c_str();
		}
		else {
			std::string error("unexpected SMTP property: ");
			error += s1;
			ctx->expr->error(error.c_str());
		}
	}
	if (smtp_send_mail_ex(&smtp)) return zs_create_string(ctx, smtp.strlast_smtp_message);
	return 0;
}

static zoObject* sys_getarg(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	zsObjTable<zsRCP<zoObject> > &vars = ctx->expr->module()->context()->vars();
	if (args[0]->type() == ZO_INTEGER) {
		zsObjItem<zsRCP<zoObject> > *item = vars.get(((zoNumber*)args[0])->getInteger());
		if (item) return item->_obj.get();
		return 0;
	}
	if (args[0]->type() == ZO_STRING) {
		zsObjItem<zsRCP<zoObject> > *item = vars.get(((zoString*)args[0])->get());
		if (item) return item->_obj.get();
		return 0;
	}
	return 0;
}

static zoObject* sys_error(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	ctx->expr->error(get_string(ctx, args[0]).c_str());
	return 0;
}
/*
static zoObject* sys_func(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	zsFuncExpr *f = ctx->expr->module()->getFunc(get_string(ctx, args[0]), ctx->expr, true);
	return zs_create_user(ctx, f, 0, 0, 0);
}

static zoObject* sys_call(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	zsFuncExpr *f = (zsFuncExpr*)get_ptr(ctx, args[0]);
	if (f->type() != EXPR_FUNC) ctx->expr->error("function pointer object expected");
	return f->call(ctx, nargs-1, &args[1]);
}

static zoObject* sys_parse(zsContext *ctx, int nargs, zoObject **args)
{
	extern void delete_module(void *ptr);
	if (nargs < 1) input_error(ctx);
	const std::string& code = get_string(ctx, args[0]);
	char msg[256];
	msg[0] = 0;
	zsModuleExpr *module = new zsModuleExpr;
	module->context()->setContext(ctx);
	try {
		module->import(ctx->expr->module(), code.c_str(), true);
		return zs_create_user(ctx, module, 0, delete_module, 0);
	}
	catch (const char* err) {
		strcpy(msg, err);
	}
	catch (const std::string &err) {
		strcpy(msg, err.c_str());
	}
	catch (...) {
		strcpy(msg, SYSTEM_EXCEPTION);
	}
	delete module;
	return zs_create_string(ctx, msg);
}

static zoObject* sys_eval(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	zsModuleExpr *module;
	if (args[0]->type() == ZO_USER) {
		module = (zsModuleExpr*)((zoUser*)args[0])->get();
		if (module->type() != EXPR_MODULE) ctx->expr->error("object is not a module");
	}
	else {
		zoObject* o = sys_parse(ctx, nargs, args);
		if (o->type() == ZO_STRING) return o;
		module = (zsModuleExpr*)((zoUser*)o)->get();
	}
	char msg[256];
	try {
		module->exec(0);
		if (nargs > 1 && get_integer(ctx, args[1]) != 0) module->context()->saveVars(stdout);
		return 0;
	}
	catch (const char* err) {
		strcpy(msg, err);
	}
	catch (const std::string &err) {
		strcpy(msg, err.c_str());
	}
	catch (...) {
		strcpy(msg, SYSTEM_EXCEPTION);
	}
	return zs_create_string(ctx, msg);
}

static zoObject* sys_trace(zsContext *ctx, int nargs, zoObject **args)
{
	ctx->saveVars(stdout);
	fputs("press ENTER to continue", stderr);
	fgetc(stdin);
	return 0;
}
*/

static zoObject* sys_import(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	const std::string& fname = get_string(ctx, args[0]);
	zsModuleExpr *parent = ctx->expr->module();
	char msg[256];
	msg[0] = 0;
	zsModuleExpr *module = new zsModuleExpr;
	try {
		if (!module->import(parent, fname.c_str(), false)) {
			delete module;
			sprintf(msg, "Failed to import %s", fname.c_str());
		}
		else {
			module->exec(parent->context());
			strcpy(msg, "OK");
		}
	}
	catch (const char* err) {
		strcpy(msg, err);
	}
	catch (const std::string &err) {
		strcpy(msg, err.c_str());
	}
	catch (...) {
		strcpy(msg, SYSTEM_EXCEPTION);
	}
	return zs_create_string(ctx, msg);
}

static bool load_dll(const std::string& fname, const char *path)
{
	std::string tmp = fname;
	if (path) tmp = path + fname;
	HINSTANCE hinstLib = LoadLibrary(tmp.c_str()); 
	if (hinstLib) {
		// DLL needs to be kept alive
		zsModuleExpr::g_dlls.push(hinstLib, fname);
		return true;
	}
	return false;
}

static zoObject* sys_load(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	zsAutoLock lock(zsModuleExpr::g_mutex, zsModuleExpr::g_mutex_flag);
	for (int i = 0; i < nargs; i++) {
		const std::string& fname = get_string(ctx, args[i]);
		// try to load DLL
		if (!zsModuleExpr::g_dlls.isloaded(fname)) {
			if (!load_dll(fname, 0) && !load_dll(fname, "dll/")) {
				LoadError();
				ctx->expr->error("failed to load", fname.c_str());
			}
		}
	}
	return 0;
}

static zoObject* sys_exec(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	const std::string &prog = get_string(ctx, args[0]);
	return zs_create_integer(ctx, system(prog.c_str()));
}

static zoObject* sys_now(zsContext *ctx, int nargs, zoObject **args)
{
	time_t t = time(0);
	tm *st = localtime(&t);
	if (nargs > 0 && get_integer(ctx, args[0]) != 0) st = gmtime(&t);
	st->tm_year += 1900;
	st->tm_mon += 1;
	zoArray *arr = zs_create_array(ctx, 15);
	arr->set("year", zs_create_integer(0, st->tm_year));
	arr->set("0", zs_create_integer(0, st->tm_year));
	arr->set("month", zs_create_integer(0, st->tm_mon));
	arr->set("1", zs_create_integer(0, st->tm_mon));
	arr->set("day", zs_create_integer(0, st->tm_mday));
	arr->set("2", zs_create_integer(0, st->tm_mday));
	arr->set("hour", zs_create_integer(0, st->tm_hour));
	arr->set("3", zs_create_integer(0, st->tm_hour));
	arr->set("minute", zs_create_integer(0, st->tm_min));
	arr->set("4", zs_create_integer(0, st->tm_min));
	arr->set("second", zs_create_integer(0, st->tm_sec));
	arr->set("5", zs_create_integer(0, st->tm_sec));
	arr->set("wday", zs_create_integer(0, st->tm_wday));
	arr->set("yday", zs_create_integer(0, st->tm_yday));
	arr->set("time", zs_create_integer(0, t));
	real_t jul = cal2jul(st->tm_year, st->tm_mon, st->tm_mday) +
				 jultime(st->tm_hour, st->tm_min, + st->tm_sec);
	arr->set("julian", zs_create_real(0, jul));
	return arr;
}

static zoObject* sys_sleep(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	int ms = get_integer(ctx, args[0]);
	if (ms > 0) Sleep(ms);				// OS dependend
	return 0;
}

static zoObject* sys_assert(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	for (int i = 0; i < nargs; i++) {
		if (args[i]->type() != ZO_INTEGER) ctx->expr->error("boolean parameter expected");
		if (((zoNumber*)args[i])->getInteger() == 0) ctx->expr->error("assertion failed");
	}
	return 0;
}

static zoObject* sys_getenv(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	const char *name = get_string(ctx, args[0]).c_str();
#ifdef _WIN32
	char buf[1024];
	DWORD n = GetEnvironmentVariable(name, buf, 1024);
	if (n > 0 && n < 1024) {
		return zs_create_string(ctx, buf);
	}
#else
	return zs_create_string(ctx, getenv(name));
#endif
	return 0;
}

static zoObject* sys_find(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 2) input_error(ctx);
	if (args[0]->type() == ZO_STRING) {
		const std::string &s = ((zoString*)args[0])->get();
		int loc = 0;
		if (nargs > 2) loc = get_integer(ctx, args[2]);
		if (loc < 0) {
			loc = s.find_last_of(get_string(ctx, args[1]));
		}
		else {
			loc = s.find(get_string(ctx, args[1]), loc);
		}
		if (loc != std::string::npos) return zs_create_integer(ctx, loc);
		return zs_create_integer(ctx, -1);
	}
	if (args[0]->type() != ZO_ARRAY) ctx->expr->error("the first parameter must be string or array");
	zoArray *arr = (zoArray*)args[0];
	if (args[1]->type() == ZO_STRING) {
		return arr->get(ctx, ((zoString*)args[1])->get());
	}
	else if (args[1]->type() == ZO_INTEGER) {
		return arr->get(ctx, ((zoNumber*)args[1])->getInteger());
	}
	ctx->expr->error("string/integer key expected");
	return 0;
}

static zoObject* sys_size(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() == ZO_STRING) return zs_create_integer(ctx, get_string(ctx, args[0]).length());
	if (args[0]->type() == ZO_ARRAY)  return zs_create_integer(ctx, ((zoArray*)args[0])->size());
	input_error(ctx);
	return 0;
}

static zoObject* sys_escape(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() == ZO_ARRAY) {
		zoArray *arr = get_array(ctx, args[0]);
		std::vector<std::string> keys;
		arr->get(keys);
		std::string ret;
		for (int i = 0; i < keys.size(); i++) {
			if (i > 0) ret += "&";
			int n = keys[i].size()*3 + 1;
			zsBuffer buf(n);
			ret += http_escape(keys[i].c_str(), buf.u.pchar, n-1);
			ret += "=";
			char b32[32];
			const char *value = zs_object_str(arr->get(ctx, keys[i]), b32);
			n = strlen(value)*3 + 1;
			buf.resize(n);
			ret += http_escape(value, buf.u.pchar, n-1);
		}
		return zs_create_string(ctx, ret);
	}
	zoString *s = (zoString*)args[0];
	int n = s->size()*3 + 1;
	zsBuffer buf(n);
	return zs_create_string(ctx, http_escape(s->get().c_str(), buf.u.pchar, n-1));
}

/////////////////////////// IO //////////////////////////////////////

static zoObject* sys_input(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
    char buf[1024];
    fputs(get_string(ctx, args[0]).c_str(), stdout);
    if (fgets(buf, sizeof(buf)-2, stdin)) {
		return zs_create_string(ctx, buf); 
	}
	return 0;
}

static zoObject* sys_print(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	char buf[64];
	for (int i = 0; i < nargs; i++) printf("%s", zs_object_str(args[i], buf));
	fflush(stdout);
	return 0;
}

using namespace std;

static void print_array(zoArray *arr)
{
	if (arr->size() == 0) {
		printf("[]");
		return;
	}
	char buf[64];
	std::vector<std::string> keys;
	arr->get(keys);
	std::sort(keys.begin(), keys.end());
	printf("[");
	zsContext ctx(0);
	for (int k = 0; k < keys.size(); k++) {
		if (k > 0) printf(", ");
		zoObject *o = arr->get(&ctx, keys[k]);
		if (o->type() == ZO_ARRAY) {
			printf("%s=", keys[k].c_str());
			print_array((zoArray*)o);
		}
		else {
			printf("%s=%s", keys[k].c_str(), zs_object_str(o, buf));
		}
	}
	printf("]");
}

static zoObject* sys_csv(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) return 0;
	char buf[64];
	for (int i = 0; i < nargs; i++) {
		if (i > 0) printf(", ");
		if (args[i]->type() == ZO_ARRAY) {
			print_array((zoArray*)args[i]);
		}
		else {
			printf("%s", zs_object_str(args[i], buf));
		}
	}
	printf("\n");
	fflush(stdout);
	return 0;
}

static void close_file(void *ptr)
{
	FILE *f = (FILE*)ptr;
	if (f != stdin && f != stdout && f != stderr) fclose(f);
}

static zoObject* sys_open(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 2) input_error(ctx);
	const std::string &s = get_string(ctx, args[0]);
	FILE *f;
	if (s == "stdin") {
		f = stdin;
	}
	else if (s == "stdout") {
		f = stdout;
	}
	else if (s == "stderr") {
		f = stderr;
	}
	else {
		f = fopen(s.c_str(), get_string(ctx, args[1]).c_str());
		if (!f) ctx->expr->error("failed to open", s.c_str());
	}
	return zs_create_user(ctx, f, 0, close_file, FILE_TYPE);
}

static zoObject* file_close(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	FILE *fp = (FILE*)get_user(ctx, args[0], FILE_TYPE);
	fflush(fp);
	fclose(fp);
	return 0;
}

static zoObject* file_flush(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	fflush((FILE*)get_user(ctx, args[0], FILE_TYPE));
	return 0;
}

static zoObject* file_eof(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (feof((FILE*)get_user(ctx, args[0], FILE_TYPE))) return zs_create_integer(ctx, 1);
	return zs_create_integer(ctx, 0);
}

static zoObject* file_read(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	FILE *f = (FILE*)get_user(ctx, args[0], FILE_TYPE);
	if (nargs == 1) {
		zsBuffer buf(8092);
		if (!fgets(buf.u.pchar, 8090, f)) return zs_create_string(ctx, "");
		return zs_create_string(ctx, buf.u.pchar);
	}
	if (args[1]->type() == ZO_STRING) {
		const std::string& type = get_string(ctx, args[1]);
		if (type == "float") {
			float v;
			fread(&v, 1, sizeof(float), f);
			return zs_create_real(ctx, v);
		}
		if (type == "double") {
			double v;
			fread(&v, 1, sizeof(double), f);
			return zs_create_real(ctx, v);
		}
		if (type == "int") {
			int v;
			fread(&v, 1, sizeof(int), f);
			return zs_create_integer(ctx, v);
		}
		if (type == "uint") {
			unsigned int v;
			fread(&v, 1, sizeof(unsigned int), f);
			return zs_create_integer(ctx, v);
		}
		if (type == "int64") {
			int64 v;
			fread(&v, 1, sizeof(int64), f);
			return zs_create_integer(ctx, v);
		}
		if (type == "short") {
			short v;
			fread(&v, 1, sizeof(short), f);
			return zs_create_integer(ctx, v);
		}
		if (type == "ushort") {
			unsigned short v;
			fread(&v, 1, sizeof(unsigned short), f);
			return zs_create_integer(ctx, v);
		}
		if (type == "char") {
			char v;
			fread(&v, 1, sizeof(char), f);
			return zs_create_integer(ctx, v);
		}
		if (type == "uchar") {
			unsigned char v;
			fread(&v, 1, sizeof(unsigned char), f);
			return zs_create_integer(ctx, v);
		}
		if (type == "string") {
			int len = 0;
			if (nargs > 2) len = get_integer(ctx, args[2]);
			if (len <= 0) {
				int pos0 = ftell(f);
				fseek(f, 0, SEEK_END);
				int pos1 = ftell(f);
				fseek(f, pos0, SEEK_SET);
				len = pos1 - pos0;
			}
			zsBuffer buf(len+1);
			fread(buf.u.pchar, 1, len, f);
			buf.u.pchar[len] = 0;
			return zs_create_string(ctx, buf.u.pchar);
		}
		input_error(ctx);
	}
	void *ptr = get_ptr(ctx, args[1]);
	int n = get_integer(ctx, args[2]);
	if (n <= 0) input_error(ctx);
	int m = fread(ptr, 1, n, f);
	return zs_create_integer(ctx, m);
}

static zoObject* file_write(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 2) input_error(ctx);
	FILE *f = (FILE*)get_user(ctx, args[0], FILE_TYPE);
	if (nargs == 2) {
		const std::string str = get_string(ctx, args[1]);
		fprintf(f,"%s",str.c_str());
		fflush(f);
		return 0;
	}
	if (args[1]->type() >= ZO_USER) {
		void *buf = get_ptr(ctx, args[1]);
		int n = get_integer(ctx, args[2]);
		if (n <= 0) input_error(ctx);
		fwrite(buf, 1, n, f);
		fflush(f);
		return 0;
	}
	const std::string &type = get_string(ctx, args[2]);
	if (type == "float") {
		float v;
		if (args[1]->type() == ZO_REAL)
			v = get_real(ctx, args[1]);
		else if (args[1]->type() == ZO_INTEGER)
			v = get_integer(ctx, args[1]);
		else
			input_error(ctx);
		fwrite(&v, 1, sizeof(float), f);
		fflush(f);
		return 0;
	}
	if (type == "double") {
		double v;
		if (args[1]->type() == ZO_REAL)
			v = get_real(ctx, args[1]);
		else if (args[1]->type() == ZO_INTEGER)
			v = get_integer(ctx, args[1]);
		else
			input_error(ctx);
		fwrite(&v, 1, sizeof(double), f);
		fflush(f);
		return 0;
	}
	if (type == "int") {
		int v;
		if (args[1]->type() == ZO_REAL)
			v = get_real(ctx, args[1]);
		else if (args[1]->type() == ZO_INTEGER)
			v = get_integer(ctx, args[1]);
		else
			input_error(ctx);
		fwrite(&v, 1, sizeof(int), f);
		fflush(f);
		return 0;
	}
	if (type == "uint") {
		unsigned int v;
		if (args[1]->type() == ZO_REAL)
			v = get_real(ctx, args[1]);
		else if (args[1]->type() == ZO_INTEGER)
			v = get_integer(ctx, args[1]);
		else
			input_error(ctx);
		fwrite(&v, 1, sizeof(unsigned int), f);
		fflush(f);
		return 0;
	}
	if (type == "int64") {
		int64 v;
		if (args[1]->type() == ZO_REAL)
			v = get_real(ctx, args[1]);
		else if (args[1]->type() == ZO_INTEGER)
			v = get_integer(ctx, args[1]);
		else
			input_error(ctx);
		fwrite(&v, 1, sizeof(int64), f);
		fflush(f);
		return 0;
	}
	if (type == "short") {
		short v;
		if (args[1]->type() == ZO_REAL)
			v = get_real(ctx, args[1]);
		else if (args[1]->type() == ZO_INTEGER)
			v = get_integer(ctx, args[1]);
		else
			input_error(ctx);
		fwrite(&v, 1, sizeof(short), f);
		fflush(f);
		return 0;
	}
	if (type == "ushort") {
		unsigned short v;
		if (args[1]->type() == ZO_REAL)
			v = get_real(ctx, args[1]);
		else if (args[1]->type() == ZO_INTEGER)
			v = get_integer(ctx, args[1]);
		else
			input_error(ctx);
		fwrite(&v, 1, sizeof(unsigned short), f);
		fflush(f);
		return 0;
	}
	if (type == "char") {
		char v;
		if (args[1]->type() == ZO_REAL)
			v = get_real(ctx, args[1]);
		else if (args[1]->type() == ZO_INTEGER)
			v = get_integer(ctx, args[1]);
		else
			input_error(ctx);
		fwrite(&v, 1, sizeof(char), f);
		fflush(f);
		return 0;
	}
	if (type == "uchar") {
		unsigned char v;
		if (args[1]->type() == ZO_REAL)
			v = get_real(ctx, args[1]);
		else if (args[1]->type() == ZO_INTEGER)
			v = get_integer(ctx, args[1]);
		else
			input_error(ctx);
		fwrite(&v, 1, sizeof(unsigned char), f);
		fflush(f);
		return 0;
	}
	input_error(ctx);
	return 0;
}

static zoObject* file_seek(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 2) input_error(ctx);
	FILE *f = (FILE*)get_user(ctx, args[0], FILE_TYPE);
	if (args[1]->type() == ZO_STRING) {
		const std::string& flag = get_string(ctx, args[1]);
		long limit = 0;
		if (nargs > 2) limit = get_integer(ctx, args[2]);
		if (limit < 0) input_error(ctx);
		unsigned char b[1];
		int k = 0;
		fread(b, 1, 1, f);
		while (!feof(f)) {
			if (b[0] == (unsigned char)flag.at(k)) {
				k++;
				if (k == flag.size()) {
					return zs_create_integer(ctx, ftell(f));
				}
			}
			else {
				k = 0;
			}
			fread(b, 1, 1, f);
			if (limit > 0) {
				limit--;
				if (limit == 0) break;
			}
		}
		return zs_create_integer(ctx, 0);
	}
	if (nargs < 3) input_error(ctx);
	int offset = get_integer(ctx, args[1]);
	const std::string flag = get_string(ctx, args[2]);
	if (flag == "set") {
		fseek(f, offset, SEEK_SET);
	}
	else if (flag == "cur") {
		fseek(f, offset, SEEK_CUR);
	}
	else if (flag == "end") {
		fseek(f, offset, SEEK_END);
	}
	else {
		input_error(ctx);
	}
	return 0;
}

static zoObject* file_tell(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	return zs_create_integer(ctx, ftell((FILE*)get_user(ctx, args[0], FILE_TYPE)));
}

static zoObject* sys_diff(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 2) input_error(ctx);
	const int N = 1024;
	char b1[N], b2[N];
	const std::string &s1 = get_string(ctx, args[0]);
	FILE *f1 = fopen(s1.c_str(), "rb");
	if (!f1) ctx->expr->error("failed to open", s1.c_str());
	const std::string &s2 = get_string(ctx, args[1]);
	FILE *f2 = fopen(s2.c_str(), "rb");
	if (!f2) ctx->expr->error("failed to open", s2.c_str());
	while (!feof(f1) && !feof(f2)) {
		int n1 = fread(b1, 1, N, f1);
		int n2 = fread(b2, 1, N, f2);
		int i, n = n1 < n2 ? n1 : n2;
		for (i = 0; i < n; i++) {
			if (b1[i] != b2[i]) {
				long m = ftell(f1) - n1 + i;
				fclose(f1);
				fclose(f2);
				return zs_create_integer(ctx, m);
			}
		}
		if (n1 < n2) return zs_create_integer(ctx, ftell(f2));
		if (n2 < n1) return zs_create_integer(ctx, ftell(f1));
	}
	fclose(f1);
	fclose(f2);
	return zs_create_integer(ctx, -1);
}

static zoObject* sys_tmpnam(zsContext *ctx, int nargs, zoObject **args)
{
	char buf[L_tmpnam];
	tmpnam(buf);
	if (buf[0] == '/' || buf[0] == '\\') buf[0] = 'x';
	return zs_create_string(ctx, buf);
}

static zoObject* sys_ftime(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	const std::string &s = get_string(ctx, args[0]);
	if (nargs > 6) {
#ifdef _WIN32
		HANDLE h = CreateFile((LPCTSTR)s.c_str(),
								FILE_WRITE_ATTRIBUTES,
								FILE_SHARE_READ,
								NULL,
								OPEN_EXISTING,
								FILE_ATTRIBUTE_NORMAL,
								NULL);
		FILETIME ft;
		SYSTEMTIME st;
		st.wYear = get_integer(ctx, args[1]);
		st.wMonth = get_integer(ctx, args[2]);
		st.wDay = get_integer(ctx, args[3]);
		st.wHour = get_integer(ctx, args[4]);
		st.wMinute = get_integer(ctx, args[5]);
		st.wSecond = get_integer(ctx, args[6]);
		st.wMilliseconds = 0;

		SystemTimeToFileTime(&st, &ft);
		SetFileTime(h, (LPFILETIME)NULL, (LPFILETIME)NULL, &ft);
		CloseHandle(h);
#else
		tm st;
		memset(&st,0,sizeof(st));
		st.tm_year = get_integer(ctx, args[1]) - 1900;
		st.tm_mon = get_integer(ctx, args[2]) - 1;
		st.tm_mday = get_integer(ctx, args[3]);
		st.tm_hour = get_integer(ctx, args[4]);
		st.tm_min = get_integer(ctx, args[5]);
		st.tm_sec = get_integer(ctx, args[6]);
		
		utimbuf new_times;
		new_times.actime = mktime(&st);
		new_times.modtime = new_times.actime;
		utime(s.c_str(), &new_times);
#endif
	}
	time_t t = get_file_time(s.c_str());
	tm *st = localtime(&t);
	if (nargs > 1 && get_integer(ctx, args[1]) != 0) st = gmtime(&t);
	st->tm_year += 1900;
	st->tm_mon += 1;
	zoArray *arr = zs_create_array(ctx, 15);
	arr->set("year", zs_create_integer(0, st->tm_year));
	arr->set("0", zs_create_integer(0, st->tm_year));
	arr->set("month", zs_create_integer(0, st->tm_mon));
	arr->set("1", zs_create_integer(0, st->tm_mon));
	arr->set("day", zs_create_integer(0, st->tm_mday));
	arr->set("2", zs_create_integer(0, st->tm_mday));
	arr->set("hour", zs_create_integer(0, st->tm_hour));
	arr->set("3", zs_create_integer(0, st->tm_hour));
	arr->set("minute", zs_create_integer(0, st->tm_min));
	arr->set("4", zs_create_integer(0, st->tm_min));
	arr->set("second", zs_create_integer(0, st->tm_sec));
	arr->set("5", zs_create_integer(0, st->tm_sec));
	real_t jul = cal2jul(st->tm_year, st->tm_mon, st->tm_mday) +
				 jultime(st->tm_hour, st->tm_min, + st->tm_sec);
	arr->set("julian", zs_create_real(0, jul));
	return arr;
}

static zoObject* sys_revb(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 3) input_error(ctx);
	unsigned char *p   = (unsigned char*)get_ptr(ctx, args[0]);
	int n     = get_integer(ctx, args[1]);
	int esize = get_integer(ctx, args[2]);
	if (n < 2 || esize < 2) input_error(ctx);
	int m = esize / 2;
	for (int i = 0; i < n; i++, p += esize) {
		for (int j = 0; j < m; j++) {
			unsigned char tmp = *(p + j);
			*(p + j) = *(p + esize - j - 1);
			*(p + esize - j - 1) = tmp;
		}
	}
	return 0;
}

static zoObject* sys_format(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 2) input_error(ctx);
	const std::string & s = get_string(ctx, args[0]);
	std::string fmt, out;
	int loc1 = s.find("%", 0);
	if (loc1 > 0) out += s.substr(0, loc1);
	int loc2 = s.find("%", loc1+1);
	char buf[1024];
	for (int i = 1; i < nargs; i++) {
		if (loc1 == std::string::npos) input_error(ctx);
		if (loc2 != std::string::npos)
			fmt = s.substr(loc1, loc2-loc1);
		else
			fmt = s.substr(loc1, s.size()-loc1);
		if (args[i]->type() == ZO_INTEGER)
			sprintf(buf, fmt.c_str(), get_integer(ctx, args[i]));
		else if (args[i]->type() == ZO_REAL)
			sprintf(buf, fmt.c_str(), get_real(ctx, args[i]));
		else if (args[i]->type() == ZO_STRING)
			sprintf(buf, fmt.c_str(), get_string(ctx, args[i]).c_str());
		else
			input_error(ctx);
		out += buf;
		loc1 = loc2;
		loc2 = s.find("%", loc1+1);
	}
	return zs_create_string(ctx, out);
}

static zoObject* sys_scandir(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 2) input_error(ctx);
	zsFuncExpr *func = ctx->expr->func()->getFunc(get_string(ctx, args[0]), ctx->expr, true);
	std::string s(get_string(ctx, args[1]));
	zoObject *o[4];
    char buf[64];
	DIRST dir;
	Bool rc = open_dir(&dir, s.c_str());
	while (rc) {
		if (strcmp(dir.dir_name, ".") != 0 && strcmp(dir.dir_name, "..") != 0) {
            struct tm *t = localtime(&dir.file_time);
            sprintf(buf, "%d-%02d-%02d %02d:%02d:%02d", t->tm_year+1900,t->tm_mon+1,t->tm_mday,t->tm_hour,t->tm_min,t->tm_sec);
			o[0] = zs_create_string(ctx, dir.dir_name);
			o[1] = zs_create_string(ctx, dir.file_name);
	    	o[2] = zs_create_string(ctx, buf);
			o[3] = zs_create_integer(ctx, dir.file_size);
			func->call(ctx, 4, o);
			ctx->clear();
		}
		rc = read_dir(&dir);
	}
	close_dir(&dir);
	return 0;
}

static zoObject* sys_mkdir(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	const std::string & s = get_string(ctx, args[0]);
#ifdef _WIN32
	return zs_create_integer(ctx, CreateDirectory(s.c_str(), NULL));
#else
	return zs_create_integer(ctx, mkdir(s.c_str(),0775)!=0);
#endif
}

static zoObject* sys_curdir(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs > 0) {
		const std::string & s = get_string(ctx, args[0]);
#ifdef _WIN32
		return zs_create_integer(ctx, SetCurrentDirectory(s.c_str()));
#else
		return zs_create_integer(ctx, chdir(s.c_str()));
#endif
	}
	char curdir[1024];
#ifdef _WIN32
	GetCurrentDirectory(1020,curdir);
    strconvch(curdir, '\\', '/');
#else
	getcwd(curdir,1020);
#endif
	return zs_create_string(ctx, curdir);
}

static zoObject* sys_isdir(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() != ZO_STRING) return zs_create_integer(ctx, 0);
	const std::string & s = get_string(ctx, args[0]);
	return zs_create_integer(ctx, file_is_directory(s.c_str()) != 0);
}

static zoObject* sys_isfile(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() != ZO_STRING) return zs_create_integer(ctx, 0);
	const std::string & s = get_string(ctx, args[0]);
	FILE *fp = fopen(s.c_str(), "r");
	int flag = 0;
	if (fp) {
		flag = 1;
		fclose(fp);
	}
	return zs_create_integer(ctx, flag);
}

static zoObject* sys_base64(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (nargs == 1) {
		const std::string & s = get_string(ctx, args[0]);
		byte *ptr = (byte*)malloc(s.size()+1);
		ptr[s.size()] = 0;
		size_t n = decode_base64((byte*)s.c_str(), ptr, s.size());
		zoArray *arr = zs_create_array(ctx, 1);
		arr->set("0", zs_create_user(0, ptr, 0, free, 0));
		arr->set("1", zs_create_integer(0, n));
		return arr;
	}
	void* ptr = get_ptr(ctx, args[0]);
	long n = get_integer(ctx, args[1]);
	if (n <= 0) input_error(ctx);
	zsBuffer out(n*3);
	encode_base64((byte*)ptr, (byte*)out.u.puchar, n);
	return zs_create_string(ctx, out.u.pchar);
}

/////////////////////////// string //////////////////////////////////

static zoObject* str_substr(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 3) input_error(ctx);
	const std::string& s = get_string(ctx, args[0]);
	int k = get_integer(ctx, args[1]);
	int n = get_integer(ctx, args[2]);
	if (k < 0 || k+n > s.size()) input_error(ctx);
	return zs_create_string(ctx, s.substr(k, n));
}

static zoObject* str_left(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 2) input_error(ctx);
	const std::string& s = get_string(ctx, args[0]);
	int n = get_integer(ctx, args[1]);
	if (n <= 0 || n > s.size()) input_error(ctx);
	return zs_create_string(ctx, s.substr(0, n));
}

static zoObject* str_right(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 2) input_error(ctx);
	const std::string& s = get_string(ctx, args[0]);
	int k = get_integer(ctx, args[1]);
	if (k < 0 || k >= s.size()) input_error(ctx);
	return zs_create_string(ctx, s.substr(k, s.size()-k));
}

static zoObject* str_tolower(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	std::string s = get_string(ctx, args[0]);
	for (int i = 0; i < s.length(); i++) s.at(i) = tolower(s.at(i));
	return zs_create_string(ctx, s);
}

static zoObject* str_toupper(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	std::string s = get_string(ctx, args[0]);
	for (int i = 0; i < s.length(); i++) s.at(i) = toupper(s.at(i));
	return zs_create_string(ctx, s);
}

static std::string trim_string(const std::string &s)
{
	if (s.size() == 0) return s;
	int i = 0;
	while (i < s.size() && isspace(s.at(i))) i++;
	int j = s.size() - 1;
	while (j > 0 && isspace(s.at(j))) j--;
	j++;
	if (j > i) return s.substr(i, j-i);
	return "";
}

static zoObject* str_trim(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	const std::string& s = get_string(ctx, args[0]);
	return zs_create_string(ctx, trim_string(s));
}

static zoObject* str_numstr(zsContext *ctx, int nargs, zoObject **args)
{
	extern char *scan_number(char *p);
	if (nargs < 1) input_error(ctx);
	const char *cur = get_string(ctx, args[0]).c_str();
	if (atof(cur)!=0) return zs_create_integer(ctx, 1);
	while (cur[0]) {
		if (cur[0]!='0' && cur[0]!='-' && cur[0]!='.' && cur[0]!='e' && cur[0]!='E') return zs_create_integer(ctx, 0);
		cur++;
	}
	return zs_create_integer(ctx, 1);
}

static zoObject* str_replace(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 3) input_error(ctx);
	const std::string& s = get_string(ctx, args[0]);
	const std::string& o = get_string(ctx, args[1]);
	const std::string& r = get_string(ctx, args[2]);
	size_t loc = s.find(o, 0);
	if (loc == std::string::npos) return args[0];
	std::string str = s.substr(0, loc) + r;
	loc += o.size();
	size_t loc2 = s.find(o, loc);
	while (loc2 != std::string::npos) {
		str += s.substr(loc, loc2-loc) + r;
		loc = loc2 + o.size();
		loc2 = s.find(o, loc);
	}
	if (loc < s.size()) {
		str += s.substr(loc, s.size()-loc);
	}
	return zs_create_string(ctx, str);
}

static zoObject* str_mbs2utf(zsContext *ctx, int nargs, zoObject **args)
{
#ifdef _WIN32
	if (nargs < 1) input_error(ctx);
	const char* s = get_string(ctx, args[0]).c_str();
	int m = strlen(s) + 2;
	zsBuffer wc(m*sizeof(unsigned short));
	m = MultiByteToWideChar(CP_ACP, 0, s, -1, (LPWSTR)wc.u.ptr, m);
	if (m <= 0) ctx->expr->error("MultiByteToWideChar failed");
	m = WideCharToMultiByte(CP_UTF8, 0, (LPWSTR)wc.u.ptr, -1, 0, 0, 0, 0) + 2;
	zsBuffer utf(m);
	m = WideCharToMultiByte(CP_UTF8, 0, (LPWSTR)wc.u.ptr, -1, utf.u.pchar, m, 0, 0);
	if (m <= 0) ctx->expr->error("WideCharToMultiByte failed");
#else
	if (nargs < 2) input_error(ctx);
	char* s = (char*)get_string(ctx, args[0]).c_str();
	const char* charset = get_string(ctx, args[1]).c_str();
	iconv_t cd = iconv_open("UTF-8", charset);
	if (cd == (iconv_t)(-1)) ctx->expr->error("iconv_open failed");
	size_t insize = strlen(s);
	size_t outsize = 3*insize + 1;
	zsBuffer utf(outsize);
	iconv(cd, &s, &insize, &utf.u.pchar, &outsize);
	iconv_close(cd);
#endif
	return zs_create_string(ctx, utf.u.pchar);
}

static zoObject* str_utf2mbs(zsContext *ctx, int nargs, zoObject **args)
{
#ifdef _WIN32
	if (nargs < 1) input_error(ctx);
	const char* utf = get_string(ctx, args[0]).c_str();
	int m = MultiByteToWideChar(CP_UTF8, 0, utf, -1, 0, 0) + 2;
	zsBuffer wc(m*sizeof(unsigned short));
	m = MultiByteToWideChar(CP_UTF8, 0, utf, -1, (LPWSTR)wc.u.ptr, m);
	if (m <= 0) ctx->expr->error("MultiByteToWideChar failed");
	m = WideCharToMultiByte(CP_ACP, 0, (LPWSTR)wc.u.ptr, -1, 0, 0, 0, 0) + 2;
	zsBuffer s(m);
	m = WideCharToMultiByte(CP_ACP, 0, (LPWSTR)wc.u.ptr, -1, s.u.pchar, m, 0, 0);
	if (m <= 0) ctx->expr->error("WideCharToMultiByte");
#else
	if (nargs < 2) input_error(ctx);
	char* utf = (char*)get_string(ctx, args[0]).c_str();
	const char* charset = get_string(ctx, args[1]).c_str();
	iconv_t cd = iconv_open(charset, "UTF-8");
	if (cd == (iconv_t)(-1)) ctx->expr->error("iconv_open failed");
	size_t insize = strlen(utf);
	size_t outsize = 2*insize + 1;
	zsBuffer s(outsize);
	iconv(cd, &utf, &insize, &s.u.pchar, &outsize);
	iconv_close(cd);
#endif
	return zs_create_string(ctx, s.u.pchar);
}

void tokenize_blank(zsContext *ctx, zoArray *arr, const std::string &s)
{
	int loc = 0, count = 0;
	while (loc < s.size()) {
		while (loc < s.size() && isspace(s.at(loc))) loc++;
		if (loc >= s.size()) break;
		int loc2 = loc + 1;
		while (loc2 < s.size() && !isspace(s.at(loc2))) loc2++;
		arr->set(count++, zs_create_string(0, s.substr(loc, loc2-loc)));
		loc = loc2 + 1;
	}
}

void tokenize_string(zsContext *ctx, zoArray *arr, const std::string &s, const std::string &d, bool trim)
{
	size_t loc = 0, count = 0, loc2 = s.find(d, 0);
	while (loc2 != std::string::npos) {
		std::string s2 = s.substr(loc, loc2-loc);
		if (trim) s2 = trim_string(s2);
		arr->set(count++, zs_create_string(0, s2));
		loc = loc2 + d.size();
		loc2 = s.find(d, loc);
	}
	if (s.size() >= loc) {
		std::string s2 = s.substr(loc, s.size()-loc);
		if (trim) s2 = trim_string(s2);
		arr->set(count, zs_create_string(0, s2));
	}
}

static zoObject* str_tokenize(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	const std::string & s = get_string(ctx, args[0]);
	zoArray *arr = zs_create_array(ctx, 31);
	if (s=="") return arr;
	if (nargs == 1) {
		tokenize_blank(ctx,arr,s);
		return arr;
	}
	const std::string & d = get_string(ctx, args[1]);
	bool trim = true;
	if (nargs > 2) trim = get_integer(ctx, args[2]) != 0;
	tokenize_string(ctx,arr,s,d,trim);
	return arr;
}

static zoObject* str_regex(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 2) input_error(ctx);
	const std::string & s = get_string(ctx, args[0]);
	const std::string & r = get_string(ctx, args[1]);
	int p = 0;
	if (nargs > 2) p = get_integer(ctx, args[2]);
	zoArray *arr = zs_create_array(ctx, 2);

    CRegexpT <char> regexp(r.c_str());
    MatchResult result = regexp.Match(s.c_str());
	arr->set(0, zs_create_integer(ctx, result.GetStart()));
	arr->set(0, zs_create_integer(ctx, result.GetEnd()));

    return arr;
}

//////////////////////////// type ///////////////////////////////////

static zoObject* sys_integer(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
/*
#ifdef USE_INT64
	if (args[0]->type() == ZO_STRING) return zs_create_integer(ctx, _atoi64(get_string(ctx, args[0]).c_str()));
#else
	if (args[0]->type() == ZO_STRING) return zs_create_integer(ctx, atoi(get_string(ctx, args[0]).c_str()));
#endif
*/
	if (args[0]->type() == ZO_STRING) return zs_create_integer(ctx, atof(get_string(ctx, args[0]).c_str()));
	if (args[0]->type() == ZO_REAL) return zs_create_integer(ctx, ((zoNumber*)args[0])->getReal());
	if (args[0]->type() == ZO_INTEGER) return args[0];
	input_error(ctx);
	return 0;
}

static zoObject* sys_real(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() == ZO_STRING) return zs_create_real(ctx, atof(get_string(ctx, args[0]).c_str()));
	if (args[0]->type() == ZO_INTEGER) return zs_create_real(ctx, ((zoNumber*)args[0])->getInteger());
	if (args[0]->type() == ZO_REAL) return args[0];
	input_error(ctx);
	return 0;
}

static zoObject* sys_string(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() == ZO_USER) return zs_create_string(ctx, (char*)((zoUser*)args[0])->get());
	char s[128];
	return zs_create_string(ctx, zs_object_str(args[0], s));
	return 0;
}

static zoObject* sys_isnull(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	return zs_create_integer(ctx, args[0]->type() == ZO_NULL);
}

static zoObject* sys_isinteger(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	return zs_create_integer(ctx, args[0]->type() == ZO_INTEGER);
}

static zoObject* sys_isreal(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	return zs_create_integer(ctx, args[0]->type() == ZO_REAL);
}

static zoObject* sys_isnumber(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	return zs_create_integer(ctx, args[0]->type() == ZO_INTEGER || args[0]->type() == ZO_REAL);
}

static zoObject* sys_isstring(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	return zs_create_integer(ctx, args[0]->type() == ZO_STRING);
}

static zoObject* sys_isarray(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	return zs_create_integer(ctx, args[0]->type() == ZO_ARRAY);
}

static zoObject* sys_isclass(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	return zs_create_integer(ctx, args[0]->type() == ZO_CLASS);
}

static zoObject* sys_isuser(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	return zs_create_integer(ctx, args[0]->type() >= ZO_USER);
}

static zoObject* sys_cal2jul(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs == 1 && args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "cal2jul");
	if (nargs < 3) input_error(ctx);
	real_t d = cal2jul(get_number(ctx, args[0]), get_number(ctx, args[1]), get_number(ctx, args[2]));
	if (nargs > 3) d += get_number(ctx, args[3]) / 24.0;
	if (nargs > 4) d += get_number(ctx, args[4]) / 1440.0;
	if (nargs > 5) d += get_number(ctx, args[5]) / 86400.0;
	return zs_create_real(ctx, d);
}

static zoObject* sys_jul2cal(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "jul2cal");
	int yy, mm, dd, hh, mn, sc;
	real_t jul = get_number(ctx, args[0]);
	jul2cal(&yy, &mm, &dd, jul);
	real_t res = jul2tim(&hh, &mn, &sc, jul);
	zoArray *arr = zs_create_array(ctx, 15);
	arr->set("year", zs_create_integer(0, yy));
	arr->set("0", zs_create_integer(0, yy));
	arr->set("month", zs_create_integer(0, mm));
	arr->set("1", zs_create_integer(0, mm));
	arr->set("day", zs_create_integer(0, dd));
	arr->set("2", zs_create_integer(0, dd));
	arr->set("hour", zs_create_integer(0, hh));
	arr->set("3", zs_create_integer(0, hh));
	arr->set("minute", zs_create_integer(0, mn));
	arr->set("4", zs_create_integer(0, mn));
	arr->set("second", zs_create_integer(0, sc));
	arr->set("5", zs_create_integer(0, sc));
	arr->set("millisecond", zs_create_integer(0, res));
	arr->set("6", zs_create_integer(0, res));
	return arr;
}

////////////////////// array ////////////////////////////////////////

static zoObject* sys_array(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	return zs_create_array(ctx, get_integer(ctx, args[0]));
}

static zoObject* arr_keys(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	zoArray *arr = get_array(ctx, args[0]);
	std::vector<std::string> keys;
	arr->get(keys);
	if (nargs > 1 && get_integer(ctx, args[1]) != 0) {
		std::sort(keys.begin(), keys.end());
	}
	zoArray *ret = zs_create_array(ctx, keys.size());
	for (int i = 0; i < keys.size(); i++) ret->set(i, zs_create_string(0, keys[i]));
	return ret;
}

static zoObject* arr_clear(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	zoArray *arr = get_array(ctx, args[0]);
	arr->clear();
	return 0;
}

static zoObject* arr_remove(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 2) input_error(ctx);
	zoArray *arr = get_array(ctx, args[0]);
	for (int i = 1; i < nargs; i++) {
		if (args[i]->type() == ZO_INTEGER) {
			arr->remove(((zoNumber*)args[i])->getInteger());
		}
		else if (args[i]->type() == ZO_STRING) {
			arr->remove(((zoString*)args[i])->get());
		}
		else {
			input_error(ctx);
		}
	}
	return 0;
}

static zoObject* arr_foreach(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 2) input_error(ctx);
	zoArray *arr = get_array(ctx, args[0]);
	zsFuncExpr *f = ctx->expr->module()->getFunc(get_string(ctx, args[1]), ctx->expr, true);
	std::vector<std::string> keys;
	arr->get(keys);
	if (nargs > 2 && get_integer(ctx, args[2]) != 0) {
		std::sort(keys.begin(), keys.end());
	}
	for (int i = 0; i < keys.size(); i++) {
		zoObject *p[2] = { zs_create_string(ctx, keys[i]), arr->get(ctx, keys[i]) };
		f->call(ctx, 2, p);
//		ctx->clear();
	}
	return 0;
}

/////////////////////////// math ////////////////////////////////////

static zoObject* sys_cos(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "cos");
	return zs_apply_func1(ctx, cos, args[0]);
}

static zoObject* sys_acos(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "acos");
	return zs_apply_func1(ctx, acos, args[0]);
}

static zoObject* sys_cosh(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "cosh");
	return zs_apply_func1(ctx, cosh, args[0]);
}

static zoObject* sys_sin(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "sin");
	return zs_apply_func1(ctx, sin, args[0]);
}

static zoObject* sys_asin(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "asin");
	return zs_apply_func1(ctx, asin, args[0]);
}

static zoObject* sys_sinh(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "sinh");
	return zs_apply_func1(ctx, sinh, args[0]);
}

static zoObject* sys_tan(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "tan");
	return zs_apply_func1(ctx, tan, args[0]);
}

static zoObject* sys_atan(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "atan");
	return zs_apply_func1(ctx, atan, args[0]);
}

static zoObject* sys_tanh(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "tanh");
	return zs_apply_func1(ctx, tanh, args[0]);
}

static zoObject* sys_sqrt(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "sqrt");
	return zs_apply_func1(ctx, sqrt, args[0], 1);
}

static zoObject* sys_exp(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "exp");
	return zs_apply_func1(ctx, exp, args[0]);
}

static zoObject* sys_log(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "log");
	return zs_apply_func1(ctx, log, args[0], 2);
}

static zoObject* sys_log10(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "log10");
	return zs_apply_func1(ctx, log10, args[0], 2);
}

static zoObject* sys_ceil(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "ceil");
	return zs_apply_func1(ctx, ceil, args[0]);
}

static zoObject* sys_floor(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "floor");
	return zs_apply_func1(ctx, floor, args[0]);
}

static zoObject* sys_abs(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive(ctx, nargs, args, "abs");
	return zs_apply_func1(ctx, fabs, args[0]);
}

static zoObject* sys_atan2(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 2) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive (ctx, nargs, args, "atan2");
	if (args[1]->type() >= ZO_USER) return use_primitive2(ctx, nargs, args, "atan2");
	return zs_apply_func2(ctx, atan2, args[0], args[1]);
}

static zoObject* sys_pow(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 2) input_error(ctx);
	if (args[0]->type() >= ZO_USER) return use_primitive (ctx, nargs, args, "pow");
	if (args[1]->type() >= ZO_USER) return use_primitive2(ctx, nargs, args, "pow");
	return zs_apply_func2(ctx, pow, args[0], args[1]);
}

static zoObject* sys_pi(zsContext *ctx, int nargs, zoObject **args)
{
	return zs_create_real(ctx, 3.14159265358979323846);
}

/////////////////////////////////////////////////////////////////////

zsRegPrimitive::zsRegPrimitive() {
	zs_add_primitive("getarg",		0,			sys_getarg);
	zs_add_primitive("error",		0,			sys_error);
//	zs_add_primitive("func",		0,			sys_func);
//	zs_add_primitive("call",		0,			sys_call);
//	zs_add_primitive("parse",		0,			sys_parse);
//	zs_add_primitive("eval",		0,			sys_eval);
	zs_add_primitive("import",		0,			sys_import);
//	zs_add_primitive("trace",		0,			sys_trace);
	zs_add_primitive("load",		0,			sys_load);
	zs_add_primitive("exec",		0,			sys_exec);
	zs_add_primitive("now",			0,			sys_now);
	zs_add_primitive("sleep",		0,			sys_sleep);
	zs_add_primitive("assert",		0,			sys_assert);

	zs_add_primitive("getenv",		0,			sys_getenv);
	zs_add_primitive("input",		0,			sys_input);
	zs_add_primitive("print",		0,			sys_print);
	zs_add_primitive("csv",			0,			sys_csv);
	zs_add_primitive("open",		0,			sys_open);
	zs_add_primitive("malloc",		0,			sys_malloc);

	zs_add_primitive("find",		0,			sys_find);
	zs_add_primitive("find",		ZO_STRING,	sys_find);
	zs_add_primitive("find",		ZO_ARRAY,	sys_find);
	zs_add_primitive("size",		0,			sys_size);
	zs_add_primitive("size",		ZO_STRING,	sys_size);
	zs_add_primitive("size",		ZO_ARRAY,	sys_size);
		
	zs_add_primitive("flush",		FILE_TYPE,	file_flush);
	zs_add_primitive("flush",		0,			file_flush);
	zs_add_primitive("eof",			FILE_TYPE,	file_eof);
	zs_add_primitive("eof",			0,			file_eof);
	zs_add_primitive("read",		FILE_TYPE,	file_read);
	zs_add_primitive("read",		0,			file_read);
	zs_add_primitive("write",		FILE_TYPE,	file_write);
	zs_add_primitive("write",		0,			file_write);
	zs_add_primitive("seek",		FILE_TYPE,	file_seek);
	zs_add_primitive("seek",		0,			file_seek);
	zs_add_primitive("tell",		FILE_TYPE,	file_tell);
	zs_add_primitive("tell",		0,			file_tell);
	zs_add_primitive("close",		FILE_TYPE,	file_close);
	zs_add_primitive("close",		0,			file_close);

	zs_add_primitive("diff",		0,			sys_diff);
	zs_add_primitive("tmpnam",		0,			sys_tmpnam);
	zs_add_primitive("revb",		0,			sys_revb);
	zs_add_primitive("format",		0,			sys_format);
	zs_add_primitive("isfile",		0,			sys_isfile);

	zs_add_primitive("substr",		ZO_STRING,	str_substr);
	zs_add_primitive("substr",		0,			str_substr);
	zs_add_primitive("left",		ZO_STRING,	str_left);
	zs_add_primitive("left",		0,			str_left);
	zs_add_primitive("right",		ZO_STRING,	str_right);
	zs_add_primitive("right",		0,			str_right);
	zs_add_primitive("tolower",		ZO_STRING,	str_tolower);
	zs_add_primitive("tolower",		0,			str_tolower);
	zs_add_primitive("toupper",		ZO_STRING,	str_toupper);
	zs_add_primitive("toupper",		0,			str_toupper);
	zs_add_primitive("trim",		ZO_STRING,	str_trim);
	zs_add_primitive("trim",		0,			str_trim);
	zs_add_primitive("numstr",		ZO_STRING,	str_numstr);
	zs_add_primitive("numstr",		0,			str_numstr);
	zs_add_primitive("replace",		ZO_STRING,	str_replace);
	zs_add_primitive("replace",		0,			str_replace);
	zs_add_primitive("tokenize",	ZO_STRING,	str_tokenize);
	zs_add_primitive("tokenize",	0,			str_tokenize);
	zs_add_primitive("regex",		ZO_STRING,	str_regex);
	zs_add_primitive("regex",		0,			str_regex);

	zs_add_primitive("integer",		0,			sys_integer);
	zs_add_primitive("real",		0,			sys_real);
	zs_add_primitive("string",		0,			sys_string);
	zs_add_primitive("isnull",		0,			sys_isnull);
	zs_add_primitive("isinteger",	0,			sys_isinteger);
	zs_add_primitive("isreal",		0,			sys_isreal);
	zs_add_primitive("isnumber",	0,			sys_isnumber);
	zs_add_primitive("isstring",	0,			sys_isstring);
	zs_add_primitive("isarray",		0,			sys_isarray);
	zs_add_primitive("isclass",		0,			sys_isclass);
	zs_add_primitive("isuser",		0,			sys_isuser);

	zs_add_primitive("array",		0,			sys_array);

	zs_add_primitive("keys",		ZO_ARRAY,	arr_keys);
	zs_add_primitive("keys",		0,			arr_keys);
	zs_add_primitive("clear",		ZO_ARRAY,	arr_clear);
	zs_add_primitive("clear",		0,			arr_clear);
	zs_add_primitive("remove",		ZO_ARRAY,	arr_remove);
	zs_add_primitive("remove",		0,			arr_remove);
	zs_add_primitive("foreach",		ZO_ARRAY,	arr_foreach);
	zs_add_primitive("foreach",		0,			arr_foreach);

	zs_add_primitive("cos",			0,			sys_cos);
	zs_add_primitive("acos",		0,			sys_acos);
	zs_add_primitive("cosh",		0,			sys_cosh);
	zs_add_primitive("sin",			0,			sys_sin);
	zs_add_primitive("asin",		0,			sys_asin);
	zs_add_primitive("sinh",		0,			sys_sinh);
	zs_add_primitive("tan",			0,			sys_tan);
	zs_add_primitive("atan",		0,			sys_atan);
	zs_add_primitive("atan2",		0,			sys_atan2);
	zs_add_primitive("tanh",		0,			sys_tanh);
	zs_add_primitive("sqrt",		0,			sys_sqrt);
	zs_add_primitive("exp",			0,			sys_exp);
	zs_add_primitive("log",			0,			sys_log);
	zs_add_primitive("log10",		0,			sys_log10);
	zs_add_primitive("ceil",		0,			sys_ceil);
	zs_add_primitive("floor",		0,			sys_floor);
	zs_add_primitive("abs",			0,			sys_abs);
	zs_add_primitive("pow",			0,			sys_pow);
	zs_add_primitive("pi",			0,			sys_pi);
	zs_add_primitive("cal2jul",		0,			sys_cal2jul);
	zs_add_primitive("jul2cal",		0,			sys_jul2cal);

	zs_add_primitive("scandir",		0,			sys_scandir);
	zs_add_primitive("isdir",		0,			sys_isdir);
	zs_add_primitive("mkdir",		0,			sys_mkdir);
	zs_add_primitive("curdir",		0,			sys_curdir);
	zs_add_primitive("escape",		ZO_STRING,	sys_escape);
	zs_add_primitive("escape",		ZO_ARRAY,	sys_escape);
	zs_add_primitive("escape",		0,			sys_escape);
	zs_add_primitive("base64",		0,			sys_base64);
	zs_add_primitive("sendmail",	0,			sys_sendmail);
	zs_add_primitive("ftime",		0,			sys_ftime);

	zs_add_primitive("mbs2utf",		ZO_STRING,	str_mbs2utf);
	zs_add_primitive("mbs2utf",		0,			str_mbs2utf);
	zs_add_primitive("utf2mbs",		ZO_STRING,	str_utf2mbs);
	zs_add_primitive("utf2mbs",		0,			str_utf2mbs);
}


