/*
Copyright (c) 2005, Jiye Zeng (http://www.zegraph.com/)

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

[1] Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer. 

[2] Neither the name of the organization nor the names of its contributors
may be used to endorse or promote products derived from this software
without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "config.h"
#include "pool.h"
#include "object.h"
#include "compiler.h"
#include <cassert>

//#define CHECK_LEAK

///
/// Memory pool
///
template <class T, int nObjects>
class zsPool
{
public:
	zsPool() : _head(0) { }

	/// create object in the pool -- extend the pool if necessary
	T* create()
	{
		if (_head == 0) {
			const int last = nObjects - 1;
			Block* pool = reinterpret_cast<Block*>(malloc(nObjects * sizeof(Block)));
			_head = pool;
			for (int i = 0; i < last; ++i) {
				pool[i].next = &pool[i+1];
			}
			pool[last].next = 0;
		}

		Block* cur = _head;
		_head = cur->next;

		return new(cur) T;
	}

	/// remove object from the pool
	void destroy(T* t)
	{
		assert(t != 0);
		if (t == 0) return;
		t->~T();
		reinterpret_cast<Block*>(t)->next = _head;
		_head = reinterpret_cast<Block*>(t);
	}

private:

   struct Block {
      T object;
      Block* next;
   };

   Block* _head;
};

#ifdef USE_POOL
	static zsPool<zoObject,  POOL_SIZE>		g_oPool;
	static zsPool<zoNumber,  POOL_SIZE*2>	g_nPool;
	static zsPool<zoString,  POOL_SIZE>		g_sPool;
	static zsPool<zoUser,    POOL_SIZE>		g_uPool;
	static zsPool<zoArray,   POOL_SIZE>		g_aPool;
	static zsPool<zoClass,   POOL_SIZE>		g_cPool;
#endif

/// create null
zoObject *zs_create_null(zsContext *ctx)
{
#ifdef USE_POOL
	zoObject *o = g_oPool.create();
#else
	zoObject *o = new zoObject;
#endif
	if (ctx) ctx->keep(o);
#ifdef CHECK_LEAK
	printf("object:   %p\n", o);
#endif
	return o;
}

/// create integer
zoNumber *zs_create_integer(zsContext *ctx, integer_t value)
{
#ifdef USE_POOL
	zoNumber *o = g_nPool.create();
#else
	zoNumber *o = new zoNumber;
#endif
	o->setInteger(value);
	if (ctx) ctx->keep(o);
#ifdef CHECK_LEAK
	printf("integer:  %p  %d\n", o, o->getInteger());
#endif
	return o;
}

/// create real
zoNumber *zs_create_real(zsContext *ctx, real_t value)
{
#ifdef USE_POOL
	zoNumber *o = g_nPool.create();
#else
	zoNumber *o = new zoNumber;
#endif
	o->setReal(value);
	if (ctx) ctx->keep(o);
#ifdef CHECK_LEAK
	printf("real:     %p  %f\n", o, o->getReal());
#endif
	return o;
}

/// create string
zoString *zs_create_string(zsContext *ctx, const std::string& s)
{
#ifdef USE_POOL
	zoString *o = g_sPool.create();
#else
	zoString *o = new zoString;
#endif
	o->set(s);
	if (ctx) ctx->keep(o);
#ifdef CHECK_LEAK
	printf("string:   %p  %s\n", o, o->get().c_str());
#endif
	return o;
}

/// create user
zoUser *zs_create_user(zsContext *ctx, void *ptr, zsPrimitiveFunc opfunc, zsDestroyFunc destroy, int type)
{
#ifdef USE_POOL
	zoUser *o = g_uPool.create();
#else
	zoUser *o = new zoUser;
#endif
	o->init(ptr, opfunc, destroy, type);
	if (ctx) ctx->keep(o);
#ifdef CHECK_LEAK
	printf("user:     %p\n", o);
#endif
	return o;
}

/// create class
zoClass *zs_create_class(zsContext *ctx, zsClassExpr *cls)
{
	assert(ctx);
#ifdef USE_POOL
	zoClass *o = g_cPool.create();
#else
	zoClass *o = new zoClass;
#endif
	o->init(cls);
	if (ctx) ctx->keep(o);
#ifdef CHECK_LEAK
	printf("class:    %p\n", o);
#endif
	return o;
}

/// create array
zoArray *zs_create_array(zsContext *ctx, int size)
{
#ifdef USE_POOL
	zoArray *o = g_aPool.create();
#else
	zoArray *o = new zoArray;
#endif
	o->resize(size);
	if(ctx) ctx->keep(o);
#ifdef CHECK_LEAK
	printf("array:    %p\n", o);
#endif
	return o;
}

/// destroy object
void zs_destroy(zoObject *o)
{
#ifdef CHECK_LEAK
	printf("destroy:  %p\n", o);
#endif
#ifdef USE_POOL
	switch (o->type()) {
	case ZO_NULL:
		g_oPool.destroy((zoObject*)o);
		break;
	case ZO_INTEGER:
	case ZO_REAL:
		g_nPool.destroy((zoNumber*)o);
		break;
	case ZO_STRING:
		g_sPool.destroy((zoString*)o);
		break;
	case ZO_CLASS:
		g_cPool.destroy((zoClass*)o);
		break;
	case ZO_ARRAY:
		g_aPool.destroy((zoArray*)o);
		break;
	default:
		g_uPool.destroy((zoUser*)o);
		break;
	}
#else
	switch (o->type()) {
	case ZO_NULL:
		delete o;
		break;
	case ZO_INTEGER:
	case ZO_REAL:
		delete (zoNumber*)o;
		break;
	case ZO_STRING:
		delete (zoString*)o;
		break;
	case ZO_CLASS:
		delete (zoClass*)o;
		break;
	case ZO_ARRAY:
		delete (zoArray*)o;
		break;
	default:
		delete (zoUser*)o;
		break;
	}
#endif
	o = 0;
}
