/*
Copyright (c) 2005, Jiye Zeng (http://www.zegraph.com/)

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

[1] Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer. 

[2] Neither the name of the organization nor the names of its contributors
may be used to endorse or promote products derived from this software
without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "config.h"
#include "parse.h"
#include "compiler.h"
#include <iostream>

#include <stack>

static bool zs_reduce(std::stack<int>& operators, std::stack<int>& ranks, std::stack<zsBaseExpr*>& operands)
{
	if (ranks.top() == 1) {
		// unary operators
		if (operands.size() < 1) return false;
		zsSimpleExpr *expr = new zsSimpleExpr(operands.top(), operators.top(), 0);
		operands.pop();
		operands.push(expr);
		operators.pop();
		ranks.pop();
	}
	else {
		// binary operator
		if (operands.size() < 2) return false;
		zsBaseExpr *right = operands.top();
		operands.pop();
		zsBaseExpr *left = operands.top();
		operands.pop();
		int op = operators.top();
		operators.pop();
		ranks.pop();
		if (op == '.') {
			if (right->type() == EXPR_CALL) {
				operands.push(new zsMethodExpr(left, (zsCallExpr*)right));
			}
			else if (right->type() == EXPR_ACCESS) {
				operands.push(new zsMemberExpr(left, right));
			}
			else if (right->type() == EXPR_BASE && right->id() == TOK_ID) {
				operands.push(new zsMemberExpr(left, right));
			}
			else {
				right->error();
			}
		}
		else if (op == '=') {
			operands.push(new zsSetExpr(left, right));
		}
		else if (op == ':') {
			operands.push(new zsRangeExpr(left, right));
		}
		else if (op == '?') {
			operands.push(new zsOptionExpr(left, right));
		}
		else {
			operands.push(new zsSimpleExpr(left, op, right));
		}
	}
	return true;
}

zsBaseExpr* zs_parse(zsBaseExpr* caller, int &idx, int opening, int closure)
{
	const zsTokens* toks = caller->tokens();

	if (opening != 0) idx++;

	std::stack<int>	operators;				// operator stack
	std::stack<int>	ranks;					// operator rank stack
	std::stack<zsBaseExpr*> operands;		// operand stack
	
	while (toks->id(idx) != closure) {
		switch (toks->id(idx)) {

		case TOK_IMPORT:
			if (toks->id(idx+1) == '(') {
				// function call, e.g., f(...)
				zsCallExpr *expr = new zsCallExpr(caller);
				operands.push(expr);
				expr->parse(idx);
				break;
			}
			// import module, ...;
			if (!operators.empty() || !operands.empty()) {
				toks->error(idx);
			}
			else {
				idx++;
				zsModuleExpr *module = 0;
				for (;;) {
					if (toks->id(idx) != TOK_ID && toks->id(idx) != TOK_STRING) {
						toks->error(idx);
					}
					// new module to be added to the caller module
					module = new zsModuleExpr;
					if (!module->import(caller->module(), toks->str(idx), false)) {
						// module has been imported
						delete module;
						module = 0;
					}
					else {
						// initialize imported module
						module->exec(caller->module()->context());
					}
					idx++;
					if (toks->id(idx) == ';') break;
					if (toks->id(idx) != ',') toks->error(idx);
					idx++;
				}
				idx++;
				if (!operators.empty() || !operands.empty()) toks->error(idx);
			}
			break;

		case TOK_FUNCTION:
			// function definition, e.g. function f(a, b, c) {...}
			if (caller->type() != EXPR_MODULE && caller->type() != EXPR_CLASS && caller->type() != EXPR_FUNC) {
				caller->error("improper place to define function");
			}
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			{
				zsFuncExpr *expr = new zsFuncExpr(caller);
				expr->parse(idx);
				((zsFuncExpr*)caller)->addFuncCls(expr);
			}
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			break;

		case TOK_CLASS:
			if (caller->type() != EXPR_MODULE && caller->type() != EXPR_CLASS && caller->type() != EXPR_FUNC) {
				caller->error("improper place to define class");
			}
			// class definition, e.g., class A {...}
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			{
				zsClassExpr *expr = new zsClassExpr(caller);
				expr->parse(idx);
				((zsFuncExpr*)caller)->addFuncCls(expr);
			}
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			break;

		case TOK_NEW:
			// instantiate a class, e.g. a = new A;
			operands.push(new zsNewExpr(caller, idx));
			break;

		case TOK_SWITCH:
			// swith (..) { case a: ...; case b: ...; default: ...; }
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			{
				zsSwitchExpr *expr = new zsSwitchExpr(caller);
				caller->addExpr(expr);
				expr->parse(idx);
			}
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			break;

		case TOK_DEFAULT:
			// default case, e.g., default: ...
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			caller->addExpr(new zsDefaultExpr(caller, idx));
			break;

		case TOK_CASE:
			// case, e.g., case 100: ....
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			caller->addExpr(new zsCaseExpr(caller, idx));
			break;

		case TOK_GOTO:
			// goto, e.g., goto 100;
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			caller->addExpr(new zsGotoExpr(caller, idx));
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			break;

		case TOK_GLOBAL:
			// global.variable to be handled at runtime
			idx++;
			break;

		case ',':
			// closure of argument list of function or array
			if (closure != ']' && closure != ')') toks->error(idx);
			while (!operators.empty()) {
				if (!zs_reduce(operators, ranks, operands)) toks->error(idx);
			}
			if (operands.size() != 1) toks->error(idx);
			caller->addExpr(operands.top());
			operands.pop();
			idx++;
			break;

		case ';':
			// end of statement
			while (!operators.empty()) {
				if (!zs_reduce(operators, ranks, operands)) toks->error(idx);
			}
			if (operands.size() > 0) {
				caller->addExpr(operands.top());
				operands.pop();
			}
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			caller->addExpr(new zsEndExpr(caller, idx));
			break;

		case '(':
			// group, e.g., a+(b-c);
			{
			zsBlockExpr *expr = new zsBlockExpr(caller);
			operands.push(expr);
			expr->parseBlock(idx, '(', ')');
			}
			break;

		case '[':
			// array creation; e.g., a + [1, 10, c="abc", "d" = 100]; 
			{
			zsArrayExpr *expr = new zsArrayExpr(caller);
			operands.push(expr);
			expr->parseBlock(idx, '[', ']');
			}
			break;

		case TOK_TRY:
			// try {... }
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			{
				zsTryExpr *expr = new zsTryExpr(caller);
				caller->addExpr(expr);
				expr->parse(idx);
			}
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			break;

		case TOK_WHILE:
		case TOK_DO:
			// while (expr) {...} or do {...} while (expr)
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			{
				zsWhileExpr *expr = new zsWhileExpr(caller);
				caller->addExpr(expr);
				expr->parse(idx);
			}
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			break;

		case TOK_FOR:
			// for (expr; expr; expr;) {...}
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			{
				zsForExpr *expr = new zsForExpr(caller);
				caller->addExpr(expr);
				expr->parse(idx);
			}
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			break;

		case TOK_IF:
			// if (expr) {...} elseif (expr) {...} else {...}
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			{
				zsIfExpr *expr = new zsIfExpr(caller);
				caller->addExpr(expr);
				expr->parse(idx);
			}
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			break;

		case TOK_RETURN:
			// e.g., return; return a; return a+b;
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			{
				zsReturnExpr *expr = new zsReturnExpr(caller);
				caller->addExpr(expr);
				expr->parse(idx);
			}
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			break;

		case TOK_BREAK:
			// e.g., break;
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			caller->addExpr(new zsBreakExpr(caller, idx));
			break;

		case TOK_CONTINUE:
			// e.g., continue;
			if (!operators.empty() || !operands.empty()) toks->error(idx);
			caller->addExpr(new zsContinueExpr(caller, idx));
			break;

		case TOK_FALSE:
		case TOK_NIL:
		case TOK_TRUE:
		case TOK_INTEGER:
		case TOK_HEX:
		case TOK_REAL:
		case TOK_STRING:
			// cosstant
			{
			zsBaseExpr *expr = new zsBaseExpr(caller);
			operands.push(expr);
			expr->parse(idx);
			}
			break;

		case TOK_ID:
			if (toks->id(idx+1) == '(') {
				// function call, e.g., f(...)
				zsCallExpr *expr = new zsCallExpr(caller);
				operands.push(expr);
				expr->parse(idx);
			}
			else if (toks->id(idx+1) == '[') {
				//array access a[...]
				zsAccessExpr *expr = new zsAccessExpr(caller);
				operands.push(expr);
				expr->parse(idx);
			}
			else {
				// variable
				zsBaseExpr *expr = new zsBaseExpr(caller);
				operands.push(expr);
				expr->parse(idx);
			}
			break;

		default:
			{
			// must be operator
			int rank = zs_precedence(toks->id(idx));		// the smaller the rank, the higher the precedence.
			if (rank < 0) toks->error(idx);
			if (!operators.empty()) {
				while(ranks.top() <= rank) {
					// reduce
					if (!zs_reduce(operators, ranks, operands)) toks->error(idx);
					if (operators.empty()) break;
				}
			}
			// shift
			operators.push(toks->id(idx++));
			ranks.push(rank);
			}
			break;
		}
	}

	idx++;		// pass over the closure

	while (!operators.empty()) {
		if (!zs_reduce(operators, ranks, operands)) toks->error(idx);
	}

	if (operands.size() > 1) toks->error(idx);

	if (operands.size() == 1) return operands.top();

	return 0;
}
