/*
Copyright (c) 2005, Jiye Zeng (http://www.zegraph.com/)

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

[1] Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer. 

[2] Neither the name of the organization nor the names of its contributors
may be used to endorse or promote products derived from this software
without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef __ZS_OBJECT__
#define __ZS_OBJECT__

#include "pool.h"
#include "hash.h"
#include "refcount.h"

enum {
	ZO_NULL = 0,
    ZO_INTEGER, 
    ZO_REAL,
    ZO_STRING, 
    ZO_ARRAY,
	ZO_CLASS,
	ZO_USER
};

class zsContext;
class zsBaseExpr;
class zsBlockExpr;
class zsFuncExpr;
class zsClassExpr;
class zoObject;

//! Reference counted base object for null in script.
//
class zoObject : public rcBase<zoObject>
{
public:

	//! Constructor.
	zoObject() : _type(ZO_NULL) { }

	//! Destructor.
	virtual ~zoObject() { }

	//! Null operator function to be called by expression.
	virtual zoObject* opfunc(zsContext *ctx, int op, zoObject *o);

	//! Duplication.
	virtual zoObject* copy(zsContext *ctx) { return this; }

	//! Type checking.
	int type() const { return _type; }

protected:

    int		_type;	// object type
};


//! Object of integer or real.
//
class zoNumber : public zoObject
{
public:
	//! Constructor.
	zoNumber() { _type = ZO_REAL; }

	//! Destructor.
	~zoNumber() { }

	//! Numeric operator function to be called by expression
	zoObject* opfunc(zsContext *ctx, int op, zoObject *o);

	//! Duplication
	zoObject* copy(zsContext *ctx)
	{
		if (_type == ZO_INTEGER) {
			return zs_create_integer(ctx, _u.ivalue);
		}
		return zs_create_real(ctx, _u.rvalue);
	}

	//! Set integer value.
	void setInteger(integer_t value)
	{
		_type = ZO_INTEGER;
		_u.ivalue = value;
	}

	//! Set real value.
	void setReal(real_t value)
	{
		_type = ZO_REAL;
		_u.rvalue = value;
	}

	//! Get integer value.
	integer_t getInteger() const { return _u.ivalue; }

	//! Get real value.
	real_t getReal() const { return _u.rvalue; }

	//! Increment of value by one.
	void incr()
    {
		if (_type == ZO_INTEGER)
			_u.ivalue++;
		else
			_u.rvalue++;
    }

	//! Decrement of value by one
	void decr()
    {
		if (_type == ZO_INTEGER)
			_u.ivalue--;
		else
			_u.rvalue--;
    }

	//! Checking zero.
	bool zero() const
    {
		if (_type == ZO_INTEGER)
			return _u.ivalue == 0;
		else
			return _u.rvalue == 0;
    }

protected:

	union {
		integer_t ivalue;
		real_t rvalue;
	} _u;
};

//! String object
//
class zoString : public zoObject
{
public:
	//! Constructor.
	 zoString() { _type = ZO_STRING; }

	//! Destructor.
	~zoString() { }

	//! String operator function to be called by expression
	zoObject* opfunc(zsContext *ctx, int op, zoObject *o);

	//! Duplication
	zoObject* copy(zsContext *ctx) { return zs_create_string(ctx, _str); }

	//! Member access functions

	const std::string& get() const { return _str; }

	char get(size_t idx) const { return _str.at(idx); }

	void set(const std::string& str) { _str = str;  }

	void set(size_t idx, char c) { _str.at(idx) = c; }

	size_t size() const { return _str.size(); }

private:

	std::string		_str;
};

//! User defined object
//
class zoUser : public zoObject
{
public:
	//! Constructor.
	zoUser() : _ptr(0), _copy(0), _set(0), _get(0), _opfunc(0), _destroy(0) { _type = ZO_USER; }
	
	//! Destructor.
	~zoUser() { if(_destroy != 0) _destroy(_ptr); }

	//! Initialization. Must call this after object creation.
	void init(void* ptr, zsPrimitiveFunc opfunc, zsDestroyFunc destroy, int type);

	//! User operator function to be called be expression.
	zoObject* opfunc(zsContext *ctx, int op, zoObject *o);

	//! Duplication.
	zoObject* copy(zsContext *ctx);

	//! To be called by expression a[...];
	zoObject* get(zsContext *ctx, int nargs, zoObject **args);

	//! To be called by expression a[...]=b;
	void set(zsContext *ctx, int nargs, zoObject **args);

	//! Member access functions

	const void* get() const { return _ptr; }
	      void* get()       { return _ptr; }

private:

	zsPrimitiveFunc		_opfunc;	// user defined operation function

	zsDestroyFunc		_destroy;	// user defined destructor

	zsPrimitiveFunc		_copy;		// called by a = b

	zsPrimitiveFunc		_get;		// called by a[...]

	zsPrimitiveFunc		_set;		// caller by a[...] = b

	void*				_ptr;		// user defined pointer
};

//! User defined class object
//
class zoClass : public zoObject
{
public:
	//! Constructor.
	zoClass() : _class(), _copy(0), _ctx(0) { _type = ZO_CLASS; }
	
	//! Destructor.
	~zoClass();

	//! Initialization. Must call this after object creation.
	void init(zsClassExpr* cls);

	//! Class operator function to be call by expression.
	zoObject* opfunc(zsContext *ctx, int op, zoObject *o);

	//! Duplication.
	zoObject* copy(zsContext *ctx);

	//! Call functions declared in the class.
	zoObject* call(zsContext *ctx, zsBlockExpr *args, zsBaseExpr *expr);

	//! Set class varaible as in c.m=expr;
	void set(const std::string &name, zoObject* o, zsBaseExpr *expr);

	//! Set class variable by expression as in a[...]=c;
	void set(zsContext *ctx, zsBaseExpr *expr, zoObject *o);

	//! Get class varaible as in c.m;
	zoObject* get(const std::string &name, zsBaseExpr *expr);

	//! Get class variable by expression as in a[...];
	zoObject* get(zsContext *ctx, zsBaseExpr *expr);

	//! Get class context.
	zsContext* getContext() { return _ctx; }

private:

	zoObject* callopfunc(const std::string &name, zsContext *ctx, zoObject *right);

private:

	zsFuncExpr*			_copy;

	zsClassExpr*		_class;		// instantiate to this class

	zsContext*			_ctx;		// to be used by _class
};

//! Hash array object
//
class zoArray : public zoObject
{
public:
	//! Constructor.
	zoArray() { _type = ZO_ARRAY; }

	//! Destructor.
	~zoArray() { }

	//! Array operator function to be called by expression.
	zoObject* opfunc(zsContext *ctx, int op, zoObject *o);

	//! Duplication.
	zoObject* copy(zsContext *ctx);

	//! Add a key/value pair to the array as in a.b = c;
	void set(const std::string& key, zoObject *o);

	//! Add a integer-key/value pair to the array as in a[i] = c;
	void set(int key, zoObject *o);

	//! Add object by expression as in a[...] = c;
	void set(zsContext *ctx, zsBaseExpr *expr, zoObject *o);

	//! Get object by name as in  b.c;
//	zoObject* get(const std::string& key);
	zoObject* get(zsContext *ctx, const std::string& key);

	//! Get object by integer-key as in  b[i];
//	zoObject* get(int key);
	zoObject* get(zsContext *ctx, int key);

	//! Get object by expression as in a[...];
	zoObject* get(zsContext *ctx, zsBaseExpr *expr);

	//! Get all keys in the array.
	void get(std::vector<std::string>& keys);

	//! Get number of objects.
	int size() const { return _table.size(); }

	//! Resize and initialize array.
	void resize(int size);

	//! Remove the named object.
	void remove(const std::string& key);

	//! Remove the object with name equivalent to integer string
	void remove(int key);

	//! Remove all objects.
	void clear();

private:

	zsObjTable<zsRCP<zoObject> >	_table;
};


// Prototypes for applying math functions to array
typedef double (*ApplyFunc1)(double);

typedef double (*ApplyFunc2)(double, double);

// Apply one-argument math function to array members
zoObject *zs_apply_func1(zsContext *ctx, ApplyFunc1 f, zoObject *o, int check=0);

// Apply two-arguments math function to array members
zoObject *zs_apply_func2(zsContext *ctx, ApplyFunc2 f, zoObject *o1, zoObject *o2);

// String representation of object
const char* zs_object_str(const zoObject *o, char* buf);

// Format real number
const char *format_real(real_t d, char *buf);

#endif
