/*
Copyright (c) 2005, Jiye Zeng (http://www.zegraph.com/)

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

[1] Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer. 

[2] Neither the name of the organization nor the names of its contributors
may be used to endorse or promote products derived from this software
without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "config.h"
#include "compiler.h"
#include "object.h"
#include <cmath>
#include <cstring>
#include <iostream>

#include "scan.h"

// use global null to save memory
//static zsRCP<zoObject> g_null(zs_create_null(0));

/////////////////////////////////////////////////////////////////////

static void zo_zero_divider(zsContext *ctx)
{
	ctx->expr->error("divided by zero");
}

static const char *zo_type_name(zoObject *o, char *buf)
{
	buf[0] = 0;
	if (o) {
		switch (o->type()) {
		case ZO_NULL: strcpy(buf, "null"); break;
		case ZO_INTEGER: strcpy(buf, "integer"); break;
	    case ZO_REAL: strcpy(buf, "real"); break;
		case ZO_STRING: strcpy(buf, "string"); break; 
		case ZO_ARRAY: strcpy(buf, "array"); break;
		case ZO_CLASS: strcpy(buf, "class"); break;
		default: strcpy(buf, "user"); break;
		}
	}
	return buf;
}

static void zo_op_error(zsContext *ctx, zoObject *left, int op, zoObject *right)
{
	char error[128], buf[32], buf1[16], buf2[16];
	zo_type_name(left, buf1);
	zo_type_name(right, buf2);
	switch (op) {
	case TOK_NEGATE:
		sprintf(buf, "-%s", buf1); break;
	case TOK_INCR:
		sprintf(buf, "%s++", buf1); break;
	case TOK_DECR:
		sprintf(buf, "%s--", buf1); break;
	case '!':
		sprintf(buf, "!%s", buf1); break;
	case '~':
		sprintf(buf, "~%s", buf1); break;
	case '*':
		sprintf(buf, "%s * %s", buf1, buf2); break;
	case '/':
		sprintf(buf, "%s / %s", buf1, buf2); break;
	case TOK_DIV2:
		sprintf(buf, "%s / %s", buf2, buf1); break;
	case '%':
		sprintf(buf, "%s % %s", buf1, buf2); break;
	case TOK_MOD2:
		sprintf(buf, "%s % %s", buf2, buf1); break;
	case '-':
		sprintf(buf, "%s - %s", buf1, buf2); break;
	case TOK_SUB2:
		sprintf(buf, "%s - %s", buf2, buf1); break;
	case '+':
		sprintf(buf, "%s + %s", buf1, buf2); break;
	case TOK_LEQ:
		sprintf(buf, "%s <= %s", buf1, buf2); break;
	case TOK_GEQ:
		sprintf(buf, "%s >= %s", buf1, buf2); break;
	case '<':
		sprintf(buf, "%s < %s", buf1, buf2); break;
	case '>':
		sprintf(buf, "%s > %s", buf1, buf2); break;
	case TOK_EQL:
		sprintf(buf, "%s == %s", buf1, buf2); break;
	case TOK_NEQ:
		sprintf(buf, "%s != %s", buf1, buf2); break;
	case TOK_LAND:
		sprintf(buf, "%s && %s", buf1, buf2); break;
	case TOK_LOR:
		sprintf(buf, "%s || %s", buf1, buf2); break;
	case TOK_MULEQ:
		sprintf(buf, "%s *= %s", buf1, buf2); break;
	case TOK_DIVEQ:
		sprintf(buf, "%s /= %s", buf1, buf2); break;
	case TOK_MODEQ:
		sprintf(buf, "%s %= %s", buf1, buf2); break;
	case TOK_SUBEQ:
		sprintf(buf, "%s -= %s", buf1, buf2); break;
	case TOK_ADDEQ:
		sprintf(buf, "%s += %s", buf1, buf2); break;
	case TOK_RSHIFT:
		sprintf(buf, "%s >> %s", buf1, buf2); break;
	case TOK_LSHIFT:
		sprintf(buf, "%s << %s", buf1, buf2); break;
	case '^':
		sprintf(buf, "%s ^ %s", buf1, buf2); break;
	case '&':
		sprintf(buf, "%s & %s", buf1, buf2); break;
	case '|':
		sprintf(buf, "%s | %s", buf1, buf2); break;
	case TOK_RSHIFTEQ:
		sprintf(buf, "%s >>= %s", buf1, buf2); break;
	case TOK_LSHIFTEQ:
		sprintf(buf, "%s <<= %s", buf1, buf2); break;
	case TOK_XOREQ:
		sprintf(buf, "%s ^= %s", buf1, buf2); break;
	case TOK_ANDEQ:
		sprintf(buf, "%s &= %s", buf1, buf2); break;
	case TOK_OREQ:
		sprintf(buf, "%s |= %s", buf1, buf2); break;
	default:
		sprintf(buf, "%s op %s", buf1, buf2); break;
	}
	sprintf(error, "undefined operation (%s)", buf);
	ctx->expr->error(error);
}

/////////////////////////////////////////////////////////////////////

zoObject* zoObject::opfunc(zsContext *ctx, int op, zoObject *o)
{
	switch (op) {
	case '+':
		return o;
	case '!':
		return zs_create_integer(ctx, 1);
	case TOK_EQL:
		if (o->type() == ZO_NULL) return zs_create_integer(ctx, 1);
		return zs_create_integer(ctx, 0);
	case TOK_NEQ:
		if (o->type() != ZO_NULL) return zs_create_integer(ctx, 1);
		return zs_create_integer(ctx, 0);
	}
	zo_op_error(ctx, this, op, o);
	return 0;
}

/////////////////////////////////////////////////////////////////////

zoObject* zoNumber::opfunc(zsContext *ctx, int op, zoObject *o)
{
	if (o && o->type() == ZO_NULL) {
		// o is null
		if (op == TOK_EQL) return zs_create_integer(ctx, 0);
		if (op == TOK_NEQ) return zs_create_integer(ctx, 1);
		if (op == '+') return this;
		zo_op_error(ctx, this, op, o);
	}
	switch (op) {
	case TOK_INCR:
		this->incr();
		return this;
	case TOK_DECR:
		this->decr();
		return this;
	case TOK_NEGATE:
		if (type() == ZO_INTEGER)
			return zs_create_integer(ctx, -getInteger());
		else
			return zs_create_real(ctx, -getReal());
	case '!':
		if (type() == ZO_INTEGER)
			return zs_create_integer(ctx, !getInteger());
		else
			return zs_create_integer(ctx, !getReal());
	case '~':
		if (type() != ZO_INTEGER) zo_op_error(ctx, this, op, o);
		return zs_create_integer(ctx, ~getInteger());
	case '*':
		if (o->type() > ZO_REAL) return o->opfunc(ctx, op, this);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getInteger()*((zoNumber*)o)->getInteger());
			else
				return zs_create_real(ctx, getInteger()*((zoNumber*)o)->getReal());
		}
		else {
			if (o->type() == ZO_INTEGER)
				return zs_create_real(ctx, getReal()*((zoNumber*)o)->getInteger());
			else
				return zs_create_real(ctx, getReal()*((zoNumber*)o)->getReal());
		}
	case '/':
		if (o->type() > ZO_REAL) return o->opfunc(ctx, TOK_DIV2, this);
		if (((zoNumber*)o)->zero()) zo_zero_divider(ctx);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getInteger()/((zoNumber*)o)->getInteger());
			else
				return zs_create_real(ctx, getInteger()/((zoNumber*)o)->getReal());
		}
		else {
			if (o->type() == ZO_INTEGER)
				return zs_create_real(ctx, getReal()/((zoNumber*)o)->getInteger());
			else
				return zs_create_real(ctx, getReal()/((zoNumber*)o)->getReal());
		}
	case TOK_DIV2:
		if (o->type() > ZO_REAL) zo_op_error(ctx, this, op, o);
		if (zero()) zo_zero_divider(ctx);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, ((zoNumber*)o)->getInteger()/getInteger());
			else
				return zs_create_real(ctx, ((zoNumber*)o)->getReal()/getInteger());
		}
		else {
			if (o->type() == ZO_INTEGER)
				return zs_create_real(ctx, ((zoNumber*)o)->getInteger()/getReal());
			else
				return zs_create_real(ctx, ((zoNumber*)o)->getReal()/getReal());
		}
	case '%':
		if (o->type() > ZO_REAL) return o->opfunc(ctx, TOK_MOD2, this);
		if (((zoNumber*)o)->zero()) zo_zero_divider(ctx);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getInteger()%((zoNumber*)o)->getInteger());
			else
				return zs_create_real(ctx, fmod(getInteger(),((zoNumber*)o)->getReal()));
		}
		else {
			if (o->type() == ZO_INTEGER)
				return zs_create_real(ctx, fmod(getReal(),((zoNumber*)o)->getInteger()));
			else
				return zs_create_real(ctx, fmod(getReal(),((zoNumber*)o)->getReal()));
		}
	case TOK_MOD2:
		if (o->type() > ZO_REAL) zo_op_error(ctx, this, op, o);
		if (zero()) zo_zero_divider(ctx);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, ((zoNumber*)o)->getInteger()%getInteger());
			else
				return zs_create_real(ctx, fmod(((zoNumber*)o)->getReal(),getInteger()));
		}
		else {
			if (o->type() == ZO_INTEGER)
				return zs_create_real(ctx, fmod(((zoNumber*)o)->getInteger(),getReal()));
			else
				return zs_create_real(ctx, fmod(((zoNumber*)o)->getReal(),getReal()));
		}
	case '-':
		if (o->type() > ZO_REAL) return o->opfunc(ctx, TOK_SUB2, this);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getInteger()-((zoNumber*)o)->getInteger());
			else
				return zs_create_real(ctx, getInteger()-((zoNumber*)o)->getReal());
		}
		else {
			if (o->type() == ZO_INTEGER)
				return zs_create_real(ctx, getReal()-((zoNumber*)o)->getInteger());
			else
				return zs_create_real(ctx, getReal()-((zoNumber*)o)->getReal());
		}
	case TOK_SUB2:
		if (o->type() > ZO_REAL) zo_op_error(ctx, this, op, o);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, ((zoNumber*)o)->getInteger()-getInteger());
			else
				return zs_create_real(ctx, ((zoNumber*)o)->getReal()-getInteger());
		}
		else {
			if (o->type() == ZO_INTEGER)
				return zs_create_real(ctx, ((zoNumber*)o)->getInteger()-getReal());
			else
				return zs_create_real(ctx, ((zoNumber*)o)->getReal()-getReal());
		}
	case '+':
		if (o->type() > ZO_REAL) {
			if (o->type() == ZO_STRING) {
				char buf[64];
				std::string s(zs_object_str(this, buf));
				s += ((zoString*)o)->get();
				return zs_create_string(ctx, s);
			}
			return o->opfunc(ctx, op, this);
		}
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getInteger()+((zoNumber*)o)->getInteger());
			else
				return zs_create_real(ctx, getInteger()+((zoNumber*)o)->getReal());
		}
		else {
			if (o->type() == ZO_INTEGER)
				return zs_create_real(ctx, getReal()+((zoNumber*)o)->getInteger());
			else
				return zs_create_real(ctx, getReal()+((zoNumber*)o)->getReal());
		}
	case TOK_LEQ:
		if (o->type() > ZO_REAL) return o->opfunc(ctx, TOK_GEQ, this);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getInteger()<=((zoNumber*)o)->getInteger());
			else
				return zs_create_integer(ctx, getInteger()<=((zoNumber*)o)->getReal());
		}
		else {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getReal()<=((zoNumber*)o)->getInteger());
			else
				return zs_create_integer(ctx, getReal()<=((zoNumber*)o)->getReal());
		}
	case TOK_GEQ:
		if (o->type() > ZO_REAL) return o->opfunc(ctx, TOK_LEQ, this);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getInteger()>=((zoNumber*)o)->getInteger());
			else
				return zs_create_integer(ctx, getInteger()>=((zoNumber*)o)->getReal());
		}
		else {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getReal()>=((zoNumber*)o)->getInteger());
			else
				return zs_create_integer(ctx, getReal()>=((zoNumber*)o)->getReal());
		}
	case '<':
		if (o->type() > ZO_REAL) return o->opfunc(ctx, '>', this);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getInteger()<((zoNumber*)o)->getInteger());
			else
				return zs_create_integer(ctx, getInteger()<((zoNumber*)o)->getReal());
		}
		else {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getReal()<((zoNumber*)o)->getInteger());
			else
				return zs_create_integer(ctx, getReal()<((zoNumber*)o)->getReal());
		}
	case '>':
		if (o->type() > ZO_REAL) return o->opfunc(ctx, '<', this);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getInteger()>((zoNumber*)o)->getInteger());
			else
				return zs_create_integer(ctx, getInteger()>((zoNumber*)o)->getReal());
		}
		else {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getReal()>((zoNumber*)o)->getInteger());
			else
				return zs_create_integer(ctx, getReal()>((zoNumber*)o)->getReal());
		}
	case TOK_EQL:
		if (o->type() > ZO_REAL) return o->opfunc(ctx, op, this);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getInteger()==((zoNumber*)o)->getInteger());
			else
				return zs_create_integer(ctx, getInteger()==((zoNumber*)o)->getReal());
		}
		else {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getReal()==((zoNumber*)o)->getInteger());
			else
				return zs_create_integer(ctx, getReal()==((zoNumber*)o)->getReal());
		}
	case TOK_NEQ:
		if (o->type() > ZO_REAL) return o->opfunc(ctx, op, this);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getInteger()!=((zoNumber*)o)->getInteger());
			else
				return zs_create_integer(ctx, getInteger()!=((zoNumber*)o)->getReal());
		}
		else {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getReal()!=((zoNumber*)o)->getInteger());
			else
				return zs_create_integer(ctx, getReal()!=((zoNumber*)o)->getReal());
		}
	case TOK_LAND:
		if (o->type() > ZO_REAL) return o->opfunc(ctx, op, this);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getInteger()&&((zoNumber*)o)->getInteger());
			else
				return zs_create_integer(ctx, getInteger()&&((zoNumber*)o)->getReal());
		}
		else {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getReal()&&((zoNumber*)o)->getInteger());
			else
				return zs_create_integer(ctx, getReal()&&((zoNumber*)o)->getReal());
		}
	case TOK_LOR:
		if (o->type() > ZO_REAL) return o->opfunc(ctx, op, this);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getInteger()||((zoNumber*)o)->getInteger());
			else
				return zs_create_integer(ctx, getInteger()||((zoNumber*)o)->getReal());
		}
		else {
			if (o->type() == ZO_INTEGER)
				return zs_create_integer(ctx, getReal()||((zoNumber*)o)->getInteger());
			else
				return zs_create_integer(ctx, getReal()||((zoNumber*)o)->getReal());
		}
	case TOK_RSHIFT:
		if (o->type() > ZO_REAL) return o->opfunc(ctx, op, this);
		return zs_create_integer(ctx, getInteger()>>((zoNumber*)o)->getInteger());
	case TOK_LSHIFT:
		if (o->type() > ZO_REAL) return o->opfunc(ctx, op, this);
		return zs_create_integer(ctx, getInteger()<<((zoNumber*)o)->getInteger());
	case '^':
		if (o->type() > ZO_REAL) return o->opfunc(ctx, op, this);
		return zs_create_integer(ctx, getInteger()^((zoNumber*)o)->getInteger());
	case '&':
		if (o->type() > ZO_REAL) return o->opfunc(ctx, op, this);
		return zs_create_integer(ctx, getInteger()&((zoNumber*)o)->getInteger());
	case '|':
		if (o->type() > ZO_REAL) return o->opfunc(ctx, op, this);
		return zs_create_integer(ctx, getInteger()|((zoNumber*)o)->getInteger());
	case TOK_MULEQ:
		if (o->type() > ZO_REAL) zo_op_error(ctx, this, op, o);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				setInteger(getInteger()*((zoNumber*)o)->getInteger());
			else
				setReal(getInteger()*((zoNumber*)o)->getReal());
		}
		else {
			if (o->type() == ZO_INTEGER)
				setReal(getReal()*((zoNumber*)o)->getInteger());
			else
				setReal(getReal()*((zoNumber*)o)->getReal());
		}
		return this;
	case TOK_DIVEQ:
		if (o->type() > ZO_REAL) zo_op_error(ctx, this, op, o);
		if (((zoNumber*)o)->zero()) zo_zero_divider(ctx);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				setInteger(getInteger()/((zoNumber*)o)->getInteger());
			else
				setReal(getInteger()/((zoNumber*)o)->getReal());
		}
		else {
			if (o->type() == ZO_INTEGER)
				setReal(getReal()/((zoNumber*)o)->getInteger());
			else
				setReal(getReal()/((zoNumber*)o)->getReal());
		}
		return this;
	case TOK_MODEQ:
		if (o->type() > ZO_REAL) zo_op_error(ctx, this, op, o);
		if (((zoNumber*)o)->zero()) zo_zero_divider(ctx);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				setInteger(getInteger()%((zoNumber*)o)->getInteger());
			else
				setReal(fmod(getInteger(),((zoNumber*)o)->getReal()));
		}
		else {
			if (o->type() == ZO_INTEGER)
				setReal(fmod(getReal(),((zoNumber*)o)->getInteger()));
			else
				setReal(fmod(getReal(),((zoNumber*)o)->getReal()));
		}
		return this;
	case TOK_SUBEQ:
		if (o->type() > ZO_REAL) zo_op_error(ctx, this, op, o);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				setInteger(getInteger()-((zoNumber*)o)->getInteger());
			else
				setReal(getInteger()-((zoNumber*)o)->getReal());
		}
		else {
			if (o->type() == ZO_INTEGER)
				setReal(getReal()-((zoNumber*)o)->getInteger());
			else
				setReal(getReal()-((zoNumber*)o)->getReal());
		}
		return this;
	case TOK_ADDEQ:
		if (o->type() > ZO_REAL) zo_op_error(ctx, this, op, o);
		if (type() == ZO_INTEGER) {
			if (o->type() == ZO_INTEGER)
				setInteger(getInteger()+((zoNumber*)o)->getInteger());
			else
				setReal(getInteger()+((zoNumber*)o)->getReal());
		}
		else {
			if (o->type() == ZO_INTEGER)
				setReal(getReal()+((zoNumber*)o)->getInteger());
			else
				setReal(getReal()+((zoNumber*)o)->getReal());
		}
		return this;
	case TOK_RSHIFTEQ:
		if (type() != ZO_INTEGER || o->type() != ZO_INTEGER) zo_op_error(ctx, this, op, o);
		setInteger(getInteger()>>((zoNumber*)o)->getInteger());
		return this;
	case TOK_LSHIFTEQ:
		if (type() != ZO_INTEGER || o->type() != ZO_INTEGER) zo_op_error(ctx, this, op, o);
		setInteger(getInteger()<<((zoNumber*)o)->getInteger());
		return this;
	case TOK_XOREQ:
		if (type() != ZO_INTEGER || o->type() != ZO_INTEGER) zo_op_error(ctx, this, op, o);
		setInteger(getInteger()^((zoNumber*)o)->getInteger());
		return this;
	case TOK_ANDEQ:
		if (type() != ZO_INTEGER || o->type() != ZO_INTEGER) zo_op_error(ctx, this, op, o);
		setInteger(getInteger()&((zoNumber*)o)->getInteger());
		return this;
	case TOK_OREQ:
		if (type() != ZO_INTEGER || o->type() != ZO_INTEGER) zo_op_error(ctx, this, op, o);
		setInteger(getInteger()|((zoNumber*)o)->getInteger());
		return this;
	}
	zo_op_error(ctx, this, op, o);
	return 0;
}

/////////////////////////////////////////////////////////////////////

zoObject* zoString::opfunc(zsContext *ctx, int op, zoObject *o)
{
	if (o && o->type() == ZO_NULL) {
		if (op == TOK_EQL) return zs_create_integer(ctx, 0);
		if (op == TOK_NEQ) return zs_create_integer(ctx, 1);
		if (op == '+') return this;
		zo_op_error(ctx, this, op, o);
	}
	char buf[256];
	switch (op) {
	case TOK_ADDEQ:
		if (o->type() == ZO_STRING)
			set(get()+((zoString*)o)->get());
		else
			set(get()+zs_object_str(o, buf));
		return this;
	case '+':
		if (o->type() == ZO_ARRAY) {
			zoArray *arr = (zoArray*)o;
			zoArray *ret = zs_create_array(ctx, arr->size());
			std::vector<std::string> keys;
			arr->get(keys);
			for (int i = 0; i < keys.size(); i++) {
				zoObject *val = arr->get(ctx, keys[i]);
				if (val->type() == ZO_ARRAY) {
					ret->set(keys[i], opfunc(ctx, op, val));
				}
				else {
					char buf[64];
					std::string s(get());
					s += zs_object_str(val, buf);
					ret->set(keys[i], zs_create_string(ctx, s));
				}
			}
			return ret;
		}
		if (o->type() == ZO_STRING)
			return zs_create_string(ctx, get()+((zoString*)o)->get());
		else
			return zs_create_string(ctx, get()+zs_object_str(o, buf));
	case TOK_EQL:
		if (o->type() == ZO_STRING)
			return zs_create_integer(ctx, get()==((zoString*)o)->get());
		else
			return zs_create_integer(ctx, 0);
	case TOK_NEQ:
		if (o->type() == ZO_STRING)
			return zs_create_integer(ctx, get()!=((zoString*)o)->get());
		else
			return zs_create_integer(ctx, 1);
	case '>':
		if (o->type() != ZO_STRING) zo_op_error(ctx, this, op, o);
		return zs_create_integer(ctx, get()>((zoString*)o)->get());
	case '<':
		if (o->type() != ZO_STRING) zo_op_error(ctx, this, op, o);
		return zs_create_integer(ctx, get()<((zoString*)o)->get());
	case TOK_GEQ:
		if (o->type() != ZO_STRING) zo_op_error(ctx, this, op, o);
		return zs_create_integer(ctx, get()>=((zoString*)o)->get());
	case TOK_LEQ:
		if (o->type() != ZO_STRING) zo_op_error(ctx, this, op, o);
		return zs_create_integer(ctx, get()<=((zoString*)o)->get());
	}
	zo_op_error(ctx, this, op, o);
	return 0;
}

/////////////////////////////////////////////////////////////////////

void zoUser::init(void* ptr, zsPrimitiveFunc opfunc, zsDestroyFunc destroy, int type)
{
	_ptr = ptr;
	_opfunc = opfunc;
	_destroy = destroy;
	_type = ZO_USER + type;
	if (type > 0) {
		_copy = zs_get_primitive("__copy", _type, 0);
		_set = zs_get_primitive("__set", _type, 0);
		_get = zs_get_primitive("__get", _type, 0);
	}
}

zoObject* zoUser::opfunc(zsContext *ctx, int op, zoObject *o)
{
	if (o && o->type() == ZO_NULL) {
		if (op == TOK_EQL) return zs_create_integer(ctx, 0);
		if (op == TOK_NEQ) return zs_create_integer(ctx, 1);
		if (op == '+') return this;
		zo_op_error(ctx, this, op, o);
	}
	if (!_opfunc) {
		switch (op) {
		case TOK_EQL:
			if (o->type() != ZO_USER)
				return zs_create_integer(ctx, 0);
			else
				return zs_create_integer(ctx, get()==((zoUser*)o)->get());
		case TOK_NEQ:
			if (o->type() != ZO_USER)
				return zs_create_integer(ctx, 1);
			else
				return zs_create_integer(ctx, get()!=((zoUser*)o)->get());
		}
		zo_op_error(ctx, this, op, o);
	}
	zoNumber p;
	p.setInteger(op);
	zoObject *args[3] = {this, &p, o};
	return _opfunc(ctx, o?3:2, args);
}

zoObject* zoUser::copy(zsContext *ctx)
{
	if (_copy) {
		zoObject *args[1] = { this };
		return _copy(ctx, 1, args);
	}
	return this;
}

zoObject* zoUser::get(zsContext *ctx, int nargs, zoObject **args)
{
	if (!_get) ctx->expr->error("__get primitive undefined for this type of user object");
	zoObject *ret = _get(ctx, nargs, args);
//	return (ret?ret:zsModuleExpr::g_null.get());
	return (ret?ret:zs_create_null(ctx));
}

void zoUser::set(zsContext *ctx, int nargs, zoObject **args)
{
	if (!_set) ctx->expr->error("__set primitive undefined for this type of user object");
	_set(ctx, nargs, args);
}

/////////////////////////////////////////////////////////////////////

zoClass::~zoClass()
{
	delete _ctx;
}

void zoClass::init(zsClassExpr* cls)
{
	_ctx = new zsContext(cls->module()->context());
	_class = cls;
	_ctx->setVar("this->", this, false);
	cls->exec(_ctx);
	_copy = cls->getFunc("__copy", 0, false);
}

zoObject* zoClass::opfunc(zsContext *ctx, int op, zoObject *o)
{
	if (o && o->type() == ZO_NULL) {
		if (op == TOK_EQL) return zs_create_integer(ctx, 0);
		if (op == TOK_NEQ) return zs_create_integer(ctx, 1);
		if (op == '+') return this;
		zo_op_error(ctx, this, op, o);
	}
	switch (op) {
	case TOK_NEGATE:
		return callopfunc("__neg", ctx, o);
	case TOK_INCR:
		return callopfunc("__incr", ctx, o);
	case TOK_DECR:
		return callopfunc("__decr", ctx, o);
	case '!':
		return callopfunc("__not", ctx, o);
	case '~':
		return callopfunc("__cmpl", ctx, o);
	case '*':
		return callopfunc("__mul", ctx, o);
	case '/':
		return callopfunc("__div", ctx, o);
	case TOK_DIV2:
		return callopfunc("__div2", ctx, o);
	case '%':
		return callopfunc("__mod", ctx, o);
	case TOK_MOD2:
		return callopfunc("__mod2", ctx, o);
	case '-':
		return callopfunc("__sub", ctx, o);
	case TOK_SUB2:
		return callopfunc("__sub2", ctx, o);
	case '+':
		return callopfunc("__add", ctx, o);
	case TOK_LEQ:
		return callopfunc("__le", ctx, o);
	case TOK_GEQ:
		return callopfunc("__ge", ctx, o);
	case '<':
		return callopfunc("__lt", ctx, o);
	case '>':
		return callopfunc("__gt", ctx, o);
	case TOK_EQL:
		return callopfunc("__eq", ctx, o);
	case TOK_NEQ:
		return callopfunc("__ne", ctx, o);
	case '&':
		return callopfunc("__and", ctx, o);
	case '|':
		return callopfunc("__or", ctx, o);
	case '^':
		return callopfunc("__xor", ctx, o);
	case TOK_LAND:
		return callopfunc("__nn", ctx, o);
	case TOK_LOR:
		return callopfunc("__oo", ctx, o);
	case TOK_RSHIFT:
		return callopfunc("__rsh", ctx, o);
	case TOK_LSHIFT:
		return callopfunc("__lsh", ctx, o);
	case TOK_MULEQ:
		return callopfunc("__mul_eq", ctx, o);
	case TOK_DIVEQ:
		return callopfunc("__div_eq", ctx, o);
	case TOK_MODEQ:
		return callopfunc("__mod_eq", ctx, o);
	case TOK_SUBEQ:
		return callopfunc("__sub_eq", ctx, o);
	case TOK_ADDEQ:
		return callopfunc("__add_eq", ctx, o);
	case TOK_RSHIFTEQ:
		return callopfunc("__rsh_eq", ctx, o);
	case TOK_LSHIFTEQ:
		return callopfunc("__lsh_eq", ctx, o);
	case TOK_XOREQ:
		return callopfunc("__xor_eq", ctx, o);
	case TOK_ANDEQ:
		return callopfunc("__and_eq", ctx, o);
	case TOK_OREQ:
		return callopfunc("__or_eq", ctx, o);
	}
	zo_op_error(ctx, this, op, o);
	return 0;
}

zoObject* zoClass::copy(zsContext *ctx)
{
	if (_copy) {
		zoObject *args[1] = { this };
		return _copy->call(ctx, 1, args);
	}
	return this;
}

zoObject* zoClass::callopfunc(const std::string &name, zsContext *ctx, zoObject *right)
{
	zsFuncExpr *f = _class->getFunc(name, 0, false);
	if (!f) ctx->expr->error("class operator function undefined");
	zoObject *args[2] = { this, right };
	int nargs = right ? 2 : 1;
	return f->call(_ctx, nargs, args);
}

zoObject* zoClass::call(zsContext *ctx, zsBlockExpr *args, zsBaseExpr *expr)
{
	zsFuncExpr *f = _class->getFunc(args->name(), expr, true);
	ctx->expr = args;
	return f->call(ctx, args, this);
}

void zoClass::set(const std::string &name, zoObject* o, zsBaseExpr *expr)
{
	_ctx->setVar(name, o, false);
}

void zoClass::set(zsContext *ctx, zsBaseExpr *expr, zoObject *o)
{
	if (expr->type() == EXPR_BASE && (expr->id() == TOK_STRING || expr->id() == TOK_INTEGER)) {
		set(expr->name(), o, expr);
	}
	else {
		zoObject *key = expr->exec(ctx);
		if (key->type() == ZO_STRING) {
			set(((zoString*)key)->get(), o, expr);
		}
		else {
			expr->error("class member name expected");
		}
	}
}

zoObject* zoClass::get(const std::string &name, zsBaseExpr *expr)
{
	zsObjTable<zsRCP<zoObject> > &vars = _ctx->vars();
	zsObjItem<zsRCP<zoObject> > *item = vars.get(name);
	if (!item) expr->error("class member undefined");
	return item->_obj.get();
}

zoObject *zoClass::get(zsContext *ctx, zsBaseExpr *expr)
{
	if (expr->type() == EXPR_BASE && (expr->id() == TOK_ID || expr->id() == TOK_INTEGER)) {
		return get(expr->name(), expr);
	}
	zoObject *key = expr->exec(ctx);
	if (key->type() == ZO_STRING) {
		return get(((zoString*)key)->get(), expr);
	}
	expr->error("class member name expected");
	return 0;
}

/////////////////////////////////////////////////////////////////////

zoObject* zoArray::opfunc(zsContext *ctx, int op, zoObject *o)
{
	if (o && o->type() == ZO_NULL) {
		if (op == TOK_EQL) return zs_create_integer(ctx, 0);
		if (op == TOK_NEQ) return zs_create_integer(ctx, 1);
		if (op == '+') return this;
		zo_op_error(ctx, this, op, o);
	}
	int i, n = size();
	zoArray *arr;
	zoArray *src = 0;
	if (o != 0 && o->type() == ZO_ARRAY) {
		src = (zoArray*)o;
		if (src->size() != n) ctx->expr->error("arrays have different sizes");
	}
	switch (op) {
	case TOK_INCR:
	case TOK_DECR:
		for (i = 0; i < n; i++) {
			zoObject* m = get(ctx, i);
			m->opfunc(ctx, op, 0);
		}
		return this;
	case TOK_NEGATE:
	case '!':
	case '~':
		arr = zs_create_array(ctx, n);
		for (i = 0; i < n; i++) {
			zoObject* m = get(ctx, i);
			zoObject* r = m->opfunc(ctx, op, 0);
			arr->set(i, r);
		}
		return arr;
	case '*':
	case '/':
	case '%':
	case '-':
	case '+':
	case TOK_LEQ:
	case TOK_GEQ:
	case '<':
	case '>':
	case TOK_EQL:
	case TOK_NEQ:
	case '^':
	case '&':
	case '|':
	case TOK_LAND:
	case TOK_LOR:
		arr = zs_create_array(ctx, n);
		if (src != 0) {
			for (i = 0; i < n; i++) {
				zoObject* m = get(ctx, i);
				zoObject* s = src->get(ctx, i);
				zoObject* r = m->opfunc(ctx, op, s);
				arr->set(i, r);
			}
		}
		else {
			for (i = 0; i < n; i++) {
				zoObject* m = get(ctx, i);
				zoObject* r = m->opfunc(ctx, op, o);
				arr->set(i, r);
			}
		}
		return arr;
	case TOK_DIV2:
	case TOK_MOD2:
	case TOK_SUB2:
		arr = zs_create_array(ctx, n);
		for (i = 0; i < n; i++) {
			zoObject* m = get(ctx, i);
			zoObject* r = m->opfunc(ctx, op, o);
			arr->set(i, r);
		}
		return arr;
	case TOK_MULEQ:
	case TOK_DIVEQ:
	case TOK_MODEQ:
	case TOK_SUBEQ:
	case TOK_ADDEQ:
	case TOK_XOREQ:
	case TOK_ANDEQ:
	case TOK_OREQ:
		if (src != 0) {
			for (i = 0; i < n; i++) {
				zoObject* m = get(ctx, i);
				zoObject* s = src->get(ctx, i);
				zoObject* r = m->opfunc(ctx, op, s);
				set(i, r);
			}
		}
		else {
			for (i = 0; i < n; i++) {
				zoObject* m = get(ctx, i);
				zoObject* r = m->opfunc(ctx, op, o);
				set(i, r);
			}
		}
		return this;
	}
	zo_op_error(ctx, this, op, o);
	return 0;
}

zoObject* zoArray::copy(zsContext *ctx)
{
	zoArray *arr = zs_create_array(ctx, size());
	std::vector<std::string> keys;
	get(keys);
	for (int i = 0; i < keys.size(); i++) {
		arr->set(keys[i], get(ctx, keys[i])->copy(ctx));
	}
	return arr;
}

zoObject* zoArray::get(zsContext *ctx, const std::string& key)
{
	zsObjItem<zsRCP<zoObject> > *item = _table.get(key);
	if (item) return item->_obj.get();
	return zs_create_null(ctx);
}

zoObject* zoArray::get(zsContext *ctx, int key)
{
	char buf[32];
	sprintf(buf, "%d", key);
	return get(ctx, buf);
}

void zoArray::get(std::vector<std::string>& keys)
{
	_table.get(keys);
}

zoObject *zoArray::get(zsContext *ctx, zsBaseExpr *expr)
{
	if (expr->type() == EXPR_BASE && (expr->id() == TOK_STRING || expr->id() == TOK_INTEGER)) {
		return get(ctx, expr->name());
	}
	zoObject *key = expr->exec(ctx);
	if (key->type() == ZO_INTEGER) {
		return get(ctx, ((zoNumber*)key)->getInteger());
	}
	if (key->type() == ZO_STRING) {
		return get(ctx, ((zoString*)key)->get());
	}
	expr->error("string/integer key expected");
	return 0;
}

void zoArray::set(const std::string& key, zoObject *o)
{
//	if (!o) o = zsModuleExpr::g_null.get();
	if (!o) o = zs_create_null(0);
	_table.set(key, zsRCP<zoObject>(o));
	_table.expand();
}

void zoArray::set(int key, zoObject *o)
{
	char buf[32];
	sprintf(buf, "%d", key);
	set(buf, o);
}

void zoArray::set(zsContext *ctx, zsBaseExpr *expr, zoObject *o)
{
	if (expr->type() == EXPR_BASE && (expr->id() == TOK_STRING || expr->id() == TOK_INTEGER)) {
		set(expr->name(), o);
	}
	else {
		zoObject *key = expr->exec(ctx);
		if (key->type() == ZO_INTEGER) {
			set(((zoNumber*)key)->getInteger(), o);
		}
		else if (key->type() == ZO_STRING) {
			set(((zoString*)key)->get(), o);
		}
		else {
			expr->error("string/integer key expected");
		}
	}
}

void zoArray::clear()
{
	_table.clear();
}

void zoArray::resize(int size)
{
	_table.clear();
	_table.resize(size);
}

void zoArray::remove(const std::string& key)
{
	_table.remove(key);
}

void zoArray::remove(int key) 
{
	_table.remove(key);
}

/////////////////////////////////////////////////////////////////////

const char *format_real(real_t d, char *buf)
{
	if (d == 0) {
		sprintf(buf, "0.0");
		return buf;
	}
	double a = fabs(d);
	if (a > 1.e12) {
		sprintf(buf, "%1.12e", d);
		return buf;
	}
	for (int i = 1; i <= 12; i++) {
		a *= 10.0;
		if (a == floor(a)) {
			char fmt[32];
			sprintf(fmt, "%%0.%df", i);
			sprintf(buf, fmt, d);
			return buf;
		}
	}
	sprintf(buf, "%1.12e", d);
	return buf;
}

const char* zs_object_str(const zoObject *o, char* buf)
{
	switch (o->type()) {
	case ZO_NULL:
		sprintf(buf, "null");
		break;
	case ZO_INTEGER:
		sprintf(buf, "%d", ((const zoNumber*)o)->getInteger());
		break;
	case ZO_REAL:
		sprintf(buf, "%.9g", ((const zoNumber*)o)->getReal());
		break;
	case ZO_STRING:
		sprintf(buf, "srting");
		return ((zoString*)o)->get().c_str();
		break;
	case ZO_CLASS:
		sprintf(buf, "<%p:CLASS>", o);
		break;
	case ZO_ARRAY:
		sprintf(buf, "<%p:ARRAY>", o);
		break;
	default:
		sprintf(buf, "<%p:USER>", o);
		break;
    }
	return buf;
}


inline real_t check_number(zsContext *ctx, real_t d, int check)
{
	if (check > 0) {
		if (check == 1) {
			if (d < 0) ctx->expr->error("input < 0");
		}
		else {
			if (d <= 0) ctx->expr->error("input <= 0");
		}
	}
	return d;
}


zoObject *zs_apply_func1(zsContext *ctx, ApplyFunc1 f, zoObject *o, int check)
{
	switch (o->type()) {
	case ZO_INTEGER:
		return zs_create_real(ctx, f(check_number(ctx, ((zoNumber*)o)->getInteger(), check)));
	case ZO_REAL:
		return zs_create_real(ctx, f(check_number(ctx, ((zoNumber*)o)->getReal(), check)));
	case ZO_ARRAY:
		{
		zoArray *src = (zoArray*)o;
		int i, n = src->size();
		zoArray *arr = zs_create_array(ctx, n);
		for (i = 0; i < n; i++) {
			zoObject* m = src->get(ctx, i);
			switch (m->type()) {
			case ZO_INTEGER:
				arr->set(i, zs_create_real(0, f(check_number(ctx, ((zoNumber*)m)->getInteger(), check))));
				break;
			case ZO_REAL:
				arr->set(i, zs_create_real(0, f(check_number(ctx, ((zoNumber*)m)->getReal(), check))));
				break;
			case ZO_ARRAY:
				arr->set(i, zs_apply_func1(ctx, f, m, check));
				break;
			default:
				ctx->expr->error("array has non-number object");
			}
		}
		return arr;
		}
	}
	ctx->expr->error("cannot apply math function to the obbject");
	return 0;
}


zoObject *zs_apply_func2(zsContext *ctx, ApplyFunc2 f, zoObject *o1, zoObject *o2)
{
	switch (o1->type()) {
	case ZO_INTEGER:
	case ZO_REAL:
		{
		real_t value;
		if (o1->type() == ZO_INTEGER) {
			value = ((zoNumber*)o1)->getInteger();
		}
		else {
			value = ((zoNumber*)o1)->getReal();
		}
		switch (o2->type()) {
		case ZO_INTEGER:
			return zs_create_real(ctx, f(value, ((zoNumber*)o2)->getInteger()));
		case ZO_REAL:
			return zs_create_real(ctx, f(value, ((zoNumber*)o2)->getReal()));
		case ZO_ARRAY:
			{
			zoArray *src = (zoArray*)o2;
			int i, n = src->size();
			zoArray *arr = zs_create_array(ctx, n);
			for (i = 0; i < n; i++) {
				zoObject* m = src->get(ctx, i);
				switch (m->type()) {
				case ZO_INTEGER:
					arr->set(i, zs_create_real(0, f(value, ((zoNumber*)m)->getInteger())));
					break;
				case ZO_REAL:
					arr->set(i, zs_create_real(0, f(value, ((zoNumber*)m)->getReal())));
					break;
				case ZO_ARRAY:
					arr->set(i, zs_apply_func2(ctx, f, o1, m));
					break;
				default:
					ctx->expr->error("array has non-number object");
				}
			}
			return arr;
			}
		}
		}
		break;
	case ZO_ARRAY:
		{
		zoArray *ar1 = (zoArray*)o1;
		int i, n = ar1->size();
		zoArray *arr = zs_create_array(ctx, n);
		if (o2->type() == ZO_ARRAY) {
			zoArray *ar2 = (zoArray*)o2;
			if (ar2->size() != n) ctx->expr->error("arrays have different sizes");
			for (i = 0; i < n; i++) {
				zoObject* a1 = ar1->get(ctx, i);
				zoObject* a2 = ar2->get(ctx, i);
				arr->set(i, zs_apply_func2(ctx, f, a1, a2));
			}
		}
		else {
			for (i = 0; i < n; i++) {
				zoObject* a1 = ar1->get(ctx, i);
				arr->set(i, zs_apply_func2(ctx, f, a1, o2));
			}
		}
		return arr;
		}
	}
	ctx->expr->error("cannot apply math function to the object");
	return 0;
}
