/*
Copyright (c) 2005, Jiye Zeng (http://www.zegraph.com/)

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

[1] Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer. 

[2] Neither the name of the organization nor the names of its contributors
may be used to endorse or promote products derived from this software
without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef __ZS_HASH__
#define __ZS_HASH__

#include <cstdio>
#include <cstdlib>
#include <vector>
#include <string>
#include "config.h"

//#define N_PRIMES	1000
#define N_PRIMES	17

extern size_t g_primes[N_PRIMES];

typedef void (*zsDeleteFunc)(void*);

//! Hash function borrowed from C-Talk.
inline int string_hash_code(const std::string& name)
{ 
    int h = 0;
    unsigned char* p = (unsigned char*)name.c_str();
    while (*p != 0) { 
		h = h*31 + *p++;
	}
    return h;
}

//! Pointer item.
//
template<class T> class zsPtrItem
{
public:
	zsPtrItem() : _ptr(0), _next(0), _delete(0) { }

	zsPtrItem(const std::string& name, T* ptr, zsDeleteFunc func) : _name(name), _ptr(ptr), _delete(func), _next(0) { }

	~zsPtrItem() { if (_delete && _ptr) _delete(_ptr); }

	std::string				_name;

	T*						_ptr;

	zsPtrItem*				_next;

	zsDeleteFunc			_delete;
};

//! Hash table class for pointers.
//
template<class T> class zsPtrTable
{
public:
	explicit  zsPtrTable(size_t size = DEFAULT_TABLE_SIZE) : _table(0), _nitem(0), _size(0)
	{
		resize(size);
	}

	~zsPtrTable()
	{
		clear();
		free(_table);
	}

	void resize(size_t size)
	{
		if (size == 0) return;
		size_t i;
		for (i = 0; i < N_PRIMES-1; i++) {
			if (g_primes[i] >= size) break;
		}
		_nitem = 0;
		_size = g_primes[i];
		if (_table) free(_table);
		_table = (zsPtrItem<T>**)malloc(sizeof(zsPtrItem<T>*)*_size);
		for (i = 0; i < _size; i++) _table[i] = 0;
	}

	void expand()
	{
		if (_nitem <= _size) return;
		size_t i, size = _size;
		for (i = 0; i < N_PRIMES-1; i++) {
			if (g_primes[i] > _size) {
				size = g_primes[i];
				break;
			}
		}
		if (size <= _size) return;
		zsPtrItem<T> **table = (zsPtrItem<T>**)malloc(sizeof(zsPtrItem<T>*)*size);
		for (i = 0; i < size; i++) table[i] = 0;
		for (i = 0; i < _size; i++) {
			zsPtrItem<T>* p = _table[i];
			while (p != 0) {
				zsPtrItem<T>* tmp = p->_next;
				size_t h = string_hash_code(p->_name);
				size_t j = h % size;
				p->_next = table[j];
				table[j] = p;
				p = tmp;
			}
			_table[i] = 0;
		}
		free(_table);
		_table = table;
		_size = size;
	}

	void add(const std::string& name, T* ptr, zsDeleteFunc func)
	{
		size_t h = string_hash_code(name);
		size_t i = h % _size;
		zsPtrItem<T>* item = new zsPtrItem<T>(name, ptr, func);
		item->_next = _table[i];
		_table[i] = item;
		_nitem++;
	}

	void add(size_t key, T* ptr, zsDeleteFunc func)
	{
		char buf[64];
		sprintf(buf, "%u", key);
		add(buf, ptr, func);
	}

	T* get(const std::string& name) const
	{
		size_t h = string_hash_code(name);
		size_t i = h % _size;
		for (zsPtrItem<T>* p = _table[i]; p != 0; p = p->_next) {
			if (p->_name == name) return p->_ptr; 
		}
		return 0;
	}

	T* get(size_t key) const
	{
		char buf[64];
		sprintf(buf, "%u", key);
		return get(buf);
	}

	void get(std::vector<std::string>& names) const
	{
		for (size_t i = 0; i < _size; i++) {
			for (zsPtrItem<T>* p = _table[i]; p != 0; p = p->_next) {
				names.push_back(p->_name);
			}
		}
	}

	void get(std::vector<T*>& ptr) const
	{
		for (size_t i = 0; i < _size; i++) {
			for (zsPtrItem<T>* p = _table[i]; p != 0; p = p->_next) {
				ptr.push_back(p->_ptr);
			}
		}
	}

	size_t remove(const std::string& name)
	{
		size_t h = string_hash_code(name);
		size_t i = h % _size;
		zsPtrItem<T> *p    = _table[i];
		zsPtrItem<T> *prev;
		while (p != 0) {
			if (p->_name == name) {
				if (p == _table[i]) {
					_table[i] = p->_next;
					delete p;
				}
				else {
					prev->_next = p->_next;
					delete p;
				}
				_nitem--;
				return i;
			}
			prev = p;
			p = p->_next;
		}
		return i;
	}

	size_t remove(size_t key)
	{
		char buf[64];
		sprintf(buf, "%d", key);
		return remove(buf);
	}

	void set(const std::string& name, T* ptr, zsDeleteFunc func)
	{
		size_t i = remove(name);
		zsPtrItem<T>* item = new zsPtrItem<T>(name, ptr, func);
		item->_next = _table[i];
		_table[i] = item;
		_nitem++;
	}

	void set(size_t key, T* ptr, zsDeleteFunc func)
	{
		char buf[64];
		sprintf(buf, "%u", key);
		set(buf, ptr, func);
	}

	void clear()
	{
		if (_nitem == 0) return;
		for (size_t i = 0; i < _size; i++) {
			zsPtrItem<T>* p = _table[i];
			while (p != 0) {
				zsPtrItem<T>* tmp = p->_next;
				delete p;
				p = tmp;
			}
			_table[i] = 0;
		}
		_nitem = 0;
	}

	size_t size() const { return _nitem; }

	size_t capacity() const { return _size; }

private:

	size_t _nitem, _size;

	zsPtrItem<T>** _table;
};


//! Object item
//
template<class T> class zsObjItem
{
public:
	zsObjItem() : _next(0) { }

	zsObjItem(const std::string& name, const T &obj) : _name(name), _obj(obj), _next(0) { }

	~zsObjItem() { }

	std::string				_name;

	T						_obj;

	zsObjItem*				_next;
};

//! Hash table for objects
//
template<class T> class zsObjTable
{
public:
	explicit zsObjTable(size_t size = DEFAULT_TABLE_SIZE) : _table(0), _nitem(0), _size(0)
	{
		resize(size);
	}

	~zsObjTable() {
		clear();
		free(_table);
	}

	void resize(size_t size)
	{
		if (size == 0) return;
		size_t i;
		for (i = 0; i < N_PRIMES-1; i++) {
			if (g_primes[i] >= size) break;
		}
		_nitem = 0;
		_size = g_primes[i];
		if (_table) free(_table);
		_table = (zsObjItem<T>**)malloc(sizeof(zsObjItem<T>*)*_size);
		for (i = 0; i < _size; i++) _table[i] = 0;
	}

	void expand()
	{
		if (_nitem <= _size) return;
		size_t i, size = _size;
		for (i = 0; i < N_PRIMES-1; i++) {
			if (g_primes[i] > _size) {
				size = g_primes[i];
				break;
			}
		}
		if (size <= _size) return;
		zsObjItem<T> **table = (zsObjItem<T>**)malloc(sizeof(zsObjItem<T>*)*size);
		for (i = 0; i < size; i++) table[i] = 0;
		for (i = 0; i < _size; i++) {
			zsObjItem<T>* p = _table[i];
			while (p != 0) {
				zsObjItem<T>* tmp = p->_next;
				size_t h = string_hash_code(p->_name);
				size_t j = h % size;
				p->_next = table[j];
				table[j] = p;
				p = tmp;
			}
			_table[i] = 0;
		}
		free(_table);
		_table = table;
		_size = size;
	}

	void add(const std::string& name, const T& obj)
	{
		size_t h = string_hash_code(name);
		size_t i = h % _size;
		zsObjItem<T>* item = new zsObjItem<T>(name, obj);
		item->_next = _table[i];
		_table[i] = item;
		_nitem++;
	}

	void add(size_t key, const T &obj)
	{
		char buf[64];
		sprintf(buf, "%u", key);
		add(buf, obj);
	}

	zsObjItem<T>* get(const std::string& name) const
	{
		size_t h = string_hash_code(name);
		size_t i = h % _size;
		for (zsObjItem<T>* p = _table[i]; p != 0; p = p->_next) {
			if (p->_name == name) return p; 
		}
		return 0;
	}

	zsObjItem<T>* get(size_t key) const
	{
		char buf[64];
		sprintf(buf, "%u", key);
		return get(buf);
	}

	void get(std::vector<std::string>& names) const
	{
		for (size_t i = 0; i < _size; i++) {
			for (zsObjItem<T>* p = _table[i]; p != 0; p = p->_next) {
				names.push_back(p->_name);
			}
		}
	}

	void get(std::vector<T>& objs) const
	{
		for (size_t i = 0; i < _size; i++) {
			for (zsObjItem<T>* p = _table[i]; p != 0; p = p->_next) {
				objs.push_back(p->_obj);
			}
		}
	}

	size_t remove(const std::string& name)
	{
		size_t h = string_hash_code(name);
		size_t i = h % _size;
		zsObjItem<T> *p    = _table[i];
		zsObjItem<T> *prev;
		while (p != 0) {
			if (p->_name == name) {
				if (p == _table[i]) {
					_table[i] = p->_next;
					delete p;
				}
				else {
					prev->_next = p->_next;
					delete p;
				}
				_nitem--;
				return i;
			}
			prev = p;
			p = p->_next;
		}
		return i;
	}

	size_t remove(size_t key)
	{
		char buf[64];
		sprintf(buf, "%d", key);
		return remove(buf);
	}

	void set(const std::string& name, const T& obj)
	{
		size_t i = remove(name);
		zsObjItem<T>* item = new zsObjItem<T>(name, obj);
		item->_next = _table[i];
		_table[i] = item;
		_nitem++;
	}

	void set(size_t key, const T &obj)
	{
		char buf[64];
		sprintf(buf, "%u", key);
		set(buf, obj);
	}

	void clear()
	{
		if (_nitem == 0) return;
		for (size_t i = 0; i < _size; i++) {
			zsObjItem<T>* p = _table[i];
			while (p != 0) {
				zsObjItem<T>* tmp = p->_next;
				delete p;
				p = tmp;
			}
			_table[i] = 0;
		}
		_nitem = 0;
	}

	size_t size() const { return _nitem; }

	size_t capacity() const { return _size; }

private:

	size_t _nitem, _size;

	zsObjItem<T>** _table;
};


#endif	// __ZS_HASH__
