/*
Copyright (c) 2005, Jiye Zeng (http://www.zegraph.com/)

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

[1] Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer. 

[2] Neither the name of the organization nor the names of its contributors
may be used to endorse or promote products derived from this software
without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef __ZS_COMPILER__
#define __ZS_COMPILER__

#include "object.h"
#include "token.h"
#include "scan.h"
#include <stack>

enum {
	EXPR_CONTEXT,
	EXPR_BASE,
	EXPR_END,
	EXPR_NEW,
	EXPR_BREAK,
	EXPR_CONTINUE,
	EXPR_SIMPLE,
	EXPR_RANGE,
	EXPR_OPTION,
	EXPR_SET,
	EXPR_MEMBER,
	EXPR_METHOD,
	EXPR_RETURN,
	EXPR_CASE,
	EXPR_DEFAULT,
	EXPR_GOTO,
	EXPR_BLOCK,
	EXPR_SWITCH,
	EXPR_ARRAY,
	EXPR_ACCESS,
	EXPR_CALL,
	EXPR_WHILE,
	EXPR_FOR,
	EXPR_IF,
	EXPR_TRY,
	EXPR_FUNC,
	EXPR_CLASS,
	EXPR_MODULE
};

class zsFuncExpr;
class zsModuleExpr;
class zsCallExpr;
class zsDllLibs;
class zsRegPrimitive;
class zsMutex;

//! Context for expression excution. 
/*!
 It contains objects of variables created by expressions. Context is managed by module.
 A class object also has its own context. When a function is called, a new context is
 created for holding objects created by expressions in the function. When an expression
 is executed, it can access objects in the context of fucntion, class, or module that
 owns it.
*/
class zsContext
{
public:
	explicit zsContext(zsContext *ctx) : _ctx(ctx), _type(EXPR_CONTEXT) { expr = 0; }

	~zsContext() { }

	//! Peek variable by name. Retrun NULL if the variable does not exist.
	zoObject* peekVar(const std::string& name);

	//! Get variable by name. Throw error if the variable does not exist.
	zoObject* getVar(const std::string& name, bool global);

	//! Set object to variable. If global=true, the variable must exist.
	void setVar(const std::string& name, zoObject* o, bool global);

	//! Set object to variable. (For API function call.)
	void setObj(const std::string& name, zoObject* o);

	//! Save variables file. (For debugging.)
	void saveVars(FILE *f) const;

	//! Set return object.
	void setReturn(zoObject *ret);
	
	//! Get return object.
	zoObject *getReturn() { return _ret.get(); }
	
	//! Temporally keep newly created object
	void keep(zoObject *o) { _tmpvars.push(zsRCP<zoObject>(o)); }

	//! clear temporal objects
	void clear();

	//! Keep track of the expression that uses the context.
	zsBaseExpr *expr;

	//! Access object table.
	const zsObjTable<zsRCP<zoObject> > &vars() const { return _vars; }
	      zsObjTable<zsRCP<zoObject> > &vars()       { return _vars; }

	//! Set parent context.
	void setContext(zsContext *ctx);

	//! Get parent context.
	zsContext *getContext() { return _ctx; }

	//! For type checking.
	int type() const { return _type; }

private:
	// get global variable by name
	zoObject* getGlobalVar(const std::string& name, zsBaseExpr *caller);

	// set global variable by name
	void setGlobalVar(const std::string& name, zoObject* o, zsBaseExpr *caller);

private:
	zsContext							*_ctx;		// context of the owner of expr

	zsObjTable<zsRCP<zoObject> >		_vars;		// variable object table

	zsRCP<zoObject>						_ret;		// return object

	std::stack<zsRCP<zoObject> >		_tmpvars;	// temporal objects 

	int									_type;		// for type checking by API
};


//! Base expression representing a constant or variable.
//
class zsBaseExpr
{
public:
	//! Constructor.
	explicit zsBaseExpr(zsBaseExpr *expr);

	//! Destructor.
	virtual ~zsBaseExpr() { }

	//! Evaluation.
	virtual zoObject* exec(zsContext *ctx);

	//! Parsing.
	virtual void parse(int &idx);

	//! To be used by block expression.
	virtual void addExpr(zsBaseExpr *expr) { _toks->error(_idx, 0); }

	//! Report error.
	void error(const char *msg=0, const char *msg2=0) const {  _toks->error(_idx, msg, msg2); }

	//! Find the owner module.
	zsModuleExpr* module();

	//! Member access functions
	
	int index() const { return _idx;  }

	int type() const { return _type; }

	int id() const { return _toks->id(_idx); }

	const std::string& name() const { return _toks->str(_idx); }

	const zsTokens *tokens() const { return _toks; } 
	      zsTokens *tokens()       { return _toks; } 

	const zsFuncExpr* func() const { return _owner; }
	      zsFuncExpr* func()       { return _owner; }

	bool global() const { return _toks->id(_idx-1) == TOK_GLOBAL; }

protected:

	int					_type;      // expression type

	int					_idx;		// token index

	zsTokens*			_toks;		// tokens

	zsFuncExpr*			_owner;		// owner of this expression
};

//! Simple expression representing operation, e.g., a+b;
//
class zsSimpleExpr : public zsBaseExpr
{
public:
	//! Constructor.
	zsSimpleExpr(zsBaseExpr* left, int op, zsBaseExpr* right);

	//! Destructor.
	~zsSimpleExpr();

	//! Evaluation.
	zoObject* exec(zsContext *ctx);

	//! Member access functions

	const zsBaseExpr *left() const { return _left;  }
	      zsBaseExpr *left()       { return _left;  }

	const zsBaseExpr *right() const { return _right; }
	      zsBaseExpr *right()       { return _right; }

	int op() const { return _op; }

private:

	int			_op;				// operator ID

	zsBaseExpr	*_left, *_right;	// operands
};

//! Range expression, e.g., a:b;
//
class zsRangeExpr : public zsBaseExpr
{
public:
	//! Constructor.
	zsRangeExpr(zsBaseExpr* left, zsBaseExpr* right);

	//! Destructor.
	~zsRangeExpr();

	//! Evaluation.
	zoObject* exec(zsContext *ctx);

	//! Evaluate range.
	bool getRange(zsContext *ctx, integer_t ir[], real_t rr[], integer_t last); 

	//! Member access functions.
	const zsBaseExpr *left() const { return _left;  }
	      zsBaseExpr *left()       { return _left;  }

	const zsBaseExpr *right() const { return _right; }
	      zsBaseExpr *right()       { return _right; }

private:

	zsBaseExpr	*_left,	*_right;	// operands of :
};

//! Option expression, e.g., a > b ? a : b;
//
class zsOptionExpr : public zsBaseExpr
{
public:
	//! Constructor.
	zsOptionExpr(zsBaseExpr* left, zsBaseExpr* right);

	//! Destructor.
	~zsOptionExpr();

	//! Evaluation.
	zoObject* exec(zsContext *ctx);

private:

	zsBaseExpr	*_left, *_right;	// operands of ?
};

//! Set expression, e.g., a = b; a.b = c; a.b[i] = c;  a = b = c = d;
//
class zsSetExpr : public zsBaseExpr
{
public:
	//! Constructor.
	zsSetExpr(zsBaseExpr* left, zsBaseExpr* right);

	//! Destructor.
	~zsSetExpr();

	//! Evaluation.
	zoObject* exec(zsContext *ctx);

	//! For setting a = b = c = ...;
	void set(zsContext *ctx, zoObject *right);

	//! Member access functions

	const zsBaseExpr *left() const { return _left;  }
	      zsBaseExpr *left()       { return _left;  }

	const zsBaseExpr *right() const { return _right; }
	      zsBaseExpr *right()       { return _right; }

private:

	zsBaseExpr	*_left, *_right;	// operands of =
};

//! Member expression, e.g., a.b; a.b[i]; 
//
class zsMemberExpr : public zsBaseExpr
{
public:
	//! Constructor.
	zsMemberExpr(zsBaseExpr* left, zsBaseExpr* right);

	//! Destructor.
	~zsMemberExpr();

	//! Evaluation.
	zoObject* exec(zsContext *ctx);

	//! Member access functions.
	const zsBaseExpr *left() const { return _left;  }
	      zsBaseExpr *left()       { return _left;  }

	const zsBaseExpr *right() const { return _right; }
	      zsBaseExpr *right()       { return _right; }

private:

	zsBaseExpr	*_left, *_right;	// operands of .
};

//! Method expression, e.g., a.f(); 
//
class zsMethodExpr : public zsBaseExpr
{
public:
	//! Constructor.
	zsMethodExpr(zsBaseExpr* left, zsCallExpr* right);

	//! Destructor.
	~zsMethodExpr();

	//! Evaluation.
	zoObject* exec(zsContext *ctx);

private:

	zsBaseExpr	*_left;		// left of .

	zsCallExpr	*_right;	// right of .

	int					_otype;		// object type

	zsFuncExpr*			_f;		// script function

	zsPrimitiveFunc		_prim;		// primitive function
};

//! End expression, e.g., expr;
//
class zsEndExpr : public zsBaseExpr
{
public:
	//! Constructor.
	zsEndExpr(zsBaseExpr *expr, int &idx);

	//! Destructor.
	~zsEndExpr() { }

	//! Evaluation.
	zoObject* exec(zsContext *ctx);
};

//! New expression, e.g., a = new aClass;
//
class zsNewExpr : public zsBaseExpr
{
public:
	//! Constructor.
	zsNewExpr(zsBaseExpr *expr, int &idx);

	//! Destructor.
	~zsNewExpr() { }

	//! Evaluation.
	zoObject* exec(zsContext *ctx);

private:

	zsClassExpr*		_cls;	// the class expression
};

//! Break expression, e.g., break;
//
class zsBreakExpr : public zsBaseExpr
{
public:
	//! Constructor.
	zsBreakExpr(zsBaseExpr *expr, int &idx);

	//! Destructor.
	~zsBreakExpr() { }

	//! Evaluation.
	zoObject* exec(zsContext *ctx);
};

//! Continue expression, e.g., continue;
//
class zsContinueExpr : public zsBaseExpr
{
public:
	//! Constructor.
	explicit zsContinueExpr(zsBaseExpr *expr, int &idx);

	//! Destructor.
	~zsContinueExpr() { }

	//! Evaluation.
	zoObject* exec(zsContext *ctx);
};

//! Return expression, e.g., return; return a+b;
//
class zsReturnExpr : public zsBaseExpr
{
public:
	//! Constructor.
	zsReturnExpr(zsBaseExpr *expr);

	//! Destructor.
	~zsReturnExpr();

	//! Evaluation.
	zoObject* exec(zsContext *ctx);

	//! To be called by derived class.
	void parse(int &idx);

private:

	zsBaseExpr*		_expr;		// what to return
};

// !Case expression, e.g., case 1: ...
//
class zsCaseExpr : public zsBaseExpr
{
public:
	//! Constructor.
	zsCaseExpr(zsBaseExpr *expr, int &idx);

	//! Destructor.
	~zsCaseExpr() { }

	//! Evaluation.
	zoObject* exec(zsContext *ctx);
};


//! Default case expression, e.g., default: ...
//
class zsDefaultExpr : public zsBaseExpr
{
public:
	//! Constructor.
	zsDefaultExpr(zsBaseExpr *expr, int &idx);

	//! Destructor.
	~zsDefaultExpr() { }

	//! Evaluation.
	zoObject* exec(zsContext *ctx);
};


//! Go to expression, e.g., goto 300;
//
class zsGotoExpr : public zsBaseExpr
{
public:
	//! Constructor.
	zsGotoExpr(zsBaseExpr *expr, int &idx);

	//! Destructor.
	~zsGotoExpr() { }

	//! Evaluation.
	zoObject* exec(zsContext *ctx);
};

//! Block expression ([...], {...}, (...), etc).
//
class zsBlockExpr : public zsBaseExpr
{
public:
	//! Constructor.
	zsBlockExpr(zsBaseExpr *expr);

	//! Destructor.
	~zsBlockExpr();

	//! Evaluation.
	zoObject* exec(zsContext *ctx) { return execFrom(ctx, 0); }

	zoObject* execFrom(zsContext *ctx, int idx);

	//! Parsing a block of tokens.
	void parse(int &idx) { parseBlock(idx, '{', '}'); }

	//! Parsing a block of tokens.
	void parseBlock(int &idx, int opening, int closure);

	//! Add expression.
	void addExpr(zsBaseExpr *expr);

	//! member access functions
	int size() const { return _block.size(); }

	zsBaseExpr* getExpr(int idx) { return _block[idx]; }

protected:

	std::vector<zsBaseExpr*>	_block;		// expresions
};

//! Switch expression, e.g., swith (...) { case 1: ...}
//
class zsSwitchExpr : public zsBlockExpr
{
public:
	//! Constructor.
	zsSwitchExpr(zsBaseExpr *expr);

	//! Destructor.
	~zsSwitchExpr();

	//! Evaluation.
	zoObject* exec(zsContext *ctx);

	//! Parsing.
	void parse(int &idx);

	//! Overide block function.
	void addExpr(zsBaseExpr* expr);

	//! Find "case".
	int findCase(zsContext *ctx, const zoObject *cond);

private:

	std::vector<int>			_marks;		// block expr marks ;

	std::vector<zsBaseExpr*>	_cases;		// case 1: ...

	zsBaseExpr*					_expr;		// switch argument

	zsContext*					_ctx;		// speed up execution for constant cases
};

//! Array expression, e.g., [1, 2, "key"=value, ...];
//
class zsArrayExpr : public zsBlockExpr
{
public:
	//! Constructor.
	zsArrayExpr(zsBaseExpr *expr);

	//! Destructor.
	~zsArrayExpr() { }

	//! Parsing
	void parse(int &idx);

	//! Evaluation.
	zoObject* exec(zsContext *ctx);
};

//! Array/class/user access expression, e.g., a[1]; or a["key"];
//
class zsAccessExpr : public zsBlockExpr
{
public:
	//! Constructor.
	zsAccessExpr(zsBaseExpr *expr);

	//! Destructor.
	~zsAccessExpr() { }

	//! Parsing.
	void parse(int &idx);

	//! Evaluation.
	zoObject* exec(zsContext *ctx);
};

//! Function call expression, e.g., func(...).
//
class zsCallExpr : public zsBlockExpr
{
public:
	//! Constructor.
	zsCallExpr(zsBaseExpr *expr);

	//! Destructor.
	~zsCallExpr() { }

	//! Evaluation.
	zoObject* exec(zsContext *ctx);

	//! Parsing.
	void parse(int &idx);

private:

	zsFuncExpr*			_f;		// script function

	zsPrimitiveFunc		_prim;	// primitive function
};

//! Try-catch expression, e.g., try {...} catch(error) {...}
//
class zsTryExpr : public zsBlockExpr
{
public:
	//! Constructor.
	zsTryExpr(zsBaseExpr *expr);

	//! Destructor.
	~zsTryExpr() { }

	//! Evaluation.
	zoObject* exec(zsContext *ctx);

	//! Parsing
	void parse(int &idx);

private:

	int				_err;		// token index of expression that throws the error

	zsBlockExpr		_catch;		// catch block
};

//! While/Do expression, e.g., while(...) {...} or do {...} while (..).
//
class zsWhileExpr : public zsBlockExpr
{
public:
	//! Constructor.
	zsWhileExpr(zsBaseExpr *expr);

	//! Destructor.
	~zsWhileExpr();

	//! Evaluation.
	zoObject* exec(zsContext *ctx);

	//! Parsing.
	void parse(int &idx);

private:

	zsBaseExpr*		_cond;		// condition check
};

//! For expression, e.g., for(i=0; i<n; i++) {...}
//
class zsForExpr : public zsBlockExpr
{
public:
	//! Constructor.
	zsForExpr(zsBaseExpr *expr);

	//! Destructor.
	~zsForExpr();

	//! Evaluation.
	zoObject* exec(zsContext *ctx);

	//! Parsing.
	void parse(int &idx);

private:

	zsBaseExpr	*_expr1;		// initialize

	zsBaseExpr  *_expr2;		// contidtion check

	zsBaseExpr  *_expr3;		// post evaluation
};

// If expression, e.g., if (...) {...} else if (...) {...} else {...}
//
class zsIfExpr : public zsBlockExpr
{
public:
	//! Constructor.
	zsIfExpr(zsBaseExpr *expr);

	//! Destructor.
	~zsIfExpr();
	
	//! Evaluation.
	zoObject* exec(zsContext *ctx);

	//! Parsing.
	void parse(int &idx);

private:
	
	zsBaseExpr*		_cond;		// condition check

	zsIfExpr*		_next;		// else-if block
};

//! Function expression, e.g., function (...) { ...}
//
class zsFuncExpr : public zsSwitchExpr
{
public:
	//! Constructor.
	zsFuncExpr(zsBaseExpr *expr);

	//! Destructor.
	~zsFuncExpr() { }

	//! Evaluation.
	zoObject* exec(zsContext *ctx);

	//! Parsing.
	void parse(int &idx);

	//! Call this function.
	zoObject* call(zsContext *ctx, zsBlockExpr *args, zoObject *caller);

	zoObject* call(zsContext *ctx, int nargs, zoObject **args);

	//! Add class or function expression to this expression.
	void addFuncCls(zsBaseExpr *expr) { _funcls.addExpr(expr); }

	//! Find function added to this expression
	zsFuncExpr* getFunc(const std::string& name, zsBaseExpr *expr, bool global);

	//! Find class added to this expression.
	zsClassExpr* getClass(const std::string& name, zsBaseExpr *expr, bool global);

	//! Access function expressions
	zsBlockExpr &funcls() { return _funcls; }

private:
	/// initialize argument defaults
	void init(zsContext *ctx, zsContext &ctx2);

	zsBlockExpr		_args;		// function arguments

	zsBlockExpr		_funcls;	// class and functions defined internally
};

//! Class expression, e.g., class ClassName { ...}
/*!
 A class is like a module, but requires instantiation. 
*/
class zsClassExpr : public zsFuncExpr
{
public:
	//! Constructor.
	zsClassExpr(zsBaseExpr *expr);

	//! Destructor.
	~zsClassExpr() { }

	//! Evaluation.
	zoObject* exec(zsContext *ctx);

	//! Parsing.
	void parse(int &idx);

	//! Assert exclusive, i.e., the class does not create itself.
	bool exclusive(const std::string &name);
};

//! Module expression.
/*!
 A module is like a function without argument, but has a context for objects of global variables.
*/
class zsModuleExpr : public zsFuncExpr
{
public:
	//! Constructor.
	zsModuleExpr();

	//! Destructor.
	~zsModuleExpr();

	//! Module initialization.
	zoObject* exec(zsContext *ctx);

	//! Parse file or string.
	bool import(zsModuleExpr *importer, const std::string& file, bool codestring, const char *path=0, bool local=false);

	//! This should never be called
	void parse(int &idx) { throw "Should not be called."; }

	const zsPtrTable<zsModuleExpr> &modules() const { return _modules; }
	      zsPtrTable<zsModuleExpr> &modules()       { return _modules; }

	const char *path() const { return _path; } 
	      char *path()       { return _path; } 

	const zsPtrTable<zsModuleExpr> *imported() const { return _imported; } 
	      zsPtrTable<zsModuleExpr> *imported()       { return _imported; } 

	const zsContext *context() const { return &_ctx; }
	      zsContext *context()       { return &_ctx; }

	bool root() const { return _root; }

public:

	// Global variables
	static zsObjTable<zsPrimitiveFunc> g_primitives;
//	static zsRCP<zoObject> g_null;
	static zsDllLibs g_dlls;
	static zsRegPrimitive sys_register;
	static zsMutex g_mutex;
	static bool g_mutex_flag;

private:

	bool							_root;			// root module flag

	bool							_status;		// initialize status

	zsContext						_ctx;			// module context

	zsPtrTable<zsModuleExpr>		_modules;		// local table for imported modules
	zsPtrTable<zsModuleExpr>		*_imported;		// master table for imported modules

	char							*_path;			// physical path of root module
};


// Add a primitive function to the global table 
void zs_add_primitive(const std::string& name, int type, zsPrimitiveFunc func);

// Find a primitive function in the global table 
zsPrimitiveFunc zs_get_primitive(const std::string& name, int type, zsBaseExpr *expr);


#endif
