/*
Copyright (c) 2005, Jiye Zeng (http://www.zegraph.com/)

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

[1] Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer. 

[2] Neither the name of the organization nor the names of its contributors
may be used to endorse or promote products derived from this software
without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "config.h"
#include "parse.h"
#include "compiler.h"
#include "pool.h"
#include "syslib.h"
#include "thread.h"
#include <cmath>
#include <cstring>
#include <cassert>
#include <iostream>

#define MIN(a,b) ((a)<(b)?(a):(b))

// primitive functions
//static zsObjTable<zsPrimitiveFunc> g_primitives(PRIMITIVE_TABLE_SIZE);
zsObjTable<zsPrimitiveFunc> zsModuleExpr::g_primitives(PRIMITIVE_TABLE_SIZE);

// use global null to save memory
//static zsRCP<zoObject> g_null(zs_create_null(0));
//zsRCP<zoObject> zsModuleExpr::g_null(zs_create_null(0));

zsDllLibs zsModuleExpr::g_dlls;
zsRegPrimitive zsModuleExpr::sys_register;

// mutex for thread safe
zsMutex zsModuleExpr::g_mutex;
bool zsModuleExpr::g_mutex_flag = false;

/////////////////////////////////////////////////////////////////////

// Code based on http://bdn.borland.com/article/17203
static integer_t hextoi(const char *hex)
{
	int n = 0;         // position in string
	int m = 0;         // position in digit[] to shift
	int count;         // loop index
	integer_t intValue = 0;  // integer value of hex string
	int digit[9];      // hold values to convert
	int len = strlen(hex);
#ifdef USE_INT64
	len = len > 8 ? 8 : len;
#else
	len = len > 4 ? 4 : len;
#endif
	while (n < len) {
		if (hex[n]=='\0') break;
		if (hex[n] > 0x29 && hex[n] < 0x40 )		//if 0 to 9
			digit[n] = hex[n] & 0x0f;				//convert to int
		else if (hex[n] >='a' && hex[n] <= 'f')		//if a to f
			digit[n] = (hex[n] & 0x0f) + 9;			//convert to int
		else if (hex[n] >='A' && hex[n] <= 'F')		//if A to F
			digit[n] = (hex[n] & 0x0f) + 9;			//convert to int
		else
			break;
		n++;
	}
	count = n;
	m = n - 1;
	n = 0;
	while(n < count) {
		// digit[n] is value of hex digit at position n
		// (m << 2) is the number of positions to shift
		// OR the bits into return value
		intValue = intValue | (digit[n] << (m << 2));
		m--;   // adjust the position to set
		n++;   // next digit to process
	}
	return (intValue);
}

void zs_add_primitive(const std::string& name, int type, zsPrimitiveFunc func)
{
	char buf[64];
	type = type <= ZO_USER ? type : type+ZO_USER;
	sprintf(buf, "%d", type);
	zsAutoLock lock(zsModuleExpr::g_mutex, zsModuleExpr::g_mutex_flag);
	zsModuleExpr::g_primitives.set(name+buf, func);
	zsModuleExpr::g_primitives.expand();
}

zsPrimitiveFunc zs_get_primitive(const std::string& name, int type, zsBaseExpr *expr)
{
	char buf[64];
	sprintf(buf, "%d", type);
	zsObjItem<zsPrimitiveFunc>* t = zsModuleExpr::g_primitives.get(name+buf);
	if (t != 0) return t->_obj;
	if (expr) expr->error("primitive function undefined");
	return 0;
}

static bool zs_check_cond(const zoObject *o)
{
	if (!o) return false;
	if (o->type() == ZO_NULL) return false;
	if (o->type() == ZO_INTEGER) return (((zoNumber*)o)->getInteger() != 0);
	return false;
}

static zoObject* zs_call_primitive(zsContext *ctx, zsPrimitiveFunc prim, zsBlockExpr *block, zoObject *caller)
{
	std::stack<zsRCP<zoObject> > tmp;
	std::vector<zoObject*> args;
	if (caller) args.push_back(caller);
	for (int i = 0; i < block->size(); i++) {
		zoObject* o = block->getExpr(i)->exec(ctx);
		if (!o) {
//			o = zsModuleExpr::g_null.get();
			o = zs_create_null(ctx);
		}
		else {
			tmp.push(zsRCP<zoObject>(o));
		}
		args.push_back(o);
	}
	ctx->expr = block;
	zoObject *ret = 0;
	if (args.size() > 0)
		ret = prim(ctx, args.size(), &args[0]);
	else
		ret = prim(ctx, 0, 0);
//	return (ret?ret:zsModuleExpr::g_null.get());
	return (ret?ret:zs_create_null(ctx));
}

static void zs_set_block(zsContext *ctx, zsBlockExpr *block, zoObject *obj, zoObject *right)
{
	// obj[...] = right
	real_t rr[3];
	integer_t j, jr[3];
	if (obj->type() > ZO_USER) {
		std::vector<zoObject*> args;
		args.push_back(obj);
		for (int i = 0; i < block->size(); i++) {
			args.push_back(block->getExpr(i)->exec(ctx));
		}
		args.push_back(right);
		ctx->expr = block;
		((zoUser*)obj)->set(ctx, args.size(), &args[0]);
		return;
	}
	if (obj->type() == ZO_ARRAY) {
		zoArray *arr = (zoArray*)obj;
		if (block->size() == 1 && block->getExpr(0)->type() != EXPR_RANGE) {
//			arr->set(ctx, block->getExpr(0), right);
			arr->set(ctx, block->getExpr(0), right->copy(ctx));
			return;
		}
		if (right->type() != ZO_ARRAY) {
			for (int i = 0; i < block->size(); i++) {
				if (block->getExpr(i)->type() == EXPR_RANGE) {
					((zsRangeExpr*)block->getExpr(i))->getRange(ctx, jr, rr, arr->size());
//					for (j = jr[0]; j <= jr[1]; j += jr[2]) arr->set(j, right);
					for (j = jr[0]; j <= jr[1]; j += jr[2]) arr->set(j, right->copy(ctx));
				}
				else {
//					arr->set(ctx, block->getExpr(i), right);
					arr->set(ctx, block->getExpr(i), right->copy(ctx));
				}
			}
			return;
		}
		zoArray *tmp = (zoArray*)right;
		for (int i = 0; i < block->size(); i++) {
			if (block->getExpr(i)->type() == EXPR_RANGE) {
				((zsRangeExpr*)block->getExpr(i))->getRange(ctx, jr, rr, arr->size());
//				for (j = jr[0]; j <= jr[1]; j += jr[2]) arr->set(j, tmp->get(i));
				for (j = jr[0]; j <= jr[1]; j += jr[2]) arr->set(j, tmp->get(ctx, i)->copy(ctx));
			}
			else {
//				arr->set(ctx, block->getExpr(i), tmp->get(i));
				arr->set(ctx, block->getExpr(i), tmp->get(ctx, i)->copy(ctx));
			}
		}
		return;
	}
	if (obj->type() == ZO_STRING) {
		zoString* sobj = (zoString*)obj;
		const char *msg1 = "index out of range";
		const char *msg2 = "assignment to string accepts only integer or string";
		for (int i = 0; i < block->size(); i++) {
			if (block->getExpr(i)->type() == EXPR_RANGE) {
				((zsRangeExpr*)block->getExpr(i))->getRange(ctx, jr, rr, sobj->size());
				for (j = jr[0]; j <= jr[1]; j += jr[2]) {
					if (j < 0 || j >= sobj->size()) block->getExpr(i)->error(msg1);
					if (right->type() == ZO_INTEGER) {
						sobj->set(j, ((zoNumber*)right)->getInteger());
					}
					else if (right->type() == ZO_STRING) {
						const std::string &s = ((zoString*)right)->get();
						sobj->set(j, s.at(i%s.size()));
					}
					else {
						block->error(msg2);
					}
				}
			}
			else {
				zoObject* o = block->getExpr(i)->exec(ctx);
				if (o->type() != ZO_INTEGER) block->getExpr(i)->error("integer index expected");
				int idx = ((zoNumber*)o)->getInteger();
				if (idx < 0 || idx >= sobj->size()) block->getExpr(i)->error(msg1);
				if (right->type() == ZO_INTEGER) {
					sobj->set(idx, ((zoNumber*)right)->getInteger());
				}
				else if (right->type() == ZO_STRING) {
					const std::string &s = ((zoString*)right)->get();
					sobj->set(idx, s.at(i%s.size()));
				}
				else {
					block->error(msg2);
				}
			}
		}
		return;
	}
	block->error("bad array/string/user assignment");
}

static zoObject *zs_get_block(zsContext *ctx, zsBlockExpr *block, zoObject *obj)
{
	// obj[...];
	real_t rr[3];
	integer_t j, jr[3];
	if (obj->type() > ZO_USER) {
		std::vector<zoObject*> args;
		args.push_back(obj);
		for (int i = 0; i < block->size(); i++) {
			args.push_back(block->getExpr(i)->exec(ctx));
		}
		ctx->expr = block;
		return ((zoUser*)obj)->get(ctx, args.size(), &args[0]);
	}
	if (obj->type() == ZO_ARRAY) {
		zoArray *arr = (zoArray*)obj;
		if (block->size() == 1 && block->getExpr(0)->type() != EXPR_RANGE) {
			return arr->get(ctx, block->getExpr(0));
		}
		zoArray *ret = zs_create_array(ctx, block->size());
		for (int i = 0; i < block->size(); i++) {
			if (block->getExpr(i)->type() == EXPR_RANGE) {
				((zsRangeExpr*)block->getExpr(i))->getRange(ctx, jr, rr, arr->size());
				for (j = jr[0]; j <= jr[1]; j += jr[2]) ret->set(i+j, arr->get(ctx, j));
			}
			else {
				ret->set(i, arr->get(ctx, block->getExpr(i)));
			}
		}
		return ret;
	}
	if (obj->type() == ZO_STRING) {
		const char *msg = "index out of range";
		const std::string &r = ((zoString*)obj)->get();
		std::string s;
		for (int i = 0; i < block->size(); i++) {
			if (block->getExpr(i)->type() == EXPR_RANGE) {
				((zsRangeExpr*)block->getExpr(i))->getRange(ctx, jr, rr, r.size());
				for (j = jr[0]; j <= jr[1]; j += jr[2]) {
					if (j < 0 || j >= r.size()) block->getExpr(i)->error(msg);
					s += r.at(j);
				}
			}
			else {
				zoObject* o = block->getExpr(i)->exec(ctx);
				if (o->type() != ZO_INTEGER) block->getExpr(i)->error("integer index expected");
				int idx = ((zoNumber*)o)->getInteger();
				if (idx < 0 || idx >= r.size()) block->getExpr(i)->error(msg);
				s += r.at(idx);
			}
		}
		return zs_create_string(ctx, s);
	}
	block->error("bad array/string/user access");
	return 0;
}

/////////////////////////////////////////////////////////////////////

zoObject* zsContext::peekVar(const std::string& name)
{
	zsObjItem<zsRCP<zoObject> > *item = _vars.get(name);
	if (!item) return 0;
	return item->_obj.get();
}

void zsContext::clear()
{
	zsAutoLock lock(zsModuleExpr::g_mutex, zsModuleExpr::g_mutex_flag);
	while(!_tmpvars.empty()) _tmpvars.pop();
}

zoObject* zsContext::getVar(const std::string& name, bool global)
{
	if (global) {
		if (!_ctx) expr->error("failed to get variable", name.c_str());
		return _ctx->getGlobalVar(name, expr);
	}
	zsObjItem<zsRCP<zoObject> > *item = _vars.get(name);
	if (!item) {
		if (_ctx) return _ctx->getGlobalVar(name, expr);
		expr->error("failed to get variable", name.c_str());
	}
	return item->_obj.get();
}

zoObject* zsContext::getGlobalVar(const std::string& name, zsBaseExpr *caller)
{
	zsObjItem<zsRCP<zoObject> > *item = _vars.get(name);
	if (item) return item->_obj.get();
	if (!_ctx) caller->error("failed to get variable", name.c_str());
	return _ctx->getGlobalVar(name, caller);
}

void zsContext::setVar(const std::string& name, zoObject* o, bool global)
{
	if (!o) {
//		expr->error("set NULL to variable", name.c_str());
		o = zs_create_null(this);
	}
	if (!global) {
		zsAutoLock lock(zsModuleExpr::g_mutex, zsModuleExpr::g_mutex_flag);
		_vars.set(name, zsRCP<zoObject>(o->copy(this)));
	}
	else {
		if (_ctx) {
			_ctx->setGlobalVar(name, o, expr);
		}
		else {
			expr->error("set object to undefined variable", name.c_str());
		}
	}
}

void zsContext::setGlobalVar(const std::string& name, zoObject* o, zsBaseExpr *caller)
{
	zsObjItem<zsRCP<zoObject> > *item = _vars.get(name);
	if (item) {
		zsAutoLock lock(zsModuleExpr::g_mutex, zsModuleExpr::g_mutex_flag);
		_vars.set(name, zsRCP<zoObject>(o->copy(this)));
	}
	else {
		if (_ctx) {
			_ctx->setGlobalVar(name, o, caller);
		}
		else {
			caller->error("set object to undefined variable", name.c_str());
		}
	}
}

void zsContext::setObj(const std::string& name, zoObject* o)
{
	zsAutoLock lock(zsModuleExpr::g_mutex, zsModuleExpr::g_mutex_flag);
	_vars.set(name, zsRCP<zoObject>(o));
}

void zsContext::setReturn(zoObject *ret)
{
	zsAutoLock lock(zsModuleExpr::g_mutex, zsModuleExpr::g_mutex_flag);
	_ret = zsRCP<zoObject>(ret);
}

void zsContext::setContext(zsContext *ctx)
{
	zsAutoLock lock(zsModuleExpr::g_mutex, zsModuleExpr::g_mutex_flag);
	_ctx = ctx;
}

void zsContext::saveVars(FILE *f) const
{
	char buf[64];
	std::vector<std::string> keys;
	_vars.get(keys);
	for (int i = 0; i < keys.size(); i++) {
		zoObject *o = _vars.get(keys[i])->_obj.get();
		fprintf(f, "%s = %s\n", keys[i].c_str(), zs_object_str(o, buf));
	}
}

/////////////////////////////////////////////////////////////////////

zsBaseExpr::zsBaseExpr(zsBaseExpr *expr)
: _type(EXPR_BASE), _idx(0), _toks(0), _owner(0)
{
	if (expr) {
		_toks = expr->tokens();
		_idx = expr->index();
		if (expr->type() == EXPR_FUNC) {
			_owner = (zsFuncExpr*)expr;
		}
		else if (expr->type() == EXPR_CLASS) {
			_owner = (zsClassExpr*)expr;
		}
		else if (expr->type() == EXPR_MODULE) {
			_owner = (zsModuleExpr*)expr;
		}
		else {
			_owner = expr->func();
		}
	}
}

void zsBaseExpr::parse(int &idx)
{
	_idx  = idx++;
	switch (id()) {
	case TOK_NIL:
	case TOK_FALSE:
	case TOK_TRUE:
	case TOK_INTEGER:
	case TOK_HEX:
	case TOK_REAL:
	case TOK_STRING:
	case TOK_ID:
		break;
	default:
		error();
	}
}

zoObject* zsBaseExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	switch (id()) {
	case TOK_NIL:
//		return zsModuleExpr::g_null.get();
		return zs_create_null(ctx);
	case TOK_FALSE:
		return zs_create_integer(ctx, 0);
	case TOK_TRUE:
		return zs_create_integer(ctx, 1);
	case TOK_INTEGER:
#ifdef USE_INT64
		return zs_create_integer(ctx, atoi64(name().c_str()));
#else
		return zs_create_integer(ctx, atoi(name().c_str()));
#endif
	case TOK_HEX:
		return zs_create_integer(ctx, hextoi(name().c_str()+2));
	case TOK_REAL:
		return zs_create_real(ctx, atof(name().c_str()));
	case TOK_STRING:
		return zs_create_string(ctx, name());
	case TOK_ID:
		return ctx->getVar(name(), global());
	default:
		error();
	}
	return 0;
}

zsModuleExpr* zsBaseExpr::module()
{
	zsFuncExpr *f = _owner;
	while (f->type() != EXPR_MODULE) f = f->func();
	return (zsModuleExpr*)f;
}

/////////////////////////////////////////////////////////////////////

zsSimpleExpr::zsSimpleExpr(zsBaseExpr* left, int op, zsBaseExpr* right)
 : zsBaseExpr(left), _left(left), _op(op), _right(right)
{
	_type  = EXPR_SIMPLE;
}

zsSimpleExpr::~zsSimpleExpr()
{
	delete _left;
	delete _right;
}

zoObject* zsSimpleExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	zoObject *left = _left->exec(ctx);
	if (!left) left=zs_create_null(ctx);
	if (_right) {
		return left->opfunc(ctx, _op, _right->exec(ctx));
	}
	else {
		return left->opfunc(ctx, _op, 0);
	}
}

/////////////////////////////////////////////////////////////////////

zsRangeExpr::zsRangeExpr(zsBaseExpr* left, zsBaseExpr* right)
 : zsBaseExpr(right), _left(left), _right(right)
{
	_type  = EXPR_RANGE;
}

zsRangeExpr::~zsRangeExpr()
{
	delete _left;
	delete _right;
}

zoObject* zsRangeExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	zoArray *arr = zs_create_array(ctx, 3);
	if (_left->type() == EXPR_RANGE) {
		zsRangeExpr *expr = (zsRangeExpr*)_left;
		arr->set("0", expr->left()->exec(ctx));
		arr->set("1", expr->right()->exec(ctx));
		arr->set("2", _right->exec(ctx));
	}
	else {
		arr->set("0", _left->exec(ctx));
		arr->set("1", _right->exec(ctx));
	}
	return arr;
}

bool zsRangeExpr::getRange(zsContext *ctx, integer_t ir[], real_t rr[], integer_t last)
{
	const char *msg1 = "One of the range values is not number"; 
	const char *msg2 = "Range step is zero";
	const char *msg3 = "Range values must be integers";
	bool flag = false;
	if (_left->type() == EXPR_RANGE) {
		zsRangeExpr *expr = (zsRangeExpr*)_left;
		zoObject *o = expr->left()->exec(ctx);
		if (o->type() == ZO_INTEGER) {
			ir[0] = ((zoNumber*)o)->getInteger();
		}
		else if (o->type() == ZO_REAL) {
			if (last > 0) error(msg3);
			rr[0] = ((zoNumber*)o)->getReal();
			flag = true;
		}
		else {
			if (last > 0 && o->type() == ZO_NULL)
				ir[0] = 0;
			else
				error(msg1);
		}
		o = expr->right()->exec(ctx);
		if (o->type() == ZO_INTEGER) {
			ir[1] = ((zoNumber*)o)->getInteger();
		}
		else if (o->type() == ZO_REAL) {
			if (last > 0) error(msg3);
			rr[1] = ((zoNumber*)o)->getReal();
			flag = true;
		}
		else {
			if (last > 0 && o->type() == ZO_NULL)
				ir[1] = last - 1;
			else
				error(msg1);
		}
		o = _right->exec(ctx);
		if (o->type() == ZO_INTEGER) {
			ir[2] = ((zoNumber*)o)->getInteger();
			if (ir[2] == 0) error(msg2);
		}
		else if (o->type() == ZO_REAL) {
			if (last > 0) error(msg3);
			rr[2] = ((zoNumber*)o)->getReal();
			if (rr[2] == 0) error(msg2);
			flag = true;
		}
		else {
			error(msg1);
		}
	}
	else {
		zoObject *o = _left->exec(ctx);
		if (o->type() == ZO_INTEGER) {
			ir[0] = ((zoNumber*)o)->getInteger();
		}
		else if (o->type() == ZO_REAL) {
			if (last > 0) error(msg3);
			rr[0] = ((zoNumber*)o)->getReal();
			flag = true;
		}
		else {
			if (last > 0 && o->type() == ZO_NULL)
				ir[0] = 0;
			else
				error(msg1);
		}
		o = _right->exec(ctx);
		if (o->type() == ZO_INTEGER) {
			ir[1] = ((zoNumber*)o)->getInteger();
		}
		else if (o->type() == ZO_REAL) {
			if (last > 0) error(msg3);
			rr[1] = ((zoNumber*)o)->getReal();
			flag = true;
		}
		else {
			if (last > 0 && o->type() == ZO_NULL)
				ir[1] = last-1;
			else
				error(msg1);
		}
		rr[2] = 1;
		ir[2] = 1;
	}
	return flag;
}

/////////////////////////////////////////////////////////////////////

zsOptionExpr::zsOptionExpr(zsBaseExpr* left, zsBaseExpr* right)
 : zsBaseExpr(right), _left(left), _right(right)
{
	if (right->type() != EXPR_RANGE) right->error();
	_type  = EXPR_OPTION;
}

zsOptionExpr::~zsOptionExpr()
{
	delete _left;
	delete _right;
}

zoObject* zsOptionExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	if (zs_check_cond(_left->exec(ctx)))
		return ((zsRangeExpr*)_right)->left()->exec(ctx);
	else
		return ((zsRangeExpr*)_right)->right()->exec(ctx);
}

/////////////////////////////////////////////////////////////////////

zsSetExpr::zsSetExpr(zsBaseExpr* left, zsBaseExpr* right)
 : zsBaseExpr(left), _left(left), _right(right)
{
	_type  = EXPR_SET;
}

zsSetExpr::~zsSetExpr()
{
	delete _left;
	delete _right;
}

zoObject* zsSetExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	zoObject *right = _right->exec(ctx);
	ctx->expr = this;
	switch (_left->type()) {
	case EXPR_BASE:
		// a = expr
		if (id() != TOK_ID) error("variable name expected");
		ctx->setVar(name(), right, global());
		break;
	case EXPR_ARRAY:
		// [...] = expr
		if (right->type() != ZO_ARRAY) {
			zsArrayExpr *block = (zsArrayExpr*)_left;
			for (int i = 0; i < block->size(); i++) {
				zsBaseExpr *expr = block->getExpr(i);
				ctx->setVar(expr->name(), right, expr->global());
			}
		}
		else {
			zsArrayExpr *block = (zsArrayExpr*)_left;
			zoArray *arr = (zoArray*)right;
			for (int i = 0; i < block->size(); i++) {
				zsBaseExpr *expr = block->getExpr(i);
				ctx->setVar(expr->name(), arr->get(ctx, i), expr->global());
			}
		}
		break;
	case EXPR_ACCESS:
		// a[...] = expr
		zs_set_block(ctx, (zsAccessExpr*)_left, ctx->getVar(name(), global()), right);
		break;
	case EXPR_MEMBER:
		{
		zoObject *l = ((zsMemberExpr*)_left)->left()->exec(ctx);
		zsBaseExpr *expr = ((zsMemberExpr*)_left)->right();
		if (expr->type() == EXPR_BASE && expr->id() == TOK_ID) {
			// l.expr = right
			if (l->type() == ZO_ARRAY) {
				((zoArray*)l)->set(expr->name(), right);
			}
			else if (l->type() == ZO_CLASS) {
				((zoClass*)l)->set(expr->name(), right, expr);
			}
			else if (l->type() > ZO_USER) {
				zoObject *p = zs_create_string(ctx, expr->name());
				zoObject *args[3] = { l, p, right };
				((zoUser*)l)->set(ctx, 3, args);
			}
		}
		else if (expr->type() == EXPR_ACCESS) {
			// l.expr[i] = right
			if (l->type() == ZO_ARRAY) {
				zs_set_block(ctx, (zsAccessExpr*)expr, ((zoArray*)l)->get(ctx, expr->name()), right);
			}
			else if (l->type() == ZO_CLASS) {
				zs_set_block(ctx, (zsAccessExpr*)expr, ((zoClass*)l)->get(expr->name(), this), right);
			}
			else if (l->type() > ZO_USER) {
				zoObject *args[2] = { l, zs_create_string(ctx, expr->name()) };
				zs_set_block(ctx, (zsAccessExpr*)expr, ((zoUser*)l)->get(ctx, 2, args), right);
			}
			else {
				error("bad array/class/user member assignment");
			}
		}
		else {
			error("bad array/class/user member assignment");
		}
		}
		break;
	case EXPR_SET:
		// a = b = expr
		((zsSetExpr*)_left)->set(ctx, right);
		break;
	default:
		error("bad assignment");
	}
	return right;
}


void zsSetExpr::set(zsContext *ctx, zoObject *right)
{
	ctx->expr = this;
	if (_right->type() == EXPR_BASE && _right->id() == TOK_ID) {
		ctx->setVar(_right->name(), right, global());
		if (_left->type() == EXPR_SET) {
			((zsSetExpr*)_left)->set(ctx, right);
			return;
		}
		if (_left->type() == EXPR_BASE && _left->id() == TOK_ID) {
			ctx->setVar(name(), right, global());
			return;
		}
	}
	_right->error("bad assignment");
}


/////////////////////////////////////////////////////////////////////

zsMemberExpr::zsMemberExpr(zsBaseExpr* left, zsBaseExpr* right)
 : zsBaseExpr(left), _left(left), _right(right)
{
	_type  = EXPR_MEMBER;
}

zsMemberExpr::~zsMemberExpr()
{
	delete _left;
	delete _right;
}

zoObject* zsMemberExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	zoObject *l = _left->exec(ctx);
	if (_right->type() == EXPR_BASE && _right->id() == TOK_ID) {
		// l.r
		if (l->type() == ZO_ARRAY) return ((zoArray*)l)->get(ctx, _right->name());
		if (l->type() == ZO_CLASS) return ((zoClass*)l)->get(_right->name(), this);
		if (l->type() > ZO_USER) {
			zoObject *args[2] = { l, zs_create_string(ctx, _right->name()) };
			return ((zoUser*)l)->get(ctx, 2, args);
		}
	}
	else if (_right->type() == EXPR_ACCESS) {
		// l.r[...]
		if (l->type() == ZO_ARRAY) return zs_get_block(ctx, (zsAccessExpr*)_right, ((zoArray*)l)->get(ctx, _right->name()));
		if (l->type() == ZO_CLASS) return zs_get_block(ctx, (zsAccessExpr*)_right, ((zoClass*)l)->get(_right->name(), this));
		if (l->type() > ZO_USER) {
			zoObject *args[2] = { l, zs_create_string(ctx, _right->name()) };
			return zs_get_block(ctx, (zsAccessExpr*)_right, ((zoUser*)l)->get(ctx, 2, args));
		}
	}
	error("bad member access");
	return 0;
}

/////////////////////////////////////////////////////////////////////

zsMethodExpr::zsMethodExpr(zsBaseExpr* left, zsCallExpr* right)
 : zsBaseExpr(left), _left(left), _right(right), _f(0), _prim(0), _otype(0)
{
	_type  = EXPR_METHOD;
}

zsMethodExpr::~zsMethodExpr()
{
	delete _left;
	delete _right;
}

zoObject* zsMethodExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	zoObject *left = _left->exec(ctx);
	if (left->type() == ZO_CLASS) {
		return ((zoClass*)left)->call(ctx, _right, this);
	}
	if (_prim == 0 || _otype != left->type()) {
		_otype = left->type();
		_prim = zs_get_primitive(_right->name(), _otype, 0);
	}
	if (!_prim) {
		_f = _owner->getFunc(_right->name(), _right, true);
	}
	if (_prim) {
		return zs_call_primitive(ctx, _prim, _right, left);
	}
	return _f->call(ctx, _right, left);
}

/////////////////////////////////////////////////////////////////////

zsEndExpr::zsEndExpr(zsBaseExpr *expr, int &idx)
: zsBaseExpr(expr)
{
	_type = EXPR_END;
	idx++;
}

zoObject* zsEndExpr::exec(zsContext *ctx)
{
	ctx->clear();
	return 0;
}

/////////////////////////////////////////////////////////////////////

zsNewExpr::zsNewExpr(zsBaseExpr *expr, int &idx)
: zsBaseExpr(expr), _cls(0)
{
	_type = EXPR_NEW;
	_idx = ++idx;
	idx++;
	if (id() != TOK_ID || _toks->id(idx) != ';') error();
}

zoObject* zsNewExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	if (!_cls) {
		_cls = _owner->getClass(name(), this, true);
		if (_cls == _owner) error("self creation of class");
		if (_owner->type() == EXPR_CLASS && !_cls->exclusive(_owner->name())) error("mutual creation of class");
	}
	return zs_create_class(ctx, _cls);
}

/////////////////////////////////////////////////////////////////////

zsBreakExpr::zsBreakExpr(zsBaseExpr *expr, int &idx)
: zsBaseExpr(expr)
{
	_type = EXPR_BREAK;
	_idx  = idx++;
	if (_toks->id(idx) != ';') error();
}

zoObject* zsBreakExpr::exec(zsContext *ctx)
{
	throw zsRCP<zoObject>(zs_create_integer(0, EXPR_BREAK)); 
	return 0;
}

/////////////////////////////////////////////////////////////////////

zsContinueExpr::zsContinueExpr(zsBaseExpr *expr, int &idx)
: zsBaseExpr(expr)
{
	_type = EXPR_CONTINUE;
	_idx = idx++;
	if (_toks->id(idx) != ';') error();
}

zoObject* zsContinueExpr::exec(zsContext *ctx)
{
	throw zsRCP<zoObject>(zs_create_integer(0, EXPR_CONTINUE));
	return 0;
}

/////////////////////////////////////////////////////////////////////

zsReturnExpr::zsReturnExpr(zsBaseExpr *expr)
: zsBaseExpr(expr), _expr(0)
{
	_type = EXPR_RETURN;
}

zsReturnExpr::~zsReturnExpr()
{
	delete _expr;
}

void zsReturnExpr::parse(int &idx)
{
	_idx = idx++;
	if (_toks->id(idx) == ';') return;	// return;
	// return ...;
	_expr = zs_parse(this, idx, 0, ';');
	// leave ; to the care of other exprs
	idx--;
}

zoObject* zsReturnExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	if (_expr) {
		ctx->setReturn(_expr->exec(ctx));
	}
	else {
//		ctx->setReturn(zsModuleExpr::g_null.get());
		ctx->setReturn(zs_create_null(ctx));
	}
	throw zsRCP<zoObject>(zs_create_integer(0, EXPR_RETURN));
	return 0;
}

/////////////////////////////////////////////////////////////////////

zsCaseExpr::zsCaseExpr(zsBaseExpr *expr, int &idx)
: zsBaseExpr(expr)
{
	_type = EXPR_CASE;
	idx++;
	parse(idx);
	if (_toks->id(idx) != ':') error();
	idx++;
}

zoObject* zsCaseExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	zoObject *ret = zsBaseExpr::exec(ctx);
	if (ret->type() != ZO_INTEGER &&
		ret->type() != ZO_STRING) error("case does not evaluate to integer or string");
	return ret;
}

/////////////////////////////////////////////////////////////////////

zsDefaultExpr::zsDefaultExpr(zsBaseExpr *expr, int &idx)
: zsBaseExpr(expr)
{
	_type = EXPR_DEFAULT;
	_idx = idx++;
	if (_toks->id(idx) != ':') error();
	idx++;
}

zoObject* zsDefaultExpr::exec(zsContext *ctx)
{
	// should not be called
	ctx->expr = this;
	error("exec default");
	return 0;
}

/////////////////////////////////////////////////////////////////////

zsGotoExpr::zsGotoExpr(zsBaseExpr *expr, int &idx)
: zsBaseExpr(expr)
{ 
	_type = EXPR_GOTO;
	idx++;
	parse(idx);
	if (_toks->id(idx) != ';') error();
}

zoObject* zsGotoExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	zoObject *ret = zsBaseExpr::exec(ctx);
	if (ret->type() != ZO_INTEGER &&
		ret->type() != ZO_STRING) error("goto does not evaluate to integer or string");
	_owner->execFrom(ctx, _owner->findCase(ctx, ret));
	throw zsRCP<zoObject>(zs_create_integer(0, EXPR_GOTO));
	return 0;
}

/////////////////////////////////////////////////////////////////////

zsBlockExpr::zsBlockExpr(zsBaseExpr *expr)
: zsBaseExpr(expr)
{
	_type = EXPR_BLOCK;
}

zsBlockExpr::~zsBlockExpr()
{
	for (int i = 0; i < _block.size(); i++) {
		delete _block[i];
        _block[i] = 0;
	}
}

void zsBlockExpr::parseBlock(int &idx, int opening, int closure)
{
	zsBaseExpr *expr = zs_parse(this, idx, opening, closure);
	if (expr != 0) _block.push_back(expr);
}

zoObject* zsBlockExpr::execFrom(zsContext *ctx, int n)
{
	ctx->expr = this;
	zoObject *ret;
	for (int i = n; i < _block.size(); i++) {
		ret = _block[i]->exec(ctx);
	}
	return ret;
}

void zsBlockExpr::addExpr(zsBaseExpr *expr)
{
	_block.push_back(expr);
}

/////////////////////////////////////////////////////////////////////

zsSwitchExpr::zsSwitchExpr(zsBaseExpr *expr)
: zsBlockExpr(expr), _expr(0), _ctx(0)
{
	_type = EXPR_SWITCH;
}

zsSwitchExpr::~zsSwitchExpr()
{
	delete _expr;
	delete _ctx;
	for (int i = 0; i < _cases.size(); i++) {
		delete _cases[i];
        _cases[i] = 0;
	}
}

void zsSwitchExpr::addExpr(zsBaseExpr* expr)
{
	if (expr->type() == EXPR_CASE || expr->type() == EXPR_DEFAULT) {
		_cases.push_back(expr);
		_marks.push_back(size());
	}
	else {
		_block.push_back(expr);
	}
}

void zsSwitchExpr::parse(int &idx)
{
	_idx = idx++;
	_expr = zs_parse(this, idx, '(', ')');
	if (!_expr) error("switch has no arguement");
	parseBlock(idx, '{', '}');
	int i, k = -1;
	for (i = 0; i < _cases.size(); i++) {
		if (_cases[i]->id() == TOK_INTEGER || _cases[i]->id() == TOK_STRING) {
			if (!_ctx) _ctx = new zsContext(0);
			_ctx->setVar(_cases[i]->name(), zs_create_integer(0, _marks[i]), false);
		}
		else if (_cases[i]->type() == EXPR_DEFAULT) {
			if (k >= 0) error("duplciated default");
			k = i;
		}
	}
	if (k < 0) error("no default in switch block");
	if (k != _cases.size()-1) error("default must be the last case");
}

zoObject* zsSwitchExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	try {
		zoObject *ret = _expr->exec(ctx);
		if (ret->type() != ZO_INTEGER &&
			ret->type() != ZO_STRING) error("switch does not evaluate to integer or string");
		execFrom(ctx, findCase(ctx, ret));
	}
	catch (zsRCP<zoObject> o) {
		if (((zoNumber*)o.get())->getInteger() != EXPR_BREAK) throw;
	}
	return 0;
}

int zsSwitchExpr::findCase(zsContext *ctx, const zoObject *cond)
{
	if (_ctx) {
		char buf[64];
		zoObject *o = _ctx->peekVar(zs_object_str(cond, buf));
		if (o) return ((zoNumber*)o)->getInteger();
	}
	for (int i = 0; i < _cases.size(); i++) {
		if (_cases[i]->type() == EXPR_DEFAULT) return _marks[i];
		zoObject *ret = _cases[i]->exec(ctx);
		if (ret->type() == ZO_INTEGER) {
			integer_t flag = ((zoNumber*)ret)->getInteger();
			if (cond->type() == ZO_INTEGER) {
				if (((zoNumber*)cond)->getInteger() == flag) return _marks[i];
			}
			else {
#ifdef USE_INT64
				if (atoi64(((zoString*)cond)->get().c_str()) == flag) return _marks[i];
#else
				if (atoi(((zoString*)cond)->get().c_str()) == flag) return _marks[i];
#endif
			}
		}
		else {
			const std::string &s = ((zoString*)ret)->get();
			if (cond->type() == ZO_STRING) {
				if (((zoString*)cond)->get() == s) return _marks[i];
			}
			else {
#ifdef USE_INT64
				if (atoi64(s.c_str()) == ((zoNumber*)cond)->getInteger()) return _marks[i];
#else
				if (atoi(s.c_str()) == ((zoNumber*)cond)->getInteger()) return _marks[i];
#endif
			}
		}
	}
	error("failed to find a matching case");
	return 0;
}

///////////////////////////////////////////////////////////

zsArrayExpr::zsArrayExpr(zsBaseExpr *expr)
: zsBlockExpr(expr)
{
	_type = EXPR_ARRAY;
}

void zsArrayExpr::parse(int &idx)
{
	_idx = idx++;
	parseBlock(idx, '[', ']');
	if (idx == '=') {
		if (size() < 1) error("empty array list in assignment");
		// [...] = expr
		for (int i = 0; i < size(); i++) {
			if (_block[i]->type() != EXPR_BASE || _block[i]->id() != TOK_ID)
				error("variable names expected in [...] = expr");
		}
	}
}

zoObject* zsArrayExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	if (_block.empty()) {
		// []
		return zs_create_array(ctx, 13);
	}
	// [...]
	zoArray *p = zs_create_array(ctx, _block.size());
	for (int i = 0; i < _block.size(); i++) {
		if (_block[i]->type() == EXPR_SET) {
			// [.., "key" = value, key = value, ...]
			zsSetExpr *expr = (zsSetExpr*)_block[i];
			p->set(ctx, expr->left(), expr->right()->exec(ctx));
		}
		else if (_block[i]->type() == EXPR_RANGE) {
			real_t r, rr[3];
			integer_t j, jr[3];
			if (((zsRangeExpr*)_block[i])->getRange(ctx, jr, rr, -1)) {
				int k = 0;
				for (r = rr[0]; r <= rr[1]; r += rr[2], k++) p->set(i+k, zs_create_real(0, r));
			}
			else {
				int k = 0;
				for (j = jr[0]; j <= jr[1]; j += jr[2], k++) p->set(i+k, zs_create_integer(0, j));
			}
		}
		else {
			p->set(i, _block[i]->exec(ctx));
		}
	}
	return p;
}

///////////////////////////////////////////////////////////

zsAccessExpr::zsAccessExpr(zsBaseExpr *expr)
: zsBlockExpr(expr)
{
	_type = EXPR_ACCESS;
}

void zsAccessExpr::parse(int &idx)
{
	_idx = idx++;
	parseBlock(idx, '[', ']');
	if (_block.empty()) error("empty list in a[]");
}

zoObject* zsAccessExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	return zs_get_block(ctx, this, ctx->getVar(name(), global()));
}

/////////////////////////////////////////////////////////////////////

zsCallExpr::zsCallExpr(zsBaseExpr *expr)
: zsBlockExpr(expr), _f(0), _prim(0)
{
	_type = EXPR_CALL;
}

void zsCallExpr::parse(int &idx)
{
	_idx = idx++;
	parseBlock(idx, '(', ')');
}

zoObject* zsCallExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	if (!_prim && !_f) {
		_prim = zs_get_primitive(name(), 0, 0);
		if (!_prim) _f = _owner->getFunc(name(), this, true);
		// note: null _prim & _f cause trouble in a callback app!
	}
	if (_prim) {
		return zs_call_primitive(ctx, _prim, this, 0);
	}
	if (_f->func()->type() == EXPR_CLASS) {
		zoObject *o = ctx->getVar("this->", false);
		if (o->type() != ZO_CLASS) ctx->expr->error("this-> is not a calss object");
		return _f->call(ctx, this, o);
	}
	else {
		return _f->call(ctx, this, 0);
	}
}

/////////////////////////////////////////////////////////////////////

zsTryExpr::zsTryExpr(zsBaseExpr *expr)
: zsBlockExpr(expr), _catch(expr), _err(0)
{
	_type = EXPR_TRY;
}

void zsTryExpr::parse(int &idx)
{
	_idx  = idx++;
	// try block
	parseBlock(idx, '{', '}');
	if (_block.empty()                ||
		_toks->id(idx  ) != TOK_CATCH ||
		_toks->id(idx+1) != '('       ||
		_toks->id(idx+2) != TOK_ID    ||
		_toks->id(idx+3) != ')') error();
	// catch block
	_err = idx + 2;
	idx += 4;
	_catch.parse(idx);
}

zoObject* zsTryExpr::exec(zsContext *ctx)
{
	try {
		zsBlockExpr::exec(ctx);
	}
	catch (const char *err) {
		ctx->setVar(_toks->str(_err), zs_create_string(0, err), false);
		_catch.exec(ctx);
	}
	catch (const std::string &err) {
		ctx->setVar(_toks->str(_err), zs_create_string(0, err), false);
		_catch.exec(ctx);
	}
	return 0;
}

/////////////////////////////////////////////////////////////////////

zsWhileExpr::zsWhileExpr(zsBaseExpr *expr)
: zsBlockExpr(expr), _cond(0)
{
	_type = EXPR_WHILE;
}

zsWhileExpr::~zsWhileExpr()
{
	delete _cond;
}

void zsWhileExpr::parse(int &idx)
{
	_idx  = idx++;
	if (id() == TOK_DO) {
		// do block
		parseBlock(idx, '{', '}');
		if (_block.empty()) error();
		if (_toks->id(idx) != TOK_WHILE) error();
		// while (expr);
		idx++;
		_cond = zs_parse(this, idx, '(', ')');
		if (_cond == 0) error();
		return;
	}
	// condition expression
	_cond = zs_parse(this, idx, '(', ')');
	if (_cond == 0) error();
	// while block
	if (_toks->id(idx) == '{') {
		// while(...) { ... }
		parseBlock(idx, '{', '}');
	}
	else {
		// while(...) expr;
		parseBlock(idx, 0, ';');
	}
	if (_block.empty()) error();
}

zoObject* zsWhileExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	for (;;) {
		if (id() == TOK_DO) {
			try {
				zsBlockExpr::exec(ctx);
			}
			catch (zsRCP<zoObject> o) {
				if (((zoNumber*)o.get())->getInteger() == EXPR_BREAK) break;
				if (((zoNumber*)o.get())->getInteger() == EXPR_CONTINUE) continue;
				throw;
			}
			if (!zs_check_cond(_cond->exec(ctx))) return 0;
		}
		else {
			if (!zs_check_cond(_cond->exec(ctx))) return 0;
			try {
				zsBlockExpr::exec(ctx);
			}
			catch (zsRCP<zoObject> o) {
				if (((zoNumber*)o.get())->getInteger() == EXPR_BREAK) break;
				if (((zoNumber*)o.get())->getInteger() == EXPR_CONTINUE) continue;
				throw;
			}
		}
	}
	return 0;
}

/////////////////////////////////////////////////////////////////////

zsForExpr::zsForExpr(zsBaseExpr *expr)
: zsBlockExpr(expr), _expr1(0), _expr2(0), _expr3(0)
{
	_type = EXPR_FOR;
}

zsForExpr::~zsForExpr()
{
	delete _expr1;
	delete _expr2;
	delete _expr3;
}

void zsForExpr::parse(int &idx)
{
	_idx  = idx++;
	_expr1 = zs_parse(this, idx, '(', ';');
	_expr2 = zs_parse(this, idx,  0,  ';');
	_expr3 = zs_parse(this, idx,  0,  ')');
	if (_toks->id(idx) == '{') {
		// for (...) {...}
		parseBlock(idx, '{', '}');
	}
	else {
		// for (...) expr;
		parseBlock(idx, 0, ';');
	}
	if (_block.empty()) error("empty for expr");
}

zoObject* zsForExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	if (_expr1) _expr1->exec(ctx);
	for (;;) {
		if (_expr2) {
			if (!zs_check_cond(_expr2->exec(ctx))) return 0;
		}
		try {
			zsBlockExpr::exec(ctx);
		}
		catch (zsRCP<zoObject> o) {
			if (((zoNumber*)o.get())->getInteger() == EXPR_BREAK) break;
			if (((zoNumber*)o.get())->getInteger() == EXPR_CONTINUE) {
				if (_expr3) _expr3->exec(ctx);
				continue;
			}
			throw;
		}
		if (_expr3) _expr3->exec(ctx);
	}
	return 0;
}

/////////////////////////////////////////////////////////////////////

zsIfExpr::zsIfExpr(zsBaseExpr *expr)
: zsBlockExpr(expr), _cond(0), _next(0)
{
	_type = EXPR_IF;
}

zsIfExpr::~zsIfExpr()
{
	delete _cond;
	delete _next;
}

void zsIfExpr::parse(int &idx)
{
	_type = _toks->id(idx);
	_idx  = idx++;
	if (_type != TOK_ELSE) {
		_cond = zs_parse(this, idx, '(', ')');
		if (_toks->id(idx) == '{') {
			// if (...) {....}
			parseBlock(idx, '{', '}');
		}
		else {
			// if (...) expr;
			parseBlock(idx, 0, ';');
		}
		if (_toks->id(idx) == TOK_ELSEIF || _toks->id(idx) == TOK_ELSE) {
			_next = new zsIfExpr(this);
			_next->parse(idx);
		}
	}
	else {
		if (_toks->id(idx) == '{') {
			// else {...}
			parseBlock(idx, '{', '}');
		}
		else {
			// else expr;
			parseBlock(idx, 0, ';');
		}
	}
}

zoObject* zsIfExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	if (_cond != 0) {
		// if
		if (zs_check_cond(_cond->exec(ctx))) {
			zsBlockExpr::exec(ctx);
		}
		else {
			// else if
			if (_next) _next->exec(ctx);
		}
	}
	else {
		// else
		zsBlockExpr::exec(ctx);
	}
	return 0;
}

/////////////////////////////////////////////////////////////////////

zsFuncExpr::zsFuncExpr(zsBaseExpr *expr)
: zsSwitchExpr(expr), _args(expr), _funcls(expr)
{
	_type = EXPR_FUNC;
}

void zsFuncExpr::parse(int &idx)
{
	// skip function
	idx++;
	// function name
	_idx = idx++;
	if (_owner->getFunc(name(), 0, false)) error("function already defined");
	// function arguments
	if (_toks->id(idx) != '(') error();
	if (_toks->id(idx+1) != ')') {
		_args.parseBlock(idx, '(', ')');
		for (int i = 0; i < _args.size(); i++) {
			// cehcking (a, b=1.0,...)
			if (_args.getExpr(i)->type() == EXPR_SET) {
				if (((zsSetExpr*)_args.getExpr(i))->left()->id() != TOK_ID) error();
			}
			else {
				if (_args.getExpr(i)->type() != EXPR_BASE ||
					_args.getExpr(i)->id() != TOK_ID) error();
			}
		}
	}
	else {
		// no argument
		idx += 2;
	}
	// function block
	parseBlock(idx, '{', '}');
	// sanitary check
	for (int i = 0; i < _block.size(); i++) {
		if (_block[i]->type() == EXPR_BREAK || _block[i]->type() == EXPR_CONTINUE)
		_block[i]->error("meaningless break/continue");
	}
}

zoObject* zsFuncExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
//	ctx->setReturn(zsModuleExpr::g_null.get());
	ctx->setReturn(zs_create_null(ctx));
	try {
		zsBlockExpr::exec(ctx);
	}
	catch (zsRCP<zoObject> o) {
		// do nonthing
	}
	return ctx->getReturn();
}

zoObject* zsFuncExpr::call(zsContext *ctx, zsBlockExpr *args, zoObject *caller)
{
	// new context for executing function block
	zsContext ctx2(ctx);
	if (caller && caller->type() == ZO_CLASS) {
		ctx2.setContext(((zoClass*)caller)->getContext());
		init(((zoClass*)caller)->getContext(), ctx2);
	}
	else {
		if (_owner->type() == EXPR_MODULE) ctx2.setContext(((zsModuleExpr*)_owner)->context());
		init(ctx, ctx2);
	}

	int i, j = 0;
	int nargs = MIN(args->size(), _args.size());	// ignore extra arguments

	// set argument
	if (caller && caller->type() != ZO_CLASS) {
		ctx2.expr = _args.getExpr(0);
		ctx2.setVar(_args.getExpr(0)->name(), caller, false);
		j++;
	}

	for (i = 0; i < nargs; i++, j++) {
		ctx2.expr = args->getExpr(i);
		if (args->getExpr(i)->type() == EXPR_SET) {
			zsSetExpr *expr = (zsSetExpr*)args->getExpr(i);
			zoObject *o = expr->right()->exec(ctx);
//			if (o->type() < ZO_ARRAY) o = o->copy(ctx);
			ctx2.setVar(expr->name(), o, false);
		}
		else {
			zoObject *o = args->getExpr(i)->exec(ctx);
//			if (o->type() < ZO_ARRAY) o = o->copy(ctx);
			ctx2.setVar(_args.getExpr(j)->name(), o, false);
		}
	}

	zoObject *ret = exec(&ctx2);
	ctx->setReturn(ret);
// leep ret to avoid error in cases like o1+o2
	if (ret && ret->type() != ZO_NULL) ctx->keep(ret);
	return ret;
}

zoObject* zsFuncExpr::call(zsContext *ctx, int nargs, zoObject **args)
{
	// new context for executing function block
	zsContext ctx2(ctx);
	if (_owner->type() == EXPR_MODULE) ctx2.setContext(((zsModuleExpr*)_owner)->context());

	init(ctx, ctx2);

	nargs = MIN(nargs, _args.size());		// ignore extra arguments

	for (int i = 0; i < nargs; i++) {
		ctx2.expr = _args.getExpr(i);
		ctx2.setVar(_args.getExpr(i)->name(), args[i], false);
	}

	zoObject *ret = exec(&ctx2);
	ctx->setReturn(ret);
// leep ret to avoid error in cases like o1+o2
	if (ret && ret->type() != ZO_NULL) ctx->keep(ret);
	return ret;
}

void zsFuncExpr::init(zsContext *ctx, zsContext &ctx2)
{
	for (int i = 0; i < _args.size(); i++) {
		ctx2.expr = _args.getExpr(i);
		if (_args.getExpr(i)->type() == EXPR_SET) {
			// as in function call func(..., a=expr, ...)
			zsSetExpr *expr = (zsSetExpr*)_args.getExpr(i);
			zoObject *o = expr->right()->exec(ctx);
//			if (o->type() < ZO_ARRAY) o = o->copy(ctx);
			ctx2.setVar(expr->name(), o, false);
		}
		else {
			// as in function call func(..., a, ...)
//			ctx2.setVar(_args.getExpr(i)->name(), zsModuleExpr::g_null.get(), false);
			ctx2.setVar(_args.getExpr(i)->name(), zs_create_null(0), false);
		}
	}
}

zsFuncExpr* zsFuncExpr::getFunc(const std::string& name, zsBaseExpr *expr, bool global)
{
	for (int i = 0; i < _funcls.size(); i++) {
		if (_funcls.getExpr(i)->type() == EXPR_FUNC && _funcls.getExpr(i)->name() == name) {
			return ((zsFuncExpr*)_funcls.getExpr(i));
		}
	}
	if (global) {
		// global search
		zsFuncExpr* func = 0;
		if (type() != EXPR_MODULE) {
			// search parent
			func = _owner->getFunc(name, expr, true);
			if (func) return func;
		}
		// search in imported modules
		if (type() == EXPR_MODULE) {
			std::vector<zsModuleExpr*> modules;
			((zsModuleExpr*)this)->modules().get(modules);
			for (int i = 0; i < modules.size(); i++) {
				func = modules[i]->getFunc(name, modules[i], false);
				if (func) return func;
			}
		}
		if (expr) expr->error("function undefined");
	}
	return 0;
}

zsClassExpr* zsFuncExpr::getClass(const std::string& name, zsBaseExpr *expr, bool global)
{
	for (int i = 0; i < _funcls.size(); i++) {
		if (_funcls.getExpr(i)->type() == EXPR_CLASS && _funcls.getExpr(i)->name() == name) {
			return ((zsClassExpr*)_funcls.getExpr(i));
		}
	}
	if (global) {
		// global search
		zsClassExpr* cls = 0;
		if (type() != EXPR_MODULE) {
			// search parent
			cls = _owner->getClass(name, expr, true);
			if (cls) return cls;
		}
		// search in imported module
		std::vector<zsModuleExpr*> modules;
		((zsModuleExpr*)this)->modules().get(modules);
		for (int i = 0; i < modules.size(); i++) {
			cls = modules[i]->getClass(name, modules[i], false);
			if (cls) return cls;
		}
		if (expr) expr->error("class undefined");
	}
	return 0;
}

/////////////////////////////////////////////////////////////////////

zsClassExpr::zsClassExpr(zsBaseExpr *expr)
: zsFuncExpr(expr)
{
	_type = EXPR_CLASS;
}

void zsClassExpr::parse(int &idx)
{
	// skip class
	idx++;
	// class name
	_idx = idx++;
	if (_owner->getClass(name(), 0, false)) error("class already defined");
	// class expressions
	parseBlock(idx, '{', '}');
	// sanitary check
	for (int i = 0; i < _block.size(); i++) {
		if (_block[i]->type() == EXPR_BREAK ||
			_block[i]->type() == EXPR_CONTINUE ||
			_block[i]->type() == EXPR_RETURN)
		_block[i]->error("meaningless break/continue/return");
	}
}

zoObject* zsClassExpr::exec(zsContext *ctx)
{
	ctx->expr = this;
	try {
		zsBlockExpr::exec(ctx);
	}
	catch (zsRCP<zoObject> o) {
		// do nonthing;
	}
	return 0;
}

bool zsClassExpr::exclusive(const std::string &name)
{
	for (int i = 0; i < size(); i++) {
		if (_block[i]->type() == EXPR_SET) {
			zsSetExpr *expr = (zsSetExpr*)_block[i];
			if (expr->right()->type() == EXPR_NEW && expr->right()->name() == name) {
				return false;
			}
		}
	}
	return true;
}

/////////////////////////////////////////////////////////////////////

zsModuleExpr::zsModuleExpr()
: zsFuncExpr(0), _path(0), _root(false), _status(false), _ctx(0), _imported(0)
{
	_ctx.expr = this;
	_type = EXPR_MODULE;
	_owner = this;
	_idx = 0;
	_toks = new zsTokens;
}

zsModuleExpr::~zsModuleExpr()
{
	delete _toks;
	if (_root) {
		delete[] _path;
		delete _imported;
	}
}

zoObject *zsModuleExpr::exec(zsContext *ctx)
{
	// ctx is NULL
	if (_status) return 0;
	try {
		zsBlockExpr::exec(&_ctx);
	}
	catch (zsRCP<zoObject> o) {
		// do nonthing;
	}
	return 0;
}

bool open_file(std::string& fname, const std::string& file, const char *path)
{
	fname = file;
	if (fname.find(".zs") == std::string::npos) fname += ".zs";
	FILE *f = fopen(fname.c_str(), "r");
	if (!f) {
		if (path) {
			fname = path + fname;
			f = fopen(fname.c_str(), "r");
		}
		if (!f) return false;
	}
	fclose(f);
	return true;
}

void delete_module(void *module)
{
	delete (zsModuleExpr*)module;
    module = 0;
}

bool zsModuleExpr::import(zsModuleExpr *importer, const std::string& file, bool codestring, const char *path, bool local)
{
	if (!importer) {
		// root module
		if (path) {
			_path = new char[strlen(path)+1];
			strcpy(_path, path);
		}
		_imported = new zsPtrTable<zsModuleExpr>;
		_root = true;
	}
	else {
		_path = importer->path();
		_imported = importer->imported();
	}

	if (!codestring) {
		// parse script in file
		std::string fname;
		if (!open_file(fname, file, _path)) {
			fname = std::string("failed to import ") + file;
			throw (fname);
		}

		if (importer) {
			if (fname == importer->tokens()->fname()) importer->error("self import");
			zsModuleExpr *m = _imported->get(fname);
			if (m) {
				// already imported
				importer->modules().add(fname, m, 0);
				importer->modules().expand();
				return false;
			}
			else {
				importer->modules().add(fname, this, delete_module);
				importer->modules().expand();
				if (!local) {
					_imported->add(fname, this, 0);
					_imported->expand();
				}
			}
		}

		_toks->scan(fname.c_str(), file.c_str(), codestring);
	}
	else {
		// parse script in string; user deletes this module.
		_toks->scan(file.c_str(), file.c_str(), codestring);
	}

	int idx = 1;
	parseBlock(idx, '{', '}');

	// sanitary check
	for (int i = 0; i < _block.size(); i++) {
		if (_block[i]->type() == EXPR_BREAK || _block[i]->type() == EXPR_CONTINUE)
		_block[i]->error("meaningless break/continue");
	}

	return true;
}

#ifndef USE_SYSLIB

void print_array(zoArray *arr)
{
	if (arr->size() == 0) {
		printf("[]");
		return;
	}
	char buf[64];
	std::vector<std::string> keys;
	arr->get(keys);
	std::sort(keys.begin(), keys.end());
	printf("[");
	for (int k = 0; k < keys.size(); k++) {
		if (k > 0) printf(", ");
		zoObject *o = arr->get(keys[k]);
		if (o->type() == ZO_ARRAY) {
			printf("%s=", keys[k].c_str());
			print_array((zoArray*)o);
		}
		else {
			printf("%s=%s", keys[k].c_str(), zs_object_str(o, buf));
		}
	}
	printf("]");
}

zoObject* sys_csv(zsContext *ctx, int nargs, zoObject **args)
{
	if (nargs < 1) return 0;
	char buf[64];
	for (int i = 0; i < nargs; i++) {
		if (i > 0) printf(", ");
		if (args[i]->type() == ZO_ARRAY) {
			print_array((zoArray*)args[i]);
		}
		else {
			printf("%s", zs_object_str(args[i], buf));
		}
	}
	printf("\n");
	fflush(stdout);
	return 0;
}

// register primitives

zsRegPrimitive::zsRegPrimitive() {
	zs_add_primitive("csv", 0, sys_csv);
}

#endif
