/*
Copyright (c) 2005, Jiye Zeng (http://www.zegraph.com/)

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

[1] Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer. 

[2] Neither the name of the organization nor the names of its contributors
may be used to endorse or promote products derived from this software
without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef __ZS_API__
#define __ZS_API__

// Define to use 64-bit integer
#define USE_INT64

// Define to use syslib.cpp
#define USE_SYSLIB

#ifdef _WIN32

typedef __int64 int64;
#define atoi64(a) _atoi64(a)

#ifdef USE_INT64
typedef __int64 integer_t;
#else
typedef int integer_t;
#endif

#ifdef _EXPORT
#define DLL_API __declspec(dllexport)
#endif

#else	// _WIN32

typedef long long int64;
#define atoi64(a) atoll(a)

#ifdef USE_INT64
typedef long long integer_t;
#else
typedef int integer_t;
#endif

#endif	// _WIN32

#ifndef DLL_API
#define DLL_API
#endif

typedef double real_t;

#ifdef __cplusplus
extern "C" {
#endif

/// Runtime error should be reported through this function.
DLL_API void api_runtime_error(void *ctx, const char* msg);
DLL_API void api_runtime_error2(void *ctx, const char* msg, const char* msg2);

/// Reports default input error message in function call.
DLL_API void api_input_error(void *ctx);

/// Creates a null object. ctx may be NULL if the created object will be inserted in an array immediately.
DLL_API void* api_create_null(void *ctx);

DLL_API void* api_create_integer(void *ctx, integer_t value);
/// Creates an object for integer. ctx may be NULL if the created object will be inserted in an array immediately.

DLL_API void* api_create_real(void *ctx, real_t value);
/// Creates an object for real number. ctx may be NULL if the created object will be inserted in an array immediately.

DLL_API void* api_create_string(void *ctx, const char* str);
/// Creates an object for string. ctx may be NULL if the created object will be inserted in an array immediately.

DLL_API void* api_create_array(void *ctx, int size);
/// Creats an array object of given size. ctx may be NULL if the created object will be inserted in an array immediately.
//
// Note that you can add to an array as many objects as you want.
// The array will resize itself when necessary, but using a proper size parameter
// will prevent array resizing and will thus be more efficient. 

DLL_API void* api_create_user(void *ctx, void* ptr, void*(*opfunc)(void*,int,void**), void(*destroy)(void*), int type);
// Creats a user object for ptr. If ctx is not NULL, the created object will be kept temporally.
//
// By setting opfunc to a function of prototype
//		void *func(void *ctx, int nargs, **args)
// You can redefine operations of operators on user objects.
//
// By setting destroy to a function of prototype
//		void func(void *ptr)
// The fucntion will be called when the variable of the user object
// is no longer needed by Z-Script.
//
// By setting type to a positive integer, you can have "methods" for
// the user object, i.e., functions registered with the type
// can only be called though the object in such an expression as user.func(a, b, c).

DLL_API void *api_create_context(void *ctx);
// Creates a variable context.

DLL_API void api_delete_context(void *ctx);
// Deletes a variable context created by api_create_context.

DLL_API int api_is_null(void* object);
// Returns 1 if the object is null type.

DLL_API int api_is_integer(void* object);
// Returns 1 if the object is integer type.

DLL_API int api_is_real(void* object);
// Returns 1 if the object is of real type.

DLL_API int api_is_number(void *object);
// Returns 1 if the object is either integer or real type.

DLL_API int api_is_string(void *object);
// Returns 1 if the object is string type.

DLL_API int api_is_array(void *object);
// Returns 1 if the object is array type.

DLL_API int api_is_user(void *object);
// Returns 1 if the object is user type.

DLL_API int api_get_type(void *object);
// Returns object type.

DLL_API	integer_t api_get_integer(void *ctx, void* object);
// Extracts integer value from the object. Reports error if the object is not integer type.

DLL_API	real_t api_get_real(void *ctx, void* object);
// Extracts real value from the object. Reports error if the object is not real type.

DLL_API	real_t api_get_number(void *ctx, void* object);
// Extracts real value from the object. Reports error if the object is neither real nor integer type.

DLL_API const char* api_get_string(void* ctx, void* object);
// Extracts string from the object. Reports error if the object is not string type.

DLL_API void* api_get_ptr(void *ctx, void *object);
// Extracts pointer from the object. Reports error if the object is not user type.

DLL_API void* api_get_user(void *ctx, void *object, int type);
// Extracts pointer from the object. Reports error if the object is not the specified user type.

DLL_API int api_get_array_size(void* ctx, void* array);
// Returns the number of objects in the array. Reports error if the object is not array-type.

DLL_API void* api_get_array_object(void *ctx, void* array, const char* key);
DLL_API void* api_get_array_object2(void *ctx, void* array, int key);
// Returns the object with the string or integer key in the array.
// Reports error if array is not array-type or array does not have the key. 

DLL_API void api_set_array_object(void *ctx, void* array, const char* key, void* value);
DLL_API void api_set_array_object2(void *ctx, void* array, int key, void* value);
// Sets the object to the array and assigned the string or integer key to the object.
// Reports error if array is not array-type. 

DLL_API void* api_get_func(void *ctx, const char* name);
// Returns the pointer to the named script function declaired in a module. Reports error if failed.

DLL_API void* api_peek_func(void *ctx, const char* name);
// Returns the pointer to the named script function declaired in a module. Returns NULL if failed.

DLL_API void* api_call_func(void *ctx, void *func, int nargs, void** args);
// Calls script function with nargs number of arguments args.

DLL_API void api_add_primitive(const char* name, int type, void*(*func)(void*,int,void**));
// Adds a primitive function to Z-Script virtual machine.
// The type should be zero or the same as a user object.
// The func must be of prototype
//		void *f(void *ctx, int nargs, void **args)

DLL_API void* api_parse_file(const char* fname, char *error);
// Parse Z-Script code in fname file and returns the pointer to the module.
// The error must point to a string of at lest 256 bytes.

DLL_API void* api_parse_file2(const char* fname, char *error, const char* path);
// Same as api_parse_file(...) but set the physical path of entry module.
// When Z-Script is used as the IIS extension, the application path will be IIS's, not that of Z-Script's.
// Use this interface to set the application path for importing modules.

DLL_API void* api_parse_string(const char* str, char *error);
// Parse Z-Script code in the string and returns the pointer to the module.

DLL_API void* api_parse_string2(const char* str, char *error, const char* path);
// Same as api_parse_string(...) but set the physical path of entry module.

DLL_API void* api_get_context(void *expr);
// Returns pointer to the module context of the expression, which may be a module or function pointer.
// The context may be used to get/set variables.

DLL_API void api_set_object(void *ctx, const char *name, void *object);
// Sets the object to the named variable in the context.

DLL_API void* api_peek_object(void *ctx, const char *name);
// Return pointer to an object if the named variable has been defined. Returns NULL otherwise.

DLL_API int api_exec_module(void *module, char *error);
// Excutes the module and reports error.
// The error variable must point to a string of at lest 256 bytes.

DLL_API void api_delete_module(void *module);
// Deletes the module.

DLL_API void api_mutex(int flag);
// Activate/deactivate mutex for thread safe.

DLL_API const char *format_real(real_t, char *buf);
// Converts real number to string.
// The variable buf must point to a string of at lest 32 bytes.

#ifdef __cplusplus
};
#endif

#endif	//__ZS_API__

/***

////////////////////// CPP example /////////////////////////

#include "api.h"
#include <stdio.h>
#include <string.h>

#pragma warning(disable: 4244)

#define P3D_TYPE		'P3D'

///////////////////////////////////////////////////////
// 3D point class
//
class zsP3D
{
public:
	zsP3D() : x(0), y(0), z(0) { }
	~zsP3D() { }
	double x, y, z;
};

///////////////////////////////////////////////////////
// This will be called when reference counts to
// a zsP3D object are reduced to zero.
//
void p3d_destroy(void* ptr) { delete (zsP3D*)ptr; }

///////////////////////////////////////////////////////
// Define + operator for zsP3D 
//
void* p3d_opfunc(void *ctx, int nargs, void** args)
{
	if (nargs != 3) api_input_error(ctx);
	zsP3D *l = (zsP3D*)api_get_user(ctx, args[0], P3D_TYPE);
	int op = api_get_integer(ctx, args[1]);
	switch (op) {
	case '+':
		if (api_is_number(args[2])) {
			// p3d + number
			real_t v = api_get_number(ctx, args[2]);
			zsP3D *o = new zsP3D;
			o->x = l->x + v;
			o->y = l->x + v;
			o->z = l->x + v;
			return api_create_user(ctx, o, p3d_opfunc, p3d_destroy, P3D_TYPE);
		}
		else {
			// p3d + p3d
			zsP3D *r = (zsP3D*)api_get_user(ctx, args[2], P3D_TYPE);
			zsP3D *o = new zsP3D;
			o->x = l->x + r->x;
			o->y = l->x + r->y;
			o->z = l->x + r->z;
			return api_create_user(ctx, o, p3d_opfunc, p3d_destroy, P3D_TYPE);
		}
		break;
		// more operator definitions here...
	}
	api_runtime_error(ctx, "undefined operation for P3D object");
	return 0;
}

///////////////////////////////////////////////////////
// Create a zsP3D object. This may be called as
// a = p3p();
// a = p3p(0.1);
// a = p3p(0.1, 0.2);
// a = p3p(0.1, 0.2, 3);
//
void* p3d_create(void *ctx, int nargs, void** args)
{
	zsP3D *o = new zsP3D;
	if (nargs > 0) o->x = api_get_number(ctx, args[0]);
	if (nargs > 1) o->y = api_get_number(ctx, args[1]);
	if (nargs > 2) o->z = api_get_number(ctx, args[2]);
	return api_create_user(ctx, o, p3d_opfunc, p3d_destroy, P3D_TYPE);
}

///////////////////////////////////////////////////////
// Define ___get to be called by such a expression as
// a = p.name;
// a = p[idx];
//
void* p3d_get(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zsP3D *o = (zsP3D*)api_get_user(ctx, args[0], P3D_TYPE);
	if (api_is_string(args[1])) {
		// p.x
		const char *name = api_get_string(ctx, args[1]);
		if (strcmp(name, "x") == 0) return api_create_real(ctx, o->x);
		if (strcmp(name, "y") == 0) return api_create_real(ctx, o->y);
		if (strcmp(name, "z") == 0) return api_create_real(ctx, o->z);
		api_runtime_error(ctx, "bad P3D member access");
	}
	// p[idx]
	int idx = api_get_integer(ctx, args[1]);
	if (idx == 0) return api_create_real(ctx, o->x);
	if (idx == 1) return api_create_real(ctx, o->y);
	if (idx == 2) return api_create_real(ctx, o->z);
	api_runtime_error(ctx, "bad P3D member access");
	return 0;
}

///////////////////////////////////////////////////////
// Define ___set to be called by such a expression as
// p.name = v;
// p[idx] = v;
//
void* p3d_set(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zsP3D *o = (zsP3D*)api_get_user(ctx, args[0], P3D_TYPE);
	real_t v = api_get_number(ctx, args[2]);
	if (api_is_string(args[1])) {
		// p.name = v
		const char *name = api_get_string(ctx, args[1]);
		if      (strcmp(name, "x") == 0) o->x = v;
		else if (strcmp(name, "y") == 0) o->y = v;
		else if (strcmp(name, "z") == 0) o->z = v;
		else api_runtime_error(ctx, "bad P3D member assignment");
		return 0;
	}
	// p[idx] = v
	int idx = api_get_integer(ctx, args[1]);
	if      (idx == 0) o->x = v;
	else if (idx == 1) o->y = v;
	else if (idx == 2) o->z = v;
	else api_runtime_error(ctx, "bad P3D member assignment2");
	return 0;
}

///////////////////////////////////////////////////////
// Define ___copy to be called by such a expression as
// p2 = p1;
//
void* p3d_copy(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zsP3D *o = (zsP3D*)api_get_user(ctx, args[0], P3D_TYPE);
	zsP3D *ret = new zsP3D;
	ret->x = o->x;
	ret->y = o->y;
	ret->z = o->z;
	return api_create_user(ctx, ret, p3d_opfunc, p3d_destroy, P3D_TYPE);
}

///////////////////////////////////////////////////////
// Display class members
//
void* p3d_print(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zsP3D *o = (zsP3D*)api_get_user(ctx, args[0], P3D_TYPE);
	printf("x=%f, y=%f, z=%f\n", o->x, o->y, o->z);
	return 0;
}

///////////////////////////////////////////////////////
// Class for registering primitive function
//
class zsRegPrimitive
{
public:
	zsRegPrimitive()
	{
		api_add_primitive("p3d",		0,			p3d_create);
		api_add_primitive("__get",		P3D_TYPE,	p3d_get);
		api_add_primitive("__set",		P3D_TYPE,	p3d_set);
		api_add_primitive("__copy",		P3D_TYPE,	p3d_copy);
		api_add_primitive("print",		P3D_TYPE,	p3d_print);
	}

	~zsRegPrimitive() { }
};

zsRegPrimitive p3d_register;

//////////////////// Z-Script example ////////////////////

  // load primitive functions in the library
load("point3d.dll");

// create 3D point object
p1 = p3d(1, 2, 3);

// show its members
p1.print();

// copy by assignment
p2 = p1;

// set members
p2.x = 10;
p2[1] = 20;

p2.print();

// get members of p1 and show them using buildin csv function
csv(p1.x, p1.y, p1[2]);

// + operator
p1 = p1 + 0.1;
p1.print();
p1 = p1 + p2;
p1.print();

***/
