/*
Copyright (c) 2005, Jiye Zeng (http://www.zegraph.com/)

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

[1] Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer. 

[2] Neither the name of the organization nor the names of its contributors
may be used to endorse or promote products derived from this software
without specific prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "config.h"
#include "compiler.h"
#include "thread.h"
#include <cstring>

#define DEFAULT_EXCEPTION "error: default exception"

#define IS_INTEGER(o) ((o) && ((zoObject*)o)->type() == ZO_INTEGER)
#define IS_REAL(o) ((o) && ((zoObject*)o)->type() == ZO_REAL)
#define IS_STRING(o) ((o) && ((zoObject*)o)->type() == ZO_STRING)
#define IS_ARRAY(o) ((o) && ((zoObject*)o)->type() == ZO_ARRAY)
#define IS_USER(o) ((o) && ((zoObject*)o)->type() >= ZO_USER)
#define IS_MODULE(m) ((m) && ((zsBaseExpr*)m)->type() == EXPR_MODULE)


/////////////////////////////////////////////////////////////////////

static zsContext *check_context(void *ctx)
{
	static char msg[256];
	if (!ctx) {
		strcpy(msg,"NULL context used in calling api function");
		throw(msg);
	}
	zsContext *c = (zsContext*)ctx;
	if (c->type() != EXPR_CONTEXT) {
		strcpy(msg,"non-context object used in calling api function");
		throw(msg);
	}
	return c;
}

void api_input_error(void *ctx)
{
	zsContext *c = check_context(ctx);
	c->expr->error("improper number/type of parameter input in function call");
}

void api_runtime_error(void *ctx, const char* msg)
{
	zsContext *c = check_context(ctx);
	c->expr->error(msg);
}

void api_runtime_error2(void *ctx, const char* msg, const char* msg2)
{
	zsContext *c = check_context(ctx);
	c->expr->error(msg, msg2);
}

void* api_create_null(void *ctx)
{
	if (ctx) check_context(ctx);
	return zs_create_null((zsContext*)ctx);
}

void *api_create_integer(void *ctx, integer_t value)
{
	if (ctx) check_context(ctx);
	return zs_create_integer((zsContext*)ctx, value);
}

void *api_create_real(void *ctx, real_t value)
{
	if (ctx) check_context(ctx);
	return zs_create_real((zsContext*)ctx, value);
}

void *api_create_string(void *ctx, const char* str)
{
	if (ctx) check_context(ctx);
	return zs_create_string((zsContext*)ctx, str);
}

void *api_create_array(void *ctx, int size)
{
	if (ctx) check_context(ctx);
	return zs_create_array((zsContext*)ctx, size);
}

void* api_create_user(void *ctx, void* ptr, void*(*opfunc)(void*,int,void**), void(*destroy)(void*), int type)
{
	if (ctx) check_context(ctx);
	return zs_create_user((zsContext*)ctx, ptr, (zsPrimitiveFunc)opfunc, (zsDestroyFunc)destroy, type);
}

void *api_create_context(void *ctx)
{
	return (new zsContext(check_context(ctx)));
}

void api_delete_context(void *ctx)
{
	delete check_context(ctx);
}

int api_is_null(void* object)
{
	if (object && ((zoObject*)object)->type() == ZO_NULL) return true;
	return false;
}

int api_is_integer(void* object)
{
	if (IS_INTEGER(object)) return true;
	return false;
}

int api_is_real(void* object)
{
	if (IS_REAL(object)) return true;
	return false;
}

int api_is_number(void* object)
{
	if (IS_INTEGER(object) || IS_REAL(object)) return true;
	return false;
}

int api_is_string(void* object)
{
	if (IS_STRING(object)) return true;
	return false;
}

int api_is_array(void* object)
{
	if (IS_ARRAY(object)) return true;
	return false;
}

int api_is_user(void* object)
{
	if (IS_USER(object)) return true;
	return false;
}

integer_t api_get_integer(void *ctx, void* object)
{
	if (!IS_INTEGER(object)) api_runtime_error(ctx, "integer expected");
	return ((zoNumber*)object)->getInteger();
}

real_t api_get_real(void *ctx, void* object)
{
	if (!IS_REAL(object)) api_runtime_error(ctx, "real expected");
	return ((zoNumber*)object)->getReal();
}

real_t api_get_number(void *ctx, void* object)
{
	if (IS_REAL(object)) return ((zoNumber*)object)->getReal();
	if (IS_INTEGER(object)) return ((zoNumber*)object)->getInteger();
	api_runtime_error(ctx, "real/integer expected");
	return 0;
}

const char* api_get_string(void *ctx, void* object)
{
	if (!IS_STRING(object)) api_runtime_error(ctx, "string expected");
	return ((zoString*)object)->get().c_str();
}

void* api_get_ptr(void *ctx, void* object)
{
	if (!IS_USER(object)) api_runtime_error(ctx, "user type expected");
	void *ptr = ((zoUser*)object)->get();
	if (ptr == 0) api_runtime_error(ctx, "NULL ptr object");
	return ptr;
}

void* api_get_user(void *ctx, void *object, int type)
{
	if (api_get_type(object) != type) {
		char b[5], s[64], *c;
		int endian = 1, i = 0;
		c = (char*)(&endian);
		if (c[0] == 0) {
			c = (char*)(&type);
			if (c[0] != 0) { b[i++] = c[0]; }
			if (c[1] != 0) { b[i++] = c[1]; }
			if (c[2] != 0) { b[i++] = c[2]; }
			if (c[3] != 0) { b[i++] = c[3]; }
		}
		else {
			c = (char*)(&type);
			if (c[3] != 0) { b[i++] = c[3]; }
			if (c[2] != 0) { b[i++] = c[2]; }
			if (c[1] != 0) { b[i++] = c[1]; }
			if (c[0] != 0) { b[i++] = c[0]; }
		}
		b[i] = 0;
		sprintf(s, "not a %s type", b);
		api_runtime_error(ctx, s);
	}
	void *ptr = ((zoUser*)object)->get();
	if (ptr == 0) api_runtime_error(ctx, "NULL user object");
	return ptr;
}

int api_get_type(void *object)
{
	if (!object) return -1;
	int type = ((zoObject*)object)->type();
	if (type >= ZO_NULL && type <= ZO_USER) return type;
	if (type > ZO_USER) return type - ZO_USER;
	return -1;
}

int api_get_array_size(void *ctx, void* array)
{
	if (!IS_ARRAY(array)) api_runtime_error(ctx, "array expected");
	return ((zoArray*)array)->size();
}

void* api_get_array_object(void *ctx, void* array, const char* key)
{
	if (!IS_ARRAY(array)) api_runtime_error(ctx, "array expected");
	return ((zoArray*)array)->get(check_context(ctx), key);
}

void* api_get_array_object2(void *ctx, void* array, int key)
{
	if (!IS_ARRAY(array)) api_runtime_error(ctx, "array expected");
	return ((zoArray*)array)->get(check_context(ctx), key);
}

void api_set_array_object(void *ctx, void* array, const char* key, void* value)
{
	if (!IS_ARRAY(array)) api_runtime_error(ctx, "array expected");
	if (!value) api_runtime_error(ctx, "add NULL pointer to array");
	((zoArray*)array)->set(key, (zoObject*)value);
}

void api_set_array_object2(void *ctx, void* array, int key, void* value)
{
	if (!IS_ARRAY(array)) api_runtime_error(ctx, "array expected");
	if (!value) api_runtime_error(ctx, "add NULL pointer to array");
	((zoArray*)array)->set(key, (zoObject*)value);
}

void* api_get_func(void *ctx, const char* name)
{
	zsContext *c = check_context(ctx);
	return c->expr->func()->getFunc(name, c->expr, true);
}

void* api_peek_func(void *ctx, const char* name)
{
	zsContext *c = check_context(ctx);
	return c->expr->func()->getFunc(name, 0, true);
}

void* api_call_func(void *ctx, void *func, int nargs, void** args)
{
	zsContext *c = check_context(ctx);
	zsFuncExpr *f = (zsFuncExpr*)func;
	if (f->type() != EXPR_FUNC) c->expr->error("pointer to function expected");
	zoObject *ret = f->call(c, nargs, (zoObject**)args);
	c->clear();
	return ret;
}

void api_add_primitive(const char* name, int type, void*(*func)(void*,int,void**))
{
	zs_add_primitive(name, type, (zsPrimitiveFunc)func);
}

static zsModuleExpr* api_parse(const char* str, char *error, bool flag, const char* path)
{
	if (error) error[0] = 0;
	zsModuleExpr *module = new zsModuleExpr;
	try {
		module->import(0, str, flag, path);
		return module;
	}
	catch (const char* err) {
		if (error) strcpy(error, err);
	}
	catch (const std::string &err) {
		if (error) strcpy(error, err.c_str());
	}
	catch (std::bad_alloc &err) {
		if (error) strcpy(error, "bad_alloc");
	}
	catch (...) {
		if (error) strcpy(error, DEFAULT_EXCEPTION);
	}
	delete module;
	return 0;
}

void* api_parse_file(const char* fname, char *error)
{
	return api_parse(fname, error, false, 0);
}

void* api_parse_file2(const char* fname, char *error, const char* path)
{
	return api_parse(fname, error, false, path);
}

void* api_parse_string(const char* str, char *error)
{
	return api_parse(str, error, true, 0);
}

void* api_parse_string2(const char* str, char *error, const char* path)
{
	return api_parse(str, error, true, path);
}

void* api_get_context(void *module)
{
	zsModuleExpr *m  = ((zsFuncExpr*)module)->module();
	if (IS_MODULE(m)) {
		return m->context();
	}
	return 0;
}

void api_set_object(void *ctx, const char *name, void *object)
{
	zsContext *c = check_context(ctx);
	c->setObj(name, (zoObject*)object);
}

void *api_peek_object(void *ctx, const char *name)
{
	zsContext *c = check_context(ctx);
	return c->peekVar(name);
}

int api_exec_module(void *module, char *error)
{
	if (error) error[0] = 0;
	if (!IS_MODULE(module)) {
		if (error) strcpy(error, "module object expected");
		return 0;
	}
	zsModuleExpr *m = (zsModuleExpr*)module;
	try {
		m->exec(0);
		return 1;
	}
	catch (const char* err) {
		if (error) strcpy(error, err);
	}
	catch (const std::string &err) {
		if (error) strcpy(error, err.c_str());
	}
	catch (std::bad_alloc &err) {
		if (error) strcpy(error, "bad_alloc");
	}
	catch (...) {
		if (error) strcpy(error, DEFAULT_EXCEPTION);
	}
	return 0;
}

void api_delete_module(void *module)
{
	if (IS_MODULE(module)) {
		delete ((zsModuleExpr*)module);
        module = 0;
	}
}

void api_mutex(int flag)
{
	zsModuleExpr::g_mutex_flag = flag;
}
