#include "colorbar.h"
#include "freefont.h"
#include "light.h"
#include "line.h"
#include "material.h"
#include "plot.h"
#include "point.h"
#include "polygon.h"
#include "scene.h"
#include "text_tex.h"
#include "texture.h"
#include "factory.h"
#include "globe.h"
#include "utility.h"

#include "matrix.h"

#include "api.h"

#include <math.h>
#include <list>

#define R2D		57.29577951
#define D2R		0.017453293

using namespace std;

#pragma warning(disable: 4244)

#define __TRY__ try {
#define __CATCH__ } catch (const char* error) { api_runtime_error(ctx, error); }

enum { IMG_GIF, IMG_PNG, IMG_JPG, IMG_TIF, IMG_JP2, IMG_BMP };


static zeObject *zg_create(void *ctx, const char* name, zeFreetype *font)
{
	if (strcmp(name, "axis") == 0) {
		zeAxis *o = new zeAxis;
		o->setFont(font);
		return o;
	}
	if (strcmp(name, "colorbar") == 0) {
		zeColorBar *o = new zeColorBar;
		o->setFont(font);
		return o;
	}
	if (strcmp(name, "globe") == 0) {
		zeGlobe *o = new zeGlobe;
		return o;
	}
	if (strcmp(name, "light") == 0) {
		zeLight *o = new zeLight;
		return o;
	}
	if (strcmp(name, "line") == 0) {
		zeLine *o = new zeLine;
		return o;
	}
	if (strcmp(name, "node") == 0) {
		zeNode *o = new zeNode;
		return o;
	}
	if (strcmp(name, "plot") == 0) {
		zePlot *o = new zePlot;
		o->xaxis()->setFont(font);
		o->yaxis()->setFont(font);
		o->zaxis()->setFont(font);
		return o;
	}
	if (strcmp(name, "point") == 0) {
		zePoint *o = new zePoint;
		return o;
	}
	if (strcmp(name, "polygon") == 0) {
		zePolygon *o = new zePolygon;
		return o;
	}
	if (strcmp(name, "text") == 0) {
		zeText *o = new zeText;
		o->setFont(font);
		return o;
	}
	api_runtime_error2(ctx, "unknown type name: ", name);
	return 0;
}

static void* set_rotatex(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeObject *o = (zeObject*)api_get_ptr(ctx, args[0]);
	double a = api_get_number(ctx, args[1]);
	switch (o->getType()) {
		case ZE_AXIS:
		case ZE_COLOR_BAR:
		case ZE_GLOBE:
		case ZE_LINE:
		case ZE_NODE:
		case ZE_PLOT:
		case ZE_POINT:
		case ZE_POLYGON:
		case ZE_TEXT:
			((zeTransform*)o)->rotateX(a);
			break;
		default:
			api_runtime_error(ctx, "unexpected object type in calling rotatex");
	}
	return 0;
}

static void* set_rotatey(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeObject *o = (zeObject*)api_get_ptr(ctx, args[0]);
	double a = api_get_number(ctx, args[1]);
	switch (o->getType()) {
		case ZE_AXIS:
		case ZE_COLOR_BAR:
		case ZE_GLOBE:
		case ZE_LINE:
		case ZE_NODE:
		case ZE_PLOT:
		case ZE_POINT:
		case ZE_POLYGON:
		case ZE_TEXT:
			((zeTransform*)o)->rotateY(a);
			break;
		default:
			api_runtime_error(ctx, "unexpected object type in calling rotatey");
	}
	return 0;
}

static void* set_rotatez(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeObject *o = (zeObject*)api_get_ptr(ctx, args[0]);
	double a = api_get_number(ctx, args[1]);
	switch (o->getType()) {
		case ZE_AXIS:
		case ZE_COLOR_BAR:
		case ZE_GLOBE:
		case ZE_LINE:
		case ZE_NODE:
		case ZE_PLOT:
		case ZE_POINT:
		case ZE_POLYGON:
		case ZE_TEXT:
			((zeTransform*)o)->rotateZ(a);
			break;
		default:
			api_runtime_error(ctx, "unexpected object type in calling rotatez");
	}
	return 0;
}

static void* set_scale(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeObject *o = (zeObject*)api_get_ptr(ctx, args[0]);
	double x = api_get_number(ctx, args[1]);
	double y = api_get_number(ctx, args[2]);
	double z = api_get_number(ctx, args[3]);
	ZExyz xyz = {x, y, z};
	switch (o->getType()) {
		case ZE_AXIS:
		case ZE_COLOR_BAR:
		case ZE_GLOBE:
		case ZE_LINE:
		case ZE_NODE:
		case ZE_PLOT:
		case ZE_POINT:
		case ZE_POLYGON:
		case ZE_TEXT:
			((zeTransform*)o)->scale(xyz);
			break;
		default:
			api_runtime_error(ctx, "unexpected object type in calling scale");
	}
	return 0;
}

static void* set_translate(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeObject *o = (zeObject*)api_get_ptr(ctx, args[0]);
	double x = api_get_number(ctx, args[1]);
	double y = api_get_number(ctx, args[2]);
	double z = api_get_number(ctx, args[3]);
	ZExyz xyz = {x, y, z};
	switch (o->getType()) {
		case ZE_AXIS:
		case ZE_COLOR_BAR:
		case ZE_GLOBE:
		case ZE_LINE:
		case ZE_NODE:
		case ZE_PLOT:
		case ZE_POINT:
		case ZE_POLYGON:
		case ZE_TEXT:
			((zeTransform*)o)->translate(xyz);
			break;
		default:
			api_runtime_error(ctx, "unexpected object type in calling translate");
	}
	return 0;
}

static void* enable_object(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeObject *o = (zeObject*)api_get_ptr(ctx, args[0]);
	switch (o->getType()) {
		case ZE_AXIS:
		case ZE_COLOR_BAR:
		case ZE_GLOBE:
		case ZE_LIGHT:
		case ZE_LINE:
		case ZE_NODE:
		case ZE_PLOT:
		case ZE_POINT:
		case ZE_POLYGON:
		case ZE_TEXT:
			o->enable();
			break;
		default:
			api_runtime_error(ctx, "unexpected object type in calling enable");
	}
	return 0;
}

static void* disable_object(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeObject *o = (zeObject*)api_get_ptr(ctx, args[0]);
	switch (o->getType()) {
		case ZE_AXIS:
		case ZE_COLOR_BAR:
		case ZE_GLOBE:
		case ZE_LIGHT:
		case ZE_LINE:
		case ZE_NODE:
		case ZE_PLOT:
		case ZE_POINT:
		case ZE_POLYGON:
		case ZE_TEXT:
			o->disable();
			break;
		default:
			api_runtime_error(ctx, "unexpected object type in calling disable");
	}
	return 0;
}

static zeFreetype* get_font(zeObject *o)
{
	zeObject *p = o->getParent();
	while (p) {
		if (p->getType()==ZE_RENDER) {
			return ((zeRender*)p)->getFont();
		}
		p = p->getParent();
	}
	return 0;
}

/////////////////////////////////////////////////////////////////////

static void* zg_type_AXIS(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeAxis *o = (zeAxis*)api_get_user(ctx, args[0], ZE_AXIS);
	const char *s = api_get_string(ctx, args[1]);
	bool flag = true;
	if (nargs > 2 && api_get_integer(ctx, args[2]) == 0) flag = false;
	if (s[0] == 'x' || s[0] == 'X')
		o->asXAxis(flag);
	else if (s[0] == 'y' || s[0] == 'Y')
		o->asYAxis(flag);
	else if (s[0] == 'z' || s[0] == 'Z')
		o->asZAxis(flag);
	else
		api_input_error(ctx);
	return 0;
}

static void* zg_font_AXIS(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeAxis *o = (zeAxis*)api_get_user(ctx, args[0], ZE_AXIS);
	o->setFontSize(api_get_integer(ctx, args[1]));
	return 0;
}

static void* zg_color_AXIS(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeAxis *o = (zeAxis*)api_get_user(ctx, args[0], ZE_AXIS);
	ZEcolor c = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]),
				  1 };
	o->setColor(c);
	return 0;
}

static void* zg_range_AXIS(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zeAxis *o = (zeAxis*)api_get_user(ctx, args[0], ZE_AXIS);
	if (!o->setAxisRange(api_get_number(ctx, args[1]),
						 api_get_number(ctx, args[2])))
		api_runtime_error(ctx, "bad axis range");
	return 0;
}

static void* zg_title_AXIS(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeAxis *o = (zeAxis*)api_get_user(ctx, args[0], ZE_AXIS);
	if (!o->setAxisLabel(api_get_string(ctx, args[1])))
		api_runtime_error(ctx, "failed to set axis title");
	return 0;
}

static void* zg_tickmarks_AXIS(void *ctx, int nargs, void** args)
{
	if (nargs < 5 || api_get_number(ctx, args[3]) == 0) api_input_error(ctx);
	zeAxis *o = (zeAxis*)api_get_user(ctx, args[0], ZE_AXIS);
	if (!o->setTickMarks(api_get_number(ctx, args[1]),
						api_get_number(ctx, args[2]),
						api_get_number(ctx, args[3]),
						api_get_integer(ctx, args[4])))
		api_runtime_error(ctx, "failed to set axis tickmarks");
	return 0;
}

static void* zg_ticksize_AXIS(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeAxis *o = (zeAxis*)api_get_user(ctx, args[0], ZE_AXIS);
	o->setTickLength(api_get_number(ctx, args[1]));
	return 0;
}

static void* zg_ticklabels_AXIS(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeAxis *o = (zeAxis*)api_get_user(ctx, args[0], ZE_AXIS);
	bool center = true;
	if (nargs > 2 && api_is_integer(args[nargs-1])) {
		center = api_get_integer(ctx, args[nargs-1]) != 0;
		nargs--;
	}
	for (int i = 1; i < nargs; i++) {
		if (!o->addTickLabel(api_get_string(ctx, args[i]), center))
			api_runtime_error(ctx, "failed to set axis ticklabels");
	}
	return 0;
}

static void* zg_tickdigits_AXIS(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeAxis *o = (zeAxis*)api_get_user(ctx, args[0], ZE_AXIS);
	bool flag = false;
	if (nargs > 2) flag = (api_get_integer(ctx, args[2]) != 0);
	o->setTickLabelDigit(api_get_integer(ctx, args[1]), flag);
	return 0;
}

static void* zg_linewidth_AXIS(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeAxis *o = (zeAxis*)api_get_user(ctx, args[0], ZE_AXIS);
	o->setLineWidth(api_get_number(ctx, args[1]));
	return 0;
}

static void* zg_showticks_AXIS(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeAxis *o = (zeAxis*)api_get_user(ctx, args[0], ZE_AXIS);
	bool yn = (api_get_integer(ctx, args[1]) != 0);
	o->showTicks(yn);
	return 0;
}

static void* zg_showlabels_AXIS(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeAxis *o = (zeAxis*)api_get_user(ctx, args[0], ZE_AXIS);
	bool yn = (api_get_integer(ctx, args[1]) != 0);
	o->showLabels(yn);
	return 0;
}

static void* zg_smooth_AXIS(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeAxis *o = (zeAxis*)api_get_user(ctx, args[0], ZE_AXIS);
	o->setAntialias(api_get_integer(ctx, args[1]) != 0);
	return 0;
}

/////////////////////////////////////////////////////////////////////

static void* zg_add_COLOR(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zeColor *o = (zeColor*)api_get_user(ctx, args[0], ZE_COLOR);
	if (api_is_user(args[1])) {
		void *ptr = api_get_ptr(ctx, args[1]);
		size_t i, n = api_get_integer(ctx, args[2]);
		for (i = 3; i < n; i += 4) {
			ZEcolor rgba = {((double*)ptr)[i-3], ((double*)ptr)[i-2], ((double*)ptr)[i-1], ((double*)ptr)[i]};
			o->add(rgba);
		}
		return 0;
	}
	if (nargs < 5) api_input_error(ctx);
	for (int i = 4; i < nargs; i += 4) {
		ZEcolor rgba = { api_get_number(ctx, args[i-3]),
						 api_get_number(ctx, args[i-2]),
						 api_get_number(ctx, args[i-1]),
						 api_get_number(ctx, args[i  ]) };
		o->add(rgba);
	}
	return 0;
}

static void* zg_clear_COLOR(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeColor *o = (zeColor*)api_get_user(ctx, args[0], ZE_COLOR);
	o->clear();
	return 0;
}

static void* zg_size_COLOR(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeColor *o = (zeColor*)api_get_user(ctx, args[0], ZE_COLOR);
	return api_create_integer(ctx, o->size());
}

static void* zg_print_COLOR(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeColor *o = (zeColor*)api_get_user(ctx, args[0], ZE_COLOR);
	FILE *f = stdout;
	if (nargs > 1) {
		f = open_file(api_get_string(ctx, args[1]), "a");
		if (!f) api_runtime_error(ctx, "failed to open file");
	}
	fprintf(f, "<color>\n");
	int n = o->size();
	for (int i = 0; i < n; i++) {
		ZEcolor c = o->get(i);
		fprintf(f, " %.2f, %.2f, %.2f, %.2f\n", c.r, c.g, c.b, c.a);
	}
	fprintf(f, "</color>\n");
	fflush(f);
	return 0;
}

static void* zg_get_COLOR(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeColor *o = (zeColor*)api_get_user(ctx, args[0], ZE_COLOR);
	int idx = api_get_integer(ctx, args[1]);
	if (idx < 0 || idx >= o->size()) api_input_error(ctx);
	ZEcolor c = o->get(idx);
	void *arr = api_create_array(ctx, 4);
	api_set_array_object(ctx, arr, "0", api_create_real(0,c.r));
	api_set_array_object(ctx, arr, "1", api_create_real(0,c.g));
	api_set_array_object(ctx, arr, "2", api_create_real(0,c.b));
	api_set_array_object(ctx, arr, "3", api_create_real(0,c.a));
	return arr;
}

static void* zg_set_COLOR(void *ctx, int nargs, void** args)
{
	if (nargs < 6) api_input_error(ctx);
	zeColor *o = (zeColor*)api_get_user(ctx, args[0], ZE_COLOR);
	int idx = api_get_integer(ctx, args[1]);
	if (idx < 0 || idx >= o->size()) api_input_error(ctx);
	ZEcolor c = { api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]),
				  api_get_number(ctx, args[4]),
				  api_get_number(ctx, args[5]) };
	o->set(idx, c);
	return 0;
}

/////////////////////////////////////////////////////////////////////

static void* zg_size_COLOR_BAR(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zeColorBar *o = (zeColorBar*)api_get_user(ctx, args[0], ZE_COLOR_BAR);
	o->setSize(api_get_number(ctx, args[1]), api_get_number(ctx, args[2]));
	return 0;
}

static void* zg_discrete_COLOR_BAR(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeColorBar *o = (zeColorBar*)api_get_user(ctx, args[0], ZE_COLOR_BAR);
	bool yn = (api_get_integer(ctx, args[1]) == 0);
	o->gradientType(yn);
	return 0;
}

static void* zg_labeldigits_COLOR_BAR(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeColorBar *o = (zeColorBar*)api_get_user(ctx, args[0], ZE_COLOR_BAR);
	bool flag = false;
	if (nargs > 2) flag = api_get_integer(ctx, args[2]) != 0;
	o->setLabelFormat(api_get_integer(ctx, args[1]), flag);
	return 0;
}

static void* zg_font_COLOR_BAR(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeColorBar *o = (zeColorBar*)api_get_user(ctx, args[0], ZE_COLOR_BAR);
	o->setFontSize(api_get_integer(ctx, args[1]));
	return 0;
}

static void* zg_color_COLOR_BAR(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeColorBar *o = (zeColorBar*)api_get_user(ctx, args[0], ZE_COLOR_BAR);
	ZEcolor rgba = { api_get_number(ctx, args[1]),
					 api_get_number(ctx, args[2]),
					 api_get_number(ctx, args[3]),
					 1 };
	o->setColor(rgba);
	return 0;
}

static void* zg_add_COLOR_BAR(void *ctx, int nargs, void** args)
{
	if (nargs < 5) api_input_error(ctx);
	zeColorBar *o = (zeColorBar*)api_get_user(ctx, args[0], ZE_COLOR_BAR);
	for (int i = 4; i < nargs; i += 4) {
		ZEcolor rgba = { api_get_number(ctx, args[i-3]),
						 api_get_number(ctx, args[i-2]),
						 api_get_number(ctx, args[i-1]),
						 1 };
		o->addColor(rgba, api_get_number(ctx, args[i]));
	}
	return 0;
}

static void* zg_get_COLOR_BAR(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeColorBar *o = (zeColorBar*)api_get_user(ctx, args[0], ZE_COLOR_BAR);
	ZEcolor rgba = o->getColor(api_get_number(ctx, args[1]));
	void *arr = api_create_array(ctx, 3);
	api_set_array_object(ctx, arr, "0", api_create_real(0, rgba.r));
	api_set_array_object(ctx, arr, "1", api_create_real(0, rgba.g));
	api_set_array_object(ctx, arr, "2", api_create_real(0, rgba.b));
	return arr;
}

static void* zg_clear_COLOR_BAR(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeColorBar *o = (zeColorBar*)api_get_user(ctx, args[0], ZE_COLOR_BAR);
	o->clear();
	return 0;
}

static void* zg_linewidth_COLOR_BAR(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeColorBar *o = (zeColorBar*)api_get_user(ctx, args[0], ZE_COLOR_BAR);
	o->lineWidth(api_get_number(ctx, args[1]));
	return 0;
}

static void* zg_interpolate_COLOR_BAR(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeColorBar *o = (zeColorBar*)api_get_user(ctx, args[0], ZE_COLOR_BAR);
	o->interpolate(api_get_integer(ctx, args[1]));
	return 0;
}

/////////////////////////////////////////////////////////////////////

static void* zg_lookat_GLOBE(void *ctx, int nargs, void** args)
{
	if (nargs < 7) api_input_error(ctx);
	zeGlobe *o = (zeGlobe*)api_get_user(ctx, args[0], ZE_GLOBE);
	o->lookAt(  api_get_number(ctx, args[1]),
				api_get_number(ctx, args[2]),
				api_get_number(ctx, args[3]),
				api_get_number(ctx, args[4]),
				api_get_number(ctx, args[5]),
				api_get_number(ctx, args[6]) );
	ZExyz xyz1 = o->getTranslate();
	ZExyz xyz2 = o->getRotate();
	void *arr = api_create_array(ctx, 5);
	api_set_array_object(ctx, arr, "0", api_create_real(0,xyz1.x));
	api_set_array_object(ctx, arr, "1", api_create_real(0,xyz1.y));
	api_set_array_object(ctx, arr, "2", api_create_real(0,xyz1.z));
	api_set_array_object(ctx, arr, "3", api_create_real(0,xyz2.z));
	api_set_array_object(ctx, arr, "4", api_create_real(0,xyz2.x));
	return arr;
}

static void* zg_surface_GLOBE(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeGlobe *o = (zeGlobe*)api_get_user(ctx, args[0], ZE_GLOBE);
	real_t r = api_get_number(ctx, args[1]);
	int iters = 0;
	if (nargs > 2) iters = api_get_integer(ctx, args[2]);
	zePolygon *p = o->surface(r, iters);
	return api_create_user(ctx, p, 0, 0, ZE_POLYGON);
}

static void* zg_texture_GLOBE(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zeGlobe *o = (zeGlobe*)api_get_user(ctx, args[0], ZE_GLOBE);
	double radius = api_get_number(ctx, args[1]);
	bool atlantic = false;
	if (nargs < 5) {
		if (nargs > 3) atlantic = api_get_integer(ctx, args[3]) != 0;
		zePolygon *p = o->texture(radius, api_get_string(ctx, args[2]), atlantic);
		return api_create_user(ctx, p, 0, 0, ZE_POLYGON);
	}
	double *data = (double*)api_get_ptr(ctx, args[2]);
	int nr = api_get_integer(ctx, args[3]);
	int nc = api_get_integer(ctx, args[4]);
	zeColorBar *cbar = (zeColorBar*)api_get_user(ctx, args[5], ZE_COLOR_BAR);
	if (nargs > 6) atlantic = api_get_integer(ctx, args[6]) != 0;
	zePolygon *p = o->texture(radius, data, nr, nc, atlantic, cbar);
	return api_create_user(ctx, p, 0, 0, ZE_POLYGON);
}

static void* zg_grid_GLOBE(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeGlobe *o = (zeGlobe*)api_get_user(ctx, args[0], ZE_GLOBE);
	double width = 1;
	if (nargs > 2) width = api_get_number(ctx, args[2]);
	zeNode *node = o->grid(api_get_integer(ctx, args[1]), width);
	return api_create_user(ctx, node, 0, 0, ZE_NODE);
}

static void* zg_focus_GLOBE(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zeGlobe *o = (zeGlobe*)api_get_user(ctx, args[0], ZE_GLOBE);
	o->focus(api_get_number(ctx, args[1]), api_get_number(ctx, args[2]));
	ZExyz xyz = o->getRotate();
	void *arr = api_create_array(ctx, 2);
	api_set_array_object(ctx, arr, "0", api_create_real(0,xyz.z));
	api_set_array_object(ctx, arr, "1", api_create_real(0,xyz.x));
	return arr;
}

static void* zg_add_GLOBE(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeGlobe *o = (zeGlobe*)api_get_user(ctx, args[0], ZE_GLOBE);
	zeObject *obj = zg_create(ctx, api_get_string(ctx, args[1]), get_font(o));
	if (nargs == 2) {
		o->add(obj);
	}
	else if (api_is_user(args[2])) {
		if (nargs < 4) api_input_error(ctx);
		void *ptr = api_get_ptr(ctx, args[2]);
		int n = api_get_integer(ctx, args[3]);
		for (int i=2; i<n; i++) {
			o->anchor(obj, ((double*)ptr)[i-1], ((double*)ptr)[i-1], ((double*)ptr)[i]);
		}
	}
	else {
		if (nargs < 4) api_input_error(ctx);
		o->anchor(obj,
				api_get_number(ctx, args[2]),
				api_get_number(ctx, args[3]),
				api_get_number(ctx, args[4]));
	}
	return api_create_user(ctx, obj, 0, 0, obj->getType());
}

static void* zg_line_GLOBE(void *ctx, int nargs, void** args)
{
	if (nargs < 5) api_input_error(ctx);
	zeGlobe *o = (zeGlobe*)api_get_user(ctx, args[0], ZE_GLOBE);
	zeLine *l = o->line(
		api_get_number(ctx, args[1]),
		api_get_number(ctx, args[2]),
		api_get_number(ctx, args[3]),
		api_get_number(ctx, args[4]));
	return api_create_user(ctx, l, 0, 0, ZE_LINE);
}

static void* zg_gshhs_GLOBE(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeGlobe *o = (zeGlobe*)api_get_user(ctx, args[0], ZE_GLOBE);
	double width = 1;
	if (nargs > 2) {
		width = api_get_number(ctx, args[2]);
		if (width <= 0) width = 1;
	}
	bool smooth = false;
	if (nargs > 3) {
		smooth = api_get_integer(ctx, args[3])!=0;
	}
	zeNode *node = o->gshhs(api_get_string(ctx, args[1]), width, smooth);
	return api_create_user(ctx, node, 0, 0, ZE_NODE);
}

static void* zg_sun_GLOBE(void *ctx, int nargs, void** args)
{
	if (nargs < 5) api_input_error(ctx);
	zeGlobe *o = (zeGlobe*)api_get_user(ctx, args[0], ZE_GLOBE);
	int day = api_get_integer(ctx, args[1]);
	int hour = api_get_integer(ctx, args[2]);
	int minute = api_get_integer(ctx, args[3]);
	int second = api_get_integer(ctx, args[4]);
	double r = 2.0*PI*(day-1+(hour-12.0)/24.0)/365.0;
	double lat = 0.006918-0.399912*cos(r)+0.070257*sin(r)-0.006758*cos(2*r)+0.000907*sin(2*r)-0.002697*cos(3*r)+0.00148*sin(3*r);
	lat *= 180.0/PI;
	double lon = (12.0-hour-minute/60.0-second/3600.0)*360.0/24.0;
	double x, y, z;
	ll2xyz(x, y, z, lon, lat, 1);
	ZExyz xyz = {x, y, z};
	o->light()->enable();
	o->light()->setPosition(xyz);
	return api_create_user(ctx, o->light(), 0, 0, ZE_LIGHT);
}

static void* zg_light_GLOBE(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zeGlobe *o = (zeGlobe*)api_get_user(ctx, args[0], ZE_GLOBE);
	double x = api_get_number(ctx, args[1]);
	double y = api_get_number(ctx, args[2]);
	double r = o->radius();
	double z;
	ll2xyz(x, y, z, x, y, r);
	ZExyz xyz = {x, y, z};
	o->light()->enable();
	o->light()->setPosition(xyz);
	return api_create_user(ctx, o->light(), 0, 0, ZE_LIGHT);
}

static void* zg_clear_GLOBE(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeGlobe *o = (zeGlobe*)api_get_user(ctx, args[0], ZE_GLOBE);
	o->clear();
	return 0;
}

void * zg_ll2xyz_GLOBE(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zeGlobe *o = (zeGlobe*)api_get_user(ctx, args[0], ZE_GLOBE);
	double r = o->radius();
	double x, y, z;
	ll2xyz(x, y, z, api_get_number(ctx, args[1]), api_get_number(ctx, args[2]), r);
	void * arr = api_create_array(ctx, 3);
	api_set_array_object(ctx, arr, "0", api_create_real(0, x));
	api_set_array_object(ctx, arr, "1", api_create_real(0, y));
	api_set_array_object(ctx, arr, "2", api_create_real(0, z));
	return arr;
}

void * zg_xyz2ll_GLOBE(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeGlobe *o = (zeGlobe*)api_get_user(ctx, args[0], ZE_GLOBE);
	double lon, lat;
	xyz2ll(lon, lat, api_get_number(ctx, args[1]), api_get_number(ctx, args[2]), api_get_number(ctx, args[3]));
	void * arr = api_create_array(ctx, 3);
	api_set_array_object(ctx, arr, "0", api_create_real(0, lon));
	api_set_array_object(ctx, arr, "1", api_create_real(0, lat));
	return arr;
}

/////////////////////////////////////////////////////////////////////

static void* zg_position_LIGHT(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeLight *o = (zeLight*)api_get_user(ctx, args[0], ZE_LIGHT);
	ZExyz xyz = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]) };
	o->setPosition(xyz);
	return 0;
}

static void* zg_ambient_LIGHT(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeLight *o = (zeLight*)api_get_user(ctx, args[0], ZE_LIGHT);
	ZEcolor c = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]),
				  1 };
	if (nargs > 4) c.a = api_get_number(ctx, args[4]);
	o->setAmbient(c);
	return 0;
}

static void* zg_add_LIGHT(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeLight *o = (zeLight*)api_get_user(ctx, args[0], ZE_LIGHT);
	zeObject *obj = zg_create(ctx, api_get_string(ctx, args[1]), get_font(o));
	o->add(obj);
	return api_create_user(ctx, obj, 0, 0, obj->getType());
}

static void* zg_clear_LIGHT(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeLight *o = (zeLight*)api_get_user(ctx, args[0], ZE_LIGHT);
	o->clear();
	return 0;
}

/////////////////////////////////////////////////////////////////////

static void* zg_solid_LINE(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeLine *o = (zeLine*)api_get_user(ctx, args[0], ZE_LINE);
	o->asSolidLine(1, api_get_number(ctx, args[1]));
	return 0;
}

static void* zg_dot_LINE(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeLine *o = (zeLine*)api_get_user(ctx, args[0], ZE_LINE);
	double a = api_get_number(ctx, args[1]);
	double b = a;
	if (nargs > 2) b = api_get_number(ctx, args[2]);
	o->asDotLine(b, a);
	return 0;
}

static void* zg_dash_LINE(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeLine *o = (zeLine*)api_get_user(ctx, args[0], ZE_LINE);
	double a = api_get_number(ctx, args[1]);
	double b = 2.5*a;
	if (nargs > 2) b = api_get_number(ctx, args[2]);
	o->asDashLine(b, a);
	return 0;
}

static void* zg_dotdash_LINE(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeLine *o = (zeLine*)api_get_user(ctx, args[0], ZE_LINE);
	double a = api_get_number(ctx, args[1]);
	double b = a;
	if (nargs > 2) b = api_get_number(ctx, args[2]);
	o->asDotDashLine(b, a);
	return 0;
}

static void* zg_type_LINE(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeLine *o = (zeLine*)api_get_user(ctx, args[0], ZE_LINE);
	const char *type = api_get_string(ctx, args[1]);
	if (strcmp(type, "lines") == 0)
		o->asLines();
	else if (strcmp(type, "loop") == 0)
		o->asLineLoop();
	else if (strcmp(type, "strip") == 0)
		o->asLineStrip();
	else
		api_input_error(ctx);
	return 0;
}

static void* zg_smooth_LINE(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeLine *o = (zeLine*)api_get_user(ctx, args[0], ZE_LINE);
	bool yn = api_get_integer(ctx, args[1]) != 0;
	o->setAntialias(yn);
	return 0;
}

static void* zg_color_LINE(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeLine *o = (zeLine*)api_get_user(ctx, args[0], ZE_LINE);
	if (nargs == 1) {
		zeColor *p = o->getVertexColor();
		if (!p) {
			p = new zeColor;
			o->setVertexColor(p);
		}
		return api_create_user(ctx, p, 0, 0, ZE_COLOR);
	}
	if (nargs == 2) {
		o->setVertexColor((zeColor*)api_get_user(ctx, args[1], ZE_COLOR));
		return 0;
	}
	if (nargs < 4) api_input_error(ctx);
  	ZEcolor c = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]), 
				  1 };
	if (nargs > 4) c.a = api_get_number(ctx, args[4]);
	o->setColor(c);
	return 0;
}

static void* zg_vertex_LINE(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeLine *o = (zeLine*)api_get_user(ctx, args[0], ZE_LINE);
	if (nargs == 1) {
		zeVertex *p = o->getVertex();
		if (!p) {
			p = new zeVertex;
			o->setVertex(p);
		}
		return api_create_user(ctx, p, 0, 0, ZE_VERTEX);
	}
	o->setVertex((zeVertex*)api_get_user(ctx, args[1], ZE_VERTEX));
	return 0;
}

static void* zg_normal_LINE(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeLine *o = (zeLine*)api_get_user(ctx, args[0], ZE_LINE);
	if (nargs == 1) {
		zeVertex *p = o->getVertex();
		if (!p) {
			p = new zeVertex;
			o->setVertexNormal(p);
		}
		return api_create_user(ctx, p, 0, 0, ZE_VERTEX);
	}
	o->setVertexNormal((zeVertex*)api_get_user(ctx, args[1], ZE_VERTEX));
	return 0;
}

void * zg_contour_LINE(void *ctx, int nargs, void** args)
{
	if (nargs < 7) api_input_error(ctx);
	zeLine *line = (zeLine*)api_get_user(ctx, args[0], ZE_LINE);
	zeVertex *vertex = new zeVertex;
	line->setVertex(vertex);
	line->asLines();
	line->setAntialias(true);
	if (api_is_user(args[3])) {
		double *X  = (double*)api_get_ptr(ctx, args[1]);
		double *Y  = (double*)api_get_ptr(ctx, args[2]);
		double *Z  = (double*)api_get_ptr(ctx, args[3]);
		int    *I  =    (int*)api_get_ptr(ctx, args[4]);
		int     n  =      api_get_integer(ctx, args[5]);
		double iso =       api_get_number(ctx, args[6]);
		int i, i1, i2, i3, k = 0;
		for (i = 0; i < n; i++) {
			i1 = I[k  ];
			i2 = I[k+1];
			i3 = I[k+2];
			ZExyz p1 = {X[i1], Y[i1], Z[i1]};
			ZExyz p2 = {X[i2], Y[i2], Z[i2]};
			ZExyz p3 = {X[i3], Y[i3], Z[i3]};
			contour3_line(vertex, iso, p1, p2, p3);
			k += 3;
		}
	}
	else {
		double *Z  = (double*)api_get_ptr(ctx, args[1]);
		double *X  = (double*)api_get_ptr(ctx, args[2]);
		int  ncol  =      api_get_integer(ctx, args[3]);
		double *Y  = (double*)api_get_ptr(ctx, args[4]);
		int  nrow  =      api_get_integer(ctx, args[5]);
		double iso =       api_get_number(ctx, args[6]);
		for (int i = 1; i < nrow; i++) {
			int k2 = i * ncol;
			int k1 = k2 - ncol;
			double y1 = Y[i-1];
			double y2 = Y[i];
				for (int j = 1; j < ncol; j++) {
				double x1 = X[j-1];
				double x2 = X[j];
				double z1 = Z[k1 + j - 1];
				double z2 = Z[k1 + j];
				double z3 = Z[k2 + j];
				double z4 = Z[k2 + j - 1];
				ZExyz p1 = {x1, y1, z1};
				ZExyz p2 = {x2, y1, z2};
				ZExyz p3 = {x2, y2, z3};
				ZExyz p4 = {x1, y2, z4};
				contour4_line(vertex, iso, p1, p2, p3, p4);
			}
		}
	}
	return api_create_user(ctx, vertex, 0, 0, ZE_VERTEX); 
}

static void* zg_vector_LINE(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeLine *o = (zeLine*) api_get_user(ctx, args[0], ZE_LINE);
	double u = api_get_number(ctx, args[1]);
	double v = api_get_number(ctx, args[2]);
	double h = api_get_number(ctx, args[3]);
	if (h <= 0) api_input_error(ctx);
	make_vector(o, u, v, h);
	return 0;
}

static void* zg_gshhs_LINE(void *ctx, int nargs, void** args)
{
	if (nargs < 6) api_input_error(ctx);
	zeLine *o = (zeLine*) api_get_user(ctx, args[0], ZE_LINE);
	int level = 1;
	if (!gshhs_coastline(o, level,
			api_get_string(ctx, args[1]),
			api_get_number(ctx, args[2]),
			api_get_number(ctx, args[3]),
			api_get_number(ctx, args[4]),
			api_get_number(ctx, args[5]))) api_input_error(ctx);
	return 0;
}

static void* zg_translate_LINE(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeLine *o = (zeLine*)api_get_user(ctx, args[0], ZE_LINE);
  	ZExyz xyz = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]) };
	o->translate(xyz);
	return 0;
}


/////////////////////////////////////////////////////////////////////

static void* zg_ambient_MATERIAL(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeMaterial *o = (zeMaterial*)api_get_user(ctx, args[0], ZE_MATERIAL);
	double a = 1;
	if (nargs > 4) a = api_get_number(ctx, args[4]);
  	ZEcolor c = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]), 
				  a };
	o->setAmbient(c);
	return 0;
}

static void* zg_diffuse_MATERIAL(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeMaterial *o = (zeMaterial*)api_get_user(ctx, args[0], ZE_MATERIAL);
	double a = 1;
	if (nargs > 4) a = api_get_number(ctx, args[4]);
  	ZEcolor c = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]), 
				  a };
	o->setDiffuse(c);
	return 0;
}

static void* zg_emission_MATERIAL(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeMaterial *o = (zeMaterial*)api_get_user(ctx, args[0], ZE_MATERIAL);
	double a = 1;
	if (nargs > 4) a = api_get_number(ctx, args[4]);
  	ZEcolor c = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]), 
				  a };
	o->setEmission(c);
	return 0;
}

static void* zg_specular_MATERIAL(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeMaterial *o = (zeMaterial*)api_get_user(ctx, args[0], ZE_MATERIAL);
	double a = 1;
	if (nargs > 4) a = api_get_number(ctx, args[4]);
  	ZEcolor c = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]), 
				  a };
	o->setSpecular(c);
	return 0;
}

static void* zg_shininess_MATERIAL(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeMaterial *o = (zeMaterial*)api_get_user(ctx, args[0], ZE_MATERIAL);
	o->setShininess(api_get_number(ctx, args[1]));
	return 0;
}

/////////////////////////////////////////////////////////////////////

static void* zg_add_NODE(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeNode *o = (zeNode*)api_get_user(ctx, args[0], ZE_NODE);
	zeObject *obj = zg_create(ctx, api_get_string(ctx, args[1]), get_font(o));
	o->add(obj);
	return api_create_user(ctx, obj, 0, 0, obj->getType());
}

static void* zg_color_NODE(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeNode *o = (zeNode*)api_get_user(ctx, args[0], ZE_NODE);
  	ZEcolor c = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]), 
				  1 };
	if (nargs > 4) c.a = api_get_number(ctx, args[4]);
	o->setColor(c);
	return 0;
}

static void* zg_rotatex_NODE(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeNode *o = (zeNode*)api_get_user(ctx, args[0], ZE_NODE);
	o->rotateX(api_get_number(ctx, args[1]));
	return 0;
}

static void* zg_rotatey_NODE(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeNode *o = (zeNode*)api_get_user(ctx, args[0], ZE_NODE);
	o->rotateY(api_get_number(ctx, args[1]));
	return 0;
}

static void* zg_rotatez_NODE(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeNode *o = (zeNode*)api_get_user(ctx, args[0], ZE_NODE);
	o->rotateZ(api_get_number(ctx, args[1]));
	return 0;
}

static void* zg_translate_NODE(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeNode *o = (zeNode*)api_get_user(ctx, args[0], ZE_NODE);
  	ZExyz xyz = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]) };
	o->translate(xyz);
	return 0;
}

static void* zg_scale_NODE(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeNode *o = (zeNode*)api_get_user(ctx, args[0], ZE_NODE);
  	ZExyz xyz = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]) };
	o->scale(xyz);
	return 0;
}

static void* zg_reset_NODE(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeNode *o = (zeNode*)api_get_user(ctx, args[0], ZE_NODE);
	o->reset();
	return 0;
}

static void* zg_clear_NODE(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeNode *o = (zeNode*)api_get_user(ctx, args[0], ZE_NODE);
	o->clear();
	return 0;
}

static void* zg_get_NODE(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeNode *o = (zeNode*)api_get_user(ctx, args[0], ZE_NODE);
	int idx = api_get_integer(ctx, args[1]);
	if (idx < 0 || idx >= o->size()) api_input_error(ctx);
	zeObject *o2 = o->get(idx);
	return api_create_user(ctx, o2, 0, 0, o2->getType());
}

static void* zg_size_NODE(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeNode *o = (zeNode*)api_get_user(ctx, args[0], ZE_NODE);
	return api_create_integer(ctx, o->size());
}

static void* zg_remove_NODE(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeNode *o = (zeNode*)api_get_user(ctx, args[0], ZE_NODE);
	zeObject *obj = (zeObject*)api_get_ptr(ctx, args[1]);
	o->remove(obj);
	return 0;
}

/////////////////////////////////////////////////////////////////////

void zg_vewport_PLOT(void *ctx, zePlot *plot, int &w, int &h)
{
	zeObject *p = plot->getParent();
	while (p && p->getType() != ZE_SCENE) p = p->getParent();
	if (!p || p->getType() != ZE_SCENE) api_runtime_error(ctx, "plot object has not been set to a sene properly");
	w = ((zeScene*)p)->width();
	h = ((zeScene*)p)->height();
}

static void* zg_add_PLOT(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zePlot *o = (zePlot*)api_get_user(ctx, args[0], ZE_PLOT);
	zeObject *obj = zg_create(ctx, api_get_string(ctx, args[1]), o->xaxis()->getFont());
	if (nargs == 2) {
		o->add(obj);
	}
	else if (api_is_user(args[2])) {
		if (nargs < 4) api_input_error(ctx);
		double *ptr = (double*)api_get_ptr(ctx, args[2]);
		int n = api_get_integer(ctx, args[3]);
		for (int i=2; i<n; i+=3) {
		  	ZExyz xyz = { ((double*)ptr)[i-2], ((double*)ptr)[i-1], ((double*)ptr)[i] };
			o->anchor(obj, xyz);
		}
		return api_create_user(ctx, obj, 0, 0, obj->getType());
	}
	else {
		if (nargs < 5) api_input_error(ctx);
		ZExyz xyz = { api_get_number(ctx,  args[2]),
					  api_get_number(ctx,  args[3]),
					  api_get_number(ctx,  args[4]) };
		o->anchor(obj, xyz);
	}
	return api_create_user(ctx, obj, 0, 0, obj->getType());
}

static void* zg_color_PLOT(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zePlot *o = (zePlot*)api_get_user(ctx, args[0], ZE_PLOT);
	ZEcolor c = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]),
				  1 };
	o->setColor(c);
	return 0;
}

static void* zg_scale_PLOT(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zePlot *o = (zePlot*)api_get_user(ctx, args[0], ZE_PLOT);
  	ZExyz xyz = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]) };
	o->scale(xyz);
	return 0;
}

static void* zg_translate_PLOT(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zePlot *o = (zePlot*)api_get_user(ctx, args[0], ZE_PLOT);
  	ZExyz xyz = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]) };
	o->translate(xyz);
	return 0;
}

static void* zg_rotate_PLOT(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zePlot *o = (zePlot*)api_get_user(ctx, args[0], ZE_PLOT);
	o->rotateZ(api_get_number(ctx, args[1]));
	o->rotateX(api_get_number(ctx, args[2]));
	return 0;
}

static void* zg_global_PLOT(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zePlot *o = (zePlot*)api_get_user(ctx, args[0], ZE_PLOT);
	int w, h;
	zg_vewport_PLOT(ctx, o, w, h);
	ZExyz xyz = o->plot2global( w, h,
								api_get_number(ctx, args[1]),
								api_get_number(ctx, args[2]),
								api_get_number(ctx, args[3]) );
	void *arr = api_create_array(ctx, 3);
	api_set_array_object(ctx, arr, "0", api_create_real(0, xyz.x));
	api_set_array_object(ctx, arr, "1", api_create_real(0, xyz.y));
	api_set_array_object(ctx, arr, "2", api_create_real(0, xyz.z));
	return arr;
}

static void* zg_local_PLOT(void *ctx, int nargs, void** args)
{
	if (nargs < 6) api_input_error(ctx);
	zePlot *o = (zePlot*)api_get_user(ctx, args[0], ZE_PLOT);
	int w, h;
	zg_vewport_PLOT(ctx, o, w, h);
	ZExyz xyz = o->global2plot( w, h,
								api_get_number(ctx, args[1]),
								api_get_number(ctx, args[2]),
								api_get_number(ctx, args[3]) );
	void *arr = api_create_array(ctx, 3);
	api_set_array_object(ctx, arr, "0", api_create_real(0, xyz.x));
	api_set_array_object(ctx, arr, "1", api_create_real(0, xyz.y));
	api_set_array_object(ctx, arr, "2", api_create_real(0, xyz.z));
	return arr;
}

static void* zg_xaxis_PLOT(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zePlot *o = (zePlot*)api_get_user(ctx, args[0], ZE_PLOT);
	if (nargs >= 4) {
	  	ZExyz xyz = { api_get_number(ctx, args[1]),
					  api_get_number(ctx, args[2]),
					  api_get_number(ctx, args[3]) };
		o->anchorXAxis(xyz);
	}
	return api_create_user(ctx, o->xaxis(), 0, 0, ZE_AXIS);
}

static void* zg_yaxis_PLOT(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zePlot *o = (zePlot*)api_get_user(ctx, args[0], ZE_PLOT);
	if (nargs >= 4) {
	  	ZExyz xyz = { api_get_number(ctx, args[1]),
					  api_get_number(ctx, args[2]),
					  api_get_number(ctx, args[3]) };
		o->anchorYAxis(xyz);
	}
	return api_create_user(ctx, o->yaxis(), 0, 0, ZE_AXIS);
}

static void* zg_zaxis_PLOT(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zePlot *o = (zePlot*)api_get_user(ctx, args[0], ZE_PLOT);
	if (nargs >= 4) {
	  	ZExyz xyz = { api_get_number(ctx, args[1]),
					  api_get_number(ctx, args[2]),
					  api_get_number(ctx, args[3]) };
		o->anchorZAxis(xyz);
	}
	return api_create_user(ctx, o->zaxis(), 0, 0, ZE_AXIS);
}

static void* zg_clabel_PLOT(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zePlot *o = (zePlot*)api_get_user(ctx, args[0], ZE_PLOT);
	int w, h;
	zg_vewport_PLOT(ctx, o, w, h);
	zeLine *l = (zeLine*)api_get_user(ctx, args[1], ZE_LINE);
	zeText *t = (zeText*)api_get_user(ctx, args[2], ZE_TEXT);
	int density = api_get_integer(ctx, args[3]);
	contour_labels(o, l, t, density, w, h, 0);
	return 0;
}

static void* zg_clear_PLOT(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zePlot *o = (zePlot*)api_get_user(ctx, args[0], ZE_PLOT);
	o->clear();
	return 0;
}

static void* zg_clip_PLOT(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zePlot *o = (zePlot*)api_get_user(ctx, args[0], ZE_PLOT);
	o->setClip(api_get_integer(ctx, args[1]) != 0);
	return 0;
}

/////////////////////////////////////////////////////////////////////

static void* zg_smooth_POINT(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zePoint *o = (zePoint*)api_get_user(ctx, args[0], ZE_POINT);
	bool yn = api_get_integer(ctx, args[1]) != 0;
	o->setAntialias(yn);
	return 0;
}

static void* zg_color_POINT(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zePoint *o = (zePoint*)api_get_user(ctx, args[0], ZE_POINT);
	if (nargs == 1) {
		zeColor *p = o->getVertexColor();
		if (!p) {
			p = new zeColor;
			o->setVertexColor(p);
		}
		return api_create_user(ctx, p, 0, 0, ZE_COLOR);
	}
	if (nargs == 2) {
		o->setVertexColor((zeColor*)api_get_user(ctx, args[1], ZE_COLOR));
		return 0;
	}
	if (nargs < 4) api_input_error(ctx);
  	ZEcolor c = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]), 
				  1 };
	if (nargs > 4) c.a = api_get_number(ctx, args[4]);
	o->setColor(c);
	return 0;
}

static void* zg_vertex_POINT(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zePoint *o = (zePoint*)api_get_user(ctx, args[0], ZE_POINT);
	if (nargs == 1) {
		zeVertex *p = o->getVertex();
		if (!p) {
			p = new zeVertex;
			o->setVertex(p);
		}
		return api_create_user(ctx, p, 0, 0, ZE_VERTEX);
	}
	o->setVertex((zeVertex*)api_get_user(ctx, args[1], ZE_VERTEX));
	return 0;
}

static void* zg_normal_POINT(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zePoint *o = (zePoint*)api_get_user(ctx, args[0], ZE_POINT);
	if (nargs == 1) {
		zeVertex *p = o->getVertexNormal();
		if (!p) {
			p = new zeVertex;
			o->setVertexNormal(p);
		}
		return api_create_user(ctx, p, 0, 0, ZE_VERTEX);
	}
	o->setVertexNormal((zeVertex*)api_get_user(ctx, args[1], ZE_VERTEX));
	return 0;
}

static void* zg_size_POINT(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zePoint *o = (zePoint*)api_get_user(ctx, args[0], ZE_POINT);
	o->setSize(api_get_number(ctx, args[1]));
	return 0;
}

static void* zg_translate_POINT(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zePoint *o = (zePoint*)api_get_user(ctx, args[0], ZE_POINT);
  	ZExyz xyz = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]) };
	o->translate(xyz);
	return 0;
}

/////////////////////////////////////////////////////////////////////

static void* zg_smooth_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zePolygon *o = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	bool yn = api_get_integer(ctx, args[1]) != 0;
	o->setAntialias(yn);
	return 0;
}

static void* zg_color_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zePolygon *o = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	if (nargs == 1) {
		zeColor *p = o->getVertexColor();
		if (!p) {
			p = new zeColor;
			o->setVertexColor(p);
		}
		return api_create_user(ctx, p, 0, 0, ZE_COLOR);
	}
	if (nargs == 2) {
		o->setVertexColor((zeColor*)api_get_user(ctx, args[1], ZE_COLOR));
		return 0;
	}
	if (nargs < 4) api_input_error(ctx);
  	ZEcolor c = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]), 
				  1 };
	if (nargs > 4) c.a = api_get_number(ctx, args[4]);
	o->setColor(c);
	return 0;
}

static void* zg_vertex_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zePolygon *o = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	if (nargs == 1) {
		zeVertex *p = o->getVertex();
		if (!p) {
			p = new zeVertex;
			o->setVertex(p);
		}
		return api_create_user(ctx, p, 0, 0, ZE_VERTEX);
	}
	o->setVertex((zeVertex*)api_get_user(ctx, args[1], ZE_VERTEX));
	return 0;
}

static void* zg_normal_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zePolygon *o = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	if (nargs == 1) {
		zeVertex *p = o->getVertexNormal();
		if (!p) {
			p = new zeVertex;
			o->setVertexNormal(p);
		}
		return api_create_user(ctx, p, 0, 0, ZE_VERTEX);
	}
	o->setVertexNormal((zeVertex*)api_get_user(ctx, args[1], ZE_VERTEX));
	return 0;
}

static void* zg_texcoord_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zePolygon *o = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	if (nargs == 1) {
		zeTexCoord *p = o->getTexCoord();
		if (!p) {
			p = new zeTexCoord;
			o->setTexCoord(p);
		}
		return api_create_user(ctx, p, 0, 0, ZE_TEXCOORD);
	}
	o->setTexCoord((zeTexCoord*)api_get_user(ctx, args[1], ZE_TEXCOORD));
	return 0;
}

static void* zg_pattern_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zePolygon *o = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	o->setStiple(api_get_string(ctx, args[1]));
	return 0;
}

static void* zg_fill_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zePolygon *o = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	int fill = api_get_integer(ctx, args[1]);
	if (fill > 0)
		o->asFilled();
	else if (fill < 0)
		o->asLines();
	else
		o->asPoints();
	return 0;
}

static void* zg_type_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zePolygon *o = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	const char *type = api_get_string(ctx, args[1]);
	if (strcmp(type, "triangles") == 0)
		o->asTriangles();
	else if (strcmp(type, "trianglestrip") == 0)
		o->asTriangleStrip();
	else if (strcmp(type, "trianglefan") == 0)
		o->asTriangleFan();
	else if (strcmp(type, "quads") == 0)
		o->asQuads();
	else if (strcmp(type, "quadstrip") == 0)
		o->asQuadStrip();
	else if (strcmp(type, "polygon") == 0)
		o->asPolygon();
	else
		api_input_error(ctx);
	return 0;
}

static void* zg_linewidth_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zePolygon *o = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	o->lineWidth(api_get_number(ctx, args[1]));
	return 0;
}

static void* zg_cull_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zePolygon *o = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	int side = api_get_integer(ctx, args[1]);
	if (side > 0)
		o->cullFront();
	else if (side < 0)
		o->cullBack();
	else
		o->cullNone();
	return 0;
}

static void* zg_clockwise_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zePolygon *o = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	bool yn = api_get_integer(ctx, args[1]) != 0;
	if (yn)
		o->clockWise();
	else
		o->counterClockWise();
	return 0;
}

static void* zg_material_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zePolygon *o = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	if (nargs == 1) {
		zeMaterial *m = o->getMaterial();
		if (!m) {
			m = new zeMaterial;
			o->setMaterial(m);
		}
		return api_create_user(ctx, m, 0, 0, ZE_MATERIAL);
	}
	o->setMaterial((zeMaterial*)api_get_user(ctx, args[1], ZE_MATERIAL));
	return 0;
}

static void* zg_texture_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zePolygon *o = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	if (nargs == 1) {
		zeTexture *t = o->getTexture();
		if (!t) {
			t = new zeTexture;
			o->setTexture(t);
		}
		return api_create_user(ctx, t, 0, 0, ZE_TEXTURE);
	}
	o->setTexture((zeTexture*)api_get_user(ctx, args[1], ZE_TEXTURE));
	return 0;
}

void * zg_isosurface_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 9) api_input_error(ctx);

	zePolygon *poly = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	double *W = (double*)api_get_ptr(ctx, args[1]);
	double *X = (double*)api_get_ptr(ctx, args[2]);
	int ncol        =api_get_integer(ctx, args[3]);
	double *Y = (double*)api_get_ptr(ctx, args[4]);
	int nrow        =api_get_integer(ctx, args[5]);
	double *Z = (double*)api_get_ptr(ctx, args[6]);
	int nlvl        =api_get_integer(ctx, args[7]);
	double iso      = api_get_number(ctx, args[8]);

	zeVertex *vert  = new zeVertex;
	poly->setVertex(vert);
	zeVertex *norm  = new zeVertex;
	poly->setVertexNormal(norm);
	poly->asTriangles();

	for (int lvl = 1; lvl < nlvl; lvl++) {
		int k2 = lvl * nrow * ncol;
		int k1 = k2 - nrow * ncol;
		double z1 = Z[lvl-1];
		double z2 = Z[lvl];
		double z0 = 0.5 * (z1 + z2);

		for (int row = 1; row < nrow; row++) {
			int j2 = row * ncol;
			int j1 = j2 - ncol;
			double y1 = Y[row-1];
			double y2 = Y[row];
			double y0 = 0.5 * (y1 + y2);

			for (int col = 1; col < ncol; col++) {
				double x1 = X[col-1];
				double x2 = X[col];
				double x0 = 0.5 * (x1 + x2);

				double w1 = W[k1 + j1 + col - 1];
				double w2 = W[k1 + j1 + col];
				double w3 = W[k1 + j2 + col];
				double w4 = W[k1 + j2 + col - 1];
				double w5 = W[k2 + j1 + col - 1];
				double w6 = W[k2 + j1 + col];
				double w7 = W[k2 + j2 + col];
				double w8 = W[k2 + j2 + col - 1];

				double w0   = (w1 + w2 + w3 + w4 + w5 + w6 + w7 + w8) / 8.0;
				double wxy1 = (w1 + w2 + w3 + w4) / 4.0;
				double wxy2 = (w5 + w6 + w7 + w8) / 4.0;
				double wxz1 = (w1 + w2 + w5 + w6) / 4.0;
				double wxz2 = (w3 + w4 + w7 + w8) / 4.0;
				double wyz1 = (w1 + w4 + w5 + w8) / 4.0;
				double wyz2 = (w2 + w3 + w6 + w7) / 4.0;

				ZEvolume p1 = {x1, y1, z1, w1};
				ZEvolume p2 = {x2, y1, z1, w2};
				ZEvolume p3 = {x2, y2, z1, w3};
				ZEvolume p4 = {x1, y2, z1, w4};
				ZEvolume p5 = {x1, y1, z2, w5};
				ZEvolume p6 = {x2, y1, z2, w6};
				ZEvolume p7 = {x2, y2, z2, w7};
				ZEvolume p8 = {x1, y2, z2, w8};

				ZEvolume p0   = {x0, y0, z0, w0  };
				ZEvolume pxy1 = {x0, y0, z1, wxy1};
				ZEvolume pxy2 = {x0, y0, z2, wxy2};
				ZEvolume pxz1 = {x0, y1, z0, wxz1};
				ZEvolume pxz2 = {x0, y2, z0, wxz2};
				ZEvolume pyz1 = {x1, y0, z0, wyz1};
				ZEvolume pyz2 = {x2, y0, z0, wyz2};

				iso_surface(vert, iso, p0, pxy1, p1, p2);
				iso_surface(vert, iso, p0, pxy1, p2, p3);
				iso_surface(vert, iso, p0, pxy1, p3, p4);
				iso_surface(vert, iso, p0, pxy1, p4, p1);

				iso_surface(vert, iso, p0, pxy2, p6, p5);
				iso_surface(vert, iso, p0, pxy2, p7, p6);
				iso_surface(vert, iso, p0, pxy2, p8, p7);
				iso_surface(vert, iso, p0, pxy2, p5, p8);

				iso_surface(vert, iso, p0, pxz1, p2, p1);
				iso_surface(vert, iso, p0, pxz1, p6, p2);
				iso_surface(vert, iso, p0, pxz1, p5, p6);
				iso_surface(vert, iso, p0, pxz1, p1, p5);

				iso_surface(vert, iso, p0, pxz2, p4, p3);
				iso_surface(vert, iso, p0, pxz2, p3, p7);
				iso_surface(vert, iso, p0, pxz2, p7, p8);
				iso_surface(vert, iso, p0, pxz2, p8, p4);

				iso_surface(vert, iso, p0, pyz1, p1, p4);
				iso_surface(vert, iso, p0, pyz1, p4, p8);
				iso_surface(vert, iso, p0, pyz1, p8, p5);
				iso_surface(vert, iso, p0, pyz1, p5, p1);

				iso_surface(vert, iso, p0, pyz2, p3, p2);
				iso_surface(vert, iso, p0, pyz2, p7, p3);
				iso_surface(vert, iso, p0, pyz2, p6, p7);
				iso_surface(vert, iso, p0, pyz2, p2, p6);
			}
		}
	}

	for (int k = 2; k < vert->size(); k += 3) {
		ZExyz p1 = vert->get(k-2);
		ZExyz p2 = vert->get(k-1);
		ZExyz p3 = vert->get(k);
		ZExyz p = cross_product3(p1, p3, p2);
		normalize_vertex(p);
		norm->add(p);
		norm->add(p);
		norm->add(p);
	}

	return 0; 
}

void * zg_box_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zePolygon *poly = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	double size = api_get_number(ctx, args[1]);
	make_box(poly, size);
	return 0;
}

void * zg_sphere_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zePolygon *poly = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	double radius  = api_get_number(ctx, args[1]);
	if (radius < 1) api_input_error(ctx);
	int iters = 2 + log(4.0*3.1416*radius*radius/200.0)/log(4.0);
	if (nargs > 2) iters = api_get_integer(ctx, args[2]);
	if (iters < 1) iters = 1;
	if (iters > 8) iters = 8;
	make_sphere(poly, radius, iters);
	return 0;
}

void * zg_cone_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zePolygon *poly = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	double h = api_get_number(ctx, args[1]);
	double r = api_get_number(ctx, args[2]);
	if (h <= 0 || r <= 0) api_input_error(ctx);
	make_cone(poly, h, r);
	return 0;
}

void * zg_disk_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zePolygon *poly = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	double r = api_get_number(ctx, args[1]);
	if (r <= 0) api_input_error(ctx);
	make_disk(poly, r);
	return 0;
}

void * zg_cylinder_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zePolygon *poly = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	double h = api_get_number(ctx, args[1]);
	double r = api_get_number(ctx, args[2]);
	if (h <= 0 || r <= 0) api_input_error(ctx);
	make_cylinder(poly, h, r);
	return 0;
}

void * zg_torus_POLYGON(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zePolygon *poly = (zePolygon*)api_get_user(ctx, args[0], ZE_POLYGON);
	double R = api_get_number(ctx, args[1]);
	double r = api_get_number(ctx, args[2]);
	if (R <= 0 || r <= 0) api_input_error(ctx);
	make_torus(poly, R, r);
	return 0;
}

/////////////////////////////////////////////////////////////////////

void delete_render(void *ptr) {	delete (zeRender*)ptr; }

static void* zg_create_RENDER(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	int w = api_get_integer(ctx, args[0]);
	int h = api_get_integer(ctx, args[1]);
	int d = 16;
	if (nargs > 2) d = api_get_integer(ctx, args[2]);
	zeRender *o = new zeRender;
	o->initialize(w, h, d);
	return api_create_user(ctx, o, 0, delete_render, ZE_RENDER);
}

static void* zg_bgcolor_RENDER(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeRender *o = (zeRender*)api_get_user(ctx, args[0], ZE_RENDER);
  	ZEcolor c = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]), 
				  1 };
	o->setColor(c);
	return 0;
}

static void* zg_font_RENDER(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeRender *o = (zeRender*)api_get_user(ctx, args[0], ZE_RENDER);
	o->getFont()->setFont(api_get_string(ctx, args[1]));
	return 0;
}

static void* zg_tofile_RENDER(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	void *buf = 0;
	long bytes;
__TRY__
	zeRender *o = (zeRender*)api_get_user(ctx, args[0], ZE_RENDER);
	const char *s = api_get_string(ctx, args[1]);
	bytes = o->render2File(s);
	buf = o->buffer();
__CATCH__
	if (buf) {
		void *arr = api_create_array(ctx, 2);
		api_set_array_object(ctx, arr, "0", api_create_user(ctx, buf, 0, 0, 0));
		api_set_array_object(ctx, arr, "1", api_create_integer(ctx, bytes));
		return arr;
	}
	return 0;
}

static void* zg_togifa_RENDER(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
__TRY__
	zeRender *o = (zeRender*)api_get_user(ctx, args[0], ZE_RENDER);
	const char *fname = api_get_string(ctx, args[1]);
	int delay = 50;
	if (nargs > 2) delay = api_get_integer(ctx, args[2]);
	o->render2GifAnimation(fname, delay);
__CATCH__
	return 0;
}

#ifdef _WIN32
static void* zg_show_RENDER(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
__TRY__
	zeRender *o = (zeRender*)api_get_user(ctx, args[0], ZE_RENDER);
	zeObject *obj = 0;
	if (nargs > 1) {
		int type = api_get_type(args[1]);
		if (type <= ZE_UNKNOWN || type >= ZE_TOP) api_input_error(ctx);
		obj = (zeObject*)api_get_ptr(ctx, args[1]);
		if (!obj->isTransform()) api_input_error(ctx);
	}
	o->show((zeTransform*)obj);
__CATCH__
	return 0;
}
#endif

/////////////////////////////////////////////////////////////////////

static void* zg_add_RENDER(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeRender *o = (zeRender*)api_get_user(ctx, args[0], ZE_RENDER);
	zeObject *p = zg_create(ctx, api_get_string(ctx, args[1]), o->getFont());
	o->getScene()->add(p);
	return api_create_user(ctx, p, 0, 0, p->getType());
}

static void* zg_color_RENDER(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeRender *o = (zeRender*)api_get_user(ctx, args[0], ZE_RENDER);
  	ZEcolor c = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]), 
				  1 };
	o->getScene()->setColor(c);
	return 0;
}

static void* zg_perspective_RENDER(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeRender *o = (zeRender*)api_get_user(ctx, args[0], ZE_RENDER);
	bool yn = api_get_integer(ctx, args[1]) != 0;
	double f = 1.0;
	if (nargs > 2) f = api_get_number(ctx, args[2]);
	if (yn)
		o->getScene()->perspective(f);
	else
		o->getScene()->ortho();
	return 0;
}

static void* zg_lookat_RENDER(void *ctx, int nargs, void** args)
{
	if (nargs < 7) api_input_error(ctx);
	zeRender *o = (zeRender*)api_get_user(ctx, args[0], ZE_RENDER);
	if (nargs == 7) {
		o->getScene()->setLookAt(api_get_number(ctx, args[1]), api_get_number(ctx, args[2]), api_get_number(ctx, args[3]),
					 api_get_number(ctx, args[4]), api_get_number(ctx, args[5]), api_get_number(ctx, args[6]),
					 0, 0, 0);
	}
	else {
		if (nargs < 10) api_input_error(ctx);
		o->getScene()->setLookAt(api_get_number(ctx, args[1]), api_get_number(ctx, args[2]), api_get_number(ctx, args[3]),
					 api_get_number(ctx, args[4]), api_get_number(ctx, args[5]), api_get_number(ctx, args[6]),
					 api_get_number(ctx, args[7]), api_get_number(ctx, args[8]), api_get_number(ctx, args[9]));
	}
	return 0;
}

/////////////////////////////////////////////////////////////////////

static void* zg_add_TEXCOORD(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zeTexCoord *o = (zeTexCoord*)api_get_user(ctx, args[0], ZE_TEXCOORD);
	if (api_is_user(args[1])) {
		double *ptr = (double*)api_get_ptr(ctx, args[1]);
		size_t i, n = api_get_integer(ctx, args[2]);
		for (i = 1; i < n; i += 2) {
			o->add(ptr[i-1], ptr[i]);
		}
		return 0;
	}
	for (int i = 2; i < nargs; i += 2) {
		o->add(api_get_number(ctx, args[i-1]), api_get_number(ctx, args[i]));
	}
	return 0;
}

static void* zg_clear_TEXCOORD(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeTexCoord *o = (zeTexCoord*)api_get_user(ctx, args[0], ZE_TEXCOORD);
	o->clear();
	return 0;
}

static void* zg_size_TEXCOORD(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeTexCoord *o = (zeTexCoord*)api_get_user(ctx, args[0], ZE_TEXCOORD);
	return api_create_integer(ctx, o->size());
}

static void* zg_print_TEXCOORD(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeTexCoord *o = (zeTexCoord*)api_get_user(ctx, args[0], ZE_TEXCOORD);
	FILE *f = stdout;
	if (nargs > 1) {
		f = open_file(api_get_string(ctx, args[1]), "a");
		if (!f) api_runtime_error(ctx, "failed to open file");
	}
	fprintf(f, "<texcoord>\n");
	int n = o->size();
	for (int i = 0; i < n; i++) {
		ZEtexcoord t = o->get(i);
		fprintf(f, " %.2f, %.2f\n", t.s, t.t);
	}
	fprintf(f, "</texcoord>\n");
	fflush(f);
	return 0;
}

static void* zg_get_TEXCOORD(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeTexCoord *o = (zeTexCoord*)api_get_user(ctx, args[0], ZE_TEXCOORD);
	int idx = api_get_integer(ctx, args[1]);
	if (idx < 0 || idx >= o->size()) api_input_error(ctx);
	void *arr = api_create_array(ctx, 2);
	ZEtexcoord st = o->get(idx);
	api_set_array_object(ctx, arr, "0", api_create_real(0,st.s));
	api_set_array_object(ctx, arr, "1", api_create_real(0,st.t));
	return arr;
}

static void* zg_set_TEXCOORD(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeTexCoord *o = (zeTexCoord*)api_get_user(ctx, args[0], ZE_TEXCOORD);
	int idx = api_get_integer(ctx, args[1]);
	if (idx < 0 || idx >= o->size()) api_input_error(ctx);
	o->set(idx, api_get_number(ctx, args[2]), api_get_number(ctx, args[3]));
	return 0;
}

static void* zg_ptr_TEXCOORD(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeTexCoord *o = (zeTexCoord*)api_get_user(ctx, args[0], ZE_TEXCOORD);
	void *arr = api_create_array(ctx, 3);
	api_set_array_object(ctx, arr, "0", api_create_user(ctx, o->ptr(), 0, 0, 0));
	api_set_array_object(ctx, arr, "1", api_create_integer(ctx, o->size()*2));
	api_set_array_object(ctx, arr, "2", api_create_integer(ctx, sizeof(GLfloat)));
	return arr;
}

/////////////////////////////////////////////////////////////////////

static void* zg_font_TEXT(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeText *o = (zeText*)api_get_user(ctx, args[0], ZE_TEXT);
	o->setFontSize(api_get_integer(ctx, args[1]));
	return 0;
}

static void* zg_string_TEXT(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeText *o = (zeText*)api_get_user(ctx, args[0], ZE_TEXT);
__TRY__
	o->setText(api_get_string(ctx, args[1]));
  	double x = 0;
	double y = 0;
	if (nargs > 4) {
		int halign = api_get_integer(ctx, args[4]);
		if (halign > 0)
			x = -o->right();
		else if (halign < 0)
			x = -o->left();
	}
	if (nargs > 5) {
		int valign = api_get_number(ctx, args[5]);
		if (valign > 0)
			y = -o->top();
		else if (valign < 0)
			y = -o->bottom();
	}
	if (x != 0 || y != 0) {
  		ZExyz xyz = { x, y, 0 };
		o->translate(xyz);
	}
__CATCH__
	void *arr = api_create_array(ctx, 3);
	api_set_array_object(ctx, arr, "0", api_create_integer(0, o->right()-o->left()));
	api_set_array_object(ctx, arr, "1", api_create_integer(0, o->top()-o->bottom()));
	return arr;
}

static void* zg_color_TEXT(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zeText *o = (zeText*)api_get_user(ctx, args[0], ZE_TEXT);
  	ZEcolor c = { api_get_number(ctx, args[1]),
				  api_get_number(ctx, args[2]),
				  api_get_number(ctx, args[3]), 
				  1 };
	o->setColor(c);
	return 0;
}

/////////////////////////////////////////////////////////////////////

static void* zg_image_TEXTURE(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeTexture *o = (zeTexture*)api_get_user(ctx, args[0], ZE_TEXTURE);
	const char *fname = api_get_string(ctx, args[1]);
	int r=-1, g=-1, b=-1;
	if (nargs > 4) {
		r = api_get_integer(ctx, args[2]);
		g = api_get_integer(ctx, args[3]);
		b = api_get_integer(ctx, args[4]);
	}
	if (!o->setImage(fname, r, g, b)) api_runtime_error(ctx, "Failed to load image for texture");

	void *arr = api_create_array(ctx, 2);
	api_set_array_object(ctx, arr, "0", api_create_real(0, o->s()));
	api_set_array_object(ctx, arr, "1", api_create_real(0, o->t()));

	return arr;
}

static void* zg_data_TEXTURE(void *ctx, int nargs, void** args)
{
	if (nargs < 5) api_input_error(ctx);
	zeTexture *o = (zeTexture*)api_get_user(ctx, args[0], ZE_TEXTURE);
	double *Z = (double*)api_get_ptr(ctx, args[1]);
	int nrow = api_get_integer(ctx, args[2]);
	int ncol = api_get_integer(ctx, args[3]);
	zeColorBar *cbar = (zeColorBar*)api_get_user(ctx, args[4], ZE_COLOR_BAR);
	
	if (nrow <= 0 || nrow > 1024 || ncol <= 0 || ncol > 1024) api_input_error(ctx);

	o->setImage(Z, nrow, ncol, cbar);

	void *arr = api_create_array(ctx, 2);
	api_set_array_object(ctx, arr, "0", api_create_real(0, double(ncol)/o->width()));
	api_set_array_object(ctx, arr, "1", api_create_real(0, double(nrow)/o->height()));

	return arr;
}

/////////////////////////////////////////////////////////////////////

static void* zg_add_VERTEX(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zeVertex *o = (zeVertex*)api_get_user(ctx, args[0], ZE_VERTEX);
	if (api_is_user(args[1])) {
		void *ptr = api_get_ptr(ctx, args[1]);
		size_t i, n = api_get_integer(ctx, args[2]);
		for (i = 2; i < n; i += 3) {
			ZExyz xyz = { ((double*)ptr)[i-2], ((double*)ptr)[i-1], ((double*)ptr)[i] };
			o->add(xyz);
		}
		return 0;
	}
	if (nargs < 4) api_input_error(ctx);
	for (int i = 3; i < nargs; i += 3) {
		ZExyz xyz = { api_get_number(ctx, args[i-2]),
					  api_get_number(ctx, args[i-1]),
					  api_get_number(ctx, args[i  ]) };
		o->add(xyz);
	}
	return 0;
}

static void* zg_clear_VERTEX(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeVertex *o = (zeVertex*)api_get_user(ctx, args[0], ZE_VERTEX);
	o->clear();
	return 0;
}

static void* zg_size_VERTEX(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeVertex *o = (zeVertex*)api_get_user(ctx, args[0], ZE_VERTEX);
	return api_create_integer(ctx, o->size());
}

static void* zg_print_VERTEX(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeVertex *o = (zeVertex*)api_get_user(ctx, args[0], ZE_VERTEX);
	FILE *f = stdout;
	if (nargs > 1) {
		f = open_file(api_get_string(ctx, args[1]), "a");
		if (!f) api_runtime_error(ctx, "failed to open file");
	}
	fprintf(f, "<vertex>\n");
	int n = o->size();
	for (int i = 0; i < n; i++) {
		ZEvertex v = o->get(i);
		fprintf(f, " %.2f, %.2f, %.2f\n", v.x, v.y, v.z);
	}
	fprintf(f, "</vertex>\n");
	fflush(f);
	return 0;
}

static void* zg_color_VERTEX(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zeVertex *o = (zeVertex*)api_get_user(ctx, args[0], ZE_VERTEX);
	zeColorBar *b = (zeColorBar*)api_get_user(ctx, args[1], ZE_COLOR_BAR);
	zeColor *c = (zeColor*)api_get_user(ctx, args[2], ZE_COLOR);
	int n = o->size();
	c->clear();
	for (int i = 0; i < n; i++) {
		ZEvertex v = o->get(i);
		c->add(b->getColor(v.z));
	}
	return 0;
}

static void* zg_normal_VERTEX(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zeVertex *o = (zeVertex*)api_get_user(ctx, args[0], ZE_VERTEX);
	int np = api_get_integer(ctx, args[1]);
	zeVertex *p = (zeVertex*)api_get_user(ctx, args[2], ZE_VERTEX);;
	int n = o->size();
	p->clear();
	if (np == 1) {
		for (int i = 0; i < n; i++) {
			ZEvertex v = o->get(i);
			normalize_vertex(v);
			p->add(v);
		}
	}
	else if (np == 2) {
		for (int i = 2; i < n; i += 2) {
			ZEvertex v1 = o->get(i-2);
			ZEvertex v2 = o->get(i-1);
			ZEvertex v3 = o->get(i);
			ZExyz v = cross_product3(v1, v2, v3);
			normalize_vertex(v);
			p->add(v);
			p->add(v);
			p->add(v);
		}
	}
	else if (np == 3) {
		if (n >= 2) {
			ZEvertex v1 = o->get(0);
			ZEvertex v2 = o->get(1);
			ZEvertex v3;
			ZEvertex v = { v1.x-v2.x, v1.y-v2.y, v1.z-v2.z };
			normalize_vertex(v);
			p->add(v);
			for (int i = 2; i < n; i++) {
				v1 = o->get(i-2);
				v2 = o->get(i-1);
				v3 = o->get(i);
				v.x = 2*v2.x-v1.x-v3.x;
				v.y = 2*v2.y-v1.y-v3.y;
				v.z = 2*v2.z-v1.z-v3.z;
				normalize_vertex(v);
				p->add(v);
			}
			v.x = v3.x-v2.x;
			v.y = v2.y-v2.y;
			v.z = v1.z-v2.z;
			normalize_vertex(v);
			p->add(v);
		}
	}
	else if (np == 4) {
		for (int i = 3; i < n; i += 4) {
			ZEvertex v1 = o->get(i-3);
			ZEvertex v2 = o->get(i-2);
			ZEvertex v3 = o->get(i-1);
			ZEvertex v4 = o->get(i);
			ZExyz v = cross_product3(v1, v2, v4);
			normalize_vertex(v);
			p->add(v);
			v = cross_product3(v2, v3, v1);
			normalize_vertex(v);
			p->add(v);
			v = cross_product3(v3, v4, v2);
			normalize_vertex(v);
			p->add(v);
			v = cross_product3(v4, v1, v3);
			normalize_vertex(v);
			p->add(v);
		}
	}
	else {
		api_input_error(ctx);
	}
	return 0;
}

static void* zg_ll2xyz_VERTEX(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeVertex *o = (zeVertex*)api_get_user(ctx, args[0], ZE_VERTEX);
	double R = 1;
	if (nargs > 1) R = api_get_number(ctx, args[1]);
	if (R <= 0) api_input_error(ctx);
	for (int i = 0; i < o->size(); i++) {
		ZEvertex v = o->get(i);
		double x, y, z;
		ll2xyz(x, y, z, v.x, v.y, R);
		v.x = x;
		v.y = y;
		v.z = z;
		o->set(i, v);
	}
	return 0;
}

static void* zg_xyz2ll_VERTEX(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeVertex *o = (zeVertex*)api_get_user(ctx, args[0], ZE_VERTEX);
	for (int i = 0; i < o->size(); i++) {
		ZEvertex v = o->get(i);
		double lon, lat;
		xyz2ll(lon, lat, v.x, v.y, v.z);
		v.x = lon;
		v.y = lat;
		v.z = 0;
		o->set(i, v);
	}
	return 0;
}

static void* zg_map_VERTEX(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeVertex *o = (zeVertex*)api_get_user(ctx, args[0], ZE_VERTEX);
	double x0 = api_get_number(ctx, args[1]);
	double xmin = 1.e30;
	double xmax = -1.e30;
	double ymin = 1.e30;
	double ymax = -1.e30;
	for (size_t i=0; i<o->size(); i++) {
		ZEvertex v = o->get(i);
		v.x = (v.x-x0)*cos(v.y*D2R)/90.0;
		v.y /= 90.0;
		v.z = 0;
		if (v.x < xmin) xmin = v.x;
		else if (v.x > xmax) xmax = v.x;
		if (v.y < ymin) ymin = v.y;
		else if (v.y > ymax) ymax = v.y;
		o->set(i, v);
	}
	void *arr = api_create_array(ctx, 4);
	api_set_array_object(ctx, arr, "0", api_create_real(0, xmin));
	api_set_array_object(ctx, arr, "1", api_create_real(0, xmax));
	api_set_array_object(ctx, arr, "2", api_create_real(0, ymin));
	api_set_array_object(ctx, arr, "3", api_create_real(0, ymax));
	return arr;
}

static void* zg_set_VERTEX(void *ctx, int nargs, void** args)
{
	if (nargs < 5) api_input_error(ctx);
	zeVertex *o = (zeVertex*)api_get_user(ctx, args[0], ZE_VERTEX);
	int idx = api_get_integer(ctx, args[1]);
	if (idx < 0 || idx >= o->size()) api_input_error(ctx);
	ZEvertex v = {  api_get_number(ctx, args[2]),
					api_get_number(ctx, args[3]),
					api_get_number(ctx, args[4]) };
	o->set(idx, v);
	return 0;
}

static void* zg_get_VERTEX(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zeVertex *o = (zeVertex*)api_get_user(ctx, args[0], ZE_VERTEX);
	int idx = api_get_integer(ctx, args[1]);
	if (idx < 0 || idx >= o->size()) api_input_error(ctx);
	ZEvertex v = o->get(idx);
	void *arr = api_create_array(ctx, 3);
	api_set_array_object(ctx, arr, "0", api_create_real(ctx, v.x));
	api_set_array_object(ctx, arr, "1", api_create_real(ctx, v.y));
	api_set_array_object(ctx, arr, "2", api_create_real(ctx, v.z));
	return arr;
}

static void* zg_ptr_VERTEX(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeVertex *o = (zeVertex*)api_get_user(ctx, args[0], ZE_VERTEX);
	void *arr = api_create_array(ctx, 3);
	api_set_array_object(ctx, arr, "0", api_create_user(ctx, o->ptr(), 0, 0, 0));
	api_set_array_object(ctx, arr, "1", api_create_integer(ctx, o->size()*3));
	api_set_array_object(ctx, arr, "2", api_create_integer(ctx, sizeof(GLfloat)));
	return arr;
}

static void* zg_flatten_VERTEX(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zeVertex *o = (zeVertex*)api_get_user(ctx, args[0], ZE_VERTEX);
	for (int i = 0; i < o->size(); i++) {
		ZEvertex xyz = o->get(i);
		xyz.z = 0;
		o->set(i, xyz);
	}
	return 0;
}


/////////////////////////////////////////////////////////////////////

class zeplotRegPrimitive
{
public:
	zeplotRegPrimitive()
	{
		api_add_primitive("zegraph",		0,		zg_create_RENDER);

		api_add_primitive("color",		ZE_AXIS,	zg_color_AXIS);
		api_add_primitive("font",		ZE_AXIS,	zg_font_AXIS);
		api_add_primitive("linewidth",	ZE_AXIS,	zg_linewidth_AXIS);
		api_add_primitive("range",		ZE_AXIS,	zg_range_AXIS);
		api_add_primitive("showticks",	ZE_AXIS,	zg_showticks_AXIS);
		api_add_primitive("showlabels",	ZE_AXIS,	zg_showlabels_AXIS);
		api_add_primitive("smooth",		ZE_AXIS,	zg_smooth_AXIS);
		api_add_primitive("type",		ZE_AXIS,	zg_type_AXIS);
		api_add_primitive("title",		ZE_AXIS,	zg_title_AXIS);
		api_add_primitive("tickmarks",	ZE_AXIS,	zg_tickmarks_AXIS);
		api_add_primitive("ticksize",	ZE_AXIS,	zg_ticksize_AXIS);
		api_add_primitive("ticklabels",	ZE_AXIS,	zg_ticklabels_AXIS);
		api_add_primitive("tickdigits",	ZE_AXIS,	zg_tickdigits_AXIS);
		api_add_primitive("enable",		ZE_AXIS,	enable_object);
		api_add_primitive("disable",	ZE_AXIS,	disable_object);
		api_add_primitive("rotatex",	ZE_AXIS,	set_rotatex);
		api_add_primitive("rotatey",	ZE_AXIS,	set_rotatey);
		api_add_primitive("rotatez",	ZE_AXIS,	set_rotatez);
		api_add_primitive("scale",		ZE_AXIS,	set_scale);
		api_add_primitive("translate",	ZE_AXIS,	set_translate);

		api_add_primitive("add",	ZE_COLOR,	zg_add_COLOR);
		api_add_primitive("clear",	ZE_COLOR,	zg_clear_COLOR);
		api_add_primitive("print",	ZE_COLOR,	zg_print_COLOR);

		api_add_primitive("size",			ZE_COLOR_BAR,	zg_size_COLOR_BAR);
		api_add_primitive("discrete",		ZE_COLOR_BAR,	zg_discrete_COLOR_BAR);
		api_add_primitive("labeldigits",	ZE_COLOR_BAR,	zg_labeldigits_COLOR_BAR);
		api_add_primitive("font",			ZE_COLOR_BAR,	zg_font_COLOR_BAR);
		api_add_primitive("color",			ZE_COLOR_BAR,	zg_color_COLOR_BAR);
		api_add_primitive("add",			ZE_COLOR_BAR,	zg_add_COLOR_BAR);
		api_add_primitive("get",			ZE_COLOR_BAR,	zg_get_COLOR_BAR);
		api_add_primitive("clear",			ZE_COLOR_BAR,	zg_clear_COLOR_BAR);
		api_add_primitive("linewidth",		ZE_COLOR_BAR,	zg_linewidth_COLOR_BAR);
		api_add_primitive("interpolate",	ZE_COLOR_BAR,	zg_interpolate_COLOR_BAR);
		api_add_primitive("enable",			ZE_COLOR_BAR,	enable_object);
		api_add_primitive("disable",		ZE_COLOR_BAR,	disable_object);
		api_add_primitive("rotatex",		ZE_COLOR_BAR,	set_rotatex);
		api_add_primitive("rotatey",		ZE_COLOR_BAR,	set_rotatey);
		api_add_primitive("rotatez",		ZE_COLOR_BAR,	set_rotatez);
		api_add_primitive("scale",			ZE_COLOR_BAR,	set_scale);
		api_add_primitive("translate",		ZE_COLOR_BAR,	set_translate);

		api_add_primitive("add",		ZE_GLOBE,	zg_add_GLOBE);
		api_add_primitive("lookat",		ZE_GLOBE,	zg_lookat_GLOBE);
		api_add_primitive("surface",	ZE_GLOBE,	zg_surface_GLOBE);
		api_add_primitive("texture",	ZE_GLOBE,	zg_texture_GLOBE);
		api_add_primitive("grid",		ZE_GLOBE,	zg_grid_GLOBE);
		api_add_primitive("focus",		ZE_GLOBE,	zg_focus_GLOBE);
		api_add_primitive("line",		ZE_GLOBE,	zg_line_GLOBE);
		api_add_primitive("gshhs",		ZE_GLOBE,	zg_gshhs_GLOBE);
		api_add_primitive("sun",		ZE_GLOBE,	zg_sun_GLOBE);
		api_add_primitive("light",		ZE_GLOBE,	zg_light_GLOBE);
		api_add_primitive("clear",		ZE_GLOBE,	zg_clear_GLOBE);
		api_add_primitive("ll2xyz",		ZE_GLOBE,	zg_ll2xyz_GLOBE);
		api_add_primitive("xyz2ll",		ZE_GLOBE,	zg_xyz2ll_GLOBE);
		api_add_primitive("enable",		ZE_GLOBE,	enable_object);
		api_add_primitive("disable",	ZE_GLOBE,	disable_object);
		api_add_primitive("rotatex",	ZE_GLOBE,	set_rotatex);
		api_add_primitive("rotatey",	ZE_GLOBE,	set_rotatey);
		api_add_primitive("rotatez",	ZE_GLOBE,	set_rotatez);
		api_add_primitive("scale",		ZE_GLOBE,	set_scale);
		api_add_primitive("translate",	ZE_GLOBE,	set_translate);

		api_add_primitive("position",	ZE_LIGHT,	zg_position_LIGHT);
		api_add_primitive("ambient",	ZE_LIGHT,	zg_ambient_LIGHT);
		api_add_primitive("add",		ZE_LIGHT,	zg_add_LIGHT);
		api_add_primitive("clear",		ZE_LIGHT,	zg_clear_LIGHT);
		api_add_primitive("enable",		ZE_LIGHT,	enable_object);
		api_add_primitive("disable",	ZE_LIGHT,	disable_object);

		api_add_primitive("solid",		ZE_LINE,	zg_solid_LINE);
		api_add_primitive("dot",		ZE_LINE,	zg_dot_LINE);
		api_add_primitive("dash",		ZE_LINE,	zg_dash_LINE);
		api_add_primitive("dotdash",	ZE_LINE,	zg_dotdash_LINE);
		api_add_primitive("type",		ZE_LINE,	zg_type_LINE);
		api_add_primitive("smooth",		ZE_LINE,	zg_smooth_LINE);
		api_add_primitive("color",		ZE_LINE,	zg_color_LINE);
		api_add_primitive("vertex",		ZE_LINE,	zg_vertex_LINE);
		api_add_primitive("normal",		ZE_LINE,	zg_normal_LINE);
		api_add_primitive("contour",	ZE_LINE,	zg_contour_LINE);
		api_add_primitive("vector",		ZE_LINE,	zg_vector_LINE);
		api_add_primitive("gshhs",		ZE_LINE,	zg_gshhs_LINE);
		api_add_primitive("enable",		ZE_LINE,	enable_object);
		api_add_primitive("disable",	ZE_LINE,	disable_object);
		api_add_primitive("rotatex",	ZE_LINE,	set_rotatex);
		api_add_primitive("rotatey",	ZE_LINE,	set_rotatey);
		api_add_primitive("rotatez",	ZE_LINE,	set_rotatez);
		api_add_primitive("scale",		ZE_LINE,	set_scale);
		api_add_primitive("translate",	ZE_LINE,	set_translate);

		api_add_primitive("ambient",	ZE_MATERIAL,	zg_ambient_MATERIAL);
		api_add_primitive("diffuse",	ZE_MATERIAL,	zg_diffuse_MATERIAL);
		api_add_primitive("emission",	ZE_MATERIAL,	zg_emission_MATERIAL);
		api_add_primitive("specular",	ZE_MATERIAL,	zg_specular_MATERIAL);
		api_add_primitive("shininess",	ZE_MATERIAL,	zg_shininess_MATERIAL);

		api_add_primitive("add",		ZE_NODE,	zg_add_NODE);
		api_add_primitive("color",		ZE_NODE,	zg_color_NODE);
		api_add_primitive("clear",		ZE_NODE,	zg_clear_NODE);
		api_add_primitive("remove",		ZE_NODE,	zg_remove_NODE);
		api_add_primitive("enable",		ZE_NODE,	enable_object);
		api_add_primitive("disable",	ZE_NODE,	disable_object);
		api_add_primitive("rotatex",	ZE_NODE,	set_rotatex);
		api_add_primitive("rotatey",	ZE_NODE,	set_rotatey);
		api_add_primitive("rotatez",	ZE_NODE,	set_rotatez);
		api_add_primitive("scale",		ZE_NODE,	set_scale);
		api_add_primitive("translate",	ZE_NODE,	set_translate);

		api_add_primitive("add",		ZE_PLOT,	zg_add_PLOT);
		api_add_primitive("color",		ZE_PLOT,	zg_color_PLOT);
		api_add_primitive("scale",		ZE_PLOT,	zg_scale_PLOT);
		api_add_primitive("global",		ZE_PLOT,	zg_global_PLOT);
		api_add_primitive("local",		ZE_PLOT,	zg_local_PLOT);
		api_add_primitive("xaxis",		ZE_PLOT,	zg_xaxis_PLOT);
		api_add_primitive("yaxis",		ZE_PLOT,	zg_yaxis_PLOT);
		api_add_primitive("zaxis",		ZE_PLOT,	zg_zaxis_PLOT);
		api_add_primitive("clear",		ZE_PLOT,	zg_clear_PLOT);
		api_add_primitive("clip",		ZE_PLOT,	zg_clip_PLOT);
		api_add_primitive("clabel",		ZE_PLOT,	zg_clabel_PLOT);
		api_add_primitive("enable",		ZE_PLOT,	enable_object);
		api_add_primitive("disable",	ZE_PLOT,	disable_object);
		api_add_primitive("rotatex",	ZE_PLOT,	set_rotatex);
		api_add_primitive("rotatey",	ZE_PLOT,	set_rotatey);
		api_add_primitive("rotatez",	ZE_PLOT,	set_rotatez);
		api_add_primitive("scale",		ZE_PLOT,	set_scale);
		api_add_primitive("translate",	ZE_PLOT,	set_translate);

		api_add_primitive("smooth",		ZE_POINT,	zg_smooth_POINT);
		api_add_primitive("color",		ZE_POINT,	zg_color_POINT);
		api_add_primitive("vertex",		ZE_POINT,	zg_vertex_POINT);
		api_add_primitive("normal",		ZE_POINT,	zg_normal_POINT);
		api_add_primitive("size",		ZE_POINT,	zg_size_POINT);
		api_add_primitive("enable",		ZE_POINT,	enable_object);
		api_add_primitive("disable",	ZE_POINT,	disable_object);
		api_add_primitive("rotatex",	ZE_POINT,	set_rotatex);
		api_add_primitive("rotatey",	ZE_POINT,	set_rotatey);
		api_add_primitive("rotatez",	ZE_POINT,	set_rotatez);
		api_add_primitive("scale",		ZE_POINT,	set_scale);
		api_add_primitive("translate",	ZE_POINT,	set_translate);

		api_add_primitive("smooth",		ZE_POLYGON,	zg_smooth_POLYGON);
		api_add_primitive("color",		ZE_POLYGON,	zg_color_POLYGON);
		api_add_primitive("vertex",		ZE_POLYGON,	zg_vertex_POLYGON);
		api_add_primitive("normal",		ZE_POLYGON,	zg_normal_POLYGON);
		api_add_primitive("texcoord",	ZE_POLYGON,	zg_texcoord_POLYGON);
		api_add_primitive("pattern",	ZE_POLYGON,	zg_pattern_POLYGON);
		api_add_primitive("fill",		ZE_POLYGON,	zg_fill_POLYGON);
		api_add_primitive("type",		ZE_POLYGON,	zg_type_POLYGON);
		api_add_primitive("linewidth",	ZE_POLYGON,	zg_linewidth_POLYGON);
		api_add_primitive("cull",		ZE_POLYGON,	zg_cull_POLYGON);
		api_add_primitive("clockwise",	ZE_POLYGON,	zg_clockwise_POLYGON);
		api_add_primitive("material",	ZE_POLYGON,	zg_material_POLYGON);
		api_add_primitive("texture",	ZE_POLYGON,	zg_texture_POLYGON);
		api_add_primitive("isosurface",	ZE_POLYGON,	zg_isosurface_POLYGON);
		api_add_primitive("box",		ZE_POLYGON,	zg_box_POLYGON);
		api_add_primitive("sphere",		ZE_POLYGON,	zg_sphere_POLYGON);
		api_add_primitive("cone",		ZE_POLYGON,	zg_cone_POLYGON);
		api_add_primitive("disk",		ZE_POLYGON,	zg_disk_POLYGON);
		api_add_primitive("cylinder",	ZE_POLYGON,	zg_cylinder_POLYGON);
		api_add_primitive("torus",		ZE_POLYGON,	zg_torus_POLYGON);
		api_add_primitive("enable",		ZE_POLYGON,	enable_object);
		api_add_primitive("disable",	ZE_POLYGON,	disable_object);
		api_add_primitive("rotatex",	ZE_POLYGON,	set_rotatex);
		api_add_primitive("rotatey",	ZE_POLYGON,	set_rotatey);
		api_add_primitive("rotatez",	ZE_POLYGON,	set_rotatez);
		api_add_primitive("scale",		ZE_POLYGON,	set_scale);
		api_add_primitive("translate",	ZE_POLYGON,	set_translate);

		api_add_primitive("add",		ZE_RENDER,	zg_add_RENDER);
		api_add_primitive("color",		ZE_RENDER,	zg_color_RENDER);
		api_add_primitive("bgcolor",	ZE_RENDER,	zg_bgcolor_RENDER);
		api_add_primitive("font",		ZE_RENDER,	zg_font_RENDER);
		api_add_primitive("perspective",ZE_RENDER,	zg_perspective_RENDER);
		api_add_primitive("lookat",		ZE_RENDER,	zg_lookat_RENDER);
		api_add_primitive("tofile",		ZE_RENDER,	zg_tofile_RENDER);
		api_add_primitive("togifa",		ZE_RENDER,	zg_togifa_RENDER);
#ifdef _WIN32
		api_add_primitive("show",		ZE_RENDER,	zg_show_RENDER);
#endif
		api_add_primitive("add",		ZE_TEXCOORD,	zg_add_TEXCOORD);
		api_add_primitive("clear",		ZE_TEXCOORD,	zg_clear_TEXCOORD);
		api_add_primitive("print",		ZE_TEXCOORD,	zg_print_TEXCOORD);

		api_add_primitive("font",		ZE_TEXT,	zg_font_TEXT);
		api_add_primitive("string",		ZE_TEXT,	zg_string_TEXT);
		api_add_primitive("color",		ZE_TEXT,	zg_color_TEXT);
		api_add_primitive("enable",		ZE_TEXT,	enable_object);
		api_add_primitive("disable",	ZE_TEXT,	disable_object);
		api_add_primitive("rotatex",	ZE_TEXT,	set_rotatex);
		api_add_primitive("rotatey",	ZE_TEXT,	set_rotatey);
		api_add_primitive("rotatez",	ZE_TEXT,	set_rotatez);
		api_add_primitive("scale",		ZE_TEXT,	set_scale);
		api_add_primitive("translate",	ZE_TEXT,	set_translate);

		api_add_primitive("image",		ZE_TEXTURE,	zg_image_TEXTURE);
		api_add_primitive("data",		ZE_TEXTURE,	zg_data_TEXTURE);

		api_add_primitive("add",		ZE_VERTEX,	zg_add_VERTEX);
		api_add_primitive("clear",		ZE_VERTEX,	zg_clear_VERTEX);
		api_add_primitive("print",		ZE_VERTEX,	zg_print_VERTEX);
		api_add_primitive("color",		ZE_VERTEX,	zg_color_VERTEX);
		api_add_primitive("normal",		ZE_VERTEX,	zg_normal_VERTEX);
		api_add_primitive("ll2xyz",		ZE_VERTEX,	zg_ll2xyz_VERTEX);
		api_add_primitive("xyz2ll",		ZE_VERTEX,	zg_xyz2ll_VERTEX);
		api_add_primitive("map",		ZE_VERTEX,	zg_map_VERTEX);
		api_add_primitive("flatten",	ZE_VERTEX,	zg_flatten_VERTEX);
	}
};

static zeplotRegPrimitive zeplot_register;