#include "config.h"
#include "utility.h"
#include "tiff.h"
#include "tiffio.h"
#include <stdio.h>
#include <math.h>

using namespace std;

FILE *open_file(const char *fname, const char *mode)
{
	FILE *fp = NULL;
#ifdef _WIN32
	fopen_s(&fp, fname, mode);
#else
	fp = fopen(fname, mode);
#endif
	return fp;
}

void close_file(FILE *fp)
{
#ifdef _WIN32
	fclose(fp);
#else
	fclose(fp);
#endif
}

gdImagePtr load_tiff(const char *fname)
{
	TIFF* tif = TIFFOpen(fname, "r");
	if (!tif) return NULL;
	uint32 width, height, npixels, *raster;
	TIFFGetField(tif, TIFFTAG_IMAGEWIDTH, &width);
	TIFFGetField(tif, TIFFTAG_IMAGELENGTH, &height);
	npixels = width*height;
	raster = (uint32*)_TIFFmalloc(npixels*sizeof(uint32));
	if (!TIFFReadRGBAImage(tif, width, height, raster)) {
		_TIFFfree(raster);
		TIFFClose(tif);
		return NULL;
	}
	gdImagePtr img2 = gdImageCreateTrueColor(width, height);
	int k = 0;
	unsigned char *img = (unsigned char*)raster; 
	for (int i = 0; i < height; i++) {
		for (int j = 0; j < width; j++) {
			unsigned char r = img[k++];
			unsigned char g = img[k++];
			unsigned char b = img[k++];
			k++;
			int rgb2 = gdImageColorAllocate(img2, r, g, b);
			gdImageSetPixel(img2, j, height-i-1, rgb2);
		}
	}
	_TIFFfree(raster);
	TIFFClose(tif);
	return img2;
}


gdImagePtr load_image(const char *fname)
{
	gdImagePtr ptr = 0;
	int len = strlen(fname);
	if (len < 5) return ptr;
	FILE *f = open_file(fname, "rb");
	if (!f) return ptr;
	char ext[5];
	ext[0] = (char)toupper(fname[len-4]);
	ext[1] = (char)toupper(fname[len-3]);
	ext[2] = (char)toupper(fname[len-2]);
	ext[3] = (char)toupper(fname[len-1]);
	ext[4] = 0;
	if (!strcmp(ext, ".GIF")) {
		ptr = gdImageCreateFromGif(f);
	}
	else if (!strcmp(ext, ".PNG")) {
		ptr = gdImageCreateFromPng(f);
	}
	else if (!strcmp(ext, ".JPG")) {
		ptr = gdImageCreateFromJpeg(f);
	}
	else if (!strcmp(ext, ".BMP")) {
		ptr = gdImageCreateFromWBMP(f);
	}
	else if (!strcmp(ext, ".XBM")) {
		ptr = gdImageCreateFromXbm(f);
	}
	else if (!strcmp(ext, ".XPM")) {
		ptr = gdImageCreateFromXpm((char*)fname);
	}
	else if (!strcmp(ext, ".TIF")) {
		close_file(f);
		return load_tiff(fname);
	}
	close_file(f);
	return ptr;
}

bool save_tiff(const char *fname, gdImagePtr img)
{
	TIFF* tif = TIFFOpen(fname, "w");
	if (!tif) return false;
	int width = gdImageSX(img);
	int height = gdImageSY(img);
	TIFFSetField(tif, TIFFTAG_IMAGEWIDTH, width);
	TIFFSetField(tif, TIFFTAG_IMAGELENGTH, height);
	TIFFSetField(tif, TIFFTAG_COMPRESSION, COMPRESSION_DEFLATE);
	TIFFSetField(tif, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
	TIFFSetField(tif, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB);
	TIFFSetField(tif, TIFFTAG_BITSPERSAMPLE, 8);
	TIFFSetField(tif, TIFFTAG_SAMPLESPERPIXEL, 3);
	unsigned char *s = (unsigned char*)_TIFFmalloc(width*height*3);
	int k = 0;
	for (int i = 0; i < height; i++) {
		for (int j = 0; j < width; j++) {
			int rgb = gdImageGetPixel(img, j, i);
			unsigned char r = gdImageRed(img, rgb);
			unsigned char g = gdImageGreen(img, rgb);
			unsigned char b = gdImageBlue(img, rgb);
			s[k++] = r;
			s[k++] = g;
			s[k++] = b;
		}
	}
	TIFFWriteEncodedStrip(tif, 0, s, width * height * 3);
	_TIFFfree(s);
	TIFFClose(tif);
	return true;
}

bool save_image(const char *fname, gdImagePtr img)
{
	int len = strlen(fname);
	if (len < 5) return false;
	FILE *f = open_file(fname, "wb");
	if (!f) return false;
	char ext[5];
	ext[0] = (char)toupper(fname[len-4]);
	ext[1] = (char)toupper(fname[len-3]);
	ext[2] = (char)toupper(fname[len-2]);
	ext[3] = (char)toupper(fname[len-1]);
	ext[4] = 0;
	if (!strcmp(ext, ".GIF")) {
		gdImageGif(img, f);
		close_file(f);
		return true;		
	}
	if (!strcmp(ext, ".PNG")) {
		gdImagePng(img, f);
		close_file(f);
		return true;		
	}
	if (!strcmp(ext, ".JPG")) {
		gdImageJpeg(img, f, -1);
		close_file(f);
		return true;
	}
	if (!strcmp(ext, ".TIF")) {
		close_file(f);
		return save_tiff(fname, img);
	}
	close_file(f);
	return false;
}

size_t texture_size(size_t size)
{
	size_t k = 2; 
	while (k < size && k < MAX_TEXTURE_SIZE) k *= 2;
	return k;
}

string num2str(const GLfloat value, size_t precision, bool scientific)
{
	char num[32], fmt[32];

	if (scientific)
		sprintf(fmt, "%%15.%de", precision); 
	else
		sprintf(fmt, "%%15.%df", precision); 

	sprintf(num, fmt, value);

	string s(num);

	while (s.at(0) == ' ') s.erase(0, 1);

	if (scientific)
	{
		size_t k = s.find('e', 0);
		string b = s.substr(0, k);
		string e = s.substr(k, s.size()-k);
		while (e.at(e.size()-2) == '0') e.erase(e.size()-2, 1);
		s = b + e;
	}

	return s;
}

wstring num2wstr(const GLfloat value, size_t precision, bool scientific)
{
	string s = num2str(value, precision, scientific);
	wstring ws(s.size(), 0);
	for (size_t k = 0; k < s.size(); k++)
	{
		ws.at(k) = s.at(k);
	}
	return ws;
}

void ll2xyz(double &x, double &y, double &z, double lon, double lat, double r)
{
    lon *= 0.017453292519943;
    lat *= 0.017453292519943;
    z = r * sin(lat);
    double d = r * cos(lat);
    x = d * cos(lon);
    y = d * sin(lon);
}

void xyz2ll(double &lon, double &lat, double x, double y, double z)
{
	double	r = sqrt(x*x + y*y) + EPS,
			R = sqrt(x*x + y*y + z*z) + EPS;
	lat = acos(r/R) * RAD_TO_DEG;
	if (z < 0) lat = -lat;
	lon = acos(x/r) * RAD_TO_DEG;
	if (y < 0) lon = -lon;
}

void dxdy2ll(double &lon, double &lat, double dx, double dy, double re)
{
	double  a  = lon * DEG_TO_RAD,
			b  = lat * DEG_TO_RAD,
			r  = re * cos (b),
			x, y;

	if (lat > 87.5) {
		// point near north pole
		x = r * cos (a) - dx * sin (a) - dy * cos (a);
		y = r * sin (a) + dx * cos (a) - dy * sin (a);
		lon = RAD_TO_DEG * atan2 (y, x);
		lat = RAD_TO_DEG * acos (sqrt (x * x + y * y) / re);
	} else if (lat < -87.5) {
		// point near south pole
		x = r * cos (a) - dx * sin (a) + dy * cos (a);
		y = r * sin (a) + dx * cos (a) + dy * sin (a);
		lon = RAD_TO_DEG * atan2 (y, x);
		lat = - RAD_TO_DEG * acos(sqrt (x * x + y * y) / re);
	} else {
		lon += RAD_TO_DEG *dx / r;
		lat += RAD_TO_DEG * dy / re;
	}

	if (lon < 0.0) {
		lon += 360.0;
	}
	else if (lon >= 360.0) {
		lon -= 360.0;
	}
	if (lat > 90.0) {
		lat = 180.0 - lat;
		if (lon < 180.0)
			lon += 180.0;
		else
			lon -= 180.0;
	}
	else if (lat < -90.0) {
		lat = -(180.0 + lat);
		lon += 180.0;
		if (lon < 180.0)
			lon += 180.0;
		else
			lon -= 180.0;
	}
}

void aitoff(double &x, double &y, double lon, double lat)
{
	lon = DEG_TO_RAD * lon;
	lon /= 2.0;
	lat = DEG_TO_RAD * lat;
	double a = acos(cos(lat) * cos(lon));
	a /= (EPS + sin(a));
	x = 2.0 * a * cos(lat) * sin(lon);
	y = a * sin(lat);
}

void denoyer(double &x, double &y, double lon, double lat)
{
	x = DEG_TO_RAD * lon;
	y = DEG_TO_RAD * lat;
	x *= cos((0.95 - fabs(x) / 12.0 + cos(pow(x, 3)) / 600.0) * y);  
}

void echert(double &x, double &y, double lon, double lat)
{
    x = DEG_TO_RAD * lon;
    y = DEG_TO_RAD * lat;
    x *= (1 + cos(y)) *  0.441012772;
    y *= 0.882025544;
}

void wagner(double &x, double &y, double lon, double lat)
{
    x = DEG_TO_RAD * lon;
    y = DEG_TO_RAD * lat;
    double theta = asin(0.88022 * sin(0.8855 * y));
    x *= 0.92483 * cos(theta);
    y = 1.38725 * theta;
}

int axtoi(char *hexStg, int len)
{
	// from http://bdn.borland.com/article/0,1410,17203,00.html
	int n = 0;         // position in string
	int m = 0;         // position in digit[] to shift
	int count;         // loop index
	int intValue = 0;  // integer value of hex string
	int digit[5];      // hold values to convert
	while (n < len) {
		if (hexStg[n]=='\0')
			break;
		if (hexStg[n] > 0x29 && hexStg[n] < 0x40 ) //if 0 to 9
			digit[n] = hexStg[n] & 0x0f;            //convert to int
		else if (hexStg[n] >='a' && hexStg[n] <= 'f') //if a to f
			digit[n] = (hexStg[n] & 0x0f) + 9;      //convert to int
		else if (hexStg[n] >='A' && hexStg[n] <= 'F') //if A to F
			digit[n] = (hexStg[n] & 0x0f) + 9;      //convert to int
		else break;
		n++;
	}
	count = n;
	m = n - 1;
	n = 0;
	while(n < count) {
		// digit[n] is value of hex digit at position n
		// (m << 2) is the number of positions to shift
		// OR the bits into return value
		intValue = intValue | (digit[n] << (m << 2));
		m--;   // adjust the position to set
		n++;   // next digit to process
	}
	return (intValue);
}


void error_msg(const char *msg1, const char *msg2, const char *msg3)
{
	static char msg[1024];
	fprintf(stderr,"%s",msg1);
	strcpy(msg,msg1);
	if (msg2) {
		fprintf(stderr,"%s",msg2);
		strcat(msg,msg2);
	}
	if (msg3) {
		fprintf(stderr,"%s",msg3);
		strcat(msg,msg3);
	}
	fprintf(stderr,"\n");
	throw(msg);
}
