#include "text_tex.h"
#include "utility.h"
#include "matrix.h"
#include "scan_zgtext.h"
#include <cmath>

zeText::zeText() : _font(0)
{
	setType(ZE_TEXT);
	_left = _right = _bottom = _top = 0;
	_texHeight = _texWidth = 0;
	_color.a = -1;
	_dpi = 96;
	_fontSize = DEFAULT_FONT_SIZE;
}

zeText::~zeText()
{
}

void
zeText::drawFunc(ZEoption & option)
{
	if (!enabled() || !_font || _bmp.empty()) return;

	ZEcolor color = option.color;
	if (_color.a > 0) color = _color;
	glColor4f(color.r, color.g, color.b, color.a);

	glEnable(GL_TEXTURE_2D);
	glDisable(GL_CULL_FACE);
	glEnable(GL_COLOR_MATERIAL);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glDisable(GL_DEPTH_TEST);
	glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);

	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_ALPHA, _bmpW, _bmpH, 0, GL_ALPHA, GL_UNSIGNED_BYTE, &_bmp[0]);

	glBegin(GL_QUADS);
		glTexCoord2f(0, 0);
		glVertex3f(_left, _bottom, 0);
		glTexCoord2f(_s, 0);
		glVertex3f(_right, _bottom, 0);
		glTexCoord2f(_s, _t);
		glVertex3f(_right, _top, 0);
		glTexCoord2f(0, _t);
		glVertex3f(_left, _top, 0);
	glEnd();
}

void
zeText::setText(const char *str)
{
	if (!_font) return;

	_bmp.clear();

	int len = strlen(str);
	if (len == 0) return;

	_texWidth = 0;

	// create a big-enough buffer

	int row1 = 3*_fontSize, row2 = 0;
	int bufH = 3*_fontSize;
	int bufW = _fontSize*(len+2);
	std::vector<GLubyte> buf(bufH*bufW, 0);

	char *cur = (char*)str;
	int code, flag = 0;
	std::string tmp;

	while (cur[0] && (cur = scan_zgtext(cur, &code))) {
		switch (code) {
		case TOK_SUB1:
			if (flag != 0) error_msg("<sub> error");
			add_string(tmp, flag, buf, bufH, bufW, row1, row2);
			tmp = "";
			flag = -1;
			break;
		case TOK_SUB2:
			if (flag != -1) error_msg("</sub> error");
			add_string(tmp, flag, buf, bufH, bufW, row1, row2);
			tmp = "";
			flag = 0;
			break;
		case TOK_SUP1:
			if (flag != 0) error_msg("<sup> error");
			add_string(tmp, flag, buf, bufH, bufW, row1, row2);
			tmp = "";
			flag = 1;
			break;
		case TOK_SUP2:
			if (flag != 1) error_msg("</sup> error");
			add_string(tmp, flag, buf, bufH, bufW, row1, row2);
			tmp = "";
			flag = 0;
			break;
		case TOK_UNICODE:
			add_string(tmp, flag, buf, bufH, bufW, row1, row2);
			tmp = "";
			add_unicode(axtoi(cur-10, 4), flag, buf, bufH, bufW, row1, row2);
			break;
		case TOK_alpha:
		case TOK_beta:
		case TOK_gamma:
		case TOK_delta:
		case TOK_epsilon:
		case TOK_zeta:
		case TOK_eta:
		case TOK_theta:
		case TOK_iota:
		case TOK_kappa:
		case TOK_lambda:
		case TOK_mu:
		case TOK_nu:
		case TOK_xi:
		case TOK_pi:
		case TOK_rho:
		case TOK_sigma:
		case TOK_tau:
		case TOK_upsilon:
		case TOK_phi:
		case TOK_chi:
		case TOK_psi:
		case TOK_omega:
		case TOK_Gamma:
		case TOK_Delta:
		case TOK_Theta:
		case TOK_Lambda:
		case TOK_Xi:
		case TOK_Pi:
		case TOK_Sigma:
		case TOK_Phi:
		case TOK_Psi:
		case TOK_Omega:
			add_string(tmp, flag, buf, bufH, bufW, row1, row2);
			tmp = "";
			add_unicode(code, flag, buf, bufH, bufW, row1, row2);
			break;
		default:
			tmp += cur[-1];
			break;
		}
	}
	
	add_string(tmp, flag, buf, bufH, bufW, row1, row2);

	// make image for texture

	_texHeight = row2 - row1 + 1;

	_bmpW = texture_size(_texWidth);
	_bmpH = texture_size(_texHeight);
	_bmp.resize(_bmpW*_bmpH, 0);

	int i, j;

	for (i=0; i<_texHeight; i++) {
		for (j=0; j<_texWidth; j++) {
			_bmp[i*_bmpW+j] = buf[(row1+i)*bufW+j];
		}
	}

	_s = float(_texWidth)/float(_bmpW);
	_t = float(_texHeight)/float(_bmpH);

	_bottom = -_texHeight/2;
	_top = _texHeight + _bottom;
	_left = -_texWidth/2;
	_right = _texWidth + _left;
}

void
zeText::add_string(const std::string &str, int flag, std::vector<GLubyte> &buf, int bufH, int bufW, int &row1, int &row2)
{
	if (!_font) return;

	if (str.empty()) return;

	int i, m = str.length()*2 + 2;
	std::vector<unsigned short> wc(m);
#ifdef _WIN32
	m = MultiByteToWideChar(CP_ACP, 0, str.c_str(), str.length(), (LPWSTR)&wc[0], m);
#else
	m = mbstowcs((wchar_t*)&wc[0], str.c_str(), m);
#endif
	if (m <= 0) error_msg("Multibyte to wide character error"); 
	for (i=0; i<m; i++) {
		add_unicode(wc[i], flag, buf, bufH, bufW, row1, row2);
	}
}

void
zeText::add_unicode(unsigned short code, int flag, std::vector<GLubyte> &buf, int bufH, int bufW, int &row1, int &row2)
{
	if (!_font) return;

	if (code == 0) return;

	if (code == 32) {
		_texWidth += int(0.5 + 0.7*_fontSize);	// space 
		return;
	}

	int dy = font_size(flag);

	if (FT_Load_Char(_font->face(), code, FT_LOAD_DEFAULT) != 0 ||
		FT_Render_Glyph(_font->face()->glyph, FT_RENDER_MODE_NORMAL) != 0)
			error_msg("Failed to render character");

	FT_Glyph_Metrics&	met = _font->face()->glyph->metrics;
	FT_Bitmap&			bmp = _font->face()->glyph->bitmap;

	int bearingX = met.horiBearingX/64;
	int bearingY = met.horiBearingY/64;
	int advance = met.horiAdvance/64;

	if (_texWidth == 0 && bearingX < 0) _texWidth = -bearingX;

	int i, j, k, row, col;

	for (i=0; i<bmp.rows; i++) {
		row = dy + _fontSize + bearingY - i;
		for (j=0; j<bmp.width; j++ ) {
			col = _texWidth + bearingX + j;
			k = row*bufW + col;
			if (k >= buf.size()) error_msg("Bad buffer index");
			buf[k] = bmp.buffer[i*bmp.width+j];
		}
	}

	row1 = min(row1, (dy+_fontSize+bearingY-bmp.rows));
	row2 = max(row2, (dy+_fontSize+bearingY));

	if (row1 < 0 || row2 >= 3*_fontSize) error_msg("Bad row numbers");

	_texWidth += advance; 
}

int
zeText::font_size(int flag)
{
	if (!_font) return 0;

	int point = _fontSize;
	if (flag != 0) {
		point = int(0.8*_fontSize+.5);
		point = max(point, 6);
	}

	if (FT_Set_Char_Size(_font->face(), point*64, point*64, _dpi, _dpi) != 0) error_msg("Failed to set font size");
	
	if (flag > 0) return (int(0.3*_fontSize+.5)); 
	if (flag < 0) return (-int(0.3*_fontSize+.5)); 

	return 0; 
}
