#include "render_win.h"
#include "utility.h"
#include "texture.h"

using namespace std;

zeRender::zeRender()
:	_hMemDC(NULL), _hRC(NULL), _img(NULL), _fp(NULL), _buf(0), _width(0), _height(0), _depth(0)
{
	setType(ZE_RENDER);
	_color.r = _color.g = _color.b = _color.a = 1.0;
	_scene.setParent(this);
}

zeRender::~zeRender()
{
	reset();
}

void
zeRender::reset()
{
	wglMakeCurrent(NULL, NULL);
	if (_hMemDC) {
		DeleteDC(_hMemDC);
		_hMemDC = NULL;
	}
	if (_hRC) {
		wglDeleteContext(_hRC);
		_hRC = NULL;
	}
	if (_fp) close_file(_fp);
	_fp = 0;
	if (_buf) gdFree(_buf);
	_buf = 0;
	if (_img) gdImageDestroy(_img);
	_img = 0;
}

void
zeRender::render(bool flag)
{
	glClearColor(_color.r, _color.g, _color.b, _color.a);
	glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT|GL_STENCIL_BUFFER_BIT|GL_ACCUM_BUFFER_BIT);
	_scene.render();
	glFinish();
	if (flag) {
		if (!_img) {
			_img = gdImageCreateTrueColor(_width, _height);
			if (!_img) error_msg("Failed to create GD image");
		}
		for (int row = 0; row < _height; row++) {
			for (int col = 0; col < _width; col++) {
				COLORREF rgb = GetPixel(_hMemDC, col, row);
				int color = gdImageColorAllocate(_img, GetRValue(rgb), GetGValue(rgb), GetBValue(rgb)); 
				gdImageSetPixel(_img, col, row, color);
			}
		}
	}
	else {
		if (_fp) close_file(_fp);
		_fp = 0;
		if (_buf) gdFree(_buf);
		_buf = 0;
		if (_img) gdImageDestroy(_img);
		_img = 0;
	}
}

void
zeRender::initialize(size_t width, size_t height, size_t depth)
{
	reset();

	_width = max(width, 16);
	_height = max(height, 16);
	_depth = min(depth, 32);
	_depth = max(depth, 3);
	_scene.setViewport(0,0,_width,_height);

	HDC hDC = CreateDC("DISPLAY", NULL, NULL, NULL);

	int cbits = GetDeviceCaps(hDC, BITSPIXEL) * GetDeviceCaps(hDC, PLANES);
	if (cbits <= 8) {
		DeleteObject(hDC);
		error_msg("This system is not supported");
	}
	
	_hMemDC = CreateCompatibleDC(hDC);
	DeleteObject(hDC);
	if (!_hMemDC) error_msg("This system is not supported");

	BITMAPINFO bmi;
    ZeroMemory(&bmi, sizeof(BITMAPINFO));
	bmi.bmiHeader.biSize		= sizeof(BITMAPINFOHEADER);
    bmi.bmiHeader.biWidth		= _width;
    bmi.bmiHeader.biHeight		= _height;
    bmi.bmiHeader.biPlanes		= 1;
    bmi.bmiHeader.biBitCount	= 32;
    bmi.bmiHeader.biCompression	= BI_RGB;
    bmi.bmiHeader.biSizeImage	= 0;

	void *pBits;
	HBITMAP hBitmap = CreateDIBSection(hDC, &bmi, DIB_RGB_COLORS, &pBits, NULL, 0);
	if (!hBitmap) error_msg("Failed to CreateDIBSection");

	SelectObject(_hMemDC, hBitmap);
	DeleteObject(hBitmap);

	PIXELFORMATDESCRIPTOR pfd;
	ZeroMemory(&pfd, sizeof(PIXELFORMATDESCRIPTOR));
	pfd.nSize		= sizeof(PIXELFORMATDESCRIPTOR);
	pfd.dwFlags		= PFD_DRAW_TO_BITMAP | PFD_SUPPORT_OPENGL;
	pfd.nVersion	= 1;
	pfd.iPixelType	= PFD_TYPE_RGBA;
	pfd.cColorBits	= 32;
	pfd.cDepthBits	= _depth;

	int nPF = ChoosePixelFormat(_hMemDC, &pfd);
	if (nPF <= 0) error_msg("Failed to ChoosePixelFormat");

	BOOL status = SetPixelFormat(_hMemDC, nPF, &pfd);
	if (!status) error_msg("Failed to SetPixelFormat");

	_hRC = wglCreateContext(_hMemDC);
	if (!_hRC) error_msg("Failed to wglCreateContext");

	if (!wglMakeCurrent(_hMemDC, _hRC)) error_msg("Failed to wglMakeCurrent");
}

void
zeRender::render2Window(HWND hWnd, bool paint)
{
	HDC hDC;
	PAINTSTRUCT ps;
	if (paint)
		hDC = BeginPaint(hWnd, &ps); 
	else
		hDC = GetDC(hWnd);
	RECT rect;
	GetClientRect(hWnd, &rect);
	render(false);
	BitBlt(hDC, 0, 0, rect.right, rect.bottom, _hMemDC, 0, 0, SRCCOPY);
	if (paint)
		EndPaint(hWnd, &ps);
	else
		ReleaseDC(hWnd, hDC);
}

void
zeRender::render2GifAnimation(const char* fname, int delay)
{
	if (strcmp("add", fname) == 0) {
		if (!_fp) error_msg("GIF animation not initialized yet");
		render(true);
		gdImageGifAnimAdd(_img, _fp, 1, 0, 0, delay, gdDisposalNone, 0);
	}
	else if (strcmp("end", fname) == 0) {
		if (_fp) {
			gdImageGifAnimEnd(_fp);
			close_file(_fp);
			_fp = NULL;
		}
	}
	else {
		if (_fp) close_file(_fp);
		_fp = open_file(fname, "wb");
		if (!_fp) error_msg("Failed to open file for writing");
		render(true);
		gdImageGifAnimBegin(_img, _fp, 0, 0);
	}
}

long
zeRender::render2File(const char* fname)
{
	size_t len = strlen(fname);
	if (len < 4) return 0;

	char ext[5], nam[7];
	nam[0] = 0;
	if (len==10) {
		nam[0] = (char)toupper (fname[len-10]);
		nam[1] = (char)toupper (fname[len-9]);
		nam[2] = (char)toupper (fname[len-8]);
		nam[3] = (char)toupper (fname[len-7]);
		nam[4] = (char)toupper (fname[len-6]);
		nam[5] = (char)toupper (fname[len-5]);
		nam[6] = 0;
	}
	ext[0] = (char)toupper (fname[len-4]);
	ext[1] = (char)toupper (fname[len-3]);
	ext[2] = (char)toupper (fname[len-2]);
	ext[3] = (char)toupper (fname[len-1]);
	ext[4] = 0;

	render(true);

	if (!strcmp(nam, "STDOUT") || !strcmp(nam, "STREAM")) {
		int bytes;
		if (_buf) gdFree(_buf);
		if (!strcmp(ext, ".GIF"))
			_buf = (BYTE*)gdImageGifPtr(_img, &bytes);
		else if (!strcmp(ext, ".PNG"))
			_buf = (BYTE*)gdImagePngPtr(_img, &bytes);
		else if (!strcmp(ext, ".JPG"))
			_buf = (BYTE*)gdImageJpegPtr(_img, &bytes, 80);
		else {
			error_msg("Unexpected image format");
		}
		if (!strcmp(nam, "STDOUT")) {
			fflush(stdout);
			fwrite(_buf, 1, bytes, stdout);
			fflush(stdout);
		}
		return bytes;
	}
	else {
		if (!save_image(fname,_img)) error_msg("Failed to save image");
	}
	return 0;
}

void
zeRender::render2Texture(zeTexture *obj)
{
	render(false);

	unsigned char *buf = (unsigned char*)malloc(_width*_height*4);
	if (!buf) error_msg("failed to allocate memory in render2Texture");

	int k=0;
	for (int row = 0; row < _height; row++) {
		for (int col = 0; col < _width; col++) {
			COLORREF rgb = GetPixel(_hMemDC, col, row);
			buf[k++] = GetRValue(rgb);
			buf[k++] = GetGValue(rgb);
			buf[k++] = GetBValue(rgb);
			buf[k++] = 255;
		}
	}

	obj->setImage(buf, _height, _width);

	free(buf);
}

void
zeRender::render2Array(unsigned char *R, unsigned char *G, unsigned char *B, size_t size)
{
	if (size != _width * _height) error_msg("pointer size");

	render(false);
	int k = 0;
	for (int row = 0; row < _height; row++) {
		for (int col = 0; col < _width; col++) {
			COLORREF rgb = GetPixel(_hMemDC, col, _height-row);
			R[k] = GetRValue(rgb);
			G[k] = GetGValue(rgb);
			B[k] = GetBValue(rgb);
			k++;
		}
	}
}


/////////////////////////////////////////////////////////////////////

class Animation
{
	int	_x, _y,			// mouse position
		_rx0, _rz0,		// initial rotation angles
		_rx, _rz,		// current rotation angles
		_ms,			// clock time interval (ms)
		_drag,			// drag status
		_clock,			// clock status
		_key;			// key code

	zeRender *_render;
	zeTransform *_object;
	RenderCallBack _call;
	void *_ctx;			// Z-Script context
	void *_func;		// Z-Script function

public:
	Animation(zeRender *render, zeTransform *obj, int ms, RenderCallBack call, void *ctx, void *func);

	~Animation() { }

	LRESULT CALLBACK wproc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
};

Animation::Animation(zeRender *render, zeTransform *obj, int ms, RenderCallBack call, void *ctx, void *func)
{
	_x = _y = _drag = _clock = 0;
	_key = 39;
	_render = render;
	_object = obj;
	_ms = ms;
	_call = call;
	_ctx = ctx;
	_func = func;
	if (obj) {
		_rx = _rx0 = obj->getRotate().x;
		_rz = _rz0 = obj->getRotate().z;
	}
	else {
		_rx = _rx0 = _rz = _rz0 = 0;
	}
	if (_func) _clock = 1;
}

LRESULT CALLBACK Animation::wproc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch(message) {

	case WM_PAINT:
		_render->render2Window(hWnd, true);
		break;

	case WM_KEYUP:
		if (!_object) return 0;
		if (wParam >= 37 && wParam <= 40) {
			_key = wParam;
			if (_key == 37)             // left arrow
				_rz -= 10;
			else if (_key == 38)        // up arrow
				_rx -= 10;
			else if (_key == 39)        // right arrow
				_rz += 10;
			else						// down arrow
				_rx += 10;
			if (_func) {
				_call(_ctx, _func, "KEYUP", wParam, 10);
				_call(_ctx, _func, "ROTATE", _rz, _rx);
			}
			_object->rotateZ(_rz);
			_object->rotateX(_rx);
			_render->render2Window(hWnd);
		}
		return 0;

	case WM_LBUTTONUP:
		if (!_object) return 0;
		_drag = 0;
		if (_func) {
			_call(_ctx, _func, "LBUTTONUP", LOWORD(lParam), HIWORD(lParam));
			_call(_ctx, _func, "ROTATE", _rz, _rx);
		}
		return 0;

	case WM_LBUTTONDOWN:
		if (!_object) return 0;
		_drag = 1;
		_x = LOWORD(lParam);
		_y = HIWORD(lParam);
		if (_func) _call(_ctx, _func, "LBUTTONDOWN", LOWORD(lParam), HIWORD(lParam));
		return 0;

	case WM_LBUTTONDBLCLK:
		if (!_object) return 0;
		if (_ms > 0) {
			// activate/deactivate clock
			if (_clock == 0) {
				SetTimer(hWnd, (UINT)hWnd, _ms, NULL);
				_clock = 1;
			}
			else {
				KillTimer(hWnd, (UINT)hWnd);
				_clock = 0;
			}
		}
		return 0;

	case WM_MOUSEMOVE:
		if (!_object) return 0;
		if (_drag > 0) {
			int y = HIWORD(lParam);
			int x = LOWORD(lParam);
			if (abs(x-_x) < 10 && abs(y-_y) < 10) return 0;		// dragging distance too short
			int r = min(_render->width(), _render->height()) / 2;
			_rz += 57.29577951*(x-_x)/r;
			_rx += 57.29577951*(y-_y)/r;
			_x = x;
			_y = y;
			_object->rotateZ(_rz);
			_object->rotateX(_rx);
			_render->render2Window(hWnd);
		}
		return 0;

	case WM_RBUTTONUP:
		// resume to original state
		if (!_object) return 0;
		_drag = 0;
		KillTimer(hWnd, (UINT)hWnd);
		_clock = 0;
		_object->rotateZ(_rz0);
		_object->rotateX(_rx0);
		_render->render2Window(hWnd);
		return 0;

	case WM_TIMER:
		if (!_object) return 0;
		if (_key == 37)
			_rz -= 2;
		else if (_key == 38)
			_rx -= 2;
		else if (_key == 39)
			_rz += 2;
		else
			_rx += 2;
		_object->rotateZ(_rz);
		_object->rotateX(_rx);
		if (_func) _call(_ctx, _func, "TIMER", _key, 2);
		_render->render2Window(hWnd);
		return 0;

	case WM_DESTROY:
		PostQuitMessage(0);
		return 0;

	default:
		break;
	}
	return DefWindowProc(hWnd, message, wParam, lParam);
}

VOID CALLBACK TimerProc(HWND hWnd, UINT uMsg, UINT idEvent, DWORD dwTime)
{
	Animation *ani = (Animation*)GetWindowLong(hWnd, GWL_USERDATA);
	ani->wproc(hWnd, WM_TIMER, idEvent, dwTime);
}

LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	Animation *ani = (Animation*)GetWindowLong(hWnd, GWL_USERDATA);
	if (ani) {
		return ani->wproc(hWnd, message, wParam, lParam);
	}
	return DefWindowProc(hWnd, message, wParam, lParam);
}

void
zeRender::show(zeTransform *obj, int ms, RenderCallBack call, void *ctx, void *func)
{
	WNDCLASS wndClass;
   	wndClass.style          = CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS; 
	wndClass.lpfnWndProc    = WndProc;
	wndClass.cbClsExtra     = 0;
	wndClass.cbWndExtra     = 0;
	wndClass.hInstance      = 0;
	wndClass.hIcon          = LoadIcon(NULL, IDI_WINLOGO);
	wndClass.hCursor        = LoadCursor(NULL, IDC_ARROW);
	wndClass.hbrBackground  = (HBRUSH)(COLOR_WINDOW);
	wndClass.lpszMenuName   = 0;
	wndClass.lpszClassName = TEXT("ZeGraph");
	RegisterClass(&wndClass);
	RECT rect = {0, 0, _width, _height};
	AdjustWindowRect(&rect, WS_SYSMENU | WS_THICKFRAME, 1);
	HWND hWnd = CreateWindow(
			TEXT("ZeGraph"), TEXT("ZeGraph (http://www.zegraph.com/)"),
			WS_SYSMENU | WS_MINIMIZEBOX,
			CW_USEDEFAULT, CW_USEDEFAULT,
			rect.right-rect.left, rect.bottom-rect.top,
			0, 0, 0, 0);
	Animation ani(this, obj, ms, call, ctx, func);
	SetWindowLong(hWnd, GWL_USERDATA, (LONG)&ani);
	ShowWindow(hWnd, SW_SHOW);
	UpdateWindow(hWnd);
	if (func && ms > 0) SetTimer(hWnd, (UINT)hWnd, ms, TimerProc);
	MSG msg;
	while(GetMessage(&msg, NULL, 0, 0)) {
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}
}
