#include "plot.h"

using namespace std;

zePlot::zePlot()
{
	setType(ZE_PLOT);
	ZExyz xyz = { .6, .6, .6 };
	scale(xyz);

	_clip = false;
	_color.a = -1;

	_xaxis.setParent(this);
	_xaxis.asXAxis();
	_yaxis.setParent(this);
	_yaxis.asYAxis();
	_zaxis.setParent(this);
	_zaxis.asZAxis();
	_zaxis.disable();

	_xanchor.x = _xanchor.y = _xanchor.z = 0;
	_yanchor.x = _yanchor.y = _yanchor.z = 0;
	_zanchor.x = _zanchor.y = _zanchor.z = 0;

	_rmin.x = _rmin.y = _rmin.z = -1;
	_rmax.x = _rmax.y = _rmax.z =  1;

	_plotNode.setParent(this);
	_nodeNode.setParent(this);
	_textNode.setParent(this);
	_axisNode.setParent(this);
}

void
zePlot::drawFunc(ZEoption & option)
{
	if (_clip) {
		GLdouble p0[4] = { -1, 0, 0, -option.xmin };
		glEnable(GL_CLIP_PLANE0);
		glClipPlane(GL_CLIP_PLANE0, p0);

		GLdouble p1[4] = { 1, 0, 0, option.xmax };
		glEnable(GL_CLIP_PLANE1);
		glClipPlane(GL_CLIP_PLANE1, p1);

		GLdouble p2[4] = { 0, -1, 0, -option.ymin };
		glEnable(GL_CLIP_PLANE2);
		glClipPlane(GL_CLIP_PLANE2, p2);

		GLdouble p3[4] = { 0, 1, 0, option.ymax };
		glEnable(GL_CLIP_PLANE3);
		glClipPlane(GL_CLIP_PLANE3, p3);

		if (_zaxis.enabled()) {
			GLdouble p4[4] = { 0, 0, -1, -option.zmin };
			glEnable(GL_CLIP_PLANE4);
			glClipPlane(GL_CLIP_PLANE4, p4);

			GLdouble p5[4] = { 0, 0, 1, option.zmax };
			glEnable(GL_CLIP_PLANE5);
			glClipPlane(GL_CLIP_PLANE5, p5);
		}
	}

	ZEcolor color = option.color;
	if (_color.a > 0) option.color = _color;
	renderPlotNode(option);
	if (_clip) {
		for (int k = GL_CLIP_PLANE0; k <= GL_CLIP_PLANE5; k++) {
			glDisable(k);
		}
	}
	if (_color.a > 0) option.color = _color;
	renderNodeNode(option);
	if (_color.a > 0) option.color = _color;
	renderAxises(option);
	if (_color.a > 0) option.color = _color;
	renderTextNode(option);
	option.color = color;
}

void
zePlot::renderAxises(ZEoption & option)
{
	ZEoption opt = option;
	opt.xmin *= _scale.x;
	opt.xmax *= _scale.x;
	opt.ymin *= _scale.y;
	opt.ymax *= _scale.y;
	opt.zmin *= _scale.z;
	opt.zmax *= _scale.z;
	
	ZExyz scale = { 1.0/_scale.x, 1.0/_scale.y, 1.0/_scale.z };

	_xaxis.setLabelScale(scale);
	if (_xanchor.x != 0 || _xanchor.y != 0 || _xanchor.z != 0) {
		ZExyz xyz = { _xanchor.x*option.xmax, _xanchor.y*option.ymax, _xanchor.z*option.zmax };
		_xaxis.translate(xyz);
	}
	_xaxis.render(option);

	_yaxis.setLabelScale(scale);
	if (_yanchor.x != 0 || _yanchor.y != 0 || _yanchor.z != 0) {
		ZExyz xyz = { _yanchor.x*option.xmax, _yanchor.y*option.ymax, _yanchor.z*option.zmax };
		_yaxis.translate(xyz);
	}
	_yaxis.render(option);

	_zaxis.setLabelScale(scale);
	if (_zanchor.x != 0 || _zanchor.y != 0 || _zanchor.z != 0) {
		ZExyz xyz = { _zanchor.x*option.xmax, _zanchor.y*option.ymax, _zanchor.z*option.zmax };
		_zaxis.translate(xyz);
	}
	_zaxis.render(option);

	for (size_t i=0; i<_axisNode.size(); i++) {
		zeAxis* axis = (zeAxis*)_axisNode.get(i);
		axis->setLabelScale(scale);
		axis->render(option);
	}
}

void
zePlot::renderTextNode(ZEoption & option)
{
	if (_textXYZ.size() == _textNode.size()) {
		ZExyz scale = { 1.0/_scale.x, 1.0/_scale.y, 1.0/_scale.z };
		for (int i = 0; i < _textXYZ.size(); i++) {
			ZExyz xyz = { option.xmin + (option.xmax-option.xmin)*(_textXYZ[i].x-_rmin.x)/(_rmax.x-_rmin.x),
						  option.ymin + (option.ymax-option.ymin)*(_textXYZ[i].y-_rmin.y)/(_rmax.y-_rmin.y),
						  option.zmin + (option.zmax-option.zmin)*(_textXYZ[i].z-_rmin.z)/(_rmax.z-_rmin.z) };
			zeText* o = (zeText*)_textNode.get(i);
			o->scale(scale);
			o->rotateX(-_rotate.x);
			o->rotateZ(-_rotate.z);
			o->translate(xyz);
			o->render(option);
		}
	}
}

void
zePlot::renderNodeNode(ZEoption & option)
{
	if (_nodeXYZ.size() == _nodeNode.size()) {
		ZExyz scale = { 1.0/_scale.x, 1.0/_scale.y, 1.0/_scale.z };
		for (int i = 0; i < _nodeXYZ.size(); i++) {
			ZExyz xyz = { option.xmin + (option.xmax-option.xmin)*(_nodeXYZ[i].x-_rmin.x)/(_rmax.x-_rmin.x),
						  option.ymin + (option.ymax-option.ymin)*(_nodeXYZ[i].y-_rmin.y)/(_rmax.y-_rmin.y),
						  option.zmin + (option.zmax-option.zmin)*(_nodeXYZ[i].z-_rmin.z)/(_rmax.z-_rmin.z) };
			zeTransform* o = (zeTransform*)_nodeNode.get(i);
			o->scale(scale);
			o->translate(xyz);
			o->render(option);
		}
	}
}

void
zePlot::renderPlotNode(ZEoption & option)
{
	GLfloat xscale = (option.xmax - option.xmin) / (_rmax.x - _rmin.x),
			yscale = (option.ymax - option.ymin) / (_rmax.y - _rmin.y),
			zscale = (option.zmax - option.zmin) / (_rmax.z - _rmin.z);
	ZExyz xyz = { xscale, yscale, zscale };
	_plotNode.scale(xyz);
	xyz.x = option.xmin - _rmin.x * xscale;
	xyz.y = option.ymin - _rmin.y * yscale;
	xyz.z = option.zmin - _rmin.z * zscale;
	_plotNode.translate(xyz);
	_plotNode.render(option);
}

zeNode*
zePlot::add(zeObject *obj)
{
	if (obj->getType()==ZE_AXIS) {
		_axisNode.add(obj);
		obj->setParent(this);
		return &_axisNode;
	}
	_plotNode.add(obj);
	obj->setParent(this);
	return &_plotNode;
}

zeNode*
zePlot::anchor(zeObject *obj, const ZExyz& xyz)
{
	if (obj->getType() == ZE_TEXT) {
		if (_textNode.size() == 0) _textXYZ.clear();
		_textNode.add(obj);
		_textXYZ.push_back(xyz);
		obj->setParent(this);
		return &_textNode;
	}
	else if (obj->getType() == ZE_AXIS) {
		if (_nodeNode.size() == 0) _nodeXYZ.clear();
		_nodeXYZ.push_back(xyz);
		_nodeNode.add(obj);
		obj->setParent(this);
		return &_nodeNode;
	}
	else {
		if (_nodeNode.size() == 0) _nodeXYZ.clear();
		_nodeXYZ.push_back(xyz);
		zeNode *node = new zeNode;
		node->add(obj);
		_nodeNode.add(node);
		obj->setParent(this);
		return &_nodeNode;
	}
}

bool
zePlot::setXrange(double xmin, double xmax)
{
	if (xmax <= xmin) return false;
	_rmin.x = xmin;
	_rmax.x = xmax;
	_xaxis.setAxisRange(xmin, xmax, false);
	return true;
}

bool
zePlot::setYrange(double ymin, double ymax)
{
	if (ymax <= ymin) return false;
	_rmin.y = ymin;
	_rmax.y = ymax;
	_yaxis.setAxisRange(ymin, ymax, false);
	return true;
}
	
bool
zePlot::setZrange(double zmin, double zmax)
{
	if (zmax <= zmin) return false;
	_rmin.z = zmin;
	_rmax.z = zmax;
	_zaxis.setAxisRange(zmin, zmax, false);
	_zaxis.enable();
	return true;
}

ZExyz
zePlot::global2plot(double w, double h, double x, double y, double z) const
{
	if (w <= 0 || h <= 0) {
		ZExyz xyz = { 0, 0, 0 };
		return xyz;
	}

	double d = max(w, h);

	ZExyz xyz = { _rmin.x + (_rmax.x-_rmin.x)*(x-0.5*w)/w,
				  _rmin.y + (_rmax.y-_rmin.y)*(y-0.5*h)/h,
				  _rmin.z + (_rmax.z-_rmin.z)*(z-0.5*d)/d, };

	return xyz;
}

ZExyz
zePlot::plot2global(double w, double h, double x, double y, double z) const
{
	if (w <= 0 || h <= 0) {
		ZExyz xyz = { 0, 0, 0 };
		return xyz;
	}

	double d = max(w, h);

	ZExyz xyz = { -0.5*w + w*(x-_rmin.x)/(_rmax.x-_rmin.x),
				  -0.5*h + h*(y-_rmin.y)/(_rmax.y-_rmin.y),
				  -0.5*d + d*(z-_rmin.z)/(_rmax.z-_rmin.z) };

	return xyz;
}

void
zePlot::anchorXAxis(const ZExyz& anchor)
{
	if (anchor.x > 0)
		_xanchor.x = 1;
	else if (anchor.x < 0)
		_xanchor.x = -1;
	else
		_xanchor.x = 0;

	if (anchor.y > 0)
		_xanchor.y = 1;
	else if (anchor.y < 0)
		_xanchor.y = -1;
	else
		_xanchor.y = 0;

	if (anchor.z > 0)
		_xanchor.z = 1;
	else if (anchor.z < 0)
		_xanchor.z = -1;
	else
		_xanchor.z = 0;

	if (anchor.y <= 0)
		_xaxis.asXAxis(true);
	else
		_xaxis.asXAxis(false);
}

void
zePlot::anchorYAxis(const ZExyz& anchor)
{
	if (anchor.x > 0)
		_yanchor.x = 1;
	else if (anchor.x < 0)
		_yanchor.x = -1;
	else
		_yanchor.x = 0;

	if (anchor.y > 0)
		_yanchor.y = 1;
	else if (anchor.y < 0)
		_yanchor.y = -1;
	else
		_yanchor.y = 0;

	if (anchor.z > 0)
		_yanchor.z = 1;
	else if (anchor.z < 0)
		_yanchor.z = -1;
	else
		_yanchor.z = 0;

	if (anchor.x <= 0)
		_yaxis.asYAxis(true);
	else
		_yaxis.asYAxis(false);
}

void
zePlot::anchorZAxis(const ZExyz& anchor)
{
	if (anchor.x > 0)
		_zanchor.x = 1;
	else if (anchor.x < 0)
		_zanchor.x = -1;
	else
		_zanchor.x = 0;

	if (anchor.y > 0)
		_zanchor.y = 1;
	else if (anchor.y < 0)
		_zanchor.y = -1;
	else
		_zanchor.y = 0;

	if (anchor.z > 0)
		_zanchor.z = 1;
	else if (anchor.z < 0)
		_zanchor.z = -1;
	else
		_zanchor.z = 0;

	if (anchor.x <= 0)
		_zaxis.asZAxis(true);
	else
		_zaxis.asZAxis(false);
	_zaxis.enable();
}


void
zePlot::clear()
{
	_plotNode.clear();
	_nodeNode.clear();
	_textNode.clear();
	_axisNode.clear();
	_nodeXYZ.clear();
	_textXYZ.clear();
}