///
///   Generic 2-dimensional zeMatrix/vector mathematics class specialized for 3D usage
///
/// History:
///
///   04/13/2001 by Paul NCettle: Original creation
///   02/28/2002 modified version from FluidStudios
///   07/25/2003 first modified and bug fixed version by Jiye Zeng
///
/// nele -- number of elements in a vector
/// nvec -- number of vectors in the zeMatrix
///
/// This zeMatrix is compatible with OpenGL's colum-majored matrix.
///

#ifndef	__ZE_MATRIX__
#define __ZE_MATRIX__

#include <cmath>
#include <cassert>

#ifndef DEG_TO_RAD
#define DEG_TO_RAD		0.017453292519943
#endif

#ifndef RAD_TO_DEG 
#define RAD_TO_DEG		57.295779513082300 
#endif

template <size_t nele, size_t nvec, class T = double>
class zeMatrix
{
	T	_data[nvec * nele];

public:

	/// Constructors.
	zeMatrix() {}

	zeMatrix(const zeMatrix & m) {(*this) = m;}

	zeMatrix(const T & t1, const T & t2, const T & t3, const T & t4)
	{
		assert(nele == 4 && nvec == 1);
		_data[0] = t1;
		_data[1] = t2;
		_data[2] = t3;
		_data[3] = t4;
	}

	zeMatrix(const T & t1, const T & t2, const T & t3)
	{
		assert(nele == 3 && nvec == 1);
		_data[0] = t1;
		_data[1] = t2;
		_data[2] = t3;
	}

	zeMatrix(const T & t1, const T & t2)
	{
		assert(nele == 2 && nvec == 1);
		_data[0] = t1;
		_data[1] = t2;
	}

	/// Matrix dimensions.
	size_t numVec () const {return nvec;}
	size_t numEle () const {return nele;}
	size_t size   () const {return nvec * nele;}

	/// The infamous 'operator='.
	zeMatrix & operator =(const zeMatrix & m)
	{
		if (&m != this)
		{
			for (size_t i = 0; i < size(); i++) _data[i] = m._data[i];
		}
		return *this;
	}

	/// Comparison operator.
	bool operator ==(const zeMatrix & m) const
	{
		for (size_t i = 0; i < size(); i++)
		{
			if (_data[i] != m._data[i]) return false;
		}
		return true;
	}

	bool operator !=(const zeMatrix & m) const
	{
		return !(*this == m);
	}

	bool operator <(const zeMatrix & m) const
	{
		for (size_t i = 0; i < size(); i++)
		{
			if (_data[i] >= m._data[i]) return false;
		}
		return true;
	}

	bool operator <=(const zeMatrix & m) const
	{
		for (size_t i = 0; i < size(); i++)
		{
			if (_data[i] > m._data[i]) return false;
		}
		return true;
	}

	bool operator >(const zeMatrix & m) const
	{
		for (size_t i = 0; i < size(); i++)
		{
			if (_data[i] <= m._data[i]) return false;
		}
		return true;
	}

	bool operator >=(const zeMatrix & m) const
	{
		for (size_t i = 0; i < size(); i++)
		{
			if (_data[i] < m._data[i]) return false;
		}
		return true;
	}

	/// Component-wise multiplication.
	zeMatrix operator *(const zeMatrix & m) const
	{
		zeMatrix	result;
		for (size_t i = 0; i < size(); i++)
		{
			result._data[i] = _data[i] * m._data[i];
		}
		return result;
	}

	zeMatrix operator *(const T & value) const
	{
		zeMatrix	result;
		for (size_t i = 0; i < size(); i++)
		{
			result._data[i] = _data[i] * value;
		}
		return result;
	}

	zeMatrix operator *=(const zeMatrix & m)		// into self
	{
		for (size_t i = 0; i < size(); i++)
		{
			_data[i] *= m._data[i];
		}
		return *this;
	}

	zeMatrix operator *=(const T & value)			// into self
	{
		for (size_t i = 0; i < size(); i++)
		{
			_data[i] *= value;
		}
		return *this;
	}

	/// Component-wise division.
	zeMatrix operator /(const zeMatrix & m) const
	{
		zeMatrix	result;
		for (size_t i = 0; i < size(); i++)
		{
			assert(m._data[i] != 0.0);
			result._data[i] = _data[i] / m._data[i];
		}
		return result;
	}

	zeMatrix operator /(const T & value) const
	{
		assert(value != 0.0);
		zeMatrix result;
		for (size_t i = 0; i < size(); i++)
		{
			result._data[i] = _data[i] / value;
		}
		return result;
	}

	zeMatrix operator /=(const zeMatrix & m)		// into self
	{
		for (size_t i = 0; i < size(); i++)
		{
			assert(m._data[i] != 0.0);
			_data[i] /= m._data[i];
		}
		return *this;
	}

	zeMatrix operator /=(const T & value)			// into self
	{
		assert(value != 0.0);
		for (size_t i = 0; i < size(); i++)
		{
			_data[i] /= value;
		}
		return *this;
	}

	zeMatrix inverseDivide(const T & value) const
	{
		zeMatrix result;
		for (size_t i = 0; i < size(); i++)
		{
			assert(_data[i] != 0.0);
			result._data[i] = value / _data[i];
		}
		return result;
	}

	/// Component-wise addition.
	zeMatrix operator +(const zeMatrix & m) const
	{
		zeMatrix	result;
		for (size_t i = 0; i < size(); i++)
		{
			result._data[i] = _data[i] + m._data[i];
		}
		return result;
	}

	zeMatrix operator +(const T & value) const
	{
		zeMatrix	result;
		for (size_t i = 0; i < size(); i++)
		{
			result._data[i] = _data[i] + value;
		}
		return result;
	}

	zeMatrix operator +=(const zeMatrix & m)			// into self
	{
		for (size_t i = 0; i < size(); i++)
		{
			_data[i] += m._data[i];
		}
		return *this;
	}

	zeMatrix operator +=(const T & value)				// into self
	{
		for (size_t i = 0; i < size(); i++)
		{
			_data[i] += value;
		}
		return *this;
	}

	/// Component-wise negation.
	zeMatrix operator -() const
	{
		zeMatrix	result;
		for (size_t i = 0; i < size(); i++)
		{
			result._data[i] = -_data[i];
		}
		return result;
	}

	/// Component-wise subtraction.
	zeMatrix operator -(const zeMatrix & m) const
	{
		zeMatrix	result;
		for (size_t i = 0; i < size(); i++)
		{
			result._data[i] = _data[i] - m._data[i];
		}
		return result;
	}

	zeMatrix operator -(const T & value) const
	{
		zeMatrix result;
		for (size_t i = 0; i < size(); i++)
		{
			result._data[i] = _data[i] - value;
		}
		return result;
	}

	zeMatrix operator -=(const zeMatrix & m)		// into itself
	{
		for (size_t i = 0; i < size(); i++)
		{
			_data[i] -= m._data[i];
		}
		return *this;
	}

	zeMatrix operator -=(const T & value)			// into itself
	{
		for (size_t i = 0; i < size(); i++)
		{
			_data[i] -= value;
		}
		return *this;
	}

	zeMatrix inverseSubtract(const T & value) const
	{
		zeMatrix result;
		for (size_t i = 0; i < size(); i++)
		{
			result._data[i] = value - _data[i];
		}
		return result;
	}

	/// Indexing: ith element (of jth vector).
	const T & operator()(size_t i, size_t j) const {return _data[j * nvec + i];}
		  T & operator()(size_t i, size_t j)       {return _data[j * nvec + i];}

	const T & operator()(size_t i)           const {return _data[i];}
		  T & operator()(size_t i)                 {return _data[i];}

	/// Specialized 'convenience accessors' for vectors, points, etc.
	const T & x() const {return _data[0];}
		  T & x()       {return _data[0];}
	const T & y() const {return _data[1];}
		  T & y()       {return _data[1];}
	const T & z() const {return _data[2];}
		  T & z()       {return _data[2];}
	const T & w() const {return _data[3];}
		  T & w()       {return _data[3];}

	const T * data() const {return _data;}

	/// Convenience for using UVs.
	const T & u() const {return _data[0];}
		  T & u()       {return _data[0];}
	const T & v() const {return _data[1];}
		  T & v()       {return _data[1];}

	// Convenience for using colors (0 to 1).
	const T & r() const {return _data[0];}
		  T & r()       {return _data[0];}
	const T & g() const {return _data[1];}
		  T & g()       {return _data[1];}
	const T & b() const {return _data[2];}
		  T & b()       {return _data[2];}
	const T & a() const {return _data[3];}
		  T & a()       {return _data[3];}

	size_t rgb() const
	{
		assert(nele > 2 && nvec == 1);
		size_t nr = static_cast<size_t>(r() * static_cast<T>(255));
		size_t ng = static_cast<size_t>(g() * static_cast<T>(255));
		size_t nb = static_cast<size_t>(b() * static_cast<T>(255));
		return (nr << 16) | (ng << 8) | nb;
	}

	size_t argb() const
	{
		assert(nele > 3 && nvec == 1);
		size_t na = static_cast<size_t>(a() * static_cast<T>(255));
		size_t nr = static_cast<size_t>(r() * static_cast<T>(255));
		size_t ng = static_cast<size_t>(g() * static_cast<T>(255));
		size_t nb = static_cast<size_t>(b() * static_cast<T>(255));
		return (na << 24) | (nr << 16) | (ng << 8) | nb;
	}

	size_t bgr() const
	{
		assert(nele > 2 && nvec == 1);
		size_t nr = static_cast<size_t>(r() * static_cast<T>(255));
		size_t ng = static_cast<size_t>(g() * static_cast<T>(255));
		size_t nb = static_cast<size_t>(b() * static_cast<T>(255));
		return (nb << 16) | (ng << 8) | nr;
	}

	size_t abgr() const
	{
		assert(nele > 3 && nvec == 1);
		size_t na = static_cast<size_t>(a() * static_cast<T>(255));
		size_t nr = static_cast<size_t>(r() * static_cast<T>(255));
		size_t ng = static_cast<size_t>(g() * static_cast<T>(255));
		size_t nb = static_cast<size_t>(b() * static_cast<T>(255));
		return (na << 24) | (nb << 16) | (ng << 8) | nr;
	}

	/// Replace vector within zeMatrix.
	void setVector(const zeMatrix<nele, 1, T> & m, size_t j)
	{
		assert(j < nvec);
		for (size_t i = 0; i < nele; i++)
		{
			(*this)(i, j) = m(i);
		}
	}

	/// Extract vector from the zeMatrix.
	zeMatrix<nele, 1, T> getVector(size_t j) const
	{
		assert(j < nvec);
		zeMatrix<nele, 1, T> result;
		for (size_t i = 0; i < nele; i++) result(i) = (*this)(i, j);
		return result;
	}

	/// Vector cross product.
	void cross(const zeMatrix<nele, 1, T> & m)
	{
		assert(nele >= 3);
		T tx = y() * m.z() - z() * m.y();
		T ty = z() * m.x() - x() * m.z();
		T tz = x() * m.y() - y() * m.x();
		x() = tx;
		y() = ty;
		z() = tz;
	}

	/// Vector dot product.
	T dot(const zeMatrix<nele, 1, T> & m) const
	{
		T result = static_cast<T>(0);
		for (size_t i = 0; i < nele; i++) result += _data[i] * m._data[i];
		return result;
	}

	/// zeMatrix conatenation.
	zeMatrix & concat(const zeMatrix<nvec, nele, T> & m)
	{
		zeMatrix result;
		result.fill(static_cast<T>(0));

		for (size_t i = 0; i < nele; i++)
		{
			for (size_t j = 0; j < nvec; j++)
			{
				for (size_t k = 0; k < nvec; k++)
				{
					result(i, j) += (*this)(i, k) * m(k, j);
				}
			}
		}

		(*this) = result;

		return (*this);
	}

	/// Sum of all elements.
	T sum() const
	{
		T sum = static_cast<T>(0);
		for (size_t i = 0; i < size(); i++) sum += _data[i];
		return sum;
	}

	/// Fill the zeMatrix with a single value.
	void fill(const T & value)
	{
		for (size_t i = 0; i < size(); i++) _data[i] = value;
	}

	/// Absolute value of all elements.
	void abs()
	{
		for (size_t i = 0; i < size(); i++)
		{
			_data[i] = static_cast<T>(fabs(static_cast<double>(_data[i])));
		}
	}

	/// Confine all elements.
	void confine(const T & min, const T & max)
	{
		for (size_t i = 0; i < size(); i++)
		{
			if (_data[i] < min) _data[i] = min;
			if (_data[i] > max) _data[i] = max;
		}
	}

	/// Orthogonal transpose.
	void transpose()
	{
		zeMatrix result;
		for (size_t j = 0; j < nvec; j++)
		{
			for (size_t i = 0; i < nele; i++)
			{
				result(j, i) = (*this)(i, j);
			}
		}
		*this = result;
	}

/*	// Returns determinant of the zeMatrix (4x4 only)  // ??????

	T determinant() const
	{
		assert(nele == 4 && nvec == 4);

		zeMatrix &m = *this;

		return	  (m(0,0) * m(1,1) - m(1,0) * m(0,1)) * (m(2,2) * m(3,3) - m(3,2) * m(2,3))
				- (m(0,0) * m(2,1) - m(2,0) * m(0,1)) * (m(1,2) * m(3,3) - m(3,2) * m(1,3))
				+ (m(0,0) * m(3,1) - m(3,0) * m(0,1)) * (m(1,2) * m(2,3) - m(2,2) * m(1,3))
				+ (m(1,0) * m(2,1) - m(2,0) * m(1,1)) * (m(0,2) * m(3,3) - m(3,2) * m(0,3))
				- (m(1,0) * m(3,1) - m(3,0) * m(1,1)) * (m(0,2) * m(2,3) - m(2,2) * m(0,3))
				+ (m(2,0) * m(3,1) - m(3,0) * m(2,1)) * (m(0,2) * m(1,3) - m(1,2) * m(0,3));
	}

	// Inverts the zeMatrix (4x4 only)	  // ??????

	void invert()
	{
		assert(nele == 4 && nvec == 4);

		T d = determinant();
		if (d == 0.0) return;

		d = 1.0 / d;

		zeMatrix & m = *this;
		zeMatrix result;
		
		result(0,0) = d * (m(1,1) * (m(2,2) * m(3,3) - m(3,2) * m(2,3)) + m(2,1) * (m(3,2) * m(1,3) - m(1,2) * m(3,3)) + m(3,1) * (m(1,2) * m(2,3) - m(2,2) * m(1,3)));
		result(1,0) = d * (m(1,2) * (m(2,0) * m(3,3) - m(3,0) * m(2,3)) + m(2,2) * (m(3,0) * m(1,3) - m(1,0) * m(3,3)) + m(3,2) * (m(1,0) * m(2,3) - m(2,0) * m(1,3)));
		result(2,0) = d * (m(1,3) * (m(2,0) * m(3,1) - m(3,0) * m(2,1)) + m(2,3) * (m(3,0) * m(1,1) - m(1,0) * m(3,1)) + m(3,3) * (m(1,0) * m(2,1) - m(2,0) * m(1,1)));
		result(3,0) = d * (m(1,0) * (m(3,1) * m(2,2) - m(2,1) * m(3,2)) + m(2,0) * (m(1,1) * m(3,2) - m(3,1) * m(1,2)) + m(3,0) * (m(2,1) * m(1,2) - m(1,1) * m(2,2)));
		result(0,1) = d * (m(2,1) * (m(0,2) * m(3,3) - m(3,2) * m(0,3)) + m(3,1) * (m(2,2) * m(0,3) - m(0,2) * m(2,3)) + m(0,1) * (m(3,2) * m(2,3) - m(2,2) * m(3,3)));
		result(1,1) = d * (m(2,2) * (m(0,0) * m(3,3) - m(3,0) * m(0,3)) + m(3,2) * (m(2,0) * m(0,3) - m(0,0) * m(2,3)) + m(0,2) * (m(3,0) * m(2,3) - m(2,0) * m(3,3)));
		result(2,1) = d * (m(2,3) * (m(0,0) * m(3,1) - m(3,0) * m(0,1)) + m(3,3) * (m(2,0) * m(0,1) - m(0,0) * m(2,1)) + m(0,3) * (m(3,0) * m(2,1) - m(2,0) * m(3,1)));
		result(3,1) = d * (m(2,0) * (m(3,1) * m(0,2) - m(0,1) * m(3,2)) + m(3,0) * (m(0,1) * m(2,2) - m(2,1) * m(0,2)) + m(0,0) * (m(2,1) * m(3,2) - m(3,1) * m(2,2)));
		result(0,2) = d * (m(3,1) * (m(0,2) * m(1,3) - m(1,2) * m(0,3)) + m(0,1) * (m(1,2) * m(3,3) - m(3,2) * m(1,3)) + m(1,1) * (m(3,2) * m(0,3) - m(0,2) * m(3,3)));
		result(1,2) = d * (m(3,2) * (m(0,0) * m(1,3) - m(1,0) * m(0,3)) + m(0,2) * (m(1,0) * m(3,3) - m(3,0) * m(1,3)) + m(1,2) * (m(3,0) * m(0,3) - m(0,0) * m(3,3)));
		result(2,2) = d * (m(3,3) * (m(0,0) * m(1,1) - m(1,0) * m(0,1)) + m(0,3) * (m(1,0) * m(3,1) - m(3,0) * m(1,1)) + m(1,3) * (m(3,0) * m(0,1) - m(0,0) * m(3,1)));
		result(3,2) = d * (m(3,0) * (m(1,1) * m(0,2) - m(0,1) * m(1,2)) + m(0,0) * (m(3,1) * m(1,2) - m(1,1) * m(3,2)) + m(1,0) * (m(0,1) * m(3,2) - m(3,1) * m(0,2)));
		result(0,3) = d * (m(0,1) * (m(2,2) * m(1,3) - m(1,2) * m(2,3)) + m(1,1) * (m(0,2) * m(2,3) - m(2,2) * m(0,3)) + m(2,1) * (m(1,2) * m(0,3) - m(0,2) * m(1,3)));
		result(1,3) = d * (m(0,2) * (m(2,0) * m(1,3) - m(1,0) * m(2,3)) + m(1,2) * (m(0,0) * m(2,3) - m(2,0) * m(0,3)) + m(2,2) * (m(1,0) * m(0,3) - m(0,0) * m(1,3)));
		result(2,3) = d * (m(0,3) * (m(2,0) * m(1,1) - m(1,0) * m(2,1)) + m(1,3) * (m(0,0) * m(2,1) - m(2,0) * m(0,1)) + m(2,3) * (m(1,0) * m(0,1) - m(0,0) * m(1,1)));
		result(3,3) = d * (m(0,0) * (m(1,1) * m(2,2) - m(2,1) * m(1,2)) + m(1,0) * (m(2,1) * m(0,2) - m(0,1) * m(2,2)) + m(2,0) * (m(0,1) * m(1,2) - m(1,1) * m(0,2)));
		*this = result;
	}
*/

	/// Generate identity zeMatrix.
	static zeMatrix genIdentity()
	{
		assert(nele == nvec);
		zeMatrix result;
		result.fill(0);
		for (size_t j = 0; j < nvec; j++) result(j, j) = static_cast<T>(1);
		return result;
	}

	/// Generate rotation zeMatrix (3x3).
	//  Rotation happens in a counter-clockwise direction around X/Y/Z vector
	static zeMatrix genXRotation(const T & theta)
	{
		zeMatrix result = genIdentity();
		double a = static_cast<double>(theta) * DEG_TO_RAD;
		T ct = static_cast<T>(cos(a));
		T st = static_cast<T>(sin(a));
		result(1, 1) =  ct;
		result(1, 2) = -st;
		result(2, 1) =  st;
		result(2, 2) =  ct;
		
		return result;
	}

	static zeMatrix genYRotation(const T & theta)
	{
		zeMatrix result = genIdentity();
		double a = static_cast<double>(theta) * DEG_TO_RAD;
		T ct = static_cast<T>(cos(a));
		T st = static_cast<T>(sin(a));
		result(0, 0) =  ct;
		result(0, 2) =  st;
		result(2, 0) = -st;
		result(2, 2) =  ct;
		
		return result;
	}

	static zeMatrix genZRotation(const T & theta)
	{
		zeMatrix result = genIdentity();
		double a = static_cast<double>(theta) * DEG_TO_RAD;
		T ct = static_cast<T>(cos(a));
		T st = static_cast<T>(sin(a));
		result(0, 0) =  ct;
		result(0, 1) = -st;
		result(1, 0) =  st;
		result(1, 1) =  ct;

		return result;
	}

	/// Generate a concatenated rotation zeMatrix (3x3) for rotation about all axes in the following order: First Z, then Y and finally X.
	static zeMatrix genRotation(const T & xr, const T & yr, const T & zr)
	{
		zeMatrix result = genXRotation(xr);
		result.concat(genYRotation(yr));
		result.concat(genZRotation(zr));
		return result;
	}

	/// Generate a scale zeMatrix.
	static zeMatrix genScale(const T & xs, const T & ys, const T & zs)
	{
		assert(nele >= 3);
		zeMatrix result = genIdentity();
		result(0, 0) = xs;
		result(1, 1) = ys;
		result(2, 2) = zs;
		return result;
	}

	/// Generate a translation zeMatrix.
	static zeMatrix genTranslate(const T & xt, const T & yt, const T & zt)
	{
		assert(nele >= 3);
		zeMatrix result = genIdentity();
		result(0, nvec-1) = xt;
		result(1, nvec-1) = yt;
		result(2, nvec-1) = zt;
		return result;
	}

	/// Generate a shear zeMatrix.
	static zeMatrix genShear(const T x, const T y)
	{
		zeMatrix result = genIdentity();
		result(0, 1) = x;
		result(1, 0) = y;
		return result;
	}

	/// Generate a P1 to P2 translation matrix.
	static zeMatrix genPt2Pt(const T x1, const T y1, const T z1,
							 const T x2, const T y2, const T z2)
	{
		T x = x2 - x1;
		T y = y2 - y1;
		T z = z2 - z1;
		T d, rx, rz;
	
		d = sqrt(x * x + y * y + z * z);
		if (d == 0.0)
		{
			return genIdentity();
		}
		else
		{
			rz = acos(z / d) * RAD_TO_DEG;
			d = sqrt(x * x + y * y);
			rx = 0.0;
			if (d > 0.0) rx = acos(x / d) * RAD_TO_DEG;
			if (y < 0.0) rx = -rx;
		}

		zeMatrix<4, 4> result = genTranslate(x1, y1, z1);
		result.concat(genXRotation(rx));
		result.concat(genZRotation(rz));

		return result;
	}

	/// Extract the translation from the zeMatrix.
	zeMatrix<nele, 1, T> getTranslation() const
	{
		zeMatrix<nele, 1, T> result;
		for (size_t i = 0; i < nele; i++) result(i) = (*this)(i, nvec-1);
		return result;
	}

	/// Vector length.
	T lengthSquared() const
	{
		assert(nvec == 1);
		return dot(*this);
	}

	T length() const
	{
		assert(nvec == 1);
		return static_cast<T>(sqrt(lengthSquared()));
	}

	void setLength(const T & len)
	{
		assert(nvec == 1);
		T L = len / length();
		for (size_t i = 0; i < nele; ++i) _data[i] *= L;
	}

	T distance(const zeMatrix & m) const
	{
		assert(nvec == 1);
		zeMatrix temp = *this - m;
		return temp.length();
	}

	/// NCormalize.
	void normalize()
	{
		setLength(static_cast<T>(1));
	}

	/// set the valuse at index i.
	void set(size_t i, double v)
	{
		assert(nvec * nele > i);
		_data[i] = v;
	}

/*	// Normalize an orthogonal zeMatrix (i.e. make sure the internal vectors are all perpendicular)

	void orthoNormalize()
	{
		assert(nele == 3 && nvec == 3);

		zeMatrix<nele, 1, T> xVector = extractXVector();
		zeMatrix<nele, 1, T> yVector = extractYVector();
		zeMatrix<nele, 1, T> zVector = extractZVector();

		xVector -= yVector * (xVector * yVector);
		xVector -= zVector * (xVector * zVector);
		xVector.normalize();

		yVector -= xVector * (yVector * xVector);
		yVector -= zVector * (yVector * zVector);
		yVector.normalize();

		zVector = xVector.cross(yVector);

		setVector(xVector, 0);
		setVector(yVector, 1);
		setVector(zVector, 2);
	}
*/

};


#endif	// ZE_MATRIX
