#include "globe.h"
#include "factory.h"
#include "utility.h"

zeGlobe::zeGlobe() : _radius(0)
{
	setType(ZE_GLOBE);
	_light.disable();
	_surface.setParent(this);
	_light.setParent(this);
	_node.setParent(this);
	_map.setParent(this);
	_grid.setParent(this);
}

void 
zeGlobe::drawFunc(ZEoption & option)
{
	if (!enabled() || _radius <= 0) return;
	GLenum k = _light.on();
	_surface.render(option);
	_map.render(option);
	_grid.render(option);
	_node.render(option);
	_light.off(k);
}

zePolygon*
zeGlobe::surface(GLfloat radius, int iters)
{
	_radius = radius;
	if (iters <= 0) iters = 2 + log(4.0*3.1416*radius*radius/200.0)/log(4.0);
	if (iters > 8) iters = 8;
	make_sphere(&_surface, _radius, iters);
	ZEcolor rgba = { 0, 0, .8, 1 };
	_surface.setColor(rgba);
	return &_surface;
}

zePolygon*
zeGlobe::texture(GLfloat radius, const char* fname, bool atlantic)
{
	_radius = radius;
	double grid = radius > 100 ? 5 : 10;
	make_sphere2(&_surface, _radius, grid, atlantic);
	zeTexture *texture = new zeTexture;
	texture->setImage(fname, -1, -1, -1);
	_surface.setTexture(texture);
	ZEcolor rgba = { 1, 1, 1, 1 };
	_surface.setColor(rgba);
	if (texture->s() < 1 || texture->t() < 1) {
		zeTexCoord *st = _surface.getTexCoord();
		for (size_t i = 0; i < st->size(); i++) {
			st->s(i) *= texture->s();
			st->t(i) *= texture->t();
		}
	}
	return &_surface;
}

zePolygon*
zeGlobe::texture(GLfloat radius, double *data, int nrow, int ncol, bool atlantic, zeColorBar *cbar)
{
	_radius = radius;
	double grid = radius > 100 ? 5 : 10;
	make_sphere2(&_surface, _radius, grid, atlantic);
	zeTexture *texture = new zeTexture;
	texture->setImage(data, nrow, ncol, cbar);
	_surface.setTexture(texture);
	ZEcolor rgba = { 1, 1, 1, 1 };
	_surface.setColor(rgba);
	if (texture->s() < 1 || texture->t() < 1) {
		zeTexCoord *st = _surface.getTexCoord();
		for (size_t i = 0; i < st->size(); i++) {
			st->s(i) *= texture->s();
			st->t(i) *= texture->t();
		}
	}
	return &_surface;
}

zeNode*
zeGlobe::grid(int deg, GLfloat width)
{
	if (_radius <= 0 || deg <= 0 || width <= 0) return &_grid;

	_grid.clear();

	double r = 1.001*_radius;

	int lon, lat, lat1, lat2;

	for (lat = 0; lat < 90; lat += deg) {
		zeLine *line = new zeLine;
		_grid.add(line);
		line->asLineLoop();
		line->asSolidLine(1, width);
		zeVertex *vertex = new zeVertex;
		line->setVertex(vertex);
	    zeVertex *normal = new zeVertex;
		line->setVertexNormal(normal);
		line->setAntialias(false);
	    for (lon = 0; lon < 360; lon++) {
			double x, y, z;
			ll2xyz(x, y, z, lon, lat, 1);
			ZExyz xyz1 = { x*r, y*r, z*r };
			ZExyz nor1 = { x, y, z };
			vertex->add(xyz1);
			normal->add(nor1);
        }
		lat2 = lat;
	}

	for (lat = -deg; lat > -90; lat -= deg) {
		zeLine *line = new zeLine;
		_grid.add(line);
		line->asLineLoop();
		line->asSolidLine(1, width);
		zeVertex *vertex = new zeVertex;
		line->setVertex(vertex);
	    zeVertex *normal = new zeVertex;
		line->setVertexNormal(normal);
		line->setAntialias(false);
	    for (lon = 0; lon < 360; lon++) {
			double x, y, z;
			ll2xyz(x, y, z, lon, lat, 1);
			ZExyz xyz1 = { x*r, y*r, z*r };
			ZExyz nor1 = { x, y, z };
			vertex->add(xyz1);
			normal->add(nor1);
        }
		lat1 = lat;
	}

    for (lon = 0; lon < 360; lon+=deg) {
		zeLine *line = new zeLine;
		_grid.add(line);
		line->asLineStrip();
		line->asSolidLine(1, width);
		zeVertex *vertex = new zeVertex;
		line->setVertex(vertex);
	    zeVertex *normal = new zeVertex;
		line->setVertexNormal(normal);
		line->setAntialias(false);
		for (lat = lat1; lat <= lat2; lat++) {
			double x, y, z;
			ll2xyz(x, y, z, lon, lat, 1);
			ZExyz xyz1 = { x*r, y*r, z*r };
			ZExyz nor1 = { x, y, z };
			vertex->add(xyz1);
			normal->add(nor1);
        }
	}

	return &_grid;
}

void
zeGlobe::focus(GLfloat lon, GLfloat lat)
{
	rotateZ(-(90+lon));
	rotateX(-(90-lat));
}

zeNode*
zeGlobe::add(zeObject *obj)
{
	_node.add(obj);
	obj->setParent(this);
	return &_node;
}

zeNode*
zeGlobe::anchor(zeObject *obj, GLfloat lon, GLfloat lat, GLfloat alt)
{
	ZExyz xyz = { 1.01*_radius+alt, 0, 0 };
	zeNode *node = new zeNode;
	node->add(obj);
	_node.add(node);
	node->translate(xyz);
	node->rotateY(-lat);
	node->rotateZ(lon);
	obj->setParent(this);
	return &_node;
}

zeLine*
zeGlobe::line(double lon1, double lat1, double lon2, double lat2)
{
	if (lon1 < -180 || lon1 > 360 ||
		lon2 < -180 || lon2 > 360 ||
		lat1 < -90  || lat1 > 90  || 
		lat2 < -90  || lat2 > 90) return 0;
	zeLine *line = new zeLine;
	spherical_curve(line, _radius*1.01, lon1, lat1, lon2, lat2);
	add(line);
	return line;
}

zeNode*
zeGlobe::field(const double *U, const double *V,
			   const double *X, int nx, const double *Y, int ny,
			   double size, const zeColorBar *colorbar)
{
	int i, j, k;

	if (fabs(X[nx-1]-X[0]) > 360-EPS) nx--;
	double dX = X[nx-1] - X[0];
	if (_radius <= 0 || nx <= 1 || ny <= 1 || size <= 0 || dX <= 0) return 0;

	zeNode *node = new zeNode;
	add(node);

	for (i = 1; i < ny-1; i++) {
		double z = sin(DEG_TO_RAD*Y[i]);
		double d = cos(DEG_TO_RAD*Y[i]);
		int n = nx*d + 1;
		n = n > nx ? nx : n;
		double dx = dX/(n-1);
		double x0 = 0.5*fmod(dX, dx);
		int m = i*nx; 
		for (k = 0; k < n; k++) {
			double x = X[0] + x0 + k*dx;
			for (j = 1; j < nx; j++) {
				if (x <= X[j]) {
					double d = X[j] - X[j-1];
					double a = (X[j] - x) / d;
					double b = (x - X[j-1]) / d;
					double u = a*U[m+j-1] + b*U[m+j];
					double v = a*V[m+j-1] + b*V[m+j];
					zePolygon *cone = new zePolygon;
					make_cone_vector2(cone, _radius, x, Y[i], u, v, size);
					ZEcolor rgb = colorbar->getColor(sqrt(u*u+v*v));
					cone->setColor(rgb);
					node->add(cone);
					break;
				}
			}
		}
	}

	return node;
}

zeNode*
zeGlobe::gshhs(const char *fname, GLfloat width, bool smooth)
{
	FILE *fp = open_file(fname, "rb");
	if (!fp) return &_map;

	GSHHS h;
	int i, n, greenwich, version, flip;
	
	fread(&h,sizeof(GSHHS),1,fp);
	version=(h.flag>>8)&255;
	flip=(version!=GSHHS_DATA_RELEASE);		// Take as sign that byte-swabbing is needed
	fseek(fp,0,SEEK_SET);

	double r = 1.001*_radius;

	while (!feof(fp)) {
		if (fread(&h,sizeof(GSHHS),1,fp)!=1) break;
		if (flip) {
			h.id=swabi4((unsigned int)h.id);
			h.n=swabi4((unsigned int)h.n);
			h.west=swabi4((unsigned int)h.west);
			h.east=swabi4((unsigned int)h.east);
			h.south=swabi4((unsigned int)h.south);
			h.north=swabi4((unsigned int)h.north);
			h.area=swabi4((unsigned int)h.area);
			h.area_full=swabi4((unsigned int)h.area_full);
			h.flag=swabi4((unsigned int)h.flag);
			h.container=swabi4((unsigned int)h.container);
			h.ancestor=swabi4((unsigned int)h.ancestor);
		}
		n=h.n;
		greenwich=(h.flag>>16)&1;			// Greenwich is 0 or 1
		if ((h.flag&255)!=1) {				// not land
			fseek(fp,sizeof(GSHHS_XY)*n,SEEK_CUR);
			continue;
		}

		zeLine *line = new zeLine;
		_map.add(line);
		zeVertex *vert = new zeVertex;
		line->setVertex(vert);
		zeVertex *norm = new zeVertex;
		line->setVertexNormal(norm);
		line->asLineStrip();
		line->asSolidLine(1, width);
		line->setAntialias(smooth);
		for (i = 0; i < n; i++) {
			GSHHS_XY xy;
			fread(&xy, sizeof(GSHHS_XY), 1, fp);
			double x, y, z;
			ll2xyz(x, y, z, 1.e-6*int(swabi4((unsigned int)xy.x)), 1.e-6*int(swabi4((unsigned int)xy.y)), 1);
			ZExyz nor = { x, y, z };
			ZExyz xyz = { x*r, y*r, z*r };
			norm->add(nor);
			vert->add(xyz);
		}
	}

	close_file(fp);

	return &_map;
}

void
zeGlobe::clear()
{
	_node.clear();
}

void
zeGlobe::lookAt(GLfloat eLon, GLfloat eLat, GLfloat eAlt, GLfloat cLon, GLfloat cLat, GLfloat cAlt)
{
	if (_radius+cAlt <= 0 || _radius+eAlt <= 0) return;
	reset();
	double x, y, z, x2, y2, z2,lon, lat;
	ll2xyz(x, y, z, cLon, cLat, _radius+cAlt);
	ll2xyz(x2, y2, z2, eLon, eLat, _radius+eAlt);
	ZExyz xyz = {-x, -y, -z};
	translate(xyz);
	xyz2ll(lon, lat, x2-x, y2-y, z2-z);
	rotateZ(-(90+lon));
	rotateX(-(90-lat));
}

