#include "factory.h"
#include <list>

int circle_slices(double r, double &d)
{
	int n = PI2*r/5;
	if (n < 5) n = 5;
	d = PI2/(n-1);
	return n;
}

ZExyz vertex_interp2D(const ZExyz &p1, const ZExyz &p2, GLfloat isoValue)
{
	ZExyz p;
	if (p1.z == p2.z) {
		p = p1;
	}
	else {
		GLfloat f = (isoValue - p1.z) / (p2.z - p1.z);
		p.x = p1.x + f * (p2.x - p1.x);
		p.y = p1.y + f * (p2.y - p1.y);
	}
	p.z = isoValue;
	return p;
}

ZExyz vertex_interp3D(const ZEvolume &v1, const ZEvolume &v2, GLfloat isoValue)
{
	ZExyz p;
	if (v1.w == v2.w) {
		p.x = 0.5*(v1.x + v2.x);
		p.y = 0.5*(v1.y + v2.y);
		p.z = 0.5*(v1.z + v2.z);
	}
	else {
		GLfloat f = (isoValue - v1.w) / (v2.w - v1.w);
		p.x = v1.x + f * (v2.x - v1.x);
		p.y = v1.y + f * (v2.y - v1.y);
		p.z = v1.z + f * (v2.z - v1.z);
	}
	return p;
}

void normalize_vertex(ZExyz &p)
{
	GLfloat r = sqrt(p.x * p.x + p.y * p.y + p.z * p.z) + EPS;
	p.x /= r;
	p.y /= r;
	p.z /= r;
}

ZExyz cross_product2(const ZExyz &p1, const ZExyz &p2)
{
	ZExyz p;
	p.x = p1.y * p2.z - p1.z * p2.y;
	p.y = p1.z * p2.x - p1.x * p2.z;
	p.z = p1.x * p2.y - p1.y * p2.x;
	return p;
}

ZExyz cross_product3(const ZExyz &p0, const ZExyz &p1, const ZExyz &p2)
{
	ZExyz v1, v2, p;
	v1.x = p1.x - p0.x;
	v1.y = p1.y - p0.y;
	v1.z = p1.z - p0.z;
	v2.x = p2.x - p0.x;
	v2.y = p2.y - p0.y;
	v2.z = p2.z - p0.z;
	p.x = v1.y * v2.z - v1.z * v2.y;
	p.y = v1.z * v2.x - v1.x * v2.z;
	p.z = v1.x * v2.y - v1.y * v2.x;
	return p;
}

ZExyz mid_point(const ZExyz &p1, const ZExyz &p2)
{
	ZExyz p;
	p.x = 0.5 * (p1.x + p2.x);
	p.y = 0.5 * (p1.y + p2.y);
	p.z = 0.5 * (p1.z + p2.z);
	return p;
}

int contour3_line(zeVertex *vertex, GLfloat isoValue, const ZExyz &p0, const ZExyz &p1, const ZExyz &p2)
{
	int edges = 0;
	if (p0.z >= isoValue) edges |= 1;
	if (p1.z >= isoValue) edges |= 2;
	if (p2.z >= isoValue) edges |= 4;

	if (!vertex) return edges;

	ZExyz e;

	switch (edges) {
	case 1:
	case 6:
		e = vertex_interp2D(p0, p1, isoValue);
		vertex->add(e);
		e = vertex_interp2D(p0, p2, isoValue);
		vertex->add(e);
		break;
	case 2:
	case 5:
		e = vertex_interp2D(p1, p0, isoValue);
		vertex->add(e);
		e = vertex_interp2D(p1, p2, isoValue);
		vertex->add(e);
		break;
	case 3:
	case 4:
		e = vertex_interp2D(p2, p1, isoValue);
		vertex->add(e);
		e = vertex_interp2D(p2, p0, isoValue);
		vertex->add(e);
		break;
	}

	return edges;
}

int contour4_line(zeVertex *vertex, GLfloat isoValue, const ZExyz &p0, const ZExyz &p1, const ZExyz &p2, const ZExyz &p3)
{
	int edges = 0;
	if (p0.z >= isoValue) edges |= 1;
	if (p1.z >= isoValue) edges |= 2;
	if (p2.z >= isoValue) edges |= 4;
	if (p3.z >= isoValue) edges |= 8;

	if (!vertex) return edges;

	ZExyz e;

	switch (edges) {
	case 1:
	case 14:
		e = vertex_interp2D(p0, p1, isoValue);
		vertex->add(e);
		e = vertex_interp2D(p0, p3, isoValue);
		vertex->add(e);
		break;
	case 2:
	case 13:
		e = vertex_interp2D(p1, p0, isoValue);
		vertex->add(e);
		e = vertex_interp2D(p1, p2, isoValue);
		vertex->add(e);
		break;
	case 3:
	case 12:
		e = vertex_interp2D(p1, p2, isoValue);
		vertex->add(e);
		e = vertex_interp2D(p0, p3, isoValue);
		vertex->add(e);
		break;
	case 4:
	case 11:
		e = vertex_interp2D(p2, p1, isoValue);
		vertex->add(e);
		e = vertex_interp2D(p2, p3, isoValue);
		vertex->add(e);
		break;
	case 5:
	case 10:
		e = vertex_interp2D(p0, p1, isoValue);
		vertex->add(e);
		e = vertex_interp2D(p1, p2, isoValue);
		vertex->add(e);
		e = vertex_interp2D(p2, p3, isoValue);
		vertex->add(e);
		e = vertex_interp2D(p3, p0, isoValue);
		vertex->add(e);
		break;
	case 6:
	case 9:
		e = vertex_interp2D(p0, p1, isoValue);
		vertex->add(e);
		e = vertex_interp2D(p2, p3, isoValue);
		vertex->add(e);
		break;
	case 7:
	case 8:
		e = vertex_interp2D(p3, p2, isoValue);
		vertex->add(e);
		e = vertex_interp2D(p3, p0, isoValue);
		vertex->add(e);
		break;
	}

	return edges;
}

int iso_surface(zeVertex *vertex, GLfloat isoValue, const ZEvolume &p0, const ZEvolume &p1, const ZEvolume &p2, const ZEvolume &p3)
{
	// Enclose high point

	int edges = 0;
	if (p0.w < isoValue) edges |= 1;
	if (p1.w < isoValue) edges |= 2;
	if (p2.w < isoValue) edges |= 4;
	if (p3.w < isoValue) edges |= 8;

	if (!vertex) return edges;

	ZExyz e1, e2, e3, e4;

	switch (edges) {
	case 1:
	case 14:
		e1 = vertex_interp3D(p0, p1, isoValue);
		e2 = vertex_interp3D(p0, p2, isoValue);
		e3 = vertex_interp3D(p0, p3, isoValue);
		break;
	case 2:
	case 13:
		e1 = vertex_interp3D(p1, p0, isoValue);
		e2 = vertex_interp3D(p1, p3, isoValue);
		e3 = vertex_interp3D(p1, p2, isoValue);
		break;
	case 3:
	case 12:
		e1 = vertex_interp3D(p0, p2, isoValue);
		e2 = vertex_interp3D(p0, p3, isoValue);
		e3 = vertex_interp3D(p1, p3, isoValue);
		e4 = vertex_interp3D(p1, p2, isoValue);
		break;
	case 4:
	case 11:
		e1 = vertex_interp3D(p2, p0, isoValue);
		e2 = vertex_interp3D(p2, p1, isoValue);
		e3 = vertex_interp3D(p2, p3, isoValue);
		break;
	case 5:
	case 10:
		e1 = vertex_interp3D(p0, p3, isoValue);
		e2 = vertex_interp3D(p0, p1, isoValue);
		e3 = vertex_interp3D(p2, p1, isoValue);
		e4 = vertex_interp3D(p2, p3, isoValue);
		break;
	case 6:
	case 9:
		e1 = vertex_interp3D(p1, p0, isoValue);
		e2 = vertex_interp3D(p1, p3, isoValue);
		e3 = vertex_interp3D(p2, p3, isoValue);
		e4 = vertex_interp3D(p2, p0, isoValue);
		break;
	case 7:
	case 8:
		e1 = vertex_interp3D(p3, p0, isoValue);
		e2 = vertex_interp3D(p3, p1, isoValue);
		e3 = vertex_interp3D(p3, p2, isoValue);
		break;
	}

	switch (edges) {
	case 1:
	case 2:
	case 4:
	case 7:
		vertex->add(e1);
		vertex->add(e2);
		vertex->add(e3);
		break;
	case 8:
	case 11:
	case 13:
	case 14:
		vertex->add(e3);
		vertex->add(e2);
		vertex->add(e1);
		break;
	case 3:
	case 5:
	case 6:
		vertex->add(e1);
		vertex->add(e2);
		vertex->add(e3);
		vertex->add(e3);
		vertex->add(e4);
		vertex->add(e1);
		break;
	case 9:
	case 10:
	case 12:
		vertex->add(e3);
		vertex->add(e2);
		vertex->add(e1);
		vertex->add(e1);
		vertex->add(e4);
		vertex->add(e3);
		break;
	}

	return edges;
}

int sub_divide(int n, int iterations, ZEtriangle *faces, int size)
{
	int i, m;
	for (m = n, i = 0; i < iterations; i++) m *= 4;
	if (size < m) return 0;
	for (i = 0; i < iterations; i++) {
		int next = n;
		for (int j = 0; j < next; j++) {
			faces[n  ] = faces[j];
			faces[n+1] = faces[j];
			faces[n+2] = faces[j];
			ZExyz p1, p2, p3;
			p1 = mid_point(faces[j].p1, faces[j].p2);
			p2 = mid_point(faces[j].p2, faces[j].p3);
			p3 = mid_point(faces[j].p3, faces[j].p1);
			faces[j].p2 = p1;
			faces[j].p3 = p3;
			faces[n  ].p1 = p1;
			faces[n  ].p3 = p2;
			faces[n+1].p1 = p3;
			faces[n+1].p2 = p2;
			faces[n+2].p1 = p1;
			faces[n+2].p2 = p2;
			faces[n+2].p3 = p3;
			n += 3;
		}
	}
	return n;
}

void make_box(zePolygon *poly, double size)
{
	poly->asQuads();
	zeVertex *vert = new zeVertex;
	poly->setVertex(vert);
	zeVertex *norm = new zeVertex;
	poly->setVertexNormal(norm);
	vert->add(-size, -size, -size);
	vert->add(-size,  size, -size);
	vert->add( size,  size, -size);
	vert->add( size, -size, -size);
	norm->add(0, 0, -1); 
	norm->add(0, 0, -1); 
	norm->add(0, 0, -1); 
	norm->add(0, 0, -1);
	vert->add(-size, -size, size);
	vert->add( size, -size, size);
	vert->add( size,  size, size);
	vert->add(-size,  size, size);
	norm->add(0, 0, 1); 
	norm->add(0, 0, 1); 
	norm->add(0, 0, 1); 
	norm->add(0, 0, 1); 
	vert->add(-size, -size, -size);
	vert->add(-size, -size,  size);
	vert->add( size, -size,  size);
	vert->add( size, -size, -size);
	norm->add(0, -1, 0); 
	norm->add(0, -1, 0); 
	norm->add(0, -1, 0); 
	norm->add(0, -1, 0);
	vert->add(-size, size, -size);
	vert->add( size, size, -size);
	vert->add( size, size,  size);
	vert->add(-size, size,  size);
	norm->add(0, 1, 0); 
	norm->add(0, 1, 0); 
	norm->add(0, 1, 0); 
	norm->add(0, 1, 0); 
	vert->add(-size, -size, -size);
	vert->add(-size,  size, -size);
	vert->add(-size,  size,  size);
	vert->add(-size, -size,  size);
	norm->add(-1, 0, 0); 
	norm->add(-1, 0, 0); 
	norm->add(-1, 0, 0); 
	norm->add(-1, 0, 0);
	vert->add(size, -size, -size);
	vert->add(size, -size,  size);
	vert->add(size,  size,  size);
	vert->add(size,  size, -size);
	norm->add(1, 0, 0); 
	norm->add(1, 0, 0); 
	norm->add(1, 0, 0); 
	norm->add(1, 0, 0); 
}

void make_cone(zePolygon *poly, double h, double r)
{
	poly->asQuadStrip();
	zeVertex *vert = new zeVertex;
	poly->setVertex(vert);
	zeVertex *norm = new zeVertex;
	poly->setVertexNormal(norm);
	double d;
	int n = circle_slices(r, d);
    for (int i = 0; i < n; i++) {
		double x = r*cos(i*d);
		double y = r*sin(i*d);
		ZExyz xyz0 = { 0, 0, h };
		ZExyz xyz1 = { x, y, 0 };
		vert->add(xyz0);
		vert->add(xyz1);
		ZExyz p1 = { -y,  x, 0 };
		ZExyz p2 = { -x, -y, h };
		ZExyz xyzn = cross_product2(p1, p2);
		normalize_vertex(xyzn);
		norm->add(xyzn);
		norm->add(xyzn);
	}
}

void make_cone_vector1(zePolygon *poly, double u, double v, double w, double r)
{
	make_cone(poly, sqrt(u*u+v*v+w*w), r);
	double lat, lon;
	xyz2ll(lon, lat, u, v, w);
	poly->reset();
	poly->rotateY(90-lat);
	poly->rotateZ(lon);
}

void make_cone_vector2(zePolygon *poly, double R, double lon, double lat, double u, double v, double r)
{
	make_cone(poly, sqrt(u*u+v*v), r);
	double x1, y1, z1, x2, y2, z2;
	ll2xyz(x1, y1, z1, lon, lat, R+r);
	dxdy2ll(lon, lat, u, v, R+r);
	ll2xyz(x2, y2, z2, lon, lat, R+r);
	xyz2ll(lon, lat, x2-x1, y2-y1, z2-z1);
	poly->reset();
	poly->rotateY(90-lat);
	poly->rotateZ(lon);
	ZExyz xyz = {x1, y1, z1};
	poly->translate(xyz);
}

void make_vector(zeLine *line, double u, double v, double size)
{
	line->asLines();
	zeVertex *vert = new zeVertex;
	line->setVertex(vert);
	line->asSolidLine(1, size>5.?size/5.0:1.0);
	// arrow
    ZExyz xyz0 = {0, 0, 0};
	vert->add(xyz0);
	ZExyz xyz1 = {u, v, 0};
	vert->add(xyz1);
	// arrow head
	double ax = size;
	double ay = size / 3.0;
	if (u > 0) ax = -ax;
	double r = sqrt(u*u + v*v)+1.e-12;
	double cosphi = u/r;
	double sinphi = v/r;
    xyz0.x = u + cosphi*ax - sinphi*ay;
    xyz0.y = v + sinphi*ax + cosphi*ay;
	vert->add(xyz0);
	vert->add(xyz1);
	ay = -ay;
    xyz0.x = u + cosphi*ax - sinphi*ay;
    xyz0.y = v + sinphi*ax + cosphi*ay;
	vert->add(xyz0);
	vert->add(xyz1);
}

void make_sphere(zePolygon *poly, double radius, int iterations)
{
	zeVertex *vert  = new zeVertex;
	zeVertex *norm  = new zeVertex;
	poly->setVertex(vert);
	poly->setVertexNormal(norm);
	poly->cullBack();
	poly->asTriangles();
	if (iterations < 0) iterations = 0;
	if (iterations > 8) iterations = 8;
	// Based on Paul Bourke's code
	int i, n;
	for (n = 4, i = 0; i < iterations; i++) n *= 4;
	ZEtriangle *faces = new ZEtriangle[n];
	ZExyz p1 = { 1,  1,  1},
		  p2 = {-1, -1,  1},
		  p3 = { 1, -1, -1},
		  p4 = {-1,  1, -1};
	faces[0].p1 = p1; faces[0].p2 = p2; faces[0].p3 = p3;
	faces[1].p1 = p2; faces[1].p2 = p1; faces[1].p3 = p4;
	faces[2].p1 = p2; faces[2].p2 = p4; faces[2].p3 = p3;
	faces[3].p1 = p1; faces[3].p2 = p3; faces[3].p3 = p4;
	sub_divide(4, iterations, faces, n);
	for (i = 0; i < n; i++) {
		normalize_vertex(faces[i].p1);
		normalize_vertex(faces[i].p2);
		normalize_vertex(faces[i].p3);
		norm->add(faces[i].p1);
		norm->add(faces[i].p2);
		norm->add(faces[i].p3);
		ZExyz v1 = { faces[i].p1.x * radius, faces[i].p1.y * radius, faces[i].p1.z * radius };
		vert->add(v1);
		ZExyz v2 = { faces[i].p2.x * radius, faces[i].p2.y * radius, faces[i].p2.z * radius };
		vert->add(v2);
		ZExyz v3 = { faces[i].p3.x * radius, faces[i].p3.y * radius, faces[i].p3.z * radius };
		vert->add(v3);
	}
	delete[] faces;
}

void sphere2_data(zeVertex *vert, zeVertex *norm, zeTexCoord *texc, double lon, double lat, double radius, double lon0, double grid)
{
	double x, y, z;

	ll2xyz(x, y, z, lon, lat, radius);
	ZExyz xyz = {x, y, z};
	vert->add(xyz);
	xyz.x /= radius;
	xyz.y /= radius;
	xyz.z /= radius;
	norm->add(xyz);
	double s = (lon+lon0)/360.0;
	double t = (lat+90)/180.0;
	texc->add(s, t);

	ll2xyz(x, y, z, lon+grid, lat, radius);
	xyz.x = x;
	xyz.y = y;
	xyz.z = z;
	vert->add(xyz);
	xyz.x /= radius;
	xyz.y /= radius;
	xyz.z /= radius;
	norm->add(xyz);
	s = (lon+lon0+grid)/360.0;
	texc->add(s, t);

	ll2xyz(x, y, z, lon+grid, lat+grid, radius);
	xyz.x = x;
	xyz.y = y;
	xyz.z = z;
	vert->add(xyz);
	xyz.x /= radius;
	xyz.y /= radius;
	xyz.z /= radius;
	norm->add(xyz);
	t = (lat+90+grid)/180.0;
	texc->add(s, t);

	ll2xyz(x, y, z, lon, lat+grid, radius);
	xyz.x = x;
	xyz.y = y;
	xyz.z = z;
	vert->add(xyz);
	xyz.x /= radius;
	xyz.y /= radius;
	xyz.z /= radius;
	norm->add(xyz);
	s = (lon+lon0)/360.0;
	texc->add(s, t);
}


void make_sphere2(zePolygon *poly, double radius, int grid, bool atlantic)
{
	zeVertex *vert  = new zeVertex;
	zeVertex *norm  = new zeVertex;
	poly->setVertex(vert);
	poly->setVertexNormal(norm);
	poly->cullBack();
	poly->asQuads();
	zeTexCoord *texc = new zeTexCoord;
	poly->setTexCoord(texc);
	for (double lat = -90; lat < 90; lat += grid) {
		if (atlantic) {
			for (double lon = -180; lon < 180; lon += grid) {
				sphere2_data(vert, norm, texc, lon, lat, radius, 180, grid);
			}
		}
		else {
			for (double lon = 0; lon < 360; lon += grid) {
				sphere2_data(vert, norm, texc, lon, lat, radius, 0, grid);
			}
		}
	}
}

void spherical_curve(zeLine *line, double r, double lon1, double lat1, double lon2, double lat2)
{
	// Refer to http://williams.best.vwh.net/avform.htm
    lat1 *= DEG_TO_RAD;
    lat2 *= DEG_TO_RAD;
    lon1 *= DEG_TO_RAD;
    lon2 *= DEG_TO_RAD;
	// distance for r=1;
	double d = acos(sin(lat1)*sin(lat2)+cos(lat1)*cos(lat2)*cos(lon1-lon2));
	// estimate number of grids
	int n = 1 + 90*d/PI;
	// a fraction of distance
	d /= n-1;

	zeVertex *vertex = new zeVertex;
	zeVertex *normal = new zeVertex;
	line->asLineStrip();
	line->setVertex(vertex);
	line->setVertexNormal(normal);

	for (int i = 0; i < n; i++) {
		double f = double(i)/double(n-1);
		double a = sin((1.-f)*d)/sin(d);
		double b = sin(f*d)/sin(d);
	    double x = a*cos(lat1)*cos(lon1) +  b*cos(lat2)*cos(lon2);
		double y = a*cos(lat1)*sin(lon1) +  b*cos(lat2)*sin(lon2);
		double z = a*sin(lat1) + b*sin(lat2);
		ZExyz xyz = {x, y, z};
		normalize_vertex(xyz);
		normal->add(xyz);
		xyz.x *= r;
		xyz.y *= r;
		xyz.z *= r;
		vertex->add(xyz);
	}
}

void latitude_grid(zeLine *line, double r, int lat)
{
	// distance for r=1;
	double a = cos(DEG_TO_RAD*lat);
	// estimate number of grids
	int n = 1 + a*180;
	// a fraction of distance
	double d = PI2/(n-1);

	zeVertex *vertex = new zeVertex;
	zeVertex *normal = new zeVertex;
	line->asLineLoop();
	line->setVertex(vertex);
	line->setVertexNormal(normal);

	double z = sin(DEG_TO_RAD*lat);

	for (int i = 0; i < n; i++) {
		double lon = i*d;
		double x = a*cos(lon);
		double y = a*sin(lon);
		ZExyz xyz = {x, y, z};
		normalize_vertex(xyz);
		normal->add(xyz);
		xyz.x *= r;
		xyz.y *= r;
		xyz.z *= r;
		vertex->add(xyz);
	}
}


void longitude_grid(zeLine *line, double r, int lon, int offset)
{
	int lat1 = -90+offset;
	int lat2 =  90-offset;
	int n = 1+ (lat2-lat1)/2;
	double d = (lat2-lat1)/(n-1);

	zeVertex *vertex = new zeVertex;
	zeVertex *normal = new zeVertex;
	line->asLineStrip();
	line->setVertex(vertex);
	line->setVertexNormal(normal);

	for (int i = 0; i < n; i++) {
		double lat = DEG_TO_RAD*(lat1+ double(i)*d);
		double z = sin(lat);
		double a = cos(lat);
		double x = a*cos(DEG_TO_RAD*lon);
		double y = a*sin(DEG_TO_RAD*lon);
		ZExyz xyz = {x, y, z};
		normalize_vertex(xyz);
		normal->add(xyz);
		xyz.x *= r;
		xyz.y *= r;
		xyz.z *= r;
		vertex->add(xyz);
	}
}

void make_circle(zeLine *line, double r)
{
	zeVertex *vertex = new zeVertex;
	zeVertex *normal = new zeVertex;
	line->setVertex(vertex);
	line->setVertexNormal(normal);
	line->asLineStrip();
	double d;
	int n = circle_slices(r, d);
	for (int i = 0; i < n; i++) {
		double x = cos(d*i);
		double y = sin(d*i);
		ZExyz norm = { x, y, 0 };
		ZExyz vert = { r*x, r*y, 0 };
		vertex->add(vert);
		normal->add(norm);
	}
}

void make_disk(zePolygon *poly, double r)
{
	zeVertex *vertex = new zeVertex;
	zeVertex *normal = new zeVertex;
	poly->setVertex(vertex);
	poly->setVertexNormal(normal);
	poly->asTriangleFan();
	ZExyz vert = { 0, 0, 0 };
	vertex->add(vert);
	ZExyz norm = { 0, 0, 1 };
	normal->add(norm);
	double d;
	int n = circle_slices(r, d);
	for (int i = 0; i < n; i++) {
		double x = r*cos(d*i);
		double y = r*sin(d*i);
		ZExyz vert = { x, y, 0 };
		vertex->add(vert);
		normal->add(norm);
	}
}

void make_cylinder(zePolygon *poly, double h,  double r)
{
	// in complete
	zeVertex *vertex = new zeVertex;
	zeVertex *normal = new zeVertex;
	poly->setVertex(vertex);
	poly->setVertexNormal(normal);
	poly->asQuadStrip();
	int n = PI2*r/5;
	if (n < 16) n = 16;
	double d = PI2/(n-1);
	for (int i = 0; i < n; i++) {
		double x = cos(i*d);
		double y = sin(i*d);
		ZExyz norm = { x, y, 0 };
		ZExyz vert1 = { r*x, r*y, h };
		ZExyz vert2 = { r*x, r*y, 0 };
		vertex->add(vert1);
		vertex->add(vert2);
		normal->add(norm);
		normal->add(norm);
	}
}

void make_torus(zePolygon *poly, double R, double r)
{
	zeVertex *vertex = new zeVertex;
	zeVertex *normal = new zeVertex;
	poly->setVertex(vertex);
	poly->setVertexNormal(normal);
	poly->asQuads();

	double d, D;
	int N = circle_slices(R, D);
	int n = circle_slices(r, d);

	for (int k = 1; k < N; k++) {
		for (int i = 1; i < n; i++) {
			double u1 = (k-1)*D;
			double u2 = k*D;
			double v1 = (i-1)*d;
			double v2 = i*d;

			double x11 = (R + r*cos(v1))*cos(u1);
			double x12 = (R + r*cos(v1))*cos(u2);
			double x21 = (R + r*cos(v2))*cos(u1);
			double x22 = (R + r*cos(v2))*cos(u2);

			double y11 = (R + r*cos(v1))*sin(u1);
			double y12 = (R + r*cos(v1))*sin(u2);
			double y21 = (R + r*cos(v2))*sin(u1);
			double y22 = (R + r*cos(v2))*sin(u2);

			double z1 = r*sin(v1);
			double z2 = r*sin(v2);

			ZExyz xyz11 = { x11, y11, z1 };
			vertex->add(xyz11);
			ZExyz xyz12 = { x12, y12, z1 };
			vertex->add(xyz12);
			ZExyz xyz22 = { x22, y22, z2 };
			vertex->add(xyz22);
			ZExyz xyz21 = { x21, y21, z2 };
			vertex->add(xyz21);

			double xu11 = -y11;
			double xu12 = -y12;
			double xu21 = -y21;
			double xu22 = -y22;

			double yu11 =  x11;
			double yu12 =  x12;
			double yu21 =  x21;
			double yu22 =  x22;

			double xv11 = -r*sin(v1)*cos(u1);
			double xv12 = -r*sin(v1)*cos(u2);
			double xv21 = -r*sin(v2)*cos(u1);
			double xv22 = -r*sin(v2)*cos(u2);

			double yv11 = -r*sin(v1)*sin(u1);
			double yv12 = -r*sin(v1)*sin(u2);
			double yv21 = -r*sin(v2)*sin(u1);
			double yv22 = -r*sin(v2)*sin(u2);

			double zv1 = r*cos(v1);
			double zv2 = r*cos(v2);

			ZExyz pu1 = { xu11, yu11, 0   };
			ZExyz pv1 = { xv11, yv11, zv1 };
			ZExyz nr1 = cross_product2(pu1, pv1);
			normalize_vertex(nr1);
			normal->add(nr1);

			ZExyz pu2 = { xu12, yu12, 0   };
			ZExyz pv2 = { xv12, yv12, zv1 };
			ZExyz nr2 = cross_product2(pu2, pv2);
			normalize_vertex(nr2);
			normal->add(nr2);

			ZExyz pu3 = { xu22, yu22, 0   };
			ZExyz pv3 = { xv22, yv22, zv2 };
			ZExyz nr3 = cross_product2(pu3, pv3);
			normalize_vertex(nr3);
			normal->add(nr3);

			ZExyz pu4 = { xu21, yu21, 0   };
			ZExyz pv4 = { xv21, yv21, zv2 };
			ZExyz nr4 = cross_product2(pu4, pv4);
			normalize_vertex(nr4);
			normal->add(nr4);
		}
	}
}

bool gshhs_coastline(zeLine *line, int level, const char *fname, double west, double east, double south, double north)
{
	line->asLines();
	zeVertex *vert = new zeVertex;
	line->setVertex(vert);

	FILE *fp = open_file(fname, "rb");
	if (!fp) return false;

	if (east <= west || west < -180 || east > 360 || (east-west) > 360 ||
		north <= south || south < -90 || north > 90 ) return false;

	GSHHS h;
	int i, n, greenwich, version, flip;
	
	fread(&h,sizeof(GSHHS),1,fp);
	version=(h.flag>>8)&255;
	flip=(version!=GSHHS_DATA_RELEASE);		// Take as sign that byte-swabbing is needed
	fseek(fp,0,SEEK_SET);

	while (!feof(fp)) {
		if (fread(&h,sizeof(GSHHS),1,fp)!=1) break;
		if (flip) {
			h.id=swabi4((unsigned int)h.id);
			h.n=swabi4((unsigned int)h.n);
			h.west=swabi4((unsigned int)h.west);
			h.east=swabi4((unsigned int)h.east);
			h.south=swabi4((unsigned int)h.south);
			h.north=swabi4((unsigned int)h.north);
			h.area=swabi4((unsigned int)h.area);
			h.area_full=swabi4((unsigned int)h.area_full);
			h.flag=swabi4((unsigned int)h.flag);
			h.container=swabi4((unsigned int)h.container);
			h.ancestor=swabi4((unsigned int)h.ancestor);
		}
		n=h.n;
		greenwich=(h.flag>>16)&1;			// Greenwich is 0 or 1
		if ((h.flag&255)!=1) {				// not land
			fseek(fp,sizeof(GSHHS_XY)*n,SEEK_CUR);
			continue;
		}

		double x1, y1, x2, y2;
		GSHHS_XY xy;

		for (i = 0; i < n; i++) {
			fread(&xy, sizeof(GSHHS_XY), 1, fp);
			if (i == 0) {
				x1 = 1.e-6*int(swabi4((unsigned int)xy.x));
				y1 = 1.e-6*int(swabi4((unsigned int)xy.y));
				if (west < 0 && x1 > 180.0) x1 -= 360.0;
			}
			else {
				x2 = 1.e-6*int(swabi4((unsigned int)xy.x));
				y2 = 1.e-6*int(swabi4((unsigned int)xy.y));
				if (west < 0 && x2 > 180.0) x2 -= 360.0;
				if (fabs(x2-x1) < 30 &&
					fabs(y2-y1) < 30 &&
					x1 >= west && x1 <= east &&
					x2 >= west && x2 <= east &&
					y1 >= south && y2 <= north &&
					y2 >= south && y2 <= north ) {
					vert->add(x1, y1, 0);
					vert->add(x2, y2, 0);
				}
				x1 = x2;
				y1 = y2;
			}
		}
	}

	close_file(fp);

	return true;
}

class gluTess
{
public:
	gluTess(zeVertex *vert) { xyz = vert; }
	~gluTess() { }
	int add(GLdouble x, GLdouble y)
	{
		tmp.push_back(x);
		tmp.push_back(y);
		tmp.push_back(0);
		return tmp.size()-3;
	}
	zeVertex *xyz;
	std::vector<GLdouble> tmp;
};

void glu_edge_data(GLboolean flag, gluTess *user)
{
	// Use this to enforcing traingles output
}

void glu_vertex_data(int idx, gluTess *user)
{
	// exception problem may be solved by adding printf(...) here
	user->xyz->add(user->tmp[idx], user->tmp[idx+1], 0);
}

void glu_combine_data(GLdouble coords[3], void *vertex[4], GLfloat weight[4], int *idx, gluTess *user)
{
	*idx = user->add(coords[0], coords[1]);
}

void vertex_triangulate(zeVertex *out, zeVertex *in, rect_t* rect, int winding)
{
	int i, idx, size=in->size();
	if (in->x(0) == in->x(size-1) && in->y(0) == in->y(size-1)) {
		size--;
	}
	if (size < 3) return;
	if (size == 3) {
		out->add(in->get(0));
		out->add(in->get(1));
		out->add(in->get(2));
		return;
	}
#ifdef _WIN32
	typedef void (__stdcall *GluTessCallbackType)();
#else
	typedef void (*GluTessCallbackType)();
#endif
	gluTess user(out);
	GLUtesselator *tess = gluNewTess();
	switch (winding) {
	case 1:
		gluTessProperty(tess, GLU_TESS_WINDING_RULE, GLU_TESS_WINDING_ODD);
		break;
	case 2:
		gluTessProperty(tess, GLU_TESS_WINDING_RULE, GLU_TESS_WINDING_NONZERO);
		break;
	case 3:
		gluTessProperty(tess, GLU_TESS_WINDING_RULE, GLU_TESS_WINDING_POSITIVE);
		break;
	case 4:
		gluTessProperty(tess, GLU_TESS_WINDING_RULE, GLU_TESS_WINDING_NEGATIVE);
		break;
	case 5:
		gluTessProperty(tess, GLU_TESS_WINDING_RULE, GLU_TESS_WINDING_ABS_GEQ_TWO);
		break;
	default:
		gluTessProperty(tess, GLU_TESS_WINDING_RULE, GLU_TESS_WINDING_ODD);
	}
	gluTessCallback(tess, GLU_TESS_EDGE_FLAG_DATA, reinterpret_cast<GluTessCallbackType>(glu_edge_data));
	gluTessCallback(tess, GLU_TESS_VERTEX_DATA, reinterpret_cast<GluTessCallbackType>(glu_vertex_data));
	gluTessCallback(tess, GLU_TESS_COMBINE_DATA, reinterpret_cast<GluTessCallbackType>(glu_combine_data));
	gluTessBeginPolygon(tess, &user);
	gluTessBeginContour(tess);
	for (i = 0; i < size; i++) {
		idx = user.add(in->x(i), in->y(i));
		gluTessVertex(tess, reinterpret_cast<GLdouble*>(&user.tmp[idx]), reinterpret_cast<void*>(idx));
	}
	gluTessEndContour(tess);
	if (rect) {
		gluTessBeginContour(tess);
		idx = user.add(rect->left, rect->bottom);
		gluTessVertex(tess, reinterpret_cast<GLdouble*>(&user.tmp[idx]), reinterpret_cast<void*>(idx));
		idx = user.add(rect->right, rect->bottom);
		gluTessVertex(tess, reinterpret_cast<GLdouble*>(&user.tmp[idx]), reinterpret_cast<void*>(idx));
		idx = user.add(rect->right, rect->top);
		gluTessVertex(tess, reinterpret_cast<GLdouble*>(&user.tmp[idx]), reinterpret_cast<void*>(idx));
		idx = user.add(rect->left, rect->top);
		gluTessVertex(tess, reinterpret_cast<GLdouble*>(&user.tmp[idx]), reinterpret_cast<void*>(idx));
		gluTessEndContour(tess);
	} 
	gluTessEndPolygon(tess);
	gluDeleteTess(tess);
}

void gshhs_triangulate(zeVertex *vert, rect_t *rect, std::vector<GLdouble> &ax, std::vector<GLdouble> &ay, int winding)
{
#ifdef _WIN32
	typedef void (__stdcall *GluTessCallbackType)();
#else
	typedef void (*GluTessCallbackType)();
#endif
	gluTess user(vert);
	GLUtesselator *tess = gluNewTess();
	switch (winding) {
	case 1:
		gluTessProperty(tess, GLU_TESS_WINDING_RULE, GLU_TESS_WINDING_ODD);
		break;
	case 2:
		gluTessProperty(tess, GLU_TESS_WINDING_RULE, GLU_TESS_WINDING_NONZERO);
		break;
	case 3:
		gluTessProperty(tess, GLU_TESS_WINDING_RULE, GLU_TESS_WINDING_POSITIVE);
		break;
	case 4:
		gluTessProperty(tess, GLU_TESS_WINDING_RULE, GLU_TESS_WINDING_NEGATIVE);
		break;
	case 5:
		gluTessProperty(tess, GLU_TESS_WINDING_RULE, GLU_TESS_WINDING_ABS_GEQ_TWO);
		break;
	default:
		gluTessProperty(tess, GLU_TESS_WINDING_RULE, GLU_TESS_WINDING_ODD);
	}
	gluTessCallback(tess, GLU_TESS_EDGE_FLAG_DATA, reinterpret_cast<GluTessCallbackType>(glu_edge_data));
	gluTessCallback(tess, GLU_TESS_VERTEX_DATA, reinterpret_cast<GluTessCallbackType>(glu_vertex_data));
	gluTessCallback(tess, GLU_TESS_COMBINE_DATA, reinterpret_cast<GluTessCallbackType>(glu_combine_data));
	gluTessBeginPolygon(tess, &user);
	gluTessBeginContour(tess);
	int i, idx;
	for (i = 0; i < ax.size()-1; i++) {
		idx = user.add(ax[i], ay[i]);
		gluTessVertex(tess, reinterpret_cast<GLdouble*>(&user.tmp[idx]), reinterpret_cast<void*>(idx));
	}
	gluTessEndContour(tess);
	if (rect) {
		gluTessBeginContour(tess);
		idx = user.add(rect->left, rect->bottom);
		gluTessVertex(tess, reinterpret_cast<GLdouble*>(&user.tmp[idx]), reinterpret_cast<void*>(idx));
		idx = user.add(rect->right, rect->bottom);
		gluTessVertex(tess, reinterpret_cast<GLdouble*>(&user.tmp[idx]), reinterpret_cast<void*>(idx));
		idx = user.add(rect->right, rect->top);
		gluTessVertex(tess, reinterpret_cast<GLdouble*>(&user.tmp[idx]), reinterpret_cast<void*>(idx));
		idx = user.add(rect->left, rect->top);
		gluTessVertex(tess, reinterpret_cast<GLdouble*>(&user.tmp[idx]), reinterpret_cast<void*>(idx));
		gluTessEndContour(tess);
	}
	gluTessEndPolygon(tess);
	gluDeleteTess(tess);
}

bool gshhs_filled(zePolygon *poly, int level, const char *fname, double west, double east, double south, double north, int winding)
{
	poly->asTriangles();
	zeVertex *vert = new zeVertex;
	poly->setVertex(vert);

	FILE *fp = open_file(fname, "rb");
	if (!fp) return false;

	if (east <= west || west < -180 || east > 180 || (east-west) > 360 ||
		north <= south || south < -90 || north > 90 ) return false;

	GSHHS h;
	GSHHS_XY xy;
	int i, n, greenwich, version, flip;

	fread(&h,sizeof(GSHHS),1,fp);
	version=(h.flag>>8)&255;
	flip=(version!=GSHHS_DATA_RELEASE);		// Take as sign that byte-swabbing is needed
	fseek(fp,0,SEEK_SET);

	rect_t rect = { west, south, east, north };
	rect_t rec2 = { west+360, south, east+360, north };
	rect_t rec3 = { west-360, south, east-360, north };

	while (!feof(fp)) {
		if (fread(&h,sizeof(GSHHS),1,fp)!=1) break;
		if (flip) {
			h.id=swabi4((unsigned int)h.id);
			h.n=swabi4((unsigned int)h.n);
			h.west=swabi4((unsigned int)h.west);
			h.east=swabi4((unsigned int)h.east);
			h.south=swabi4((unsigned int)h.south);
			h.north=swabi4((unsigned int)h.north);
			h.area=swabi4((unsigned int)h.area);
			h.area_full=swabi4((unsigned int)h.area_full);
			h.flag=swabi4((unsigned int)h.flag);
			h.container=swabi4((unsigned int)h.container);
			h.ancestor=swabi4((unsigned int)h.ancestor);
		}
		n=h.n;
		greenwich=(h.flag>>16)&1;			// Greenwich is 0 or 1
		if ((h.flag&255)!=1) {				// not land
			fseek(fp,sizeof(GSHHS_XY)*n,SEEK_CUR);
			continue;
		}

		std::vector<GLdouble> ax, ay;
		GLdouble xmin=1.e30, xmax=-1.e30, ymin=1.e30, ymax=-1.e30;

		for (i = 0; i < n; i++) {
			fread(&xy, sizeof(GSHHS_XY), 1, fp);
			GLdouble a = 1.e-6*int(swabi4((unsigned int)xy.x));
			GLdouble b = 1.e-6*int(swabi4((unsigned int)xy.y));
			ax.push_back(a);
			ay.push_back(b);
			xmin = min(xmin, a);
			xmax = max(xmax, a);
			ymin = min(ymin, b);
			ymax = max(ymax, b);
		}

		if (xmin <= east && xmax >= west && ymin <= north && ymax >= south) {
			if (greenwich != 0) {
				std::vector<GLdouble> bx, cx;
				for (i = 0; i < n; i++) {
					if (ax[i] < 0) {
						bx.push_back(ax[i]+360);
						cx.push_back(ax[i]);
					}
					else {
						bx.push_back(ax[i]);
						cx.push_back(ax[i]-360);
					}
				}
				if (ymin < -60.0 && fabs(xmax-xmin) > 180.) {
						// antarctic
						bx.pop_back();
						cx.pop_back();
						ay.pop_back();
						bx.push_back(bx.back()-1.0);
						cx.push_back(cx.back()-1.0);
						ay.push_back(-91.0);
						bx.push_back(bx[0]+1.0);
						cx.push_back(cx[0]+1.0);
						ay.push_back(-92.0);
						bx.push_back(bx[0]);
						cx.push_back(cx[0]);
						ay.push_back(ay[0]);
				}
				zeVertex xyz;
				gshhs_triangulate(&xyz, &rec2, bx, ay, winding);
				for (i = 0; i < xyz.size(); i++) {
					ZEvertex v = xyz.get(i);
					v.x -= 360;
					vert->add(v);
				}
				xyz.clear();
				gshhs_triangulate(&xyz, &rec3, cx, ay, winding);
				for (i = 0; i < xyz.size(); i++) {
					ZEvertex v = xyz.get(i);
					v.x += 360;
					vert->add(v);
				}
			}
			else {
				gshhs_triangulate(vert, &rect, ax, ay, winding);
			}
		}
	}

	close_file(fp);

	return true;
}

bool connected_points(const zeVertex *vert, std::vector<int> &flag, std::list<ZExyz> &out)
{
	ZExyz xyz1, xyz2;
	int i, j;

	out.clear();
	
	// find the first line segment

	for (i = 1; i < vert->size(); i += 2) {
		if (flag[i-1] != 0) {
			xyz1 = vert->get(i-1);
			xyz2 = vert->get(i);
			flag[i-1] = 0;
			flag[i] = 0;
			out.push_back(xyz1);
			out.push_back(xyz2);
			i += 2;
			break;
		}
	}

	if (out.empty() || i >= vert->size()) return false;

	for (j = i; j < vert->size(); j += 2) {
		if (flag[j-1] == 0) continue;
		if (vert->x(j-1) == xyz2.x && vert->y(j-1) == xyz2.y) {
			xyz2 = vert->get(j);
			out.push_back(xyz2);
			flag[j-1] = 0;
			flag[j] = 0;
		}
		else if (vert->x(j) == xyz2.x && vert->y(j) == xyz2.y) {
			xyz2 = vert->get(j-1);
			out.push_back(xyz2);
			flag[j-1] = 0;
			flag[j] = 0;
		}
		else if (vert->x(j-1) == xyz1.x && vert->y(j-1) == xyz1.y) {
			xyz1 = vert->get(j);
			out.push_front(xyz1);
			flag[j-1] = 0;
			flag[j] = 0;
		}
		else if (vert->x(j) == xyz1.x && vert->y(j) == xyz1.y) {
			xyz1 = vert->get(j-1);
			out.push_front(xyz1);
			flag[j-1] = 0;
			flag[j] = 0;
		}
	}

	return true;
}

void contour_labels(zePlot *plot, zeLine *line, zeText *label,
					int density, int width, int height, double offset)
{
	if (density < 1 || line->getVertex()->size() < density) return;
	std::vector<int> flag(line->getVertex()->size(), 1);
	std::list<ZExyz> out;
	int density2 = 3*density/2;
	while(connected_points(line->getVertex(),flag,out)) {
		if (out.size()<density2) continue;
		int count = out.size() - density2;
		std::list<ZExyz>::iterator iter = out.begin();
		while (count > 0) {
			for (int k = 0; k < density; k++) {
				iter++;
				count--;
			}
			ZExyz xyz1 = *iter;
			iter++;
			count--;
			ZExyz xyz2 = *iter;
			// text layout
			ZExyz pos1 = plot->plot2global(width, height, xyz1.x, xyz1.y, 0);
			ZExyz pos2 = plot->plot2global(width, height, xyz2.x, xyz2.y, 0);
			double lat, lon;
			xyz2ll(lon, lat, pos2.x-pos1.x, pos2.y-pos1.y, 0);
			zeNode *node = new zeNode;
			ZExyz xyz = { 0.5*(xyz1.x+xyz2.x), 0.5*(xyz1.y+xyz2.y), offset };
			plot->anchor(node, xyz);
			if (lon > 90)
				node->rotateZ(lon-180);
			else if (lon < -90)
				node->rotateZ(lon+180);
			else
				node->rotateZ(lon);
			node->add(label);
		}
	}
}
