#include "colorbar.h"
#include "utility.h"

using namespace std;

zeColorBar::zeColorBar()
 : _font(0), _digit(1), _scientific(false), _gradient(true), _width(100), _height(10)
{
	setType(ZE_COLOR_BAR);
	_color.a = -1;
	_lines.asSolidLine(1, 1);
	_lines.asLines();
	_lines.setParent(this);
	_polys.setParent(this);
	_polys.asQuadStrip();
	_fontSize = DEFAULT_FONT_SIZE;
}

zeColorBar::~zeColorBar()
{
	clearLabels();
}

void
zeColorBar::drawFunc(ZEoption & option)
{
	if (!enabled() || _data.size() < 2 || !_font) return;

	ZEcolor color = option.color;
	if (_color.a > 0) option.color = _color;
	render_colors(option);
	if (_color.a > 0) option.color = _color;
	render_labels(option);
	option.color = color;
}

void
zeColorBar::clearLabels()
{
	for (int i = 0; i < _labels.size(); i++) {
		delete _labels[i];
        _labels[i] = 0;
	}
	_labels.clear();
	_lines.setVertex(0);
	_polys.setVertex(0);
}

void
zeColorBar::clear()
{
	_data.clear();
	_barColor.clear();
	clearLabels();
}

void
zeColorBar::addColor(const ZEcolor& color, GLfloat data)
{
	_data.push_back(data);
	_barColor.push_back(color);
	clearLabels();
}

void
zeColorBar::gradientType(bool yn)
{
	_gradient = yn;
	if (yn)
		_polys.asQuadStrip();
	else
		_polys.asQuads();
	clearLabels();
}

void
zeColorBar::setLabelFormat(size_t digit, bool scientific)
{
	_digit = digit;
	_scientific = scientific;
	clearLabels();
}

void
zeColorBar::interpolate(size_t n)
{
	if (n < 1) return;

	clearLabels();

	vector<ZEcolor> color;
	vector<GLfloat> data;

	color.push_back(_barColor[0]);
	data.push_back(_data[0]);

	for (size_t k = 1; k < _data.size(); k++)
	{
		ZEcolor c1 = _barColor[k-1];
		ZEcolor c2 = _barColor[k];
		GLfloat d1   = _data[k-1];
		GLfloat d2   = _data[k];

		for (size_t j = 0; j < n; j++)
		{
			ZEcolor c;
			GLfloat a = GLfloat (n - j) / (1.0 + n);
			GLfloat b = GLfloat (j + 1) / (1.0 + n);

			c.r = a * c1.r + b * c2.r;
			c.g = a * c1.g + b * c2.g;
			c.b = a * c1.b + b * c2.b;
			c.a = a * c1.a + b * c2.a;

			GLfloat d = a * d1 + b * d2;

			color.push_back(c);
			data.push_back(d);
		}

		color.push_back(c2);
		data.push_back(d2);
	}

	_barColor = color;
	_data = data;
}

ZEcolor
zeColorBar::getColor(GLfloat data) const
{
	ZEcolor c = { 0, 0, 0, -1 };
	
	size_t k, n = _data.size();
	if (n < 2) return c;
	
	for (k = 0; k < n; k++) { if (data < _data[k]) break; }

	if (k == 0) return _barColor[0];
	if (k >= n) return _barColor[n-1];
	
	if (_gradient) {
		double a = (_data[k] - data) / (_data[k] - _data[k-1]);
		double b = (data - _data[k-1]) / (_data[k] - _data[k-1]);
		c.r = a * _barColor[k-1].r + b * _barColor[k].r;
		c.g = a * _barColor[k-1].g + b * _barColor[k].g;
		c.b = a * _barColor[k-1].b + b * _barColor[k].b;
		c.a = a * _barColor[k-1].a + b * _barColor[k].a;
		return c;
	}

	return _barColor[k-1];
}

void
zeColorBar::render_colors(ZEoption & option)
{
	if (_data.size() < 2) return;

	if (!_polys.getVertex()) {
		GLfloat xstart=-0.5*_width, ystart=-0.5*_height, step;

		zeVertex *vert = new zeVertex;
		_polys.setVertex(vert);
		zeColor *rgba = new zeColor;
		_polys.setVertexColor(rgba);

		if (_width > _height) {
				// horizontal
			step = _width/(_barColor.size()-1);
		}
		else {
			// vertical
			step = _height/(_barColor.size()-1);
		}

		if (_gradient) {
			_polys.asQuadStrip();
			if (_width > _height) {
				for (int i = 0; i < _barColor.size(); i++) {
					GLfloat v = xstart + i*step;
					vert->add(v, ystart, 0);
					vert->add(v, ystart+_height, 0);
					ZEcolor color = _barColor[i];
					rgba->add(color);
					rgba->add(color);
				}
			}
			else {
				for (int i = 0; i < _barColor.size(); i++) {
					GLfloat v = ystart + i*step;
					vert->add(xstart+_width, v, 0);
					vert->add(xstart, v, 0);
					ZEcolor color = _barColor[i];
					rgba->add(color);
					rgba->add(color);
				}
			}
		}
		else {
			_polys.asQuads();
			if (_width > _height) {
				for (int i = 1; i < _barColor.size(); i++) {
					GLfloat v = xstart + i*step;
					vert->add(v-step, ystart, 0);
					vert->add(v, ystart, 0);
					vert->add(v, ystart+_height, 0);
					vert->add(v-step, ystart+_height, 0);
					ZEcolor color = _barColor[i-1];
					rgba->add(color);
					rgba->add(color);
					rgba->add(color);
					rgba->add(color);
				}
			}
			else {
				for (int i = 1; i < _barColor.size(); i++) {
					GLfloat v = ystart + i*step;
					vert->add(xstart, v-step, 0);
					vert->add(xstart+_width, v-step, 0);
					vert->add(xstart+_width, v, 0);
					vert->add(xstart, v, 0);
					ZEcolor color = _barColor[i-1];
					rgba->add(color);
					rgba->add(color);
					rgba->add(color);
					rgba->add(color);
				}
			}
		}

		vert = new zeVertex;
		_lines.setVertex(vert);

		if (_width > _height) {
			for (int i = 0; i < _barColor.size(); i++) {
				GLfloat v = xstart + i*step;
				vert->add(v, ystart, 1);
				vert->add(v, ystart+_height, 1);
			}
		}
		else {
			for (int i = 0; i < _barColor.size(); i++) {
				GLfloat v = ystart + i*step;
				vert->add(xstart, v, 1);
				vert->add(xstart+_width, v, 1);
			}
		}
	}

	_polys.render(option);
	_lines.render(option);
}

void
zeColorBar::render_labels(ZEoption & option)
{
	if (!_font) return;

	if (_labels.empty()) {
		GLfloat xstart=-.5*_width, ystart=-.5*_height, step;
		
		char fmt[32];
		char buf[32];

		if (_scientific)
			sprintf(fmt, "%%0.%de", _digit);
		else
			sprintf(fmt, "%%0.%df", _digit);

		if (_width > _height) {
			step = _width/(_data.size()-1);
		}
		else {
			step = _height/(_data.size()-1);
		}

		for (int i = 0; i < _data.size(); i++) {
			sprintf(buf, fmt, _data[i]);
			zeText *text = new zeText;
			text->setFont(_font);
			text->setFontSize(_fontSize);
			text->setText(buf);
			if (_width > _height) {
				ZExyz xyz = { xstart+i*step, ystart-0.75*_fontSize-text->top(), 0 };
				text->translate(xyz);
			}
			else {
				ZExyz xyz = { xstart+_width+0.5*_fontSize-text->left(), ystart+i*step, 0 };
				text->translate(xyz);
			}
			_labels.push_back(text);
		}
	}

	for (int i = 0; i < _labels.size(); i++) {
		_labels[i]->render(option);
	}
}
