#include "axis.h"
#include "plot.h"
#include "line.h"
#include "utility.h"

#include <math.h>

#define TICK_LENGTH	6.0

using namespace std;

zeAxis::zeAxis()
:	_font(0),
	_axisMin(-1), _axisMax(1),
	_tickStart(-1), _tickEnd(1), _tickStep(.2), _tickMinor(0),
	_tickLabelAlignCenter(false),
	_tickLabelDigit(1), _tickLabelScientific(false),
	_tickLengthScale(1),
	_tickEnabled(true), _tickLabelEnabled(true)
{
	setType(ZE_AXIS);
	_color.a = -1;
	_tickLabelTop = _tickLabelRight = -1.e9;
	_tickLabelBottom = _tickLabelLeft = 1.e9;
	asXAxis();
	_axisLines.asLines();
	_labelScale.x = _labelScale.y = _labelScale.z = 1.0;

	_axisLabel.setParent(this);
	_axisLines.setParent(this);
	_axisLines.asSolidLine(1, 1.5);

	_fontSize = DEFAULT_FONT_SIZE;
}

zeAxis::~zeAxis()
{
	clear_tickLabels();
}

void
zeAxis::drawFunc(ZEoption & option)
{
	if (!enabled()) return;
	
	ZEcolor color = option.color;
	if (_color.a > 0) option.color = _color;
	render_ticks(option);
	if (_color.a > 0) option.color = _color;
	render_ticklabels(option);
	if (_color.a > 0) option.color = _color;
	render_axislabel(option);
	option.color = color;
}

bool
zeAxis::setAxisLabel(const char *label)
{
	if (!_font) return false;
	_axisLabel.setFont(_font);
	_axisLabel.setFontSize(1.25*_fontSize);
	_axisLabel.setText(label);
	return true;
}

bool
zeAxis::setAxisRange(double amin, double amax, bool setplot)
{
	if (amax <= amin) return false;
	_axisMin = amin;
	_axisMax = amax;
	_tickStart = _axisMin;
	_tickEnd = _axisMax;
	_tickStep = (_axisMax-_axisMin)/5.0;
	if (setplot && getParent() && getParent()->getType() == ZE_PLOT) {
		zePlot *plot = (zePlot*)getParent();
		if (_axisType == X_AXIS) {
			plot->setXrange(amin, amax);
		}
		else if (_axisType == Y_AXIS) {
			plot->setYrange(amin, amax);
		}
		else {
			plot->setZrange(amin, amax);
		}
	}
	_axisLines.setVertex(0);
	return true;
}

void
zeAxis::clear_tickLabels()
{
	for (int i = 0; i < _tickLabels.size(); i++) {
		delete _tickLabels[i];
        _tickLabels[i] = 0;
	}
	_tickLabels.clear();
	_tickLabelTop = _tickLabelRight = -1.e9;
	_tickLabelBottom = _tickLabelLeft = 1.e9;
}

bool
zeAxis::addTickLabel(const char *label, bool center)
{
	if (!_font) return false;
	_tickLabelAlignCenter = center;
	zeText *text = new zeText;
	text->setFont(_font);
	_tickLabels.push_back(text);
	if (label && strlen(label) > 0) {
		text->setFontSize(_fontSize);
		text->setText(label);
		_tickLabelTop = max(_tickLabelTop, text->top());
		_tickLabelBottom = min(_tickLabelBottom, text->bottom());
		_tickLabelRight = max(_tickLabelRight, text->right());
		_tickLabelLeft = min(_tickLabelLeft, text->left());
	}
	return true;
}

void
zeAxis::asXAxis(bool tickDown)
{
	_axisType = X_AXIS;
	if (tickDown)
		_tickDirection = -1;
	else
		_tickDirection = 1;
	_axisLines.setVertex(0);
}

void
zeAxis::asYAxis(bool tickLeft)
{
	_axisType = Y_AXIS;
	if (tickLeft)
		_tickDirection = -1;
	else
		_tickDirection = 1;
	_axisLines.setVertex(0);
}

void
zeAxis::asZAxis(bool tickLeft)
{
	_axisType = Z_AXIS;
	if (tickLeft)
		_tickDirection = -1;
	else
		_tickDirection = 1;
	_axisLines.setVertex(0);
}

bool
zeAxis::setTickMarks(double start, double end, double step, size_t minor)
{
	if (step <= 0 || minor < 0) return false;
	clear_tickLabels();
	_tickStart = start;
	_tickEnd = end;
	_tickStep = step;
	_tickMinor = minor;
	_axisLines.setVertex(0);
	return true;
}

void
zeAxis::setTickLength(double lengthScale)
{
	_tickLengthScale = lengthScale;
	_axisLines.setVertex(0);
}

void
zeAxis::setTickLabelDigit(size_t digit, bool scientific)
{
	_tickLabelDigit = digit;
	_tickLabelScientific = scientific;
	_axisLines.setVertex(0);
}

void
zeAxis::render_ticks(ZEoption & option)
{
	if (!_tickEnabled) return;

	if (!_axisLines.getVertex()) {
		zeVertex *v = new zeVertex;
		_axisLines.setVertex(v);

		double tick, ticklen = TICK_LENGTH*_tickLengthScale*_tickDirection;
		double minor = _tickStep/(_tickMinor+1);
		double eps = 0.001*_tickStep;
		double scale;

		switch (_axisType) {
		case X_AXIS:
			v->add(option.xmin, 0, 0);
			v->add(option.xmax, 0, 0);
			scale = (option.xmax-option.xmin)/(_axisMax-_axisMin);
			for (tick = _tickStart; tick <= _tickEnd; tick += _tickStep) {
				GLfloat x = option.xmin + scale*(tick-_axisMin);
				if (x <= option.xmax) {
					v->add(x, 0, 0);
					v->add(x, ticklen*_labelScale.y, 0);
					if (tick < _tickEnd) {
						for (size_t k = 1; k <= _tickMinor; k++) {
							x = option.xmin + scale*(tick-_axisMin+k*minor);
							if (x <= option.xmax) {
								v->add(x, 0, 0);
								v->add(x, 0.5*ticklen*_labelScale.y, 0);
							}
						}
					}
				}
			}
			break;
		case Y_AXIS:
			v->add(0, option.ymin, 0);
			v->add(0, option.ymax, 0);
			scale = (option.ymax-option.ymin)/(_axisMax-_axisMin);
			for (tick = _tickStart; tick <= _tickEnd; tick += _tickStep) {
				GLfloat y = option.ymin + scale*(tick-_axisMin);
				if (y <= option.ymax) {
					v->add(0, y, 0);
					v->add(ticklen*_labelScale.x, y, 0);
					if (tick < _tickEnd) {
						for (size_t k = 1; k <= _tickMinor; k++) {
							y = option.ymin + scale*(tick-_axisMin+k*minor);
							if (y <= option.ymax) {
								v->add(0, y, 0);
								v->add(0.5*ticklen*_labelScale.x, y, 0);
							}
						}
					}
				}
			}
			break;
		case Z_AXIS:
			v->add(0, 0, option.zmin);
			v->add(0, 0, option.zmax);
			scale = (option.zmax-option.zmin)/(_axisMax-_axisMin);
			for (tick = _tickStart; tick <= _tickEnd; tick += _tickStep) {
				GLfloat z = option.zmin + scale*(tick-_axisMin);
				if (z < option.zmax) {
					v->add(0, 0, z);
					v->add(ticklen*_labelScale.x, 0, z);
					if (tick < _tickEnd) {
						for (size_t k = 1; k <= _tickMinor; k++) {
							z = option.zmin + scale*(tick-_axisMin+k*minor);
							if (z < option.zmax) {
								v->add(0, 0, z);
								v->add(0.5*ticklen*_labelScale.x, 0, z);
							}
						}
					}
				}
			}
			break;
		}
	}

	_axisLines.render(option);
}

void
zeAxis::render_ticklabels(ZEoption & option)
{
	if (!_tickLabelEnabled || !_font) return;

	if (_tickLabels.size() == 0) {
		char fmt[32];
		char buf[32];
		double eps = 0.01*_tickStep;
		if (_tickLabelScientific)
			sprintf(fmt, "%%0.%de", _tickLabelDigit);
		else
			sprintf(fmt, "%%0.%df", _tickLabelDigit);
		for (GLfloat tick = _tickStart; tick <= _tickEnd+eps; tick += _tickStep) {
			sprintf(buf, fmt, tick);
			zeText *text = new zeText;
			text->setFont(_font);
			text->setFontSize(_fontSize);
			_tickLabels.push_back(text);
			text->setText(buf);
			_tickLabelTop = max(_tickLabelTop, text->top());
			_tickLabelBottom = min(_tickLabelBottom, text->bottom());
			_tickLabelRight = max(_tickLabelRight, text->right());
			_tickLabelLeft = min(_tickLabelLeft, text->left());
		}
	}

	double scale, offset = TICK_LENGTH*_tickLengthScale + 0.5*_fontSize;
	int k;

	switch (_axisType) {
	case X_AXIS:
		scale = (option.xmax-option.xmin)/(_axisMax-_axisMin);
		for (k = 0; k < _tickLabels.size(); k++) {
			_tickLabels[k]->reset();
			_tickLabels[k]->scale(_labelScale);
			GLfloat tick = _tickStart + k*_tickStep;
			if (_tickLabelAlignCenter) tick += 0.5*_tickStep;
			GLfloat x = option.xmin + scale*(tick-_axisMin);
			if (_tickDirection > 0) {
				ZExyz xyz = { x, (offset-_tickLabelBottom)*_labelScale.y, 0 };
				_tickLabels[k]->translate(xyz);
			}
			else {
				ZExyz xyz = { x, (-offset-_tickLabelTop)*_labelScale.y, 0 };
				_tickLabels[k]->translate(xyz);
			}
			_tickLabels[k]->render(option);
		}
		break;
	case Y_AXIS:
		scale = (option.ymax-option.ymin)/(_axisMax-_axisMin);
		for (k = 0; k < _tickLabels.size(); k++) {
			_tickLabels[k]->reset();
			_tickLabels[k]->scale(_labelScale);
			GLfloat tick = _tickStart + k*_tickStep;
			if (_tickLabelAlignCenter) tick += 0.5*_tickStep;
			GLfloat y = option.ymin + scale*(tick-_axisMin);
			if (_tickDirection > 0) {
				ZExyz xyz = { (offset-_tickLabels[k]->left())*_labelScale.x, y, 0 };
				_tickLabels[k]->translate(xyz);
			}
			else {
				ZExyz xyz = { (-offset-_tickLabels[k]->right())*_labelScale.x, y, 0 };
				_tickLabels[k]->translate(xyz);
			}
			_tickLabels[k]->render(option);
		}
		break;
	case Z_AXIS:
		scale = (option.zmax-option.zmin)/(_axisMax-_axisMin);
		for (k = 0; k < _tickLabels.size(); k++) {
			_tickLabels[k]->reset();
			_tickLabels[k]->rotateX(90);
			_tickLabels[k]->scale(_labelScale);
			GLfloat tick = _tickStart + k*_tickStep;
			if (_tickLabelAlignCenter) tick += 0.5*_tickStep;
			GLfloat z = option.zmin + scale*(tick-_axisMin);
			if (_tickDirection > 0) {
				ZExyz xyz = { (offset-_tickLabels[k]->left())*_labelScale.x, 0, z };
				_tickLabels[k]->translate(xyz);
			}
			else {
				ZExyz xyz = { (-offset-_tickLabels[k]->right())*_labelScale.x, 0, z };
				_tickLabels[k]->translate(xyz);
			}
			_tickLabels[k]->render(option);
		}
		break;
	}
}

void
zeAxis::render_axislabel(ZEoption & option)
{
	double offset = TICK_LENGTH*_tickLengthScale + _fontSize;

	_axisLabel.reset();

	switch (_axisType) {
	case X_AXIS:
		_axisLabel.scale(_labelScale);
		if (_tickDirection > 0) {
			ZExyz xyz = { 0, (offset-_tickLabelBottom+_tickLabelTop-_tickLabelBottom-_axisLabel.bottom())*_labelScale.y, 0 };
			_axisLabel.translate(xyz);
		}
		else {
			ZExyz xyz = { 0, -(offset+_tickLabelTop+_tickLabelTop-_tickLabelBottom+_axisLabel.top())*_labelScale.y, 0 };
			_axisLabel.translate(xyz);
		}
		break;
	case Y_AXIS:
		_axisLabel.rotateZ(90);
		_axisLabel.scale(_labelScale);
		offset += .5*_fontSize;
		if (_tickDirection > 0) {
			ZExyz xyz = { (offset+_tickLabelRight-_tickLabelLeft+_axisLabel.top())*_labelScale.x, 0, 0 };
			_axisLabel.translate(xyz);
		}
		else {
			ZExyz xyz = { -(offset+_tickLabelRight-_tickLabelLeft-_axisLabel.bottom())*_labelScale.x, 0, 0 };
			_axisLabel.translate(xyz);
		}
		break;
	case Z_AXIS:
		_axisLabel.rotateZ(90);
		_axisLabel.rotateX(90);
		_axisLabel.scale(_labelScale);
		offset += .5*_fontSize;
		if (_tickDirection > 0) {
			ZExyz xyz = { (offset+_tickLabelRight-_tickLabelLeft+_axisLabel.top())*_labelScale.x, 0, 0 };
			_axisLabel.translate(xyz);
		}
		else {
			ZExyz xyz = { -(offset+_tickLabelRight-_tickLabelLeft-_axisLabel.bottom())*_labelScale.x, 0, 0 };
			_axisLabel.translate(xyz);
		}
		break;
	}
	_axisLabel.render(option);
}
