#include "netcdf.h"
#include "api.h"
#include "buffer.h"
#include <stdlib.h>
#include <math.h>
#include <stdio.h>
#include <string.h>

#define NC_TYPE			'NC'

#define GLOBAL_ATTNAME		"General"

#pragma warning(disable: 4018)
#pragma warning(disable: 4244)

/////////////////////////////////////////////////////////////////////

static char g_tname[7][10] = { "char", "byte", "short", "int", "float", "double", "unknown" };

static const char *type2name(nc_type type)
{
	switch(type) {
	case NC_CHAR:
		return g_tname[0];
	case NC_BYTE:
		return g_tname[1];
	case NC_SHORT:
		return g_tname[2];
	case NC_INT:
		return g_tname[3];
	case NC_FLOAT:
		return g_tname[4];
	case NC_DOUBLE:
		return g_tname[5];
	default:
		return g_tname[6];
	}
	return g_tname[6];
}

/////////////////////////////////////////////////////////////////////

class zsNetcdf
{
	size_t size;
public:
	zsNetcdf() : ncid(0), varid(0), ndim(0), vsize(0), esize(0), size(0), ptr(0), unlimited(0), undimid(-1)
	{
		for (int i = 0; i < NC_MAX_VAR_DIMS; i++) {
			dimid[i] = 0;
			dims[i] = 0;
		}
	}

	~zsNetcdf()
	{
		if (ncid) nc_close(ncid);
		if (ptr) free(ptr);
	}

	void checksize(size_t n)
	{
		if (size < n || size > 2*n) {
			if (ptr) free(ptr);
			ptr = malloc(n);
			size = n;
		}
	}

	int ncid, varid, ndim, unlimited, undimid, dimid[NC_MAX_VAR_DIMS];
	size_t vsize, esize, dims[NC_MAX_VAR_DIMS];
	nc_type type;
	void *ptr;
};

/////////////////////////////////////////////////////////////////////

static void type_error(void* ctx) { api_runtime_error(ctx, "unexpected data type");  }

static void cdf_check_error(void* ctx, int status) { if (status != NC_NOERR) api_runtime_error(ctx, nc_strerror(status)); }

static void check_variable(void* ctx, zsNetcdf *o) { if (o->ndim == 0) api_runtime_error(ctx, "variable not specified"); }

static void cdf_destroy(void* ptr) { delete (zsNetcdf*)ptr; ptr = 0; }

/////////////////////////////////////////////////////////////////////

static void* cdf_open(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	const char *fname = api_get_string(ctx, args[0]);
	const char *mode = "r";
	zsNetcdf *o = new zsNetcdf;
	if (nargs > 1) mode = api_get_string(ctx, args[1]);
	if (mode[0] == 'c' || mode[0] == 'C') {
		int status = nc_create(fname, NC_NOCLOBBER, &o->ncid);
		if (status != NC_NOERR) {
			delete o;
            o = 0;
			return 0;
		}
	}
	else if (mode[0] == 'w' || mode[0] == 'W') {
		int status = nc_open(fname, NC_WRITE, &o->ncid);
		if (status != NC_NOERR) {
			status = nc_create(fname, NC_NOCLOBBER, &o->ncid);
			if (status != NC_NOERR) {
				delete o;
			    o = 0;
				return 0;
			}
		}
	}
	else {
		int status = nc_open(fname, NC_NOWRITE, &o->ncid);
		if (status != NC_NOERR) {
			delete o;
            o = 0;
			return 0;
		}
	}
	return api_create_user(ctx, o, 0, cdf_destroy, NC_TYPE);
}

static void* cdf_close(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zsNetcdf *o = (zsNetcdf*)api_get_user(ctx, args[0], NC_TYPE);
	nc_close(o->ncid);
	o->ncid=0;
	return 0;
}

static void* cdf_version(void *ctx, int nargs, void** args)
{
	return api_create_string(ctx, (char*)nc_inq_libvers());
}

static void* cdf_defdim(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zsNetcdf *o = (zsNetcdf*)api_get_user(ctx, args[0], NC_TYPE);
	const char *name = api_get_string (ctx, args[1]);
	int n = api_get_integer(ctx, args[2]);

	nc_redef(o->ncid);
	if (n > 0) {
		cdf_check_error(ctx, nc_def_dim(o->ncid, name, n, &o->varid));
	}
	else {
		cdf_check_error(ctx, nc_def_dim(o->ncid, name, NC_UNLIMITED, &o->varid));
		o->undimid = o->varid;
	}
	nc_enddef(o->ncid);

	return api_create_integer(ctx, o->varid);
}

static void* cdf_defvar(void *ctx, int nargs, void** args)
{
	if (nargs < 4 || nargs-3 > NC_MAX_VAR_DIMS) api_input_error(ctx);
	zsNetcdf *o = (zsNetcdf*)api_get_user(ctx, args[0], NC_TYPE);
	const char *name = api_get_string(ctx, args[1]);
	const char *type = api_get_string(ctx, args[2]);

	if (!strcmp(type, type2name(NC_CHAR))) {
		o->type = NC_CHAR;
		o->esize = sizeof(signed char);
	}
	else if (!strcmp(type, type2name(NC_BYTE))) {
		o->type = NC_BYTE;
		o->esize = sizeof(unsigned char);
	}
	else if (!strcmp(type, type2name(NC_SHORT))) {
		o->type = NC_SHORT;
		o->esize = sizeof(short);
	}
	else if (!strcmp(type, type2name(NC_INT))) {
		o->type = NC_INT;
		o->esize = sizeof(int);
	}
	else if (!strcmp(type, type2name(NC_FLOAT))) {
		o->type = NC_FLOAT;
		o->esize = sizeof(float);
	}
	else if (!strcmp(type, type2name(NC_DOUBLE))) {
		o->type = NC_DOUBLE;
		o->esize = sizeof(double);
	}
	else {
		api_runtime_error(ctx, "unexpected type name");
	}
	
	o->ndim = 0;
	o->vsize = 1;
	for (int i = 3; i < nargs; i++) {
		o->dimid[o->ndim] = api_get_integer(ctx, args[i]);
		nc_inq_dimlen(o->ncid, o->dimid[o->ndim], &o->dims[o->ndim]);
		o->vsize *= o->dims[o->ndim];
		o->ndim++;
	}
	
	nc_redef(o->ncid);
	cdf_check_error(ctx, nc_def_var(o->ncid, name, o->type, o->ndim, o->dimid, &o->varid));
	nc_enddef(o->ncid);

	if (o->ndim == 1 && o->dimid[0] == o->undimid) {
		o->unlimited = 1;
	}
	else {
		o->unlimited = 0;
	}
	
	return 0;
}

static void get_var_info(void *ctx, const char*name, zsNetcdf *o)
{
	cdf_check_error(ctx, nc_inq_unlimdim(o->ncid, &o->undimid));
	cdf_check_error(ctx, nc_inq_varid(o->ncid, name, &o->varid));
	cdf_check_error(ctx, nc_inq_vartype(o->ncid, o->varid, &o->type));
	cdf_check_error(ctx, nc_inq_varndims(o->ncid, o->varid, &o->ndim));
	cdf_check_error(ctx, nc_inq_vardimid(o->ncid, o->varid, o->dimid));

	o->vsize = 1;
	for (int i = 0; i < o->ndim; i++) {
		cdf_check_error(ctx, nc_inq_dimlen(o->ncid, o->dimid[i], &o->dims[i]));
		o->vsize *= o->dims[i];
	}

	if (o->ndim == 1 && o->dimid[0] == o->undimid) {
		o->unlimited = 1;
	}
	else {
		o->unlimited = 0;
	}

	switch (o->type) {
	case NC_CHAR:
		o->esize = sizeof(signed char);
		break;
	case NC_BYTE:
		o->esize = sizeof(unsigned char);
		break;
	case NC_SHORT:
		o->esize = sizeof(short);
		break;
	case NC_INT:
		o->esize = sizeof(int);
		break;
	case NC_FLOAT:
		o->esize = sizeof(float);
		break;
	case NC_DOUBLE:
		o->esize = sizeof(double);
		break;
	default:
		type_error(ctx);
	}
}

static void* cdf_variable(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zsNetcdf *o = (zsNetcdf*)api_get_user(ctx, args[0], NC_TYPE);
	const char *name = api_get_string(ctx, args[1]);
	o->ndim = 0;
	if (nc_inq_varid(o->ncid, name, &o->varid) == NC_NOERR) {
		get_var_info(ctx, name, o);
		return api_create_integer(ctx, o->varid);
	}
	return 0;
}

static void* cdf_size(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zsNetcdf *o = (zsNetcdf*)api_get_user(ctx, args[0], NC_TYPE);
	check_variable(ctx, o);
	void *arr = api_create_array(ctx, 3);
	api_set_array_object(ctx, arr, "0", api_create_integer(0, o->vsize));
	api_set_array_object(ctx, arr, "1", api_create_integer(0, o->esize));
	api_set_array_object(ctx, arr, "2", api_create_string(0, type2name(o->type)));
	return arr;
}

static void* cdf_dims(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zsNetcdf *o = (zsNetcdf*)api_get_user(ctx, args[0], NC_TYPE);
	check_variable(ctx, o);
	if (o->ndim == 1) return api_create_integer(0, o->dims[0]);
	void *arr = api_create_array(ctx, o->ndim);
	for (int i = 0; i < o->ndim; i++) {
		api_set_array_object2(ctx, arr, i, api_create_integer(0, o->dims[i]));
	}
	return arr;
}

static void* cdf_getatt(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zsNetcdf *o = (zsNetcdf*)api_get_user(ctx, args[0], NC_TYPE);
	const char *name = api_get_string(ctx, args[1]);

	nc_enddef(o->ncid);

	nc_type type;
	size_t size;

	int vid = o->varid;
	if (o->ndim == 0) vid = NC_GLOBAL;

	if (nc_inq_att(o->ncid, vid, name, &type, &size) != NC_NOERR) return 0;

	zsBuffer buf(size*sizeof(double));

	switch(type) {
	case NC_CHAR:
		cdf_check_error(ctx, nc_get_att_text(o->ncid, vid, name, buf.u.pchar));
		buf.u.pchar[size] = 0;
		return api_create_string(ctx, buf.u.pchar);
		break;
	case NC_BYTE:
		cdf_check_error(ctx, nc_get_att_uchar(o->ncid, vid, name, buf.u.puchar));
		if (size > 1) {
			void *ret = api_create_array(ctx, size+1);
			for (int i = 0; i < size; i++) {
				api_set_array_object2(ctx, ret, i, api_create_integer(0, buf.u.puchar[i]));
			}
			return ret;
		}
		else {
			return api_create_integer(ctx, buf.u.puchar[0]);
		}
		break;
	case NC_SHORT:
		cdf_check_error(ctx, nc_get_att_short(o->ncid, vid, name, buf.u.pshort));
		if (size > 1) {
			void *ret = api_create_array(ctx, size+1);
			for (int i = 0; i < size; i++) {
				api_set_array_object2(ctx, ret, i, api_create_integer(0, buf.u.pshort[i]));
			}
			return ret;
		}
		else {
			return api_create_integer(ctx, buf.u.pshort[0]);
		}
		break;
	case NC_INT:
		cdf_check_error(ctx, nc_get_att_int(o->ncid, vid, name, buf.u.pint));
		if (size > 1) {
			void *ret = api_create_array(ctx, size+1);
			for (int i = 0; i < size; i++) {
				api_set_array_object2(ctx, ret, i, api_create_integer(0, buf.u.pint[i]));
			}
			return ret;
		}
		else {
			return api_create_integer(ctx, buf.u.pint[0]);
		}
		break;
	case NC_FLOAT:
		cdf_check_error(ctx, nc_get_att_float(o->ncid, vid, name, buf.u.pfloat));
		if (size > 1) {
			void *ret = api_create_array(ctx, size+1);
			for (int i = 0; i < size; i++) {
				api_set_array_object2(ctx, ret, i, api_create_real(0, buf.u.pfloat[i]));
			}
			return ret;
		}
		else {
			return api_create_real(ctx, buf.u.pfloat[0]);
		}
		break;
	case NC_DOUBLE:
		cdf_check_error(ctx, nc_get_att_double(o->ncid, vid, name, buf.u.pdouble));
		if (size > 1) {
			void *ret = api_create_array(ctx, size+1);
			for (int i = 0; i < size; i++) {
				api_set_array_object2(ctx, ret, i, api_create_real(0, buf.u.pdouble[i]));
			}
			return ret;
		}
		else {
			return api_create_real(ctx, buf.u.pdouble[0]);
		}
		break;
	default:
		api_runtime_error(ctx, "unsupported attribute type");
	}
	return 0;
}


static void* cdf_setatt(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zsNetcdf *o = (zsNetcdf*)api_get_user(ctx, args[0], NC_TYPE);
	const char *name = api_get_string(ctx, args[1]);

	int vid = o->varid;
	if (o->ndim == 0) vid = NC_GLOBAL;

	nc_redef(o->ncid);

	nc_del_att(o->ncid, vid, name);

	int offset = 2;

	if (api_is_string(args[offset])) {
		const char *str  = api_get_string(ctx, args[offset]);
		cdf_check_error(ctx, nc_put_att_text(o->ncid, vid, name, strlen(str), str));
	}
	else if (api_is_integer(args[offset])) {
		int value = api_get_integer(ctx, args[offset]);
		cdf_check_error(ctx, nc_put_att_int(o->ncid, vid, name, NC_INT, 1, &value));
	}
	else if (api_is_real(args[offset])) {
		double value = api_get_real(ctx, args[offset]);
		cdf_check_error(ctx, nc_put_att_double(o->ncid, vid, name, NC_DOUBLE, 1, &value));
	}
	else if (api_is_array(args[offset])) {
		int n = api_get_array_size(ctx, args[offset]);
		zsBuffer bi(n*sizeof(int)), br(n*sizeof(double));
		bool real_att = false;
		for (int i = 0; i < n; i++) {
			void *o = api_get_array_object2(ctx, args[offset], i);
			if (api_is_real(o)) {
				br.u.pdouble[i] = api_get_real(ctx, o);
				real_att = true;
			}
			else {
				bi.u.pint[i] = api_get_integer(ctx, o);
				br.u.pdouble[i] = bi.u.pint[i];
			}
		}
		if (real_att) {
			cdf_check_error(ctx, nc_put_att_double(o->ncid, vid, name, NC_DOUBLE, n, br.u.pdouble));
		}
		else {
			cdf_check_error(ctx, nc_put_att_int(o->ncid, vid, name, NC_INT, n, bi.u.pint));
		}
	}
	else {
		api_runtime_error(ctx, "attribute must be int, double, or array of in or double");
	}

	nc_enddef(o->ncid);

	return 0;

}

static void* cdf_attribute(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	void *ret = cdf_getatt(ctx, nargs, args);
	if (!ret) api_runtime_error(ctx, "no such an attribute exists");
	if (!api_is_array(ret)) api_runtime_error(ctx, "attribute is not array of numbers");
	int idx = api_get_integer(ctx, args[2]);
	if (idx < 0 || idx >= api_get_array_size(ctx, ret)) api_runtime_error(ctx, "bad array index");
	api_set_array_object2(ctx, ret, idx, args[3]);
	void *p[3] = { args[0], args[1], ret };
	cdf_setatt(ctx, 3, p);
	return 0;
}


static size_t cdf_check_index(void *ctx, int nargs, void** args, zsNetcdf *o, size_t *start, size_t *count)
{
	if (nargs-1 > o->ndim) api_runtime_error(ctx, "too many dimension specifiers");

	size_t size = 1;

	// set full index

	for (int k = 0; k < o->ndim; k++) {
		start[k] = 0;
		if (o->dims[k] > 0) {
			count[k] = o->dims[k];
			size *= o->dims[k];
		}
		else {
			count[k] = 1;
		}
	}

	for (int i = 1; i < nargs; i++) {
		int k = i - 1;
		if (!api_is_null(args[i])) {
			// set non-full index
			if (o->dims[k] > 0) size /= o->dims[k];
			if (api_is_integer(args[i])) {
				// single item
				start[k] = api_get_integer(ctx, args[i]);
				count[k] = 1;
			}
			else {
				// a range like [start:?]
				start[k] = api_get_integer(ctx, api_get_array_object(ctx, args[i], "0"));
				void *a1 = api_get_array_object(ctx, args[i], "1");
				if (api_is_null(a1)) {
					// a range like [start:*]
					count[k] -= start[k];
				}
				else {
					// a range like [start:end]
					count[k] = api_get_integer(ctx, a1) - start[k] + 1;
				}
				if (count[k] < 1) api_runtime_error(ctx, "invalid range specifiers");
				size *= count[k];
			}
		}
	}

	return size;
}

static void* cdf_get(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	
	if (nargs == 2 && api_is_string(args[1])) {
		// cdf.name
		return cdf_getatt(ctx, nargs, args);
	}

	// cdf[...]

	zsNetcdf *o = (zsNetcdf*)api_get_user(ctx, args[0], NC_TYPE);
	check_variable(ctx, o);

	nc_enddef(o->ncid);

	size_t size, start[NC_MAX_VAR_DIMS], count[NC_MAX_VAR_DIMS];

	size = cdf_check_index(ctx, nargs, args, o, start, count);

	switch (o->type) {
	case NC_CHAR:
		o->checksize(size+1);
		cdf_check_error(ctx, nc_get_vara_text(o->ncid, o->varid, start, count, (char*)o->ptr));
		((char*)o->ptr)[size] = 0;
		return api_create_string(ctx, (char*)o->ptr);
	case NC_BYTE:
		o->checksize(size*sizeof(unsigned char));
		cdf_check_error(ctx, nc_get_vara_uchar(o->ncid, o->varid, start, count, (unsigned char*)o->ptr));
		if (size == 1) {
			return api_create_integer(ctx, ((unsigned char*)o->ptr)[0]);
		}
		else {
			return api_create_user(ctx, o->ptr, 0, 0, 0);
		}
	case NC_SHORT:
		o->checksize(size*sizeof(short));
		cdf_check_error(ctx, nc_get_vara_short(o->ncid, o->varid, start, count, (short*)o->ptr));
		if (size == 1) {
			return api_create_integer(ctx, ((short*)o->ptr)[0]);
		}
		else {
			return api_create_user(ctx, o->ptr, 0, 0, 0);
		}
	case NC_INT:
		o->checksize(size*sizeof(int));
		cdf_check_error(ctx, nc_get_vara_int(o->ncid, o->varid, start, count, (int*)o->ptr));
		if (size == 1) {
			return api_create_integer(ctx, ((int*)o->ptr)[0]);
		}
		else {
			return api_create_user(ctx, o->ptr, 0, 0, 0);
		}
	case NC_FLOAT:
		o->checksize(size*sizeof(float));
		cdf_check_error(ctx, nc_get_vara_float(o->ncid, o->varid, start, count, (float*)o->ptr));
		if (size == 1) {
			return api_create_real(ctx, ((float*)o->ptr)[0]);
		}
		else {
			return api_create_user(ctx, o->ptr, 0, 0, 0);
		}
	case NC_DOUBLE:
		o->checksize(size*sizeof(double));
		cdf_check_error(ctx, nc_get_vara_double(o->ncid, o->varid, start, count, (double*)o->ptr));
		if (size == 1) {
			return api_create_real(ctx, ((double*)o->ptr)[0]);
		}
		else {
			return api_create_user(ctx, o->ptr, 0, 0, 0);
		}
	default:
		type_error(ctx);
	}

	return 0;
}

static void* cdf_set(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);

	if (nargs == 3 && api_is_string(args[1])) {
		// cdf.name = attribute
		return cdf_setatt(ctx, nargs, args);
	}

	// cdf[..] = data

	zsNetcdf *o = (zsNetcdf*)api_get_user(ctx, args[0], NC_TYPE);
	check_variable(ctx, o);

	nc_enddef(o->ncid);

	size_t size, start[NC_MAX_VAR_DIMS], count[NC_MAX_VAR_DIMS];

	nargs--;

	memset(start,0,sizeof(start));
	memset(count,0,sizeof(count));
	size = cdf_check_index(ctx, nargs, args, o, start, count);

	zsBuffer buf(size*sizeof(double));

	switch (o->type) {
	case NC_CHAR:
		if (api_is_string(args[nargs])) {
			const char *ptr = api_get_string(ctx, args[nargs]);
			count[o->ndim-1]=strlen(ptr);
			if (count[o->ndim-1]>o->dims[o->ndim-1]) count[o->ndim-1]=o->dims[o->ndim-1];
			cdf_check_error(ctx, nc_put_vara_text(o->ncid, o->varid, start, count, ptr));
		}
		else {
			api_runtime_error(ctx, "input value must be a string");
		}
		break;
	case NC_BYTE:
		if (api_is_user(args[nargs])) {
			void *ptr = api_get_ptr(ctx, args[nargs]);
			cdf_check_error(ctx, nc_put_vara_uchar(o->ncid, o->varid, start, count, (unsigned char*)ptr));
		}
		else {
			unsigned char value = api_get_integer(ctx, args[nargs]);
			for (size_t i = 0; i < size; i++) buf.u.puchar[i] = value;
			cdf_check_error(ctx, nc_put_vara_uchar(o->ncid, o->varid, start, count, buf.u.puchar));
		}
		break;
	case NC_SHORT:
		if (api_is_user(args[nargs])) {
			void *ptr = api_get_ptr(ctx, args[nargs]);
			cdf_check_error(ctx, nc_put_vara_short(o->ncid, o->varid, start, count, (short*)ptr));
		}
		else {
			short value = api_get_integer(ctx, args[nargs]);
			for (size_t i = 0; i < size; i++) buf.u.pshort[i] = value;
			cdf_check_error(ctx, nc_put_vara_short(o->ncid, o->varid, start, count, buf.u.pshort));
		}
		break;
	case NC_INT:
		if (api_is_user(args[nargs])) {
			void *ptr = api_get_ptr(ctx, args[nargs]);
			cdf_check_error(ctx, nc_put_vara_int(o->ncid, o->varid, start, count, (int*)ptr));
		}
		else {
			int value = api_get_integer(ctx, args[nargs]);
			for (size_t i = 0; i < size; i++) buf.u.pint[i] = value;
			cdf_check_error(ctx, nc_put_vara_int(o->ncid, o->varid, start, count, buf.u.pint));
		}
		break;
	case NC_FLOAT:
		if (api_is_user(args[nargs])) {
			void *ptr = api_get_ptr(ctx, args[nargs]);
			cdf_check_error(ctx, nc_put_vara_float(o->ncid, o->varid, start, count, (float*)ptr));
		}
		else {
			float value = api_get_real(ctx, args[nargs]);
			for (size_t i = 0; i < size; i++) buf.u.pfloat[i] = value;
			cdf_check_error(ctx, nc_put_vara_float(o->ncid, o->varid, start, count, buf.u.pfloat));
		}
		break;
	case NC_DOUBLE:
		if (api_is_user(args[nargs])) {
			void *ptr = api_get_ptr(ctx, args[nargs]);
			cdf_check_error(ctx, nc_put_vara_double(o->ncid, o->varid, start, count, (double*)ptr));
		}
		else {
			double value = api_get_real(ctx, args[nargs]);
			for (size_t i = 0; i < size; i++) buf.u.pdouble[i] = value;
			cdf_check_error(ctx, nc_put_vara_double(o->ncid, o->varid, start, count, buf.u.pdouble));
		}
		break;
	default:
		type_error(ctx);
	}
	return 0;
}


static void get_attributes(void* ctx, int ncid, FILE *f, const int ivar, const char* space)
{
	int i, status, natt;
	char name[NC_MAX_NAME];

	cdf_check_error(ctx, nc_inq_varnatts(ncid, ivar, &natt));

	for (i = 0; i < natt; i++) {
		nc_type type;
		size_t j, len;

		cdf_check_error(ctx, nc_inq_attname(ncid, ivar, i, name));
		cdf_check_error(ctx, nc_inq_att(ncid, ivar, name, &type, &len));

		fprintf(f, "%s%s:", space, name);

		if (len<=0) {
			fprintf(f, " none\n");
			continue;
		}

		zsBuffer buf(len*sizeof(double)), str(64);

		switch(type) {

		case NC_CHAR:
			status = nc_get_att_text(ncid, ivar, name, buf.u.pchar);
			if (status == NC_NOERR) {
				buf.u.pchar[len] = 0;
				fprintf(f, " %s", buf.u.pchar);
			}
			break;

		case NC_BYTE:
			status = nc_get_att_uchar(ncid, ivar, name, buf.u.puchar);
			if (status == NC_NOERR) {
				for (j = 0; j < len; j++) fprintf(f, "  %d", buf.u.puchar[j]);
			}
			break;

		case NC_SHORT:
			status = nc_get_att_short(ncid, ivar, name, buf.u.pshort);
			if (status == NC_NOERR) {
				for (j = 0; j < len; j++) fprintf(f, "  %d", buf.u.pshort[j]);
			}
			break;

		case NC_INT:
			status = nc_get_att_int(ncid, ivar, name, buf.u.pint);
			if (status == NC_NOERR) {
				for (j = 0; j < len; j++) fprintf(f, "  %d", buf.u.pint[j]);
			}
			break;

		case NC_FLOAT:
			status = nc_get_att_float(ncid, ivar, name, buf.u.pfloat);
			if (status == NC_NOERR) {
				for (j = 0; j < len; j++) fprintf(f, "  %s", format_real(buf.u.pfloat[j], str.u.pchar));
			}
			break;

		case NC_DOUBLE:
			status = nc_get_att_double(ncid, ivar, name, buf.u.pdouble);
			if (status == NC_NOERR) {
				for (j = 0; j < len; j++) fprintf(f, "  %s", format_real(buf.u.pdouble[j], str.u.pchar));
			}
			break;

		default:
			fprintf(f, "  unknuwn");
		}

		fprintf(f, "\n");

		cdf_check_error(ctx, status);
	}
	
	fflush(f);
}

static void* cdf_info(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zsNetcdf *o = (zsNetcdf*)api_get_user(ctx, args[0], NC_TYPE);
	int ncid = o->ncid;

	FILE *f = stdout;
	if (nargs > 1) f = fopen(api_get_string(ctx, args[1]), "w");
	if (!f) f = stdout;

	// dimensions

	int i, unlimited, nvar, ndim, ngatt;
	size_t len, dim[NC_MAX_VAR_DIMS];
	char name[NC_MAX_NAME];

	fprintf(f, "Dimensions:\n");

	cdf_check_error(ctx, nc_inq(ncid, &ndim, &nvar, &ngatt, &unlimited));

	for (i = 0; i < ndim; i++) {
		cdf_check_error(ctx, nc_inq_dim(ncid, i, name, &len));
		dim[i] = len;
		if (unlimited == i)
			fprintf(f, "    %s = %d (unlimited)\n", name, len);
		else
			fprintf(f, "    %s = %d\n", name, len);
	}

	// variables

	fprintf(f, "\nVariables:\n");

	for (i = 0; i < nvar; i++) 
	{
		int natt, ndim, dimid[NC_MAX_VAR_DIMS];
		nc_type type;
		cdf_check_error(ctx, nc_inq_var(ncid, i, name, &type, &ndim, dimid, &natt));

		fprintf(f, "    %s",type2name(type));
		fprintf(f, " %s", name);
		
		if (ndim > 0) {
			for (int j = 0; j < ndim; j++) {
				cdf_check_error(ctx, nc_inq_dimname(ncid, dimid[j], name));
				if (j == 0)
					fprintf(f, "(%s", name);
				else
					fprintf(f, ", %s", name);
			}
			fprintf(f, ")");
		}
		
		fprintf(f, "\n");

		get_attributes(ctx, ncid, f, i, "        ");

		fprintf(f, "\n");
	}

	// global attributes

	fprintf(f, "Global attributes:\n");

	get_attributes(ctx, ncid, f, NC_GLOBAL, "    ");
	
	fflush(f);

	if (f != stdout) fclose(f);

	return 0;
}

static void* cdf_das(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zsNetcdf *o = (zsNetcdf*)api_get_user(ctx, args[0], NC_TYPE);
	void *func = api_get_func(ctx,api_get_string(ctx,args[1]));	
	
	int ncid = o->ncid;
	int i, unlimited, nvar, ndim, ngatt;

	nc_enddef(o->ncid);

	cdf_check_error(ctx, nc_inq(ncid, &ndim, &nvar, &ngatt, &unlimited));

	for (i = 0; i < nvar; i++) 
	{
		int j, natt, dimid[NC_MAX_VAR_DIMS];
		nc_type type;
		char vname[NC_MAX_NAME];

		cdf_check_error(ctx, nc_inq_var(ncid, i, vname, &type, &ndim, dimid, &natt));

		const char *tname=type2name(type);
		
		for (j = 0; j < natt; j++) 
		{
			char aname[NC_MAX_NAME];
			size_t len;
		
			cdf_check_error(ctx, nc_inq_attname(ncid, i, j, aname));
			cdf_check_error(ctx, nc_inq_att(ncid, i, aname, &type, &len));

			if (len<=0) continue;

			zsBuffer buf(len*sizeof(double));

			switch(type) {
			case NC_CHAR:
				cdf_check_error(ctx, nc_get_att_text(ncid, i, aname, buf.u.pchar));
				break;

			case NC_BYTE:
				cdf_check_error(ctx, nc_get_att_uchar(ncid, i, aname, buf.u.puchar));
				break;

			case NC_SHORT:
				cdf_check_error(ctx, nc_get_att_short(ncid, i, aname, buf.u.pshort));
				break;

			case NC_INT:
				cdf_check_error(ctx, nc_get_att_int(ncid, i, aname, buf.u.pint));
				break;

			case NC_FLOAT:
				cdf_check_error(ctx, nc_get_att_float(ncid, i, aname, buf.u.pfloat));
				break;

			case NC_DOUBLE:
				cdf_check_error(ctx, nc_get_att_double(ncid, i, aname, buf.u.pdouble));
				break;

			default:
				break;
			}

			int narg=0;
			void *args[10];

			args[narg++]=api_create_integer(ctx,i);
			args[narg++]=api_create_integer(ctx,j);
			args[narg++]=api_create_string(ctx,vname);
			args[narg++]=api_create_string(ctx,tname);
			args[narg++]=api_create_string(ctx,aname);
			args[narg++]=api_create_string(ctx,type2name(type));
			args[narg++]=api_create_integer(ctx,len);
			args[narg++]=api_create_user(ctx,buf.u.ptr,0,0,0);

			api_call_func(ctx,func,narg,args);
		}
	}

	return 0;
}

static void* cdf_dds(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zsNetcdf *o = (zsNetcdf*)api_get_user(ctx, args[0], NC_TYPE);
	void *func = api_get_func(ctx,api_get_string(ctx,args[1]));	
	
	int ncid = o->ncid;
	int i, j, unlimited, nvar, ndim, natt, dimid[NC_MAX_VAR_DIMS];
	size_t dimlen[NC_MAX_VAR_DIMS];
	nc_type type;
	char vname[NC_MAX_NAME];

	nc_enddef(o->ncid);

	cdf_check_error(ctx, nc_inq(ncid, &ndim, &nvar, &natt, &unlimited));

	for (i = 0; i < nvar; i++) 
	{
		cdf_check_error(ctx, nc_inq_var(ncid, i, vname, &type, &ndim, dimid, &natt));

		for (j=0; j<ndim; j++) {
			cdf_check_error(ctx, nc_inq_dimlen(o->ncid, dimid[j], &dimlen[j]));
		}

		int narg=0;
		void *args[10];

		args[narg++]=api_create_integer(ctx,i);
		args[narg++]=api_create_string(ctx,vname);
		args[narg++]=api_create_string(ctx,type2name(type));
		args[narg++]=api_create_integer(ctx,ndim);
		args[narg++]=api_create_user(ctx,dimid,0,0,0);
		args[narg++]=api_create_user(ctx,dimlen,0,0,0);

		api_call_func(ctx,func,narg,args);
	}

	return 0;
}

/////////////////////////////////////////////////////////////////////

class cdfRegPrimitive
{
public:
	cdfRegPrimitive()
	{
		api_add_primitive("netcdf",		0,			cdf_open);
		api_add_primitive("close",		NC_TYPE,	cdf_close);
		api_add_primitive("version",	NC_TYPE,	cdf_version);
		api_add_primitive("defdim",		NC_TYPE,	cdf_defdim);
		api_add_primitive("defvar",		NC_TYPE,	cdf_defvar);
		api_add_primitive("variable",	NC_TYPE,	cdf_variable);
		api_add_primitive("attribute",	NC_TYPE,	cdf_attribute);
		api_add_primitive("size",		NC_TYPE,	cdf_size);
		api_add_primitive("dims",		NC_TYPE,	cdf_dims);
		api_add_primitive("__get",		NC_TYPE,	cdf_get);
		api_add_primitive("__set",		NC_TYPE,	cdf_set);
		api_add_primitive("info",		NC_TYPE,	cdf_info);
		api_add_primitive("das",		NC_TYPE,	cdf_das);
		api_add_primitive("dds",		NC_TYPE,	cdf_dds);
	}

	~cdfRegPrimitive() { }
};

static cdfRegPrimitive cdf_register;
