#include "matrixtmp.h"
#include "matrix.h"
#include "api.h"
#include "fftn.h"
#include "clipping.h"

#include <math.h>
#include <stdlib.h>

#define SWAP(a, b) { double swap=(a); (a)=(b); (b)=swap; }

#define DEG_TO_RAD		0.017453292519943 
#define NPARS			20					// max number of parameters

#pragma warning(disable: 4244)
#pragma warning(disable: 4715)
#pragma warning(disable: 4018)

extern "C" {
// ccmath
int minv(double*, int);
int ruinv(double*, int);
int psinv(double*, int);
int svduv(double*, double*, double*, int, double*, int);
double csfit(double, double*, double*, double*, int);
double tnsfit(double, double*, double*, double*, int, double);
void csplp(double*, double*, double*, int,double);
void cspl(double*, double*, double*, int, double);
double qrlsq(double*, double*, int, int, int*);
double qrvar(double*, int, int, double ssq);
//nelib/toms
int cubgcv_(double*,double*,double*,int*,double*,double*,int*,double*,int*,double*,double*,int*);
int intrc0_(int*, double*, double*, double*, double*, double*, double*, int*, int*, int*, int*, double*, int*);
int intrc1_(int*, double*, double*, double*, double*, double*, double*, int*, int*, int*, int*, double*, int*, double*, int*, double*, int*);
int gradg_(int*, double*, double*, double*, double*, int*, int*, int*, int*, double*, int*, double*, double*, int*);
int intrc0_752(double*, double*, int*, int*, int*, double*, double*, double*, int*, int*, int*, int*, double*, int*);
int intrc1_752(double*, double*, int*, int*, int*, double*, double*, double*, int*, int*, int*, int*, double*, double*, int*, int*, double*, double*, double*, int*);
int gradg_752(int*, int*, int*, double*, double*, double*, int*, int*, int*, int*, double*, int*, double*, double*, int*);
int trmesh_751(int*, double*, double*, int*, int*, int*, int*, int*, int*, double*, int*);
int trlist_751(int*, int*, int*, int*, int*, int*, int*, int*, int*, int*, int*);
int bnodes_751(int*, int*, int*, int*, int*, int*, int*, int*);
int trmesh_(int*, double*, double*, double*, int*, int*, int*, int*, int*, int*, double*, int*);
int trlist_(int*, int*, int*, int*, int*, int*, int*, int*);
int bnodes_(int*, int*, int*, int*, int*, int*, int*, int*);
};

zoDOUBLE* nlfit_func(void *ctx, void *callback, zoDOUBLE &x, zoDOUBLE &par);
double nlfit_try(void *ctx, void *callback, zoDOUBLE &x, zoDOUBLE &y, zoDOUBLE &par, zoDOUBLE &var);

void* num_interpotri(void *ctx, int nargs, void** args);
void* num_interpo1D(void *ctx, int nargs, void** args);
void* num_interpo2D(void *ctx, int nargs, void** args);
void* num_interpo2Dirregular(void *ctx, int nargs, void** args);

int get_factor(const zoDOUBLE &X, double x, int &i1, int &i2, double &f1, double &f2);


/////////////////////////////////////////////////////////////////////

void* num_nlfit(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zoDOUBLE *me = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[0], MAT_DOUBLE));
	if (me->ncol() < 2 || me->nrow() < 2) api_runtime_error(ctx, "matrix columns/rows less than 2");
	void *func = api_get_func(ctx, api_get_string(ctx, args[1]));
	zoDOUBLE *par = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[2], MAT_DOUBLE));
	int i = 0, j, np = par->ndat();
	if (np > NPARS) api_runtime_error(ctx, "too many parameters");
	zoDOUBLE *lb = 0, *ub = 0;
	if (nargs > 4) {
		lb  = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[3], MAT_DOUBLE));
		ub  = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[4], MAT_DOUBLE));
		if (lb->ndat() < np || ub->ndat() < np) api_runtime_error(ctx, "bad boundery matrix");
		for (i = 0; i < np; i++)  {
			if ((*lb)(i) >= (*ub)(i)) api_runtime_error(ctx, "bad boundery values");
		}
	}
	zoDOUBLE *var = new zoDOUBLE;
	var->resize(np, np);
	zoDOUBLE x, y;
	me->getcol(x, 0);
	me->getcol(y, 1);
	double ssq1, ssq2;
	ssq1 = ssq2 = nlfit_try(ctx, func, x, y, *par, *var);
	if (ssq1 > 0) {
		for (i = 0; i < 100; i++) {
			ssq2 = nlfit_try(ctx, func, x, y, *par, *var);
			if (ssq2 < 0 || ssq2 < 1.e-15 || ssq2 == ssq1) break;
			ssq1 = ssq2;
			if (lb) {
				for (j = 0; j < np; j++) {
					if ((*par)(j) > (*ub)(j)) (*par)(j) = (*ub)(j);
					if ((*par)(j) < (*lb)(j)) (*par)(j) = (*lb)(j);
				}
			}
		}
	}
	if (lb) {
		for (j = 0; j < np; j++) {
			if ((*par)(j) > (*ub)(j)) (*par)(j) = (*ub)(j);
			if ((*par)(j) < (*lb)(j)) (*par)(j) = (*lb)(j);
		}
	}
	
	zoDOUBLE* yf = nlfit_func(ctx, func, x, *par);
	ssq2 = 0;
	for (j = 0; j < y.ndat(); j++) {
		double d = y(j) - (*yf)(j);
		ssq2 += d*d;
	}

	void *arr = api_create_array(ctx, 3);
	api_set_array_object(ctx, arr, "0", api_create_integer(0, i));
	api_set_array_object(ctx, arr, "1", api_create_real(0, ssq2));
	api_set_array_object(ctx, arr, "2", api_create_user(0, var, mat_opfunc_DOUBLE, mat_destroy_DOUBLE, MAT_DOUBLE));
	return arr;
}
static zsRegPrimitive num1b("nlfit", MAT_DOUBLE, num_nlfit);


void* num_mlfit(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zoDOUBLE *Y = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[0], MAT_DOUBLE));
	zoDOUBLE *X = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[1], MAT_DOUBLE));
	int m = X->nrow(), n = X->ncol();
	if (m != Y->ndat()) api_input_error(ctx);
	int flag;
	double ssq = qrlsq(X->ptr(), Y->ptr(), m, n, &flag);
	if (flag < 0) api_runtime_error(ctx, "rank of X < Y\'s length");
	ssq = qrvar(X->ptr(), m, n, ssq);
	return api_create_real(ctx, ssq);
}
static zsRegPrimitive num1c("mlfit", MAT_DOUBLE, num_mlfit);


void* num_svd(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoDOUBLE *me = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[0], MAT_DOUBLE));
	int m = me->nrow();
	int n = me->ncol();
	if (n < 2 || m < 2) api_input_error(ctx);
	zoDOUBLE *u = new zoDOUBLE;
	u->resize(m, m);
	zoDOUBLE *v = new zoDOUBLE;
	v->resize(n, n);
	zoDOUBLE *d = new zoDOUBLE;
	d->resize(n, 1);
	svduv(d->ptr(), me->ptr(), u->ptr(), m, v->ptr(), n);
	void *arr = api_create_array(ctx, 3);
	api_set_array_object(ctx, arr, "0", api_create_user(0, u, mat_opfunc_DOUBLE, mat_destroy_DOUBLE, MAT_DOUBLE));
	api_set_array_object(ctx, arr, "1", api_create_user(0, d, mat_opfunc_DOUBLE, mat_destroy_DOUBLE, MAT_DOUBLE));
	api_set_array_object(ctx, arr, "2", api_create_user(0, v, mat_opfunc_DOUBLE, mat_destroy_DOUBLE, MAT_DOUBLE));
	return arr;
}
static zsRegPrimitive num1d("svd", MAT_DOUBLE, num_svd);


void* num_invert(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoDOUBLE *me = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[0], MAT_DOUBLE));
	if (me->nrow() != me->ncol()) api_runtime_error(ctx, "row != col");
	int status;
	if (nargs > 1) {
		const char* s = api_get_string(ctx, args[1]);
		if      (strcmp(s, "ps") == 0) status = psinv(me->ptr(), me->nrow());
		else if (strcmp(s, "ru") == 0) status = ruinv(me->ptr(), me->nrow());
		else                           status =  minv(me->ptr(), me->nrow());
	}
	else {
		status =  minv(me->ptr(), me->nrow());
	}
	return api_create_integer(ctx, !status);
}
static zsRegPrimitive num1("invert", MAT_DOUBLE, num_invert);


void* num_fft(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zoDOUBLE *me = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[0], MAT_DOUBLE));
	int flag = api_get_integer(ctx, args[1]);
	if (me->ncol() != 2) api_runtime_error(ctx, "matrix must contain two columns");
	int dims[1] = { me->nrow() };
	zoDOUBLE x, y;
	me->getcol(x, 0);
	me->getcol(y, 1);
	if (flag > 0) flag = fftn(1, dims, x.ptr(), y.ptr(),  1,  0.0);
	else          flag = fftn(1, dims, x.ptr(), y.ptr(), -1, -1.0);
	me->setcol(x, 0);
	me->setcol(y, 1);
	return api_create_integer(ctx, !flag);
}
static zsRegPrimitive num2("fft", MAT_DOUBLE, num_fft);


void* num_spline(void *ctx, int nargs, void** args)
{
	if (nargs < 6) api_input_error(ctx);
	zoDOUBLE *me = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[0], MAT_DOUBLE));
	double t = api_get_integer(ctx, args[1]);
	int flag = api_get_integer(ctx, args[2]);
	double a = api_get_integer(ctx, args[3]);
	double b = api_get_integer(ctx, args[4]);
	int n    = api_get_integer(ctx, args[5]);

	if (me->ncol() != 2 || n < 3) api_input_error(ctx);

	double d = (b - a) / (n - 1);

	zoDOUBLE x, y, z;
	me->getcol(x, 0);
	me->getcol(y, 1);
	z.resize(me->nrow(), 1);

	if (flag > 0) {
		cspl(x.ptr(), y.ptr(), z.ptr(), me->nrow()-1, t);
	}
	else {
		csplp(x.ptr(), y.ptr(), z.ptr(), me->nrow()-1, t);
	}

	zoDOUBLE *p = new zoDOUBLE;
	p->resize(n, 1);

	if (t > 0) {
		for (int i = 0; i < n; i++) (*p)(i) = tnsfit(a+i*d, x.ptr(), y.ptr(), z.ptr(), me->nrow()-1, t);
	}
	else {
		for (int i = 0; i < n; i++) (*p)(i) =  csfit(a+i*d, x.ptr(), y.ptr(), z.ptr(), me->nrow()-1);
	}

	return api_create_user(ctx, p, mat_opfunc_DOUBLE, mat_destroy_DOUBLE, MAT_DOUBLE);
}
static zsRegPrimitive num3("spline", MAT_DOUBLE, num_spline);


void* num_cv_spline(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zoDOUBLE *me = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[0], MAT_DOUBLE));
	double a = api_get_number(ctx, args[1]);
	double b = api_get_number(ctx, args[2]);
	int np   = api_get_integer(ctx, args[3]);

	if (me->ncol() != 2 || np < 3) api_input_error(ctx);

	double d = (b - a) / (np - 1);

	zoDOUBLE x, f, df;
	me->getcol(x, 0);
	me->getcol(f, 1);
	df.resize(me->nrow(), 1);
	df.fill(1, 0);

	int n = me->nrow();
	double var = -1.0;
	int ic = n-1;
	int job = 1;
	int ier;

	zoDOUBLE y, c, se, wk;
	y.resize(1, n);
	c.resize(1, ic*3);
	se.resize(1, n);
	wk.resize(1, 7*(n+2));

	cubgcv_(x.ptr(), f.ptr(), df.ptr(), &n, y.ptr(), c.ptr(), &ic, &var, &job, se.ptr(), wk.ptr(), &ier);

	zoDOUBLE *p = new zoDOUBLE;
	p->resize(np, 1);

	if (ier < 129 || ier > 133) {
		for (int i = 0; i < np; i++) {
			double u = a + i*d;
			if (u <= x(0)) {
				(*p)(i) = f(0);
			}
			else if (u >= x(n-1)) {
				(*p)(i) = f(n-1);
			}
			else {
				for (int j = 1; j < n; j++) {
					if (u < x(j)) break;
				}
				j--;
				double t = u - x(j);
				(*p)(i) = ((t * c(2*n-2+j) + c(n-1+j)) * t + c(j)) * t + y(j);
			}
		}
	}

	return api_create_user(ctx, p, mat_opfunc_DOUBLE, mat_destroy_DOUBLE, MAT_DOUBLE);
}
static zsRegPrimitive num3b("cvspline", MAT_DOUBLE, num_cv_spline);


void* num_dist_smooth(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zoDOUBLE *me = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[0], MAT_DOUBLE));
	int n  = api_get_integer(ctx, args[1]);
	if (n < 1) api_input_error(ctx);
	int nr = me->nrow();
	int nc = me->ncol();
	zoDOUBLE p = (*me);
	for (int i = 0; i < nr; i++) {
		int m1 = i - n;
		if (m1 < 0) m1 = 0;
		int m2 = i + n;
		if (m2 > nr) m2 = nr;
		for (int j = 0; j < nc; j++) {
			int n1 = j - n;
			if (n1 < 0) n1 = 0;
			int n2 = j + n;
			if (n2 > nc) n2 = nc;
			double fsum = 0;
			double vsum = 0;
			for (int ii = m1; ii < m2; ii++) {
				for (int jj = n1; jj < n2; jj++) {
					double tmp = 1.0 / (1.0 + (ii-i)*(ii-i)+(jj-j)*(jj-j));
					if (p(ii, jj) > -1.e35 && p(ii, jj) < 1.e35) {
						fsum += tmp;
						vsum += tmp * p(ii, jj);
					}
				}
			}
			if (fsum > 0) (*me)(i, j) = vsum / fsum;
		}
	}
	return 0;
}
static zsRegPrimitive num3c("smooth", MAT_DOUBLE, num_dist_smooth);


void sg_coef(void *ctx, zoDOUBLE &C, int m, int nl, int nr)
{
	zoDOUBLE A;
	A.resize(nl+nr+1, m+1);
	C.resize(nl+nr+1, 1);
	int i, j;
	for (i = -nl; i <= nr; i++) {
		for (j = 0; j <= m; j++) A(i+nl,j) = pow(i, j);
	}
	zoDOUBLE AT(A);
	AT.transpose();
	AT.concat(A);
	if (psinv(AT.ptr(), m+1) != 0) api_runtime_error(ctx, "matrix not positive definite.");
	A.transpose();
	for (i = -nl; i <= nr; i++) {
		C(i+nl) = 0;
		for (j = 0; j <= m; j++) C(i+nl) += AT(0,j)*A(j,i+nl);
	}
}

void* num_sg_smooth(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zoDOUBLE *X = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[0], MAT_DOUBLE));
	int m = api_get_integer(ctx, args[1]);
	int n = api_get_integer(ctx, args[2]);
	int i, k, nd = X->ndat();
	if (2*n+1 < m || m < 2 || nd < 2*n+1) api_input_error(ctx);
	zoDOUBLE T(*X), C;
	for (i = 0; i < nd; i++) {
		int nl = n > i ? i : n;
		int nr = i+n < nd ? n : nd-i-1;
		if (nl != nr || i == n+1) sg_coef(ctx, C, m, nl, nr);
		(*X)(i) = 0;
		for (k = 0; k < C.ndat(); k++) (*X)(i) += C(k)*T(i-nl+k);
	}
	return 0;
}
static zsRegPrimitive num4("sgsmooth", MAT_DOUBLE, num_sg_smooth);


void* num_interpo(void *ctx, int nargs, void** args)
{
	if (nargs > 4) return num_interpo2Dirregular(ctx, nargs, args);
	if (nargs > 3) return num_interpotri(ctx, nargs, args);
	if (nargs > 2) return num_interpo2D(ctx, nargs, args);
	return num_interpo1D(ctx, nargs, args);
}
static zsRegPrimitive num5("interpo", MAT_DOUBLE, num_interpo);


void* num_delaunay(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoDOUBLE *me = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[0], MAT_DOUBLE));
	if (me->nrow() < 3 || me->ncol() < 2) api_input_error(ctx);
	int flag = 0;
	if (nargs > 1) flag = api_get_integer(ctx, args[1]);

	zoDOUBLE x, y, z;
	me->getcol(x, 0);
	me->getcol(y, 1);
	int n = me->nrow();
	zoINT *idx = new zoINT;
	zoINT *hul = new zoINT;
	zoINT tmp;
	tmp.resize(n, 1);
	int i, k, lnew, ier, nrow = 6, nt = 2*n, na, nb, ncc = 0, lcc[1], lct[1];
	zoINT near1, next, list, lptr, lend, ltri;
	near1.resize(1, n);
	next.resize(1, n);
	list.resize(1, 6*n);
	lptr.resize(1, 6*n);
	lend.resize(1, n);
	ltri.resize(1, nrow*nt);
	zoDOUBLE dist;
	dist.resize(1, n);

	char msg[256];
	msg[0] = 0;
	if (me->ncol() == 2) {
		trmesh_751(&n, x.ptr(), y.ptr(), list.ptr(), lptr.ptr(), lend.ptr(), &lnew, near1.ptr(), next.ptr(), dist.ptr(), &ier);
		if (ier != 0) {
			sprintf(msg, "trmesh_751() error code: %d", ier);
			api_runtime_error(ctx, msg);
		}
		trlist_751(&ncc, lcc, &n, list.ptr(), lptr.ptr(), lend.ptr(), &nrow, &nt, ltri.ptr(), lct, &ier);
		if (ier != 0) {
			sprintf(msg, "trlist_751() error code: %d", ier);
			api_runtime_error(ctx, msg);
		}
		if (flag) {
			zoINT wk;
			wk.resize(nt, 3);
			int count = 0;
			for (i = 0, k = 0; i < nt; i++, k += nrow) {
				int i1 = ltri(k  ) - 1;
				int i2 = ltri(k+1) - 1;
				int i3 = ltri(k+2) - 1;
				double px = 0.5*(x(i1) + x(i2));
				double py = 0.5*(y(i1) + y(i2));
				if (!point_in_polygon(x.ptr(), y.ptr(), x.nrow(), px, py)) continue;
				px = 0.5*(x(i1) + x(i3));
				py = 0.5*(y(i1) + y(i3));
				if (!point_in_polygon(x.ptr(), y.ptr(), x.nrow(), px, py)) continue;
				px = 0.5*(x(i2) + x(i3));
				py = 0.5*(y(i2) + y(i3));
				if (!point_in_polygon(x.ptr(), y.ptr(), x.nrow(), px, py)) continue;
				wk(count, 0) = i1;
				wk(count, 1) = i2;
				wk(count, 2) = i3;
				count++;
			}
			idx->resize(count, 3);
			for (i = 0; i < count; i++) {
				(*idx)(i,0) = wk(i,0);
				(*idx)(i,1) = wk(i,1);
				(*idx)(i,2) = wk(i,2);
			}
		}
		else {
			idx->resize(nt, 3);
			for (i = 0, k = 0; i < nt; i++, k += nrow) {
				(*idx)(i, 0) = ltri(k) - 1;
				(*idx)(i, 1) = ltri(k+1) - 1;
				(*idx)(i, 2) = ltri(k+2) - 1;
			}
		}
		bnodes_751(&n, list.ptr(), lptr.ptr(), lend.ptr(), tmp.ptr(), &nb, &na, &nt);
		hul->resize(nb, 1);
		for (i = 0; i < nb; i++) (*hul)(i) = tmp(i) - 1;
	}
	else {
		me->getcol(z, 2);
		trmesh_(&n, x.ptr(), y.ptr(), z.ptr(), list.ptr(), lptr.ptr(), lend.ptr(), &lnew, near1.ptr(), next.ptr(), dist.ptr(), &ier);
		if (ier != 0) {
			sprintf(msg, "trmesh_() error code: %d", ier);
			api_runtime_error(ctx, msg);
		}
		trlist_(&n, list.ptr(), lptr.ptr(), lend.ptr(), &nrow, &nt, ltri.ptr(), &ier);
		if (ier != 0) {
			sprintf(msg, "trlist_() error code: %d", ier);
			api_runtime_error(ctx, msg);
		}
		idx->resize(nt, 3);
		for (i = 0, k = 0; i < nt; i++, k += nrow) {
			(*idx)(i, 0) = ltri(k) - 1;
			(*idx)(i, 1) = ltri(k+1) - 1;
			(*idx)(i, 2) = ltri(k+2) - 1;
		}
		bnodes_(&n, list.ptr(), lptr.ptr(), lend.ptr(), tmp.ptr(), &nb, &na, &nt);
		hul->resize(nb, 1);
		for (i = 0; i < nb; i++) (*hul)(i) = tmp(i) - 1;
	}

	void *arr = api_create_array(ctx, 2);
	api_set_array_object(ctx, arr, "0", api_create_user(0, idx, mat_opfunc_INT, mat_destroy_INT, MAT_INT));
	api_set_array_object(ctx, arr, "1", api_create_user(0, hul, mat_opfunc_INT, mat_destroy_INT, MAT_INT));
	return arr;
}
static zsRegPrimitive num7("delaunay", MAT_DOUBLE, num_delaunay);


/////////////////////////////////////////////////////////////////////

zoDOUBLE* nlfit_func(void *ctx, void *callback, zoDOUBLE &x, zoDOUBLE &par)
{
	void *args[21];
	args[0] = api_create_user(ctx, new zoDOUBLE(x), mat_opfunc_DOUBLE, mat_destroy_DOUBLE, MAT_DOUBLE);
	for (int i = 0; i < par.ndat(); i++) args[i+1] = api_create_real(ctx, par(i));
	void *ret = api_call_func(ctx, callback, par.ndat()+1, args);
	zoDOUBLE *yf = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, ret, MAT_DOUBLE));
	if (yf->ndat() != x.ndat()) api_runtime_error(ctx, "nonlinear-fitting callback returns matrix of wrong size");
	return yf;
}

double nlfit_try(void *ctx, void *callback, zoDOUBLE &x, zoDOUBLE &y, zoDOUBLE &par, zoDOUBLE &var)
{
	int n = x.ndat(), m = par.ndat();
	double *cp = (double*)calloc(2*m, sizeof(double));
	double *dp = cp + m;
	double *p, *q, *r, *s, *t;
	double err, ssq;
	int j, k;

	zoDOUBLE f(*nlfit_func(ctx, callback, x, par));
	zoDOUBLE de;
	de.resize(n, m);
	for (j = 0; j < m; j++) {
		double d = 1.e-5 * fabs(par(j)); 
		if (d < 1.e-9) d = 1.e-9;
		par(j) += d;
		zoDOUBLE *g = nlfit_func(ctx, callback, x, par);
		par(j) -= d;
		g->sub(f);
		g->div(d);
		de.setcol(*g, j);
	}

	for (j = 0; j < var.ndat(); j++) var(j) = 0;

	for (j = 0, ssq = 0.; j < n; ++j) {
		err = y(j) - f(j);
		ssq += err*err;
		for (k = 0, r = cp; k < m; ++k) *r++ = de(j,k);
		for(r = dp, s = cp, q = var.ptr(); s < dp; ++s, q += m + 1) {
			*r++ += err * *s;
			for (t = s, p = q; t < dp; ) *p++ += *s* *t++;
		}
	}

	for (j = 0, p = var.ptr(); j < m; ++j, p += m+1) {
		for (k = j+1, q = p, r = p; k < m; ++k) *(q += m) = *++r;
	}

	zoDOUBLE u, v, d;
	u.resize(m, m);
	v.resize(m, m);
	d.resize(1, m);
	svduv(d.ptr(), var.ptr(), u.ptr(), m, v.ptr(), m);
	var.fill(0,0);
	for (j=0; j < m; j++) {
		if (d(j) != 0) var(j,j) = 1./d(j);
	}
	u.transpose();
	var.concat(u);
	v.concat(var);
	for (j=0; j < var.ndat(); j++) var(j) = v(j);

	for(k = 0, p = var.ptr(), s = par.ptr(); k < m; ++k, ++s) {
		for(j = 0, t = dp; j < m; ++j) *s += *p++ * *t++;
	}
	free(cp);
	return ssq;
}

void* num_interpotri(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	zoDOUBLE *me = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[0], MAT_DOUBLE));
	zoDOUBLE *gx = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[1], MAT_DOUBLE));
	zoDOUBLE *gy = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[2], MAT_DOUBLE));
	int flag = api_get_integer(ctx, args[3]);
	if (me->nrow() < 3 || me->ncol() < 3) api_input_error(ctx);
	zoDOUBLE x, y, z, w;
	me->getcol(x, 0);
	me->getcol(y, 1);
	me->getcol(z, 2);
	int nx = gx->ndat(), ny = gy->ndat();
	zoDOUBLE *gz = new zoDOUBLE;
	gz->resize(ny, nx);
	int n = me->nrow();
	zoINT near1, next, list, lptr, lend;
	near1.resize(1, n);
	next.resize(1, n);
	list.resize(1, 6*n);
	lptr.resize(1, 6*n);
	lend.resize(1, n);
	zoDOUBLE dist, grad;
	dist.resize(1, n);
	grad.resize(1, 2*n);
	double sigma = 0, dgmax = 1.e-3;
	int i, j, ist = 1, lnew, ier, ncc = 0, lcc[1], iflgs = 0, iflgg = 1, dflag = 1, nit = 32;
	char msg[256];
	msg[0] = 0;

	if (me->ncol() == 3) {
		trmesh_751(&n, x.ptr(), y.ptr(), list.ptr(), lptr.ptr(), lend.ptr(), &lnew, near1.ptr(), next.ptr(), dist.ptr(), &ier);
		if (ier != 0) {
			sprintf(msg, "trmesh_751() error code: %d", ier);
			api_runtime_error(ctx, msg);
		}
		if (flag != 0) {
			gradg_752(&ncc, lcc, &n, x.ptr(), y.ptr(), z.ptr(), list.ptr(), lptr.ptr(), lend.ptr(), &iflgs, &sigma, &nit, &dgmax, grad.ptr(), &ier);
			if (ier != 0) {
				sprintf(msg, "gradg_752() error code: %d", ier);
				api_runtime_error(ctx, msg);
			}
		}
		for (i = 0; i < ny; i++) {
			double py = (*gy)(i);
			for (j = 0; j < nx; j++) {
				double pz, px = (*gx)(j);
				if (flag != 0) {
					intrc1_752(&px, &py, &ncc, lcc, &n, x.ptr(), y.ptr(), z.ptr(), list.ptr(), lptr.ptr(), lend.ptr(), &iflgs, &sigma, grad.ptr(), &dflag, &ist, &pz, gx->ptr(), gy->ptr(), &ier);
					if (ier < 0) {
						sprintf(msg, "intrc1_752() error code: %d", ier);
						api_runtime_error(ctx, msg);
					}
					(*gz)(i, j) = pz;
				}
				else {
					intrc0_752(&px, &py, &ncc, lcc, &n, x.ptr(), y.ptr(), z.ptr(), list.ptr(), lptr.ptr(), lend.ptr(), &ist, &pz, &ier);
					if (ier < 0) {
						sprintf(msg, "intrc0_752() error code: %d", ier);
						api_runtime_error(ctx, msg);
					}
				}
				(*gz)(i, j) = pz;
			}
		}
	}
	else {
		me->getcol(w, 3);
		trmesh_(&n, x.ptr(), y.ptr(), z.ptr(), list.ptr(), lptr.ptr(), lend.ptr(), &lnew, near1.ptr(), next.ptr(), dist.ptr(), &ier);
		if (ier != 0) {
			sprintf(msg, "trmesh_() error code: %d", ier);
			api_runtime_error(ctx, msg);
		}
		if (flag != 0) {
			gradg_(&n, x.ptr(), y.ptr(), z.ptr(), w.ptr(), list.ptr(), lptr.ptr(), lend.ptr(), &iflgs, &sigma, &nit, &dgmax, grad.ptr(), &ier);
			if (ier != 0) {
				sprintf(msg, "gradg_() error code: %d", ier);
				api_runtime_error(ctx, msg);
			}
		}
		for (i = 0; i < ny; i++) {
			double lat = DEG_TO_RAD * (*gy)(i);
			for (j = 0; j < nx; j++) {
				double lon = DEG_TO_RAD * (*gx)(j);
				double pw;
				if (flag != 0) {
					intrc1_(&n, &lat, &lon, x.ptr(), y.ptr(), z.ptr(), w.ptr(), list.ptr(), lptr.ptr(), lend.ptr(), &iflgs, &sigma, &iflgg, grad.ptr(), &ist, &pw, &ier);
					if (ier < 0) {
						sprintf(msg, "intr1_() error code: %d", ier);
						api_runtime_error(ctx, msg);
					}
				}
				else {
					intrc0_(&n, &lat, &lon, x.ptr(), y.ptr(), z.ptr(), w.ptr(), list.ptr(), lptr.ptr(), lend.ptr(), &ist, &pw, &ier);
					if (ier < 0) {
						sprintf(msg, "intr0_() error code: %d", ier);
						api_runtime_error(ctx, msg);
					}
				}
				(*gz)(i, j) = pw;
			}
		}
	}

	return api_create_user(ctx, gz, mat_opfunc_DOUBLE, mat_destroy_DOUBLE, MAT_DOUBLE);
}

void* num_interpo1D(void *ctx, int nargs, void** args)
{
	zoDOUBLE *me = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[0], MAT_DOUBLE));
	zoDOUBLE *u  = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[1], MAT_DOUBLE));
	zoDOUBLE x, y;
	me->getcol(x, 0);
	me->getcol(y, 1);
	zoDOUBLE *p = new zoDOUBLE;
	p->resize(u->ndat(), 1);
	int i1, i2, i;
	double f1, f2;
	for (i = 0; i < u->ndat(); i++) {
		get_factor(x, (*u)(i), i1, i2, f1, f2);
		(*p)(i) = f1 * y(i1) + f2 * y(i2);
	}
	return api_create_user(ctx, p, mat_opfunc_DOUBLE, mat_destroy_DOUBLE, MAT_DOUBLE);
}

void* num_interpo2D(void *ctx, int nargs, void** args)
{
	zoDOUBLE *me = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[0], MAT_DOUBLE));
	zoDOUBLE *u  = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[1], MAT_DOUBLE));
	zoDOUBLE *v  = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[2], MAT_DOUBLE));
	zoDOUBLE fu1, fu2, fv1, fv2, x, y, *p = new zoDOUBLE;
	p->resize(v->ndat(), u->ndat());
	fu1.resize(1, u->ndat());
	fu2.resize(1, u->ndat());
	fv1.resize(1, v->ndat());
	fv2.resize(1, v->ndat());
	x.resize(1, me->ncol());
	x.fill(0, 1);
	y.resize(1, me->nrow());
	y.fill(0, 1);
	zoINT iu1, iu2, iv1, iv2;
	iu1.resize(1, u->ndat());
	iu2.resize(1, u->ndat());
	iv1.resize(1, v->ndat());
	iv2.resize(1, v->ndat());
	int i, j;
	for (i = 0; i < u->ndat(); i++) get_factor(x, (*u)(i), iu1(i), iu2(i), fu1(i), fu2(i));
	for (i = 0; i < v->ndat(); i++) get_factor(y, (*v)(i), iv1(i), iv2(i), fv1(i), fv2(i));
	for (i = 0; i < v->ndat(); i++) {
		for (j = 0; j < u->ndat(); j++) {
			(*p)(i,j) = fv1(i) * fu1(j) * (*me)(iv1(i), iu1(j)) +
						fv1(i) * fu2(j) * (*me)(iv1(i), iu2(j)) + 
						fv2(i) * fu1(j) * (*me)(iv2(i), iu1(j)) +
						fv2(i) * fu2(j) * (*me)(iv2(i), iu2(j));
		}
	}
	return api_create_user(ctx, p, mat_opfunc_DOUBLE, mat_destroy_DOUBLE, MAT_DOUBLE);
}


void* num_interpo2Dirregular(void *ctx, int nargs, void** args)
{
	zoDOUBLE *me = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[0], MAT_DOUBLE));
	zoDOUBLE *x  = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[1], MAT_DOUBLE));
	zoDOUBLE *y  = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[2], MAT_DOUBLE));
	zoDOUBLE *u  = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[3], MAT_DOUBLE));
	zoDOUBLE *v  = reinterpret_cast<zoDOUBLE*>(api_get_user(ctx, args[4], MAT_DOUBLE));
	zoDOUBLE fu1, fu2, fv1, fv2, *p = new zoDOUBLE;
	p->resize(v->ndat(), u->ndat());
	fu1.resize(1, u->ndat());
	fu2.resize(1, u->ndat());
	fv1.resize(1, v->ndat());
	fv2.resize(1, v->ndat());
	zoINT iu1, iu2, iv1, iv2;
	iu1.resize(1, u->ndat());
	iu2.resize(1, u->ndat());
	iv1.resize(1, v->ndat());
	iv2.resize(1, v->ndat());
	int i, j;
	for (i = 0; i < u->ndat(); i++) get_factor(*x, (*u)(i), iu1(i), iu2(i), fu1(i), fu2(i));
	for (i = 0; i < v->ndat(); i++) get_factor(*y, (*v)(i), iv1(i), iv2(i), fv1(i), fv2(i));
	for (i = 0; i < v->ndat(); i++) {
		for (j = 0; j < u->ndat(); j++) {
			(*p)(i,j) = fv1(i) * fu1(j) * (*me)(iv1(i), iu1(j)) +
						fv1(i) * fu2(j) * (*me)(iv1(i), iu2(j)) + 
						fv2(i) * fu1(j) * (*me)(iv2(i), iu1(j)) +
						fv2(i) * fu2(j) * (*me)(iv2(i), iu2(j));
		}
	}
	return api_create_user(ctx, p, mat_opfunc_DOUBLE, mat_destroy_DOUBLE, MAT_DOUBLE);
}

int get_factor(const zoDOUBLE &X, double x, int &i1, int &i2, double &f1, double &f2)
{
	int i, n=X.ndat();

	if (n<2) return 0;

	if (X(1) > X(0)) {
		if (x <= X(0)) {
			i1 = 0;
			i2 = 0;
			f1 = 0;
			f2 = 1;
			return 1;
		}
		if (x >= X(n-1)) {
			i1 = n - 1;
			i2 = n - 1;
			f1 = 1;
			f2 = 0;
			return 1;
		}
		for (i = 1; i < n; i++) {
			if (x <= X(i)) {
				double d =  fabs(X(i)-X(i-1));
				if (d != 0) {
					f1 = fabs(X(i)-x) / d;
					f2 = fabs(X(i-1)-x) / d;
				}
				else {
					f1 = 0.5;
					f2 = 0.5;
				}
				i1 = i - 1;
				i2 = i;
				return 1;
			}
		}
	}
	else {
		if (x >= X(0)) {
			i1 = 0;
			i2 = 0;
			f1 = 0;
			f2 = 1;
			return 1;
		}
		if (x <= X(n-1)) {
			i1 = n - 1;
			i2 = n - 1;
			f1 = 1;
			f2 = 0;
			return 1;
		}
		for (i = 1; i < n; i++) {
			if (x >= X(i)) {
				double d =  fabs(X(i)-X(i-1));
				if (d != 0) {
					f1 = fabs(X(i)-x) / d;
					f2 = fabs(X(i-1)-x) / d;
				}
				else {
					f1 = 0.5;
					f2 = 0.5;
				}
				i1 = i - 1;
				i2 = i;
				return 1;
			}
		}
	}

	i1 = 0;
	i2 = 0;
	f1 = 0;
	f2 = 0;

	return 1;
}

/*
int get_factor(const zoDOUBLE &X, double x, int &i1, int &i2, double &f1, double &f2)
{
	double xmin = X(0), xmax = xmin;

	for (int i = 1; i < X.ndat(); i++) {
		xmin = X(i) < xmin ? X(i) : xmin;
		xmax = X(i) > xmax ? X(i) : xmax;
		double d =  fabs(X(i) - X(i-1)) + 1.e-30;
		f1 = fabs(X(i) - x);
		f2 = fabs(X(i-1) - x);
		if ( f1+f2 <= d) {
			f1 /= d;
			f2 /= d;
			i1 = i-1;
			i2 = i;
			return 0;
		}
	}

	if (x <= xmin) {
		f1 = 0;
		f2 = 1;
		i1 = 0;
		i2 = 0;
		return -1;
	}

	f1 = 1;
	f2 = 0;
	i1 = X.ndat() - 1;
	i2 = X.ndat() - 1;

	return 1;
}
*/