#ifndef __ZO_MATRIX_TMP__
#define __ZO_MATRIX_TMP__

#include <string.h>
#include <stdio.h>

#pragma warning(disable: 4018)
#pragma warning(disable: 4146)
#pragma warning(disable: 4244)

#define MAX(a, b) ((a) > (b) ? (a) : (b))
#define MIN(a, b) ((a) < (b) ? (a) : (b))

#define ROW_MAJOR

typedef unsigned int size_t;

enum {
	MAT_NULL = 'MA',
	MAT_CHAR, MAT_UCHAR, MAT_SHORT, MAT_USHORT, MAT_INT, MAT_UINT, MAT_INT64,
	MAT_FLOAT, MAT_DOUBLE, MAT_COMPLEX, MAT_TRIPLEX, MAT_QUADRUPLE,
	MAT_TOP
};

typedef struct sComplex  { double r, i;       } sComplex;
typedef struct sTriplex  { double x, y, z;    } sTriplex;
typedef struct sQuadruple{ double x, y, z, w; } sQuadruple;

typedef double (*ApplyFunc1)(double);
typedef double (*ApplyFunc2)(double, double);

class zoUINT;
class zoCHAR;

///
/// Matrix template
///

template<class T>
class zoMatrixTmp
{
protected:
	int				_type;

	size_t			_ncol;

	size_t			_nrow;

	T				*_data;

public:
	/// Constructor.
	zoMatrixTmp() : _type(MAT_NULL), _nrow(0), _ncol(0), _data(0) { }

	zoMatrixTmp(const zoMatrixTmp & mat) : _type(MAT_NULL), _nrow(0), _ncol(0), _data(0)
	{
		(*this) = mat;
	}

	/// Destructor.
	virtual ~zoMatrixTmp()
	{
		delete[] _data;
	}

	/// The 'operator='.
	zoMatrixTmp & operator =(const zoMatrixTmp & mat)
	{
		if (&mat != this) {
			_type = mat.type();
			resize(mat.nrow(), mat.ncol());
			memcpy(_data, mat.ptr(), ndat()*sizeof(T));
		}
		return *this;
	}

	/// Array access
	size_t ndat() const { return _ncol * _nrow; }

	size_t ncol() const { return _ncol; }

	size_t nrow() const { return _nrow; }

	int    type() const { return _type; }

	const T * ptr() const { return _data; }
		  T * ptr()       { return _data; }

	const T & operator()(size_t i) const { return _data[i]; }
		  T & operator()(size_t i)       { return _data[i]; }

#ifdef ROW_MAJOR
	const T & operator()(size_t i, size_t j) const { return _data[i * _ncol + j]; }
		  T & operator()(size_t i, size_t j)       { return _data[i * _ncol + j]; }
#else
	const T & operator()(size_t i, size_t j) const { return _data[i + _nrow * j]; }
		  T & operator()(size_t i, size_t j)       { return _data[i + _nrow * j]; }
#endif

	/// Reshape
	int reshape(size_t nrow, size_t ncol)
	{
		if (ndat() != ncol * nrow) return 0;
		_ncol = ncol;
		_nrow = nrow;
		return 1;
	}

	/// Reset size.
	int resize(size_t nrow, size_t ncol)
	{
		size_t size = nrow * ncol;
   		if (size == 0) return 0;
		if (size == ndat()) {
			_nrow = nrow;
			_ncol = ncol;
			return 1;
		}
		T *data = new T[size];
		if (!data) return 0;
		delete[] _data;
		_data = data;
		_nrow = nrow;
		_ncol = ncol;
		return 1;
	}

	int reserve(size_t size)
	{
		if (ndat() >= size) return 1;
		resize(size, 1);
		return 1;
	}

	/// Range.
	int fill(T t0, T dt)
	{
		size_t i, n = ndat();
		for (i = 0; i < n; i++) _data[i] = t0 + dt * i;
		return 1;
	}

	/// Negate.
	int negate()
	{
		size_t i, n = ndat();
		for (i = 0; i < n; i++) _data[i] = -_data[i];
		return 1;
	}

	/// Increment
	int increment()
	{
		size_t i, n = ndat();
		for (i = 0; i < n; i++) _data[i]++;
		return 1;
	}

	/// Decrement
	int decrement()
	{
		size_t i, n = ndat();
		for (i = 0; i < n; i++) _data[i]--;
		return 1;
	}

	/// Invert.
	int invert()
	{
		size_t i, n = ndat();
		for (i = 0; i < n; i++) {
			if (_data[i] == 0) return 0;
			_data[i] = 1 / _data[i];
		}
		return 1;
	}

	/// Add
	int add(T v)
	{
		if (v == 0) return 1;
		size_t i, n = ndat();
		for (i = 0; i < n; i++) _data[i] += v;
		return 1;
	}

	int add(const zoMatrixTmp<T> & mat)
	{
		size_t i, n = ndat();
		if (mat.ndat() != n) return 0;
		for (i = 0; i < n; i++) _data[i] += mat(i);
		return 1;
	}

	/// Subtract
	int sub(T v)
	{
		if (v == 0) return 1;
		size_t i, n = ndat();
		for (i = 0; i < n; i++) _data[i] -= v;
		return 1;
	}

	int sub(const zoMatrixTmp<T> & mat)
	{
		size_t i, n = ndat();
		if (mat.ndat() != n) return 0;
		for (i = 0; i < n; i++) _data[i] -= mat(i);
		return 1;
	}

	/// Multiply
	int mul(T v)
	{
		if (v == 1) return 1;
		size_t i, n = ndat();
		for (i = 0; i < n; i++) _data[i] *= v;
		return 1;
	}
	
	int mul(const zoMatrixTmp<T> & mat)
	{
		size_t i, n = ndat();
		if (mat.ndat() != n) return 0;
		for (i = 0; i < n; i++) _data[i] *= mat(i);
		return 1;
	}

	/// Divide
	int div(T v)
	{
		if (v == 1) return 1;
		if (v == 0) return 0;
		size_t i, n = ndat();
		for (i = 0; i < n; i++) _data[i] /= v;
		return 1;
	}

	int div(const zoMatrixTmp<T> & mat)
	{
		size_t i, n = ndat();
		if (mat.ndat() != n) return 0;
		for (i = 0; i < n; i++) {
			if (mat(i) == 0) return 0;
			_data[i] /= mat(i);
		}
		return 1;
	}
	
	/// Count elements that equal v
	int counteq(T v) const
	{
		size_t i, m = 0, n = ndat();
		for (i = 0; i < n; i++) {
			if (_data[i] == v) m++;
		}
		return m;
	}

	/// Count elements that do not equal v
	int countne(T v) const
	{
		size_t i, m = 0, n = ndat();
		for (i = 0; i < n; i++) {
			if (_data[i] != v) m++;
		}
		return m;
	}

	/// Get the index of  non-zero elements in both matays
	int and(zoCHAR & idx, const zoMatrixTmp<T> & mat) const
	{
		size_t  i, n = ndat();
		if (mat.ndat() != n) return 0;
		idx.resize(_nrow, _ncol);
		for (i = 0; i < n; i++) {
			if (_data[i] != 0 && mat(i) != 0) {
				idx(i) = 1;
			}
			else {
				idx(i) = 0;
			}
		}
		return 1;
	}

	/// Get the index of  non-zero elements in either matays 
	int or(zoCHAR & idx, const zoMatrixTmp<T> & mat) const
	{
		size_t  i, n = ndat();
		if (mat.ndat() != n) return 0;
		idx.resize(_nrow, _ncol);
		for (i = 0; i < n; i++) {
			if (_data[i] != 0 || mat(i) != 0) {
				idx(i) = 1;
			}
			else {
				idx(i) = 0;
			}
		}
		return 1;
	}

	/// Get the index of exclusive elements between the matays
	int xor(zoCHAR & idx, const zoMatrixTmp<T> & mat) const
	{
		size_t  i, n = ndat();
		if (mat.ndat() != n) return 0;
		idx.resize(_nrow, _ncol);
		for (i = 0; i < n; i++) {
			if (_data[i] == 0) {
				if (mat(i) != 0) {
					idx(i) = 1;
				}
				else {
					idx(i) = 0;
				}
			}
			else {
				if (mat(i) == 0) {
					idx(i) = 1;
				}
				else {
					idx(i) = 0;
				}
			}
		}
		return 1;
	}

	/// Get the index of elements that equals the input matay
	int eq(zoCHAR& idx, const zoMatrixTmp<T> & mat) const
	{
		size_t i, n = ndat();
		if (mat.ndat() != n) return 0;
		idx.resize(_nrow, _ncol);
		for (i = 0; i < n; i++) {
			if (_data[i] == mat(i)) {
				idx(i) = 1;
			}
			else {
				idx(i) = 0;
			}
		}
		return 1;
	}

	int eq(zoCHAR& idx, T v) const
	{
		size_t i, n = ndat();
		idx.resize(_nrow, _ncol);
		for (i = 0; i < n; i++) {
			if (_data[i] == v) {
				idx(i) = 1;
			}
			else {
				idx(i) = 0;
			}
		}
		return 1;
	}

	/// Get the index of elements that do not equal the input matay
	int ne(zoCHAR& idx, const zoMatrixTmp<T> & mat) const
	{
		size_t i, n = ndat();
		if (mat.ndat() != n) return 0;
		idx.resize(_nrow, _ncol);
		for (i = 0; i < n; i++) {
			if (_data[i] != mat(i)) {
				idx(i) = 1;
			}
			else {
				idx(i) = 0;
			}
		}
		return 1;
	}

	int ne(zoCHAR& idx, T v) const
	{
		size_t i, n = ndat();
		idx.resize(_nrow, _ncol);
		for (i = 0; i < n; i++) {
			if (_data[i] != v) {
				idx(i) = 1;
			}
			else {
				idx(i) = 0;
			}
		}
		return 1;
	}

	/// Get the index of elements that are less than the input matay
	int less(zoCHAR& idx, const zoMatrixTmp<T> & mat) const
	{
		size_t i, n = ndat();
		if (mat.ndat() != n) return 0;
		idx.resize(_nrow, _ncol);
		for (i = 0; i < n; i++) {
			if (_data[i] < mat(i)) {
				idx(i) = 1;
			}
			else {
				idx(i) = 0;
			}
		}
		return 1;
	}

	int less(zoCHAR& idx, T v) const
	{
		size_t i, n = ndat();
		idx.resize(_nrow, _ncol);
		for (i = 0; i < n; i++) {
			if (_data[i] < v) {
				idx(i) = 1;
			}
			else {
				idx(i) = 0;
			}
		}
		return 1;
	}

	/// Get the index of elements that are less than or equal to the input matay
	int leq(zoCHAR& idx, const zoMatrixTmp<T> & mat) const
	{
		size_t i, n = ndat();
		if (mat.ndat() != n) return 0;
		idx.resize(_nrow, _ncol);
		for (i = 0; i < n; i++) {
			if (_data[i] <= mat(i)) {
				idx(i) = 1;
			}
			else {
				idx(i) = 0;
			}
		}
		return 1;
	}

	int leq(zoCHAR& idx, T v) const
	{
		size_t i, n = ndat();
		idx.resize(_nrow, _ncol);
		for (i = 0; i < n; i++) {
			if (_data[i] <= v) {
				idx(i) = 1;
			}
			else {
				idx(i) = 0;
			}
		}
		return 1;
	}

	/// Get the index of elements that are greater than the input matay
	int greater(zoCHAR& idx, const zoMatrixTmp<T> & mat) const
	{
		size_t i, n = ndat();
		if (mat.ndat() != n) return 0;
		idx.resize(_nrow, _ncol);
		for (i = 0; i < n; i++) {
			if (_data[i] > mat(i)) {
				idx(i) = 1;
			}
			else {
				idx(i) = 0;
			}
		}
		return 1;
	}

	int greater(zoCHAR& idx, T v) const
	{
		size_t i, n = ndat();
		idx.resize(_nrow, _ncol);
		for (i = 0; i < n; i++) {
			if (_data[i] > v) {
				idx(i) = 1;
			}
			else {
				idx(i) = 0;
			}
		}
		return 1;
	}

	/// Get the index of elements that are greater than or equal to the input matay
	int geq(zoCHAR& idx, const zoMatrixTmp<T> & mat) const
	{
		size_t i, n = ndat();
		if (mat.ndat() != n) return 0;
		idx.resize(_nrow, _ncol);
		for (i = 0; i < n; i++) {
			if (_data[i] >= mat(i)) {
				idx(i) = 1;
			}
			else {
				idx(i) = 0;
			}
		}
		return 1;
	}

	int geq(zoCHAR& idx, T v) const
	{
		size_t i, n = ndat();
		idx.resize(_nrow, _ncol);
		for (i = 0; i < n; i++) {
			if (_data[i] >= v) {
				idx(i) = 1;
			}
			else {
				idx(i) = 0;
			}
		}
		return 1;
	}

	/// Replace zoMatrixTmp elements by index
	int setidx(const zoCHAR & idx, const zoMatrixTmp<T> & mat)
	{
		size_t  i, n = ndat(), m = mat.ndat();
		if (idx.ndat() != n) return 0;
		if (m == n) {
			for (i = 0; i < n; i++) {
				if (idx(i)) _data[i] = mat(i);
    		}
		}
		else {
			size_t k = 0;
			for (i = 0; i < n; i++) {
				if (idx(i)) {
					if (k >= m) return 0;
					_data[i] = mat(k++);
				}
    		}
		}
		return 1;
	}

	int setidx(const zoCHAR & idx, T v)
	{
		size_t  i, n = ndat();
		if (idx.ndat() != n) return 0;
		for (i = 0; i < n; i++) {
			if (idx(i)) _data[i] = v;
		}
		return 1;
	}

	/// Get zoMatrixTmp elements by index
	int getidx(zoMatrixTmp<T> & mat, const zoCHAR & idx) const
	{
		size_t n = idx.ndat();
		if (ndat() != n) return 0;
		size_t count = idx.countne(0);
		if (count > 0) {
			mat.resize(count, 1);
			count = 0;
			for (size_t i = 0; i < n; i++) {
				if (idx(i)) {
					mat(count++) = _data[i];
				}
			}
		}
		return 1;
	}

	/// Set block
	int setblc(const zoMatrixTmp<T> & mat, size_t i, size_t j)
	{
		if (i + mat.nrow() > nrow() || j + mat.ncol() > ncol()) return 0;
		for (size_t k = 0; k < mat.nrow(); k++) {
			for (size_t l = 0; l < mat.ncol(); l++) (*this)(i+k, j+l) = mat(k, l);
		}
		return 1;
	}

	/// Get block
	int getblc(zoMatrixTmp<T> & mat, size_t i, size_t m, size_t j, size_t n) const
	{
		if (i + m > nrow() || j + n > ncol()) return 0;
		mat.resize(m, n);
		for (size_t k = 0; k < m; k++) {
			for (size_t l = 0; l < n; l++) mat(k, l) = (*this)(i+k, j+l);
		}
		return 1;
	}

	/// Replace a column
	int setcol(const zoMatrixTmp<T> & mat, const size_t j)
	{
		size_t  i;
		if (j >= ncol() || mat.ndat() != nrow()) return 0;
		for (i = 0; i < nrow(); i++) (*this)(i, j) = mat(i);
		return 1;
	}

	int setcol(T v, const size_t j)
	{
		size_t  i;
		if (j >= ncol()) return 0;
		for (i = 0; i < nrow(); i++) (*this)(i, j) = v;
		return 1;
	}

	/// Replace a row
	int setrow(const zoMatrixTmp<T> & mat, const size_t i)
	{
		size_t  j;
		if (i >= nrow() || mat.ndat() != ncol()) return 0;
		for (j = 0; j < ncol(); j++) (*this)(i, j) = mat(j);
		return 1;
	}

	int setrow(T v, const size_t i)
	{
		size_t  j;
		if (i >= nrow()) return 0;
		for (j = 0; j < ncol(); j++) (*this)(i, j) = v;
		return 1;
	}

	/// Get a column
	int getcol(zoMatrixTmp<T> & mat, const size_t j) const
	{
		if (j >= ncol()) return 0;
		mat.resize(nrow(), 1);
		for (size_t i = 0; i < nrow(); i++) mat(i) = (*this)(i, j);
		return 1;
	}

	/// Get a row
	int getrow(zoMatrixTmp<T> & mat, const size_t i) const
	{
		if (i >= nrow()) return 0;
		mat.resize(1, ncol());
		for (size_t j = 0; j < ncol(); j++) mat(j) = (*this)(i, j);
		return 1;
	}

	/// Transpose.
	int transpose()
	{
		size_t i, j;
		if (nrow() == ncol()) {
			for (j = 0; j < ncol(); j++) {
				for (i = j+1; i < nrow(); i++) {
					T v = (*this)(i, j);
					(*this)(i, j) = (*this)(j, i);
					(*this)(j, i) = v;
				}
			}
		}
		else {
			zoMatrixTmp<T> tmp = (*this);
			resize(ncol(), nrow());
			for (j = 0; j < tmp.ncol(); j++) {
				for (i = 0; i < tmp.nrow(); i++) (*this)(j, i) = tmp(i, j);
			}
		}
		return 1;
	}

	zoMatrixTmp<T> * transpose2()
	{
		zoMatrixTmp<T> *ret = new zoMatrixTmp<T>;
		ret->resize(ncol(), nrow());
		for (size_t j = 0; j < ncol(); j++) {
			for (size_t i = 0; i < nrow(); i++) (*ret)(j, i) = (*this)(i, j);
		}
		return ret;
	}

	/// Swap columns
	int swapcol(const size_t i, const size_t j)
	{
		if (i >= ncol() || j >= ncol()) return 0;
		for (size_t k = 0; k < nrow(); k++) {
			T value = (*this)(k, i);
			(*this)(k, i) = (*this)(k, j);
			(*this)(k, j) = value;
		}
		return 1;
	}

	/// Swap rows
	int swaprow(const size_t i, const size_t j)
	{
		if (i >= nrow() || j >= nrow()) return 0;
		for (size_t k = 0; k < ncol(); k++) {
			T value = (*this)(i, k);
			(*this)(i, k) = (*this)(j, k);
			(*this)(j, k) = value;
		}
		return 1;
	}

	/// Flip columns
	int flipcol()
	{
		size_t k, n = ncol() / 2;
		for (k = 0; k < n; k++) swapcol(k, ncol() - k - 1);
		return 1;
	}

	/// Flip rows
	int fliprow()
	{
		size_t k, n = nrow() / 2;
		for (k = 0; k < n; k++) swaprow(k, nrow() - k - 1);
		return 1;
	}

	/// Shift columns
	int shiftcol(int n)
	{
		if (n < 0) n = ncol() + n;		// shift right
		if (n == 0 || n == ncol()) return 1;
		if (n > ncol()) return 0;

		zoMatrixTmp<T> tmp = (*this), p;
		size_t k;

		for (k = n; k < ncol(); k++) {
			tmp.getcol(p, k);
			setcol(p, k - n);
		}
		for (k = 0; k < n; k++) {
			tmp.getcol(p, k);
			setcol(p, ncol() - n + k);
		}

		return 1;
	}

	/// Shift rows
	int shiftrow(int n)
	{
		if (n < 0) n = nrow() + n;		// shift down
		if (n == 0 || n == nrow()) return 1;
		if (n > nrow()) return 0;

		zoMatrixTmp<T> tmp = (*this), p;
		size_t k;

		for (k = n; k < nrow(); k++) {
			tmp.getrow(p, k);
			setrow(p, k - n);
		}
		for (k = 0; k < n; k++) {
			tmp.getrow(p, k);
			setrow(p, nrow() - n + k);
		}

		return 1;
	}

	/// Insert column
	int insertcol(const zoMatrixTmp<T> & mat, const size_t j)
	{
		if (mat.nrow() != nrow() || j > ncol()) return 0;
		zoMatrixTmp<T> tmp = (*this), p;
		resize(nrow(), ncol() + mat.ncol());
		size_t k;
		for (k = 0; k < j; k++) {
			tmp.getcol(p, k);
			setcol(p, k);
		}
		for (k = 0; k < mat.ncol(); k++) {
			mat.getcol(p, k);
			setcol(p, k + j);
		}
		for (k = j; k < tmp.ncol(); k++) {
			tmp.getcol(p, k);
			setcol(p, k + mat.ncol());
		}
		return 1;
	}

	int insertcol(T v, const size_t j)
	{
		if (j > ncol()) return 0;
		zoMatrixTmp<T> tmp = (*this), p;
		resize(nrow(), ncol() + 1);
		size_t k;
		for (k = 0; k < j; k++) {
			tmp.getcol(p, k);
			setcol(p, k);
		}
		setcol(v, j);
		for (k = j; k < tmp.ncol(); k++) {
			tmp.getcol(p, k);
			setcol(p, k + 1);
		}
		return 1;
	}
	
	/// Insert row
	int insertrow(const zoMatrixTmp<T> & mat, const size_t i)
	{
		if (mat.ncol() != ncol() || i > nrow()) return 0;
		zoMatrixTmp<T> tmp = (*this), p;
		resize(nrow() + mat.nrow(), ncol());
		size_t k;
		for (k = 0; k < i; k++) {
			tmp.getrow(p, k);
			setrow(p, k);
		}
		for (k = 0; k < mat.nrow(); k++) {
			mat.getrow(p, k);
			setrow(p, k + i);
		}
		for (k = i; k < tmp.nrow(); k++) {
			tmp.getrow(p, k);
			setrow(p, k + mat.nrow());
		}
		return 1;
	}

	int insertrow(const T v, const size_t i)
	{
		if (i > nrow()) return 0;
		zoMatrixTmp<T> tmp = (*this), p;
		resize(nrow() + 1, ncol());
		size_t k;
		for (k = 0; k < i; k++) {
			tmp.getrow(p, k);
			setrow(p, k);
		}
		setrow(v, i);
		for (k = i; k < tmp.nrow(); k++) {
			tmp.getrow(p, k);
			setrow(p, k + 1);
		}
		return 1;
	}
	
	/// Delete column
	int delcol(const size_t j)
	{
		if (ncol() < 2 || j > ncol()) return 0;
		zoMatrixTmp<T> tmp = (*this), p;
		resize(nrow(), ncol()-1);
		size_t k;
		for (k = 0; k < j; k++) {
			tmp.getcol(p, k);
			setcol(p, k);
		}
		for (k = j + 1; k < tmp.ncol(); k++) {
			tmp.getcol(p, k);
			setcol(p, k - 1);
		}
		return 1;
	}

	/// Delete row
	int delrow(const size_t i)
	{
		if (nrow() < 2 || i > nrow()) return 0;
		zoMatrixTmp<T> tmp = (*this), p;
		resize(nrow()-1 , ncol());
		size_t k;
		for (k = 0; k < i; k++) {
			tmp.getrow(p, k);
			setrow(p, k);
		}
		for (k = i + 1; k < tmp.nrow(); k++) {
			tmp.getrow(p, k);
			setrow(p, k-1);
		}
		return 1;
	}

	/// Select rows
	int selectrow(const zoMatrixTmp<T> & mat, size_t i, size_t k)
	{
		if (i >= mat.nrow()) return 0;
		size_t n = 1 + (mat.nrow() - i - 1) / k;
		resize(n, mat.ncol());
		n = 0;
		zoMatrixTmp<T> p;
		while (i < mat.nrow()) {
			mat.getrow(p, i);
			setrow(p, n);
			n++;
			i += k;
		}
		return 1;
	}

	/// Select columns
	int selectcol(const zoMatrixTmp<T> & mat, size_t i, size_t k)
	{
		if (i >= mat.ncol()) return 0;
		size_t n = 1 + (mat.ncol() - i - 1) / k;
		resize(mat.nrow(), n);
		n = 0;
		zoMatrixTmp<T> p;
		while (i < mat.ncol()) {
			mat.getcol(p, i);
			setcol(p, n);
			n++;
			i += k;
		}
		return 1;
	}

	/// Concat product
	int concat(const zoMatrixTmp<T> & mat)
	{
		if (mat.nrow() != ncol()) return 0;
		
		zoMatrixTmp<T> tmp = (*this);
		resize(nrow(), mat.ncol());
		fill(static_cast<T>(0), static_cast<T>(0));

		if (mat.ptr() == ptr())
		{
			for (size_t i = 0; i < tmp.nrow(); i++) {
				for (size_t j = 0; j < mat.ncol(); j++) {
					for (size_t k = 0; k < tmp.ncol(); k++) (*this)(i, j) += tmp(i, k) * tmp(k, j);
				}
			}
		}
		else
		{
			for (size_t i = 0; i < tmp.nrow(); i++) {
				for (size_t j = 0; j < mat.ncol(); j++) {
					for (size_t k = 0; k < tmp.ncol(); k++) (*this)(i, j) += tmp(i, k) * mat(k, j);
				}
			}
		}

		return 1;
	}
	
	/// Sort in assending or dessending order.
	int sort(const size_t k, int assending)
	{
		if (k >= ncol()) return 0;
		
		zoMatrixTmp<T> col;
		getcol(col, k);

		zoUINT idx;
		idx.resize(nrow(), 1);

		HeapSort(col.ptr(), idx.ptr(), nrow(), assending);
	
		setcol(col, k);

		for (size_t j = 0; j < ncol(); j++) {
			if (j == k) continue;
			for (size_t i = 0; i < nrow(); i++) {
				col(i) = (*this)(idx(i), j);
			}
			setcol(col, j);
		}

		return 1;
	}

	/// Remove duplicated data in the matay.
	int unique(int assending)
	{
		size_t i, count = 1, n = ndat();
		if (n == 1) return 1;
		HeapSort(_data, 0, n, assending);
		for (i = 1;  i < n; i++) {
			if (_data[i-1] != _data[i]) {
				if (i != 1) _data[count] = _data[i];
				count++;
			}
		}
		resize(count, 1);
		return 1;
	}

	/// Confine
	int confine(T vmin, T vmax)
	{
		size_t k, n = ndat();
		for (k = 0; k < n; k++)
		{
			_data[k] = MIN(vmax, _data[k]);
			_data[k] = MAX(vmin, _data[k]);
		}
		return 1;
	}

	/// Join
	int join(const zoMatrixTmp<T> & mat1, const zoMatrixTmp<T> & mat2, T missing)
	{
		size_t  i, j, k = 0,
			m1 = mat1.nrow(),
			n1 = mat1.ncol(),
			m2 = mat2.nrow(),
			n2 = mat2.ncol();

		zoMatrixTmp<T> idx1, idx2;
		mat1.getcol(idx1, 0);
		mat2.getcol(idx2, 0);

		size_t m = m1, n = n1 + n2 - 1;
		resize(m, n);

		for (i = 0; i < m; i++) {
			for (j = 0; j < n1; j++) (*this)(i, j) = mat1(i, j);
			while (k < m2 && idx2(k) < idx1(i)) k++;
			if (k < m2 && idx2(k) == idx1(i)) {
				for (j = 1; j < n2; j++) (*this)(i, n1+j-1) = mat2(k, j);
			}
			else {
				for (j = 1; j < n2; j++) (*this)(i, n1+j-1) = missing;
			}
		}

		return 1;
	}

	T vmin(T v) const
	{
		size_t k, n = ndat();
		for (k = 0; k < n; k++) v = MIN(v, _data[k]);
		return v;
	}

	T vmax(T v) const
	{
		size_t k, n = ndat();
		for (k = 0; k < n; k++) v = MAX(v, _data[k]);
		return v;
	}

	double sum() const
	{
		double v = 0;
		size_t k, n = ndat();
		for (k = 0; k < n; k++) v += _data[k];
		return v;
	}

	double mean() const
	{
		return sum() / ndat();
	}

	double stdev() const
	{
		double v1 = 0, v2 = 0;
		size_t k, n = ndat();
		if (n == 1) return 0;
		for (k = 0; k < n; k++) {
			v1 += _data[k];
			v2 += _data[k]*_data[k];
		}
		return sqrt((v2-v1*v1/n)/(n-1));
	}

	int print(FILE *fp, const char *format, bool csv) const
	{
		size_t i, j;
		for (i = 0; i < nrow(); i++) {
			for (j = 0; j < ncol()-1; j++) {
				fprintf(fp, format, (*this)(i, j));
				if (csv) fprintf(fp, ",");
			}
			fprintf(fp, format, (*this)(i, j));
			fprintf(fp, "\n");
		}
		return 1;
	}

	/// Parse string to data
	int parse(const size_t i, const char *source)
	{
		if (i >= nrow()) return 0;

		std::istringstream istr(source);
		size_t j = 0;

		while (!istr.eof())
		{
			if (j >= ncol()) return 0;
			T tmp;
			istr >> tmp;
			(*this)(i, j) = tmp;
			j++;
		}

		if (j != nrow()) return 0;

		return 1;
	}

	/// applies function on elements
	int apply1(ApplyFunc1 f)
	{
		size_t k, n = ndat();
		for (k = 0; k < n; k++) _data[k] = f(_data[k]);
		return 1;
	}

	/// applies function on elements
	int apply2(ApplyFunc2 f, T v)
	{
		size_t k, n = ndat();
		for (k = 0; k < n; k++) _data[k] = f(_data[k], v);
		return 1;
	}

	int apply2(ApplyFunc2 f, const zoMatrixTmp<T> & mat)
	{
		size_t k, n = ndat();
		if (mat.ndat() != n) return 0;
		for (k = 0; k < n; k++) _data[k] = f(_data[k], mat(k));
		return 1;
	}
};

class zoCHAR : public zoMatrixTmp<char>
{
public:
	zoCHAR()  { _type = MAT_CHAR; }
	~zoCHAR() { }
};

class zoUCHAR : public zoMatrixTmp<unsigned char>
{
public:
	zoUCHAR()  { _type = MAT_UCHAR; }
	~zoUCHAR() { }
};

class zoSHORT : public zoMatrixTmp<short>
{
public:
	zoSHORT()  { _type = MAT_SHORT; }
	~zoSHORT() { }
};

class zoUSHORT : public zoMatrixTmp<unsigned short>
{
public:
	zoUSHORT()  { _type = MAT_USHORT; } 
	~zoUSHORT() { }
};

class zoINT : public zoMatrixTmp<int>
{
public:
	zoINT()  { _type = MAT_INT; }
	~zoINT() { }
};

class zoUINT : public zoMatrixTmp<unsigned int>
{
public:
	zoUINT()  { _type = MAT_UINT; }
	~zoUINT() { }
};

class zoINT64 : public zoMatrixTmp<__int64>
{
public:
	zoINT64()  { _type = MAT_INT64; }
	~zoINT64() { }
};

class zoFLOAT : public zoMatrixTmp<float>
{
public:
	zoFLOAT()  { _type = MAT_FLOAT; }
	~zoFLOAT() { }
};

class zoDOUBLE : public zoMatrixTmp<double>
{
public:
	zoDOUBLE()  { _type = MAT_DOUBLE; }
	~zoDOUBLE() { }
};

class zoCOMPLEX : public zoMatrixTmp<sComplex>
{
public:
	zoCOMPLEX()  { _type = MAT_COMPLEX; }
	~zoCOMPLEX() { }
};

class zoTRIPLEX : public zoMatrixTmp<sTriplex>
{
public:
	zoTRIPLEX()  { _type = MAT_TRIPLEX; }
	~zoTRIPLEX() { }
};

class zoQUADRUPLE : public zoMatrixTmp<sQuadruple>
{
public:
	zoQUADRUPLE()  { _type = MAT_QUADRUPLE; }
	~zoQUADRUPLE() { }
};


//////////////////////////////////////////////////////////////////////
//
template <typename T>
inline void Swap(T& a, T& b)
{
	T c = a; a = b; b = c;
}

//////////////////////////////////////////////////////////////////////
// For heap sort of assending order
//
template <typename T>
void SiftDown(T v[], size_t idx[], int parent, int n)
{
	// 0 based child-parent relationship
	int child = 2 * parent + 1;
	// find where element being sifted belongs
	while (child < n)
	{
		// use right child if it is bigger
		if (child != n - 1 && v[child] < v[child + 1])
			child++;
		// check if it is done
		if (v[parent] >= v[child])
			break;
		// if not, swap elements and move parent down
		Swap(v[parent], v[child]);
		if (idx) Swap (idx[parent], idx[child]);
		parent = child;
		child = 2 * parent + 1;
	}
}

//////////////////////////////////////////////////////////////////////
// For heap sort of decending order
//
template <typename T>
void SiftUp(T v[], size_t idx[], int parent, int n)
{
	// 0 based child-parent relationship
	int child = 2 * parent + 1;
	// find where element being sifted belongs
	while (child < n)
	{
		// use right child if it is smaller
		if (child != n - 1 && v[child] > v[child + 1])
			child++;
		if (v[parent] <= v[child])
			break;
		// if not, swap elements and move parent down
		Swap(v[parent], v[child]);
		if (idx) Swap (idx[parent], idx[child]);
		parent = child;
		child = 2 * parent + 1;
	}
}

//////////////////////////////////////////////////////////////////////
// Sort v[] in assending or dessending order according to the flag.
// If an index vector is provided, corresponding sorted indices can be
// used to rematange accompany vectors.
//
template <typename T>
void HeapSort(T v[], size_t idx[], int n, int flag = 1)
{
	int i;
	
	if (idx) for (i = 0; i < n; i++) idx[i] = i;

	// construct heap
	for (i = (n - 1) / 2; i >= 0; i--)
	{
		if (flag > 0)
			SiftDown(v, idx, i, n);
		else
			SiftUp(v, idx, i, n);
	}

	for (i = n - 1; i >= 1; i--)
	{
		// put top element in its correct position
		Swap(v[0], v[i]);
		if (idx) Swap (idx[0], idx[i]);
		// rebuild heap
		if (flag > 0)
			SiftDown(v, idx, 0, i);
		else
			SiftUp(v, idx, 0, i);
	}
}

#endif