#ifndef __ZO_MATRIX_TMP__
#define __ZO_MATRIX_TMP__

#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <vector>
#include <exception>

#pragma warning(disable: 4018)
#pragma warning(disable: 4146)
#pragma warning(disable: 4244)

#define MAX(a, b) ((a) > (b) ? (a) : (b))
#define MIN(a, b) ((a) < (b) ? (a) : (b))

enum {
	MAT_NULL = 'MA',
	MAT_CHAR, MAT_UCHAR, MAT_SHORT, MAT_USHORT, MAT_INT, MAT_UINT,
	MAT_FLOAT, MAT_DOUBLE, MAT_TOP
};

extern void *mat_opfunc_char(void *ctx, int nargs, void **args);
extern void *mat_opfunc_uchar(void *ctx, int nargs, void **args);
extern void *mat_opfunc_short(void *ctx, int nargs, void **args);
extern void *mat_opfunc_ushort(void *ctx, int nargs, void **args);
extern void *mat_opfunc_int(void *ctx, int nargs, void **args);
extern void *mat_opfunc_uint(void *ctx, int nargs, void **args);
extern void *mat_opfunc_float(void *ctx, int nargs, void **args);
extern void *mat_opfunc_double(void *ctx, int nargs, void **args);

extern void mat_destroy_char(void *ptr);
extern void mat_destroy_uchar(void *ptr);
extern void mat_destroy_short(void *ptr);
extern void mat_destroy_ushort(void *ptr);
extern void mat_destroy_int(void *ptr);
extern void mat_destroy_uint(void *ptr);
extern void mat_destroy_float(void *ptr);
extern void mat_destroy_double(void *ptr);

//////////////////////////////////////////////////////////////////////
//
template <typename T>
inline void Swap(T& a, T& b)
{
	T c = a; a = b; b = c;
}

//////////////////////////////////////////////////////////////////////
// For heap sort of assending order
//
template <typename T>
void SiftDown(T v[], size_t *idx, int parent, int n)
{
	// 0 based child-parent relationship
	int child = 2 * parent + 1;
	// find where element being sifted belongs
	while (child < n)
	{
		// use right child if it is bigger
		if (child != n - 1 && v[child] < v[child + 1])
			child++;
		// check if it is done
		if (v[parent] >= v[child])
			break;
		// if not, swap elements and move parent down
		Swap(v[parent], v[child]);
		if (idx) Swap (idx[parent], idx[child]);
		parent = child;
		child = 2 * parent + 1;
	}
}

//////////////////////////////////////////////////////////////////////
// For heap sort of decending order
//
template <typename T>
void SiftUp(T v[], size_t *idx, int parent, int n)
{
	// 0 based child-parent relationship
	int child = 2 * parent + 1;
	// find where element being sifted belongs
	while (child < n)
	{
		// use right child if it is smaller
		if (child != n - 1 && v[child] > v[child + 1])
			child++;
		if (v[parent] <= v[child])
			break;
		// if not, swap elements and move parent down
		Swap(v[parent], v[child]);
		if (idx) Swap (idx[parent], idx[child]);
		parent = child;
		child = 2 * parent + 1;
	}
}

//////////////////////////////////////////////////////////////////////
// Sort v[] in assending or dessending order according to the flag.
// If an index vector is provided, corresponding sorted indices can be
// used to re-arrange accompany vectors.
//
template <typename T>
void HeapSort(T v[], size_t *idx, int n, int flag = 1)
{
	int i;
	
	if (idx) {
		for (i = 0; i < n; i++) idx[i] = i;
	}

	// construct heap
	for (i = (n - 1) / 2; i >= 0; i--)
	{
		if (flag > 0)
			SiftDown(v, idx, i, n);
		else
			SiftUp(v, idx, i, n);
	}

	for (i = n - 1; i >= 1; i--)
	{
		// put top element in its correct position
		Swap(v[0], v[i]);
		if (idx) Swap (idx[0], idx[i]);
		// rebuild heap
		if (flag > 0)
			SiftDown(v, idx, 0, i);
		else
			SiftUp(v, idx, 0, i);
	}
}


///
/// Matrix template
///

template<class T>
class zoMatrixTmp
{
private:
	int				_type;

	size_t			_ndat;

	size_t			_ncol;

	size_t			_nrow;

	T				*_data;

public:
	/// Constructor.
	zoMatrixTmp() : _type(MAT_NULL), _nrow(0), _ncol(0), _ndat(0), _data(NULL) { }

	zoMatrixTmp(int type) : _type(type), _nrow(0), _ncol(0), _ndat(0), _data(NULL) { }

	zoMatrixTmp(const zoMatrixTmp & mat) : _type(MAT_NULL), _nrow(0), _ncol(0), _ndat(0), _data(NULL)
	{
		_type = mat.type();
		resize(mat.nrow(), mat.ncol());
		memcpy(_data, mat.ptr(), _ndat*sizeof(T));
	}

	/// Destructor.
	virtual ~zoMatrixTmp()
	{
		delete[] _data;
	}

	/// Member access
	int type() const { return _type; }

	size_t ndat() const { return _ndat; }

	size_t nrow() const { return _nrow; }

	size_t ncol() const { return _ncol; }

	const T * ptr() const { return _data; }
		  T * ptr()       { return _data; }

	const T & operator()(size_t i) const { return _data[i]; }
		  T & operator()(size_t i)       { return _data[i]; }

	const T & operator()(size_t i, size_t j) const { return _data[i*_ncol+j]; }
		  T & operator()(size_t i, size_t j)       { return _data[i*_ncol+j]; }

	void resize(size_t nrow, size_t ncol)
	{
		size_t ndat = nrow*ncol;
		if (ndat > 0) {
			try {
				T *ptr = new T[ndat];
				if (_data) {
					size_t nd = ndat < _ndat ? ndat : _ndat;
					memcpy(ptr, _data, nd*sizeof(T));
					delete[] _data;
				}
				_data = ptr;
				_nrow = nrow;
				_ncol = ncol;
				_ndat = ndat;
			}
			catch (std::bad_alloc& err) {
				throw("failed to allocate memory");
			}
		}
		else {
			throw("nrow<=0 or ncol<=0");
		}
	}

	void reshape(size_t nrow, size_t ncol)
	{
		if (ndat() == nrow*ncol) {
			_nrow = nrow;
			_ncol = ncol;
		}
	}

	size_t countne(T v) const
	{
		size_t count=0;
		for (size_t i = 0; i < ndat(); i++) {
			if (_data[i] != v) count++;
		}
		return count;
	}

	void getcol(const zoMatrixTmp<T> &mat, size_t j)
	{
		resize(mat.nrow(), 1);
		for (size_t i = 0; i < nrow(); i++) _data[i] = mat(i,j);
	}

	void setcol(const zoMatrixTmp<T> &mat, size_t j)
	{
		size_t n = MIN(nrow(), mat.ndat());
		for (size_t i = 0; i < n; i++) (*this)(i,j) = mat(i);
	}

	void fill(T v0, T dv)
	{
		for (size_t k = 0; k < ndat(); k++) (*this)(k) = v0 + dv*k;
	}

	void trans()
	{
		size_t i, j;
		if (nrow() == ncol()) {
			for (j = 0; j < ncol(); j++) {
				for (i = j+1; i < nrow(); i++) {
					T v = (*this)(i,j);
					(*this)(i,j) = (*this)(j,i);
					(*this)(j,i) = v;
				}
			}
		}
		else {
			zoMatrixTmp<T> tmp(*this);
			reshape(ncol(), nrow());
			for (j = 0; j < tmp.ncol(); j++) {
				for (i = 0; i < tmp.nrow(); i++) (*this)(j,i) = tmp(i,j);
			}
		}
	}

	void concat(const zoMatrixTmp<T> & mat)
	{
		if (mat.nrow() != ncol()) return;
		zoMatrixTmp<T> tmp = (*this);
		resize(nrow(), mat.ncol());
		memset(_data, 0, ndat()*sizeof(T));
		for (size_t i = 0; i < tmp.nrow(); i++) {
			for (size_t j = 0; j < mat.ncol(); j++) {
				for (size_t k = 0; k < tmp.ncol(); k++) (*this)(i,j) += tmp(i,k)*mat(k,j);
			}
		}
	}

	int print(const char *fname, const char *mode, const char *format, bool csv) const
	{
		FILE *fp = stdout;
		if (fname) {
			fp = fopen(fname, mode);
			if (!fp) return 0;
		}
		size_t i, j;
		for (i = 0; i < nrow(); i++) {
			for (j = 0; j < ncol()-1; j++) {
				fprintf(fp, format, (*this)(i,j));
				if (csv) fprintf(fp, ",");
			}
			fprintf(fp, format, (*this)(i,j));
			fprintf(fp, "\n");
		}
		fflush(fp);
		if (fp != stdout) fclose(fp);
		return 1;
	}

	void clone(void *ptr, int type)
	{
		switch (type) {
		case MAT_CHAR:
			{
			zoMatrixTmp<char> *src = reinterpret_cast<zoMatrixTmp<char>*>(ptr);
			resize(src->nrow(), src->ncol());
			for (size_t i = 0; i < ndat(); i++) _data[i] = (*src)(i);
			}
			break;
		case MAT_UCHAR:
			{
			zoMatrixTmp<unsigned char> *src = reinterpret_cast<zoMatrixTmp<unsigned char>*>(ptr);
			resize(src->nrow(), src->ncol());
			for (size_t i = 0; i < ndat(); i++) _data[i] = (*src)(i);
			}
			break;
		case MAT_SHORT:
			{
			zoMatrixTmp<short> *src = reinterpret_cast<zoMatrixTmp<short>*>(ptr);
			resize(src->nrow(), src->ncol());
			for (size_t i = 0; i < ndat(); i++) _data[i] = (*src)(i);
			}
			break;
		case MAT_USHORT:
			{
			zoMatrixTmp<unsigned short> *src = reinterpret_cast<zoMatrixTmp<unsigned short>*>(ptr);
			resize(src->nrow(), src->ncol());
			for (size_t i = 0; i < ndat(); i++) _data[i] = (*src)(i);
			}
			break;
		case MAT_INT:
			{
			zoMatrixTmp<int> *src = reinterpret_cast<zoMatrixTmp<int>*>(ptr);
			resize(src->nrow(), src->ncol());
			for (size_t i = 0; i < ndat(); i++) _data[i] = (*src)(i);
			}
			break;
		case MAT_UINT:
			{
			zoMatrixTmp<unsigned int> *src = reinterpret_cast<zoMatrixTmp<unsigned int>*>(ptr);
			resize(src->nrow(), src->ncol());
			for (size_t i = 0; i < ndat(); i++) _data[i] = (*src)(i);
			}
			break;
		case MAT_FLOAT:
			{
			zoMatrixTmp<float> *src = reinterpret_cast<zoMatrixTmp<float>*>(ptr);
			resize(src->nrow(), src->ncol());
			for (size_t i = 0; i < ndat(); i++) _data[i] = (*src)(i);
			}
			break;
		case MAT_DOUBLE:
			{
			zoMatrixTmp<double> *src = reinterpret_cast<zoMatrixTmp<double>*>(ptr);
			resize(src->nrow(), src->ncol());
			for (size_t i = 0; i < ndat(); i++) _data[i] = (*src)(i);
			}
			break;
		default:
			break;
		}
	}
};

#endif