#ifndef __MATRIX_IMPLEMENT__
#define __MATRIX_IMPLEMENT__

#include <string>

#include "api.h"
#include "matrix-new.h"
#include "matrixtmp-new.h"
#include "scan.h"

void *MAT_FUNC(opfunc)(void *ctx, int nargs, void** args);

void MAT_FUNC(destroy)(void* mat)
{
	delete reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(mat);
	mat = 0;
}

void *MAT_FUNC(create)(void *ctx, int nargs, void** args)
{
	zoMatrixTmp<DAT_TYPE> *P = new zoMatrixTmp<DAT_TYPE>(MAT_TYPE);
	if (nargs == 1) {
		if (api_is_user(args[0])) {
			int type = api_get_type(args[0]);
			if (type > MAT_NULL && type < MAT_TOP) {
				P->clone(api_get_ptr(ctx,args[0]), type);
			}
			else {
				matrix_error(ctx);
			}
		}
		else {
			size_t ncol = api_get_integer(ctx,args[0]);
			if (ncol == 0) size_error(ctx);
			P->resize(1, ncol);
		}
	}
	else if (nargs == 2) {
		size_t nrow = api_get_integer(ctx,args[0]);
		size_t ncol = api_get_integer(ctx,args[1]);
		if (nrow == 0 || ncol == 0) size_error(ctx);
		P->resize(nrow, ncol);
	}
	else if (nargs > 2) {
		P->resize(1, nargs);
		for (int i = 0; i < nargs; i++) (*P)(i) = api_get_number(ctx, args[i]);
	}
	return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
}

void *MAT_FUNC(csv)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	if (me->ndat() == 0) return 0;
#ifdef MAT_INTEGER
	const char* format = "  %d";
#endif
#ifdef MAT_UNSIGNED
	const char* format = "  %u";
#endif
#ifdef MAT_REAL
	const char* format = "  %.9g";
	if (me->type()==MAT_FLOAT) format = "  %.7g";
#endif
	const char *fname=0, *mode="w";
	if (nargs > 1) format = api_get_string(ctx, args[1]);
	if (nargs > 2) fname = api_get_string(ctx, args[2]);
	if (nargs > 3) mode = api_get_string(ctx, args[3]);
	if (!me->print(fname, mode, format, true)) {
		api_runtime_error2(ctx, "failed to open file for output: ", fname);
	}
	return 0;
}

void *MAT_FUNC(print)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	if (me->ndat() == 0) return 0;
#ifdef MAT_INTEGER
	const char* format = "  %d";
#endif
#ifdef MAT_UNSIGNED
	const char* format = "  %u";
#endif
#ifdef MAT_REAL
	const char* format = "  %.9g";
	if (me->type()==MAT_FLOAT) format = "  %.7g";
#endif
	const char *fname=0, *mode="w";
	if (nargs > 1) format = api_get_string(ctx, args[1]);
	if (nargs > 2) fname = api_get_string(ctx, args[2]);
	if (nargs > 3) mode = api_get_string(ctx, args[3]);
	if (!me->print(fname, mode, format, false)) {
		api_runtime_error2(ctx, "failed to open file for output: ", fname);
	}
	return 0;
}

void *MAT_FUNC(fill)(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	DAT_TYPE v0 = api_get_number(ctx, args[1]);
	DAT_TYPE dv = 0;
	if (nargs > 2) dv = api_get_number(ctx, args[2]);
	size_t k;
	if (dv == 0) {
		for (k = 0; k < me->ndat(); k++) (*me)(k) = v0;
	}
	else {
		for (k = 0; k < me->ndat(); k++) (*me)(k) = v0 + dv*k;
	}
	return 0;
}

void* MAT_FUNC(size)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	void* arr = api_create_array(ctx, 3);
	api_set_array_object(ctx, arr, "0", api_create_integer(0, me->nrow()));
	api_set_array_object(ctx, arr, "1", api_create_integer(0, me->ncol()));
	api_set_array_object(ctx, arr, "2", api_create_integer(0, me->ndat()));
	return arr;
}

void *MAT_FUNC(ptr)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	int offset = 0;
	if (nargs > 1) {
		offset = api_get_integer(ctx, args[1]);
		if (offset < 0 || offset >= me->ndat()) api_input_error(ctx);
	}
	void* arr = api_create_array(ctx, 3);
	api_set_array_object(ctx, arr, "0", api_create_user(0, me->ptr()+offset, 0, 0, 0));
	api_set_array_object(ctx, arr, "1", api_create_integer(0, me->ndat()));
	api_set_array_object(ctx, arr, "2", api_create_integer(0, sizeof(DAT_TYPE)));
	return arr;
}

void *MAT_FUNC(reshape)(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	size_t nrow = api_get_integer(ctx, args[1]);
	size_t ncol = api_get_integer(ctx, args[2]);
	if (nrow*ncol != me->ndat()) api_runtime_error(ctx, "cannot reshape to different size");
	me->reshape(nrow, ncol);
	return 0;
}

void *MAT_FUNC(resize)(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	size_t nrow = api_get_integer(ctx, args[1]);
	size_t ncol = 1;
	if (nargs > 2) ncol = api_get_integer(ctx, args[2]);
	if (nrow == 0 || ncol == 0) size_error(ctx);
	me->resize(nrow, ncol);
	return 0;
}

void *MAT_FUNC(reserve)(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	size_t n = api_get_integer(ctx, args[1]);
	if (n > me->ndat()) me->resize(n, 1);
	return 0;
}

void *MAT_FUNC(flip)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	bool column = true;
	if (nargs > 1) column = check_rcflag(ctx, args[1]);
	if (column) {
		size_t i, j1, j2, n;
		n = me->ncol()/2;
		for (i = 0; i < me->nrow(); i++) {
			for (j1 = 0, j2 = me->ncol()-1; j1 < n; j1++, j2--) {
				DAT_TYPE v = (*me)(i,j1);
				(*me)(i,j1) = (*me)(i,j2);
				(*me)(i,j2) = v;
			}
		}
	}
	else {
		size_t i1, i2, j, n;
		n = me->nrow()/2;
		for (j = 0; j < me->ncol(); j++) {
			for (i1 = 0, i2 = me->nrow()-1; i1 < n; i1++, i2--) {
				DAT_TYPE v = (*me)(i1,j);
				(*me)(i1,j) = (*me)(i2,j);
				(*me)(i2,j) = v;
			}
		}
	}
	return 0;
}

void *MAT_FUNC(insert)(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	size_t id = api_get_integer(ctx, args[1]);
	bool column = true;
	if (nargs > 3) column = check_rcflag(ctx, args[3]);
	size_t i, j, l;
	zoMatrixTmp<DAT_TYPE> tmp(*me);
	if (api_is_user(args[2])) {
		zoMatrixTmp<DAT_TYPE> *src = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[2], MAT_TYPE));
		if (column) {
			if (src->nrow() != me->nrow()) size_error(ctx);
			if (id > me->ncol()) index_error(ctx);
			me->resize(tmp.nrow(), tmp.ncol()+src->ncol());
			for (i = 0; i < me->nrow(); i++) {
				for (j = 0; j < id; j++) (*me)(i,j) = tmp(i,j);
				for (l = 0; l < src->ncol(); j++, l++) (*me)(i,j) = (*src)(i,l);
				for (l = id; l < tmp.ncol(); j++, l++) (*me)(i,j) = tmp(i,l);
			}
		}
		else {
			if (src->ncol() != me->ncol()) size_error(ctx);
			if (id > me->nrow()) size_error(ctx);
			me->resize(tmp.nrow()+src->nrow(), tmp.ncol());
			for (j = 0; j < me->ncol(); j++) {
				for (i = 0; i < id; i++) (*me)(i,j) = tmp(i,j);
				for (l = 0; l < src->nrow(); i++, l++) (*me)(i,j) = (*src)(l,j);
				for (l = id; l < tmp.nrow(); i++, l++) (*me)(i,j) = tmp(l,j);
			}
		}
	}
	else {
		DAT_TYPE v = api_get_number(ctx, args[2]);
		if (column) {
			if (id > me->ncol()) index_error(ctx);
			me->resize(tmp.nrow(), tmp.ncol()+1);
			for (i = 0; i < me->nrow(); i++) {
				for (j = 0; j < id; j++) (*me)(i,j) = tmp(i,j);
				(*me)(i,j++) = v;
				for (; j < me->ncol(); j++) (*me)(i,j) = tmp(i,j-1);
			}
		}
		else {
			if (id > me->nrow()) index_error(ctx);
			me->resize(tmp.nrow()+1, tmp.ncol());
			for (j = 0; j < me->ncol(); j++) {
				for (i = 0; i < id; i++) (*me)(i,j) = tmp(i,j);
				(*me)(i++,j) = v;
				for (; i < me->nrow(); i++) (*me)(i,j) = tmp(i-1,j);
			}
		}
	}
	return 0;
}

void* MAT_FUNC(delete)(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	size_t id = api_get_integer(ctx, args[1]);
	bool column = true;
	if (nargs > 2) column = check_rcflag(ctx, args[2]);
	zoMatrixTmp<DAT_TYPE> tmp(*me);
	size_t i, j;
	if (column) {
		if (me->ncol() == 1) api_runtime_error(ctx, "cannot delete the only column");
		if (id >= me->ncol()) index_error(ctx);
		me->resize(tmp.nrow(), tmp.ncol()-1);
		for (i = 0; i < me->nrow(); i++) {
			for (j = 0; j < id; j++) (*me)(i,j) = tmp(i,j);
			for (; j < me->ncol(); j++) (*me)(i,j) = tmp(i,j+1);
		}
	}
	else {
		if (me->nrow() == 1) api_runtime_error(ctx, "cannot delete the only row");
		if (id >= me->nrow()) index_error(ctx);
		me->resize(tmp.nrow()-1, tmp.ncol());
		for (j = 0; j < me->ncol(); j++) {
			for (i = 0; i < id; i++) (*me)(i,j) = tmp(i,j);
			for (; i < me->nrow(); i++) (*me)(i,j) = tmp(i+1,j);
		}
	}
	return 0;
}

void* MAT_FUNC(trans)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	me->trans();
	return 0;
}

void* MAT_FUNC(sort)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	size_t id = 0; 
	if (nargs > 1) {
		id = api_get_integer(ctx, args[1]);
		if (id >= me->ncol()) index_error(ctx);
	}
	int assending = 1;
	if (nargs > 2) assending = api_get_integer(ctx, args[2]);
	zoMatrixTmp<DAT_TYPE> col;
	col.resize(me->nrow(), 1);
	size_t i, j;
	for (i = 0; i < me->nrow(); i++) col(i) = (*me)(i,id);
	zoMatrixTmp<size_t> idx;
	idx.resize(me->nrow(), 1);
	HeapSort(col.ptr(), idx.ptr(), me->nrow(), assending);
	for (i = 0; i < me->nrow(); i++) (*me)(i,id) = col(i);
	for (j = 0; j < me->ncol(); j++) {
		if (j == id) continue;
		for (i = 0; i < me->nrow(); i++) col(i) = (*me)(idx(i),j);
		for (i = 0; i < me->nrow(); i++) (*me)(i,j) = col(i);
	}
	return 0;
}

void* MAT_FUNC(unique)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	int assending = 1;
	if (nargs > 1) assending = api_get_integer(ctx, args[1]);
	HeapSort(me->ptr(), 0, me->ndat(), assending);
	size_t i, count=1;
	for (i = 1;  i < me->ndat(); i++) {
		if ((*me)(i-1) != (*me)(i)) {
			(*me)(count) = (*me)(i);
			count++;
		}
	}
	if (count < me->ndat()) {
		me->resize(count, 1);
	}
	else {
		me->reshape(count, 1);
	}
	return 0;
}

void* MAT_FUNC(min)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	size_t i;
	DAT_TYPE v = DAT_MAX;
	for (i = 0; i < me->ndat(); i++) {
		v = MIN(v, (*me)(i));
	}
#ifdef MAT_REAL
	return api_create_real(ctx, v);
#else
	return api_create_integer(ctx, v);
#endif
}

void* MAT_FUNC(max)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	size_t i;
	DAT_TYPE v = DAT_MIN;
	for (i = 0; i < me->ndat(); i++) {
		v = MAX(v, (*me)(i));
	}
#ifdef MAT_REAL
	return api_create_real(ctx, v);
#else
	return api_create_integer(ctx, v);
#endif
}

void* MAT_FUNC(sum)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	double sum = 0;
	size_t i;
	for (i = 0; i < me->ndat(); i++) {
		sum += (*me)(i);
	}
	return api_create_real(ctx, sum);
}

void* MAT_FUNC(mean)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	double sum = 0;
	size_t i;
	for (i = 0; i < me->ndat(); i++) {
		sum += (*me)(i);
	}
	return api_create_real(ctx, sum/me->ndat());
}

void* MAT_FUNC(stdev)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	if (me->ndat() <= 1) return api_create_real(ctx, 0);
	int stdevp = 0;
	if (nargs>1) stdevp = api_get_integer(ctx,args[1]);
	double v, v1=0, v2=0;
	size_t i;
	for (i = 0; i < me->ndat(); i++) {
		v = (*me)(i);
		v1 += v;
		v2 += v*v;
	}
	v2 -= v1*v1/me->ndat();
	if (v2 <= 0) return api_create_real(ctx, 0);
	if (stdevp)
		v = sqrt(v2/me->ndat());
	else
		v = sqrt(v2/(me->ndat()-1));
	return api_create_real(ctx, v);
}

void* MAT_FUNC(parse)(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	std::string s(api_get_string(ctx, args[1]));
	std::vector<double> tmp;
	parse_string((char*)s.c_str(), tmp, ctx);
	if (tmp.size() == 0) return api_create_integer(ctx, 0);
	me->resize(1, tmp.size());
	for (int i = 0; i < tmp.size(); i++) (*me)(i) = tmp[i];
	return api_create_integer(ctx, tmp.size());
}

void* MAT_FUNC(clone)(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	int type = api_get_type(args[1]);
	if (type > MAT_NULL && type < MAT_TOP) {
		me->clone(api_get_ptr(ctx,args[1]), type);
	}
	else {
		matrix_error(ctx);
	}
	return 0;
}

void* MAT_FUNC(import)(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	DAT_TYPE* ptr = reinterpret_cast<DAT_TYPE*>(api_get_ptr(ctx,args[1]));
	for (size_t i = 0; i < me->ndat(); i++) (*me)(i) = ptr[i];
	return 0;
}

void* MAT_FUNC(find)(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
#ifdef MAT_REAL
	double v = api_get_number(ctx, args[1]);
#else
	int v = api_get_integer(ctx, args[1]);
#endif
	size_t i, j, i0=0, j0=0;
	if (nargs > 2) {
		i0 = api_get_integer(ctx, args[2]);
		if (i0 >= me->nrow()) index_error(ctx);
	}
	if (nargs > 3) {
		j0 = api_get_integer(ctx, args[3]);
		if (j0 >= me->ncol()) index_error(ctx);
	}
	for (i = i0; i < me->nrow(); i++) {
		for (j = j0; j < me->ncol(); j++) {
			if ((*me)(i,j) == v) {
				void *arr = api_create_array(ctx, 2);
				api_set_array_object(ctx, arr, "0", api_create_integer(0, i));
				api_set_array_object(ctx, arr, "1", api_create_integer(0, j));
				return arr;
			}
		}
	}
	return 0;
}

void* MAT_FUNC(replace)(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	zoMatrixTmp<int> *idx = reinterpret_cast<zoMatrixTmp<int>*>(api_get_user(ctx, args[1], MAT_INT));
	if (api_is_user(args[2])) {
		zoMatrixTmp<DAT_TYPE> *m2 = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[2], MAT_TYPE));
		if (idx->ndat() < m2->ndat()) size_error(ctx);
		for (int i = 0; i < m2->ndat(); i++) {
			if ((*idx)(i) < 0 || (*idx)(i) >= me->ndat()) index_error(ctx);
			(*me)((*idx)(i)) = (*m2)(i);
		}
	}
	else {
#ifdef MAT_REAL
		real_t val = api_get_number(ctx, args[2]);
#else
		integer_t val = api_get_integer(ctx, args[2]);
#endif
		for (int i = 0; i < idx->ndat(); i++) {
			if ((*idx)(i) < 0 || (*idx)(i) >= me->ndat()) index_error(ctx);
			(*me)((*idx)(i)) = val;
		}
	}
	return 0;
}

void* MAT_FUNC(abs)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	zoMatrixTmp<DAT_TYPE> *P = new zoMatrixTmp<DAT_TYPE>(*me);
	for (size_t i = 0; i < me->ndat(); i++) {
		(*P)(i) = MAT_ABS((*P)(i));
	}
	return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
}

#ifdef MAT_REAL
void* MAT_FUNC(rand)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	long seed = 10;
	if (nargs > 1) seed = abs(long(api_get_integer(ctx, args[1])));
	seed ^= 123459876;
	for (size_t i = 0; i < me->ndat(); i++) {
		long m = seed/127773;
		seed = 16807*(seed - 127773*m) - 2836*m;
		if (seed < 0) seed += 2147483647;
		(*me)(i) = DAT_TYPE(seed)/DAT_TYPE(2147483647);
	}
	return 0;
}

void* MAT_FUNC(prod)(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	zoMatrixTmp<DAT_TYPE> *m2 = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[1], MAT_TYPE));
	if (m2->nrow() != me->ncol()) size_error(ctx);
	me->concat(*m2);
	return 0;
}
#endif // MAT_REAL

#define APPLY_FUNC(func) \
	if (nargs < 1) api_input_error(ctx); \
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE)); \
	zoMatrixTmp<double> *P = new zoMatrixTmp<double>; \
	P->resize(me->nrow(), me->ncol()); \
	for (size_t i = 0; i < P->ndat(); i++) (*P)(i) = func(double((*me)(i))); \
	return api_create_user(ctx, P, mat_opfunc_double, mat_destroy_double, MAT_DOUBLE); 

void* MAT_FUNC(cos)(void *ctx, int nargs, void** args)
{
	APPLY_FUNC(cos);
}

void* MAT_FUNC(cosh)(void *ctx, int nargs, void** args)
{
	APPLY_FUNC(cosh);
}

void* MAT_FUNC(acos)(void *ctx, int nargs, void** args)
{
	APPLY_FUNC(acos);
}

void* MAT_FUNC(sin)(void *ctx, int nargs, void** args)
{
	APPLY_FUNC(sin);
}

void* MAT_FUNC(sinh)(void *ctx, int nargs, void** args)
{
	APPLY_FUNC(sinh);
}

void* MAT_FUNC(asin)(void *ctx, int nargs, void** args)
{
	APPLY_FUNC(asin);
}

void* MAT_FUNC(tan)(void *ctx, int nargs, void** args)
{
	APPLY_FUNC(tan);
}

void* MAT_FUNC(tanh)(void *ctx, int nargs, void** args)
{
	APPLY_FUNC(tanh);
}

void* MAT_FUNC(atan)(void *ctx, int nargs, void** args)
{
	APPLY_FUNC(atan);
}

void* MAT_FUNC(exp)(void *ctx, int nargs, void** args)
{
	APPLY_FUNC(exp);
}

void* MAT_FUNC(ceil)(void *ctx, int nargs, void** args)
{
	APPLY_FUNC(ceil);
}

void* MAT_FUNC(floor)(void *ctx, int nargs, void** args)
{
	APPLY_FUNC(floor);
}

#define APPLY_FUNC2(func, cmp) \
	if (nargs < 1) api_input_error(ctx); \
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE)); \
	zoMatrixTmp<double> *P = new zoMatrixTmp<double>; \
	P->resize(me->nrow(), me->ncol()); \
	for (size_t i = 0; i < me->ndat(); i++) { \
		double v = (*me)(i); \
		if (v cmp 0) { \
			delete P; \
			api_runtime_error(ctx, "bad matrix value"); \
		} \
		(*P)(i) = func(double(v)); \
	} \
	return api_create_user(ctx, P, mat_opfunc_double, mat_destroy_double, MAT_DOUBLE); 

void* MAT_FUNC(sqrt)(void *ctx, int nargs, void** args)
{
	APPLY_FUNC2(sqrt,<);
}

void* MAT_FUNC(log)(void *ctx, int nargs, void** args)
{
	APPLY_FUNC2(log,<=);
}

void* MAT_FUNC(log10)(void *ctx, int nargs, void** args)
{
	APPLY_FUNC2(log10,<=);
}

#define APPLY_FUNC3(func) \
	zoMatrixTmp<DAT_TYPE> *me; \
	zoMatrixTmp<double> *P; \
	if (api_is_number(args[0])) {	\
		double v = api_get_number(ctx,args[0]); \
		me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[1], MAT_TYPE)); \
		P = new zoMatrixTmp<double>; \
		P->resize(me->nrow(), me->ncol()); \
		for (size_t k = 0; k < me->ndat(); k++) (*P)(k) = func(double(v), double((*me)(k))); \
		return api_create_user(ctx, P, mat_opfunc_double, mat_destroy_double, MAT_DOUBLE); \
	} \
	me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE)); \
	if (api_is_user(args[1])) { \
		zoMatrixTmp<DAT_TYPE> *ma = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE)); \
		if (me->ndat() != ma->ndat()) size_error(ctx); \
		P = new zoMatrixTmp<double>; \
		P->resize(me->nrow(), me->ncol()); \
		for (size_t i = 0; i < me->ndat(); i++) (*P)(i) = func(double((*me)(i)), double((*ma)(i))); \
	} \
	else { \
		double v = api_get_number(ctx,args[1]); \
		P = new zoMatrixTmp<double>; \
		P->resize(me->nrow(), me->ncol()); \
		for (size_t i = 0; i < me->ndat(); i++) (*P)(i) = func(double((*me)(i)), double(v)); \
	} \
	return api_create_user(ctx, P, mat_opfunc_double, mat_destroy_double, MAT_DOUBLE);

void* MAT_FUNC(atan2)(void *ctx, int nargs, void** args)
{
	APPLY_FUNC3(atan2);
}

void* MAT_FUNC(pow)(void *ctx, int nargs, void** args)
{
	APPLY_FUNC3(pow);
}

void* MAT_FUNC(cal2jul)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	size_t i, m=me->nrow(), n=me->ncol();
	if (n < 3) size_error(ctx);
	zoMatrixTmp<double> *P = new zoMatrixTmp<double>;
	P->resize(m, 1);
	for (i = 0; i < m; i++) {
		double d = math_cal2jul((*me)(i,0), (*me)(i,1), (*me)(i,2));
		if (n > 3) {
			d += (*me)(i,3)/24.0;
			if (n > 4) d += (*me)(i,4)/1440.0;
			if (n > 5) d += (*me)(i,5)/86400.0;
		}
		(*P)(i) = d;
	}
	return api_create_user(ctx, P, mat_opfunc_double, mat_destroy_double, MAT_DOUBLE);
}

void* MAT_FUNC(jul2cal)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	size_t i, n = me->ndat();
	zoMatrixTmp<double> *P = new zoMatrixTmp<double>;
	P->resize(n, 6);
	int yy, mm, dd, hh, mn, sc;
	for (i = 0; i < n; i++) {
		double d = (*me)(i);
		math_jul2cal(&yy, &mm, &dd, d);
		math_jul2tim(&hh, &mn, &sc, d);
		(*P)(i,0) = yy;
		(*P)(i,1) = mm;
		(*P)(i,2) = dd;
		(*P)(i,3) = hh;
		(*P)(i,4) = mn;
		(*P)(i,5) = sc;
	}
	return api_create_user(ctx, P, mat_opfunc_double, mat_destroy_double, MAT_DOUBLE);
}

void* MAT_FUNC(__get)(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	size_t i, j, k, l, m, n, idx[3], jdx[3];
	zoMatrixTmp<char> *cidx, *cjdx;
	if (nargs == 2) {
		// A[?]
		if (api_get_type(args[1]) == MAT_CHAR) {
			// A[I]
			zoMatrixTmp<char> *cidx = reinterpret_cast<zoMatrixTmp<char>*>(api_get_ptr(ctx,args[1]));
			if (cidx->ndat() != me->ndat()) index_error(ctx);
			n = cidx->countne(0);
			if (n == 0) return 0;
			zoMatrixTmp<DAT_TYPE> *P = new zoMatrixTmp<DAT_TYPE>;
			P->resize(1, n);
			for (k = 0, i = 0; i < cidx->ndat(); i++) {
				if ((*cidx)(i) != 0) (*P)(k++) = (*me)(i);
			}
			return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
		}
		n = get_array_index(ctx, args[1], me->ndat(), idx);
		if (n == 0) return 0;
		if (n == 1) {
			// A[i]
#ifdef MAT_REAL
			return api_create_real(ctx, (*me)(idx[0]));
#else
			return api_create_integer(ctx, (*me)(idx[0]));
#endif
		}
		if (n == me->ndat()) {
			// A[*]
			return api_create_user(ctx, new zoMatrixTmp<DAT_TYPE>(*me), MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
		}
		// A[a:b:c]
		zoMatrixTmp<DAT_TYPE> *P = new zoMatrixTmp<DAT_TYPE>;
		P->resize(1, n);
		for (k = 0, i = idx[0]; i <= idx[1]; i += idx[2]) (*P)(k++) = (*me)(i);
		return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
	}
	// A[?,?]
	if (api_get_type(args[1]) == MAT_CHAR) {
		cidx = reinterpret_cast<zoMatrixTmp<char>*>(api_get_ptr(ctx, args[1]));
		if (cidx->ndat() != me->nrow()) index_error(ctx);
		m = cidx->countne(0);
	}
	else {
		m = get_array_index(ctx, args[1], me->nrow(), idx);
		cidx = 0;
	}
	if (m == 0) return 0;
	if (api_get_type(args[2]) == MAT_CHAR) {
		cjdx = reinterpret_cast<zoMatrixTmp<char>*>(api_get_ptr(ctx, args[2]));
		if (cjdx->ndat() != me->ncol()) index_error(ctx);
		n = cjdx->countne(0);
	}
	else {
		n = get_array_index(ctx, args[2], me->ncol(), jdx);
		cjdx = 0;
	}
	if (n == 0) return 0;
	if (n*m == 1) {
		// A[i.j]
#ifdef MAT_REAL
		return api_create_real(ctx, (*me)(idx[0],jdx[0]));
#else
		return api_create_integer(ctx, (*me)(idx[0],jdx[0]));
#endif
	}
	if (n*m == me->ndat()) {
		// A[*,*]
		return api_create_user(ctx, new zoMatrixTmp<DAT_TYPE>(*me), MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
	}
	zoMatrixTmp<DAT_TYPE> *P = new zoMatrixTmp<DAT_TYPE>;
	P->resize(m, n);
	if (cidx) {
		// A[I,?]
		if (cjdx) {
			// A[I,J]
			for (k = 0, i = 0; i < cidx->ndat(); i++) {
				if ((*cidx)(i) != 0) {
					for (l = 0, j = 0; j < cjdx->ndat(); j++) {
						if ((*cjdx)(j) != 0) {
							(*P)(k,l) = (*me)(i,j);
							l++;
						}
					}
					k++;
				}
			}
			return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
		}
		// A[I,j], A[I,*], or A[I,a:b:c]
		for (k = 0, i = 0; i < cidx->ndat(); i++) {
			if ((*cidx)(i) != 0) {
				for (l = 0, j = jdx[0]; j <= jdx[1]; j += jdx[2]) {
					(*P)(k,l) = (*me)(i,j);
					l++;
				}
				k++;
			}
		}
		return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
	}
	if (cjdx) {
		// A[i,J], A[*,J], or A[a:b:c,J]
		for (k = 0, i = idx[0]; i <= idx[1]; i += idx[2]) {
			for (l = 0, j = 0; j < cjdx->ndat(); j++) {
				if ((*cjdx)(j) != 0) {
					(*P)(k,l) = (*me)(i,j);
					l++;
				}
			}
			k++;
		}
		return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);

	}
	for (k = 0, i = idx[0]; i <= idx[1]; i += idx[2]) {
		for (l = 0, j = jdx[0]; j <= jdx[1]; j += jdx[2]) {
			(*P)(k,l) = (*me)(i,j);
			l++;
		}
		k++;
	}
	return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
}

void* MAT_FUNC(__set)(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zoMatrixTmp<DAT_TYPE> *me = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	size_t i, j, k, m, n, idx[3], jdx[3];
	zoMatrixTmp<char> *cidx, *cjdx;
	DAT_TYPE r;
	zoMatrixTmp<DAT_TYPE> *R=0;
	if (api_is_user(args[nargs-1])) {
		if (api_get_type(args[nargs-1]) != MAT_TYPE) type_error(ctx);
		R = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[nargs-1], MAT_TYPE));
	}
	else {
		if (!api_is_number(args[nargs-1])) matrix_error(ctx);
		r = api_get_number(ctx, args[nargs-1]);
	}
	if (nargs == 3) {
		// A[?] = ?
		if (api_get_type(args[1]) == MAT_CHAR) {
			// A[I] = ?
			cidx = reinterpret_cast<zoMatrixTmp<char>*>(api_get_ptr(ctx, args[1]));
			if (cidx->ndat() != me->ndat()) index_error(ctx);
			n = cidx->countne(0);
			if (n == 0) return 0;
			if (R) {
				if (R->ndat() < n) size_error(ctx);
				if (R->ndat() == me->ndat()) {
					for (i = 0; i < cidx->ndat(); i++) {
						if ((*cidx)(i) != 0) (*me)(i) = (*R)(i);
					}
				}
				else {
					for (k = 0, i = 0; i < cidx->ndat(); i++) {
						if ((*cidx)(i) != 0) (*me)(i) = (*R)(k++);
					}
				}
			}
			else {
				for (i = 0; i < cidx->ndat(); i++) {
					if ((*cidx)(i) != 0) (*me)(i) = r;
				}
			}
			return 0;
		}
		// // A[i]=?, A[*]=?, or A[a:b:c]=?
		n = get_array_index(ctx, args[1], me->ndat(), idx);
		if (R) {
			if (R->ndat() < n) size_error(ctx);
			if (R->ndat() == me->ndat()) {
				for (i = idx[0]; i <= idx[1]; i += idx[2]) (*me)(i) = (*R)(i);
			}
			else {
				for (k = 0, i = idx[0]; i <= idx[1]; i += idx[2]) (*me)(i) = (*R)(k++);
			}
		}
		else {
			for (k = 0, i = idx[0]; i <= idx[1]; i += idx[2]) (*me)(i) = r;
		}
		return 0;
	}
	//A[?,?] = ?
	if (api_is_user(args[1])) {
		cidx = reinterpret_cast<zoMatrixTmp<char>*>(api_get_user(ctx, args[1], MAT_CHAR));
		if (cidx->ndat() != me->nrow()) index_error(ctx);
		m = cidx->countne(0);
	}
	else {
		m = get_array_index(ctx, args[1], me->nrow(), idx);
		cidx = 0;
	}
	if (m == 0) return 0;
	if (api_is_user(args[2])) {
		cjdx = reinterpret_cast<zoMatrixTmp<char>*>(api_get_user(ctx, args[2], MAT_CHAR));
		if (cjdx->ndat() != me->ncol()) index_error(ctx);
		n = cjdx->countne(0);
	}
	else {
		n = get_array_index(ctx, args[2], me->ncol(), jdx);
		cjdx = 0;
	}
	if (n == 0) return 0;
	if (R) {
		// A[?,?]=V;
		if (R->ndat() < m*n) size_error(ctx);
		if (cidx) {
			// A[I,?]=V
			if (cjdx) {
				// A[I,J]=V
				if (R->ndat() == me->ndat()) {
					for (i = 0; i < cidx->ndat(); i++) {
						if ((*cidx)(i) != 0) {
							for (j = 0; j < cjdx->ndat(); j++) {
								if ((*cjdx)(j) != 0) (*me)(i,j) = (*R)(i,j);
							}
						}
					}
				}
				else {
					for (k = 0, i = 0; i < cidx->ndat(); i++) {
						if ((*cidx)(i) != 0) {
							for (j = 0; j < cjdx->ndat(); j++) {
								if ((*cjdx)(j) != 0) (*me)(i,j) = (*R)(k++);
							}
						}
					}
				}
				return 0;
			}
			// A[I,j]=V, A[I,*]=V, or A[I,a:b:c]=V
			if (R->ndat() == me->ndat()) {
				for (i = 0; i < cidx->ndat(); i++) {
					if ((*cidx)(i) != 0) {
						for (j = jdx[0]; j <= jdx[1]; j += jdx[2]) (*me)(i,j) = (*R)(i,j);
					}
				}
			}
			else {
				for (k = 0, i = 0; i < cidx->ndat(); i++) {
					if ((*cidx)(i) != 0) {
						for (j = jdx[0]; j <= jdx[1]; j += jdx[2]) (*me)(i,j) = (*R)(k++);
					}
				}
			}
			return 0;
		}
		if (cjdx) {
			// A[i,J]=V, A[*,J]=V, or A[a:b:c,J]=V
			if (R->ndat() == me->ndat()) {
				for (i = idx[0]; i <= idx[1]; i += idx[2]) {
					for (j = 0; j < cjdx->ndat(); j++) {
						if ((*cjdx)(j) != 0) (*me)(i,j) = (*R)(i,j);
					}
				}
			}
			else {
				for (k = 0, i = idx[0]; i <= idx[1]; i += idx[2]) {
					for (j = 0; j < cjdx->ndat(); j++) {
						if ((*cjdx)(j) != 0) (*me)(i,j) = (*R)(k++);
					}
				}
			}
			return 0;
		}
		// A[range,range]=V
		if (R->ndat() == me->ndat()) {
			for (i = idx[0]; i <= idx[1]; i += idx[2]) {
				for (j = jdx[0]; j <= jdx[1]; j += jdx[2]) (*me)(i,j) = (*R)(i,j);
			}
		}
		else {
			for (k = 0, i = idx[0]; i <= idx[1]; i += idx[2]) {
				for (j = jdx[0]; j <= jdx[1]; j += jdx[2]) (*me)(i,j) = (*R)(k++);
			}
		}
		return 0;
	}
	// A[?,?]=v;
	if (n*m == 1) {
		// A[i,j]=v
		(*me)(idx[0],jdx[0]) = r;
		return 0;
	}
	if (cidx) {
		// A[I,?]=v
		if (cjdx) {
			// A[I,J]=v
			for (i = 0; i < cidx->ndat(); i++) {
				if ((*cidx)(i) != 0) {
					for (j = 0; j < cjdx->ndat(); j++) {
						if ((*cjdx)(j) != 0) (*me)(i,j) = r;
					}
				}
			}
			return 0;
		}
		// A[I,j]=v, A[I,*]=v, or A[I,a:b:c]=v
		for (i = 0; i < cidx->ndat(); i++) {
			if ((*cidx)(i) != 0) {
				for (j = jdx[0]; j <= jdx[1]; j += jdx[2]) (*me)(i,j) = r;
			}
		}
		return 0;

	}
	if (cjdx) {
		// A[i,J], A[*,J], or A[a:b:c,J]
		for (i = idx[0]; i <= idx[1]; i += idx[2]) {
			for (j = 0; j < cjdx->ndat(); j++) {
				if ((*cjdx)(j) != 0) (*me)(i,j) = r;
			}
		}
		return 0;
	}
	// A[range,range]=v
	for (i = idx[0]; i <= idx[1]; i += idx[2]) {
		for (j = jdx[0]; j <= jdx[1]; j += jdx[2]) (*me)(i,j) = r;
	}
	return 0;
}

#define OP_COMPARE(cmp) \
	C = new zoMatrixTmp<char>; \
	C->resize(L->nrow(), L->ncol()); \
	if (!R) { \
		for (k = 0; k < L->ndat(); k++) { \
			if ((*L)(k) cmp rval) \
				(*C)(k) = 1; \
			else \
				(*C)(k) = 0; \
		} \
	} \
	else { \
		if (R->ndat() != L->ndat()) size_error(ctx); \
		for (k = 0; k < L->ndat(); k++) { \
			if ((*L)(k) cmp (*R)(k)) \
				(*C)(k) = 1; \
			else \
				(*C)(k) = 0; \
		} \
	} \
	return api_create_user(ctx, C, mat_opfunc_char, mat_destroy_char, MAT_CHAR);

#define OP_OPSET(opset) \
	P = new zoMatrixTmp<DAT_TYPE>(*L); \
	if (!R) { \
		for (k = 0; k < L->ndat(); k++) (*P)(k) opset rval; \
	} \
	else { \
		if (R->ndat() != L->ndat()) size_error(ctx); \
		for (k = 0; k < L->ndat(); k++) (*P)(k) opset (*R)(k); \
	} \
	return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);

void* MAT_FUNC(opfunc)(void *ctx, int nargs, void** args)
{
	zoMatrixTmp<char> *C;
	zoMatrixTmp<DAT_TYPE> *L = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_user(ctx, args[0], MAT_TYPE));
	int op = api_get_integer(ctx, args[1]);
	zoMatrixTmp<DAT_TYPE> *R=0, *P=0;
	size_t i, j, k;
#ifdef MAT_REAL
	double rval=0;
#else
	int rval=0;
#endif
	int ival=0;
	if (nargs > 2) {
		if (api_is_user(args[2])) {
			if (api_get_type(args[2]) != MAT_TYPE) type_error(ctx);
			R = reinterpret_cast<zoMatrixTmp<DAT_TYPE>*>(api_get_ptr(ctx, args[2]));
		}
		else {
			rval = api_get_number(ctx, args[2]);
			ival = rval;
			R = 0;
		}
	}
	switch (op) {
	case TOK_NEGATE:
		P = new zoMatrixTmp<DAT_TYPE>(*L);
		for (k = 0; k < L->ndat(); k++) (*P)(k) = -(*L)(k);
		return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
	case '~':
		P = new zoMatrixTmp<DAT_TYPE>(*L);
		P->reshape(L->ncol(), L->nrow());
		for (i = 0; i < P->nrow(); i++) {
			for (j = 0; j < P->ncol(); j++) (*P)(i,j) = (*L)(j,i);
		}
		return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
	case TOK_INCR:
		for (k = 0; k < L->ndat(); k++) (*L)(k)++;
		return args[0];
	case TOK_DECR:
		for (k = 0; k < L->ndat(); k++) (*L)(k)--;
		return args[0];
	case TOK_DIV2:
		P = new zoMatrixTmp<DAT_TYPE>(*L);
		for (k = 0; k < L->ndat(); k++) {
			DAT_TYPE d = (*P)(k);
			if (d == 0) {
				delete P;
				divide_error(ctx);
			}
			if (!R)
				(*P)(k) = rval/d;
			else
				(*P)(k) = (*R)(k)/d;
		}
		return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
	case TOK_MOD2:
		P = new zoMatrixTmp<DAT_TYPE>(*L);
		for (k = 0; k < L->ndat(); k++) {
			DAT_TYPE d = (*P)(k);
			if (d == 0) {
				delete P;
				divide_error(ctx);
			}
			if (!R)
				(*P)(k) = MAT_MOD(rval, d);
			else
				(*P)(k) = MAT_MOD((*R)(k), d);
		}
		return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
	case TOK_SUB2:
		P = new zoMatrixTmp<DAT_TYPE>(*L);
		for (k = 0; k < L->ndat(); k++) {
			if (!R)
				(*P)(k) = rval - (*P)(k);
			else
				(*P)(k) = (*R)(k) - (*P)(k);
		}
		return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
	case TOK_LAND:
		C = new zoMatrixTmp<char>;
		C->resize(L->nrow(), L->ncol());
		memset(C->ptr(), 0, C->ndat());
		for (k = 0; k < L->ndat(); k++) {
			if (!R) {
				if (rval != 0 && (*L)(k) != 0) (*C)(k) = 1;
			}
			else {
				if (R->ndat() != L->ndat()) size_error(ctx);
				if ((*R)(k) != 0 && (*L)(k) != 0) (*C)(k) = 1;
			}
		}
		return api_create_user(ctx, C, mat_opfunc_char, mat_destroy_char, MAT_CHAR);
	case TOK_LOR:
		C = new zoMatrixTmp<char>;
		C->resize(L->nrow(), L->ncol());
		memset(C->ptr(), 0, C->ndat());
		for (k = 0; k < L->ndat(); k++) {
			if (!R) {
				if (rval != 0 || (*L)(k) != 0) (*C)(k) = 1;
			}
			else {
				if (R->ndat() != L->ndat()) size_error(ctx);
				if ((*L)(k) != 0 || (*R)(k) != 0) (*C)(k) = 1;
			}

		}
		return api_create_user(ctx, C, mat_opfunc_char, mat_destroy_char, MAT_CHAR);
	case '|':
		// column append
		P = new zoMatrixTmp<DAT_TYPE>;
		if (!R) {
			P->resize(L->nrow(), L->ncol()+1);
			for (i = 0; i < L->nrow(); i++) {
				for (j = 0; j < L->ncol(); j++) (*P)(i,j) = (*L)(i,j);
				(*P)(i,j) = rval;
			}
		}
		else {
			if (R->nrow() != L->nrow()) size_error(ctx);
			P->resize(L->nrow(), L->ncol()+R->ncol());
			for (i = 0; i < L->nrow(); i++) {
				for (j = 0; j < L->ncol(); j++) (*P)(i,j) = (*L)(i,j);
				for (k = 0; k < R->ncol(); k++) (*P)(i,j++) = (*R)(i,k);
			}
		}
		return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
	case '&':
		// row append
		P = new zoMatrixTmp<DAT_TYPE>;
		if (!R) {
			P->resize(L->nrow()+1, L->ncol());
			for (j = 0; j < L->ncol(); j++) {
				for (i = 0; i < L->nrow(); i++) (*P)(i,j) = (*L)(i,j);
				(*P)(i,j) = rval;
			}
		}
		else {
			if (R->ncol() != L->ncol()) size_error(ctx);
			P->resize(L->nrow()+R->nrow(), L->ncol());
			for (j = 0; j < L->ncol(); j++) {
				for (i = 0; i < L->nrow(); i++) (*P)(i,j) = (*L)(i,j);
				for (k = 0; k < R->nrow(); k++) (*P)(i++,j) = (*R)(k,j);
			}
		}
		return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
	case '^':
		// row shift
		if (R) matrix_error(ctx);
		P = new zoMatrixTmp<DAT_TYPE>(*L);
		if (ival < 0) ival += L->nrow();
		ival %= L->nrow();
		if (ival != 0) {
			for (i = 0, k = ival; k < L->nrow(); i++, k++) {
				for (j = 0; j < L->ncol(); j++) (*P)(i,j) = (*L)(k,j);
			}
			for (i = 0, k = L->nrow()-ival; i < ival; i++, k++) {
				for (j = 0; j < L->ncol(); j++) (*P)(k,j) = (*L)(i,j);
			}
		}
		return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
	case TOK_LSHIFT:
		// left column shift
		if (R) matrix_error(ctx);
		P = new zoMatrixTmp<DAT_TYPE>(*L);
		if (ival < 0) ival += L->ncol();
		ival %= L->ncol();
		if (ival != 0) {
			for (j = 0, k = ival; k < L->ncol(); j++, k++) {
				for (i = 0; i < L->nrow(); i++) (*P)(i,j) = (*L)(i,k);
			}
			for (j = 0, k = L->ncol()-ival; j < ival; j++, k++) {
				for (i = 0; i < L->nrow(); i++) (*P)(i,k) = (*L)(i,j);
			}
		}
		return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
	case TOK_RSHIFT:
		// right column shift
		if (R) matrix_error(ctx);
		P = new zoMatrixTmp<DAT_TYPE>(*L);
		ival = -ival;		// change to left shift
		if (ival < 0) ival += L->ncol();
		ival %= L->ncol();
		if (ival != 0) {
			for (j = 0, k = ival; k < L->ncol(); j++, k++) {
				for (i = 0; i < L->nrow(); i++) (*P)(i,j) = (*L)(i,k);
			}
			for (j = 0, k = L->ncol()-ival; j < ival; j++, k++) {
				for (i = 0; i < L->nrow(); i++) (*P)(i,k) = (*L)(i,j);
			}
		}
		return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
	case TOK_LEQ:
		OP_COMPARE(<=);
	case TOK_GEQ:
		OP_COMPARE(>=);
	case '<':
		OP_COMPARE(<);
	case '>':
		OP_COMPARE(>);
	case TOK_EQL:
		OP_COMPARE(==);
	case TOK_NEQ:
		OP_COMPARE(!=);
	case '*':
		OP_OPSET(*=);
	case '/':
		P = new zoMatrixTmp<DAT_TYPE>(*L);
		if (!R) {
			if (rval == 0) divide_error(ctx);
			for (k = 0; k < L->ndat(); k++) (*P)(k) /= rval;
		}
		else {
			if (R->ndat() != L->ndat()) size_error(ctx);
			for (k = 0; k < L->ndat(); k++) {
				DAT_TYPE d = (*R)(k);
				if (d == 0) divide_error(ctx);
				(*P)(k) /= d;
			}
		}
		return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
	case '+':
		OP_OPSET(+=);
	case '-':
		OP_OPSET(-=);
	case '%':
		P = new zoMatrixTmp<DAT_TYPE>(*L);
		if (!R) {
			if (rval == 0) divide_error(ctx);
			for (k = 0; k < L->ndat(); k++) (*P)(k) = MAT_MOD((*P)(k), rval);
		}
		else {
			if (R->ndat() != L->ndat()) size_error(ctx);
			for (k = 0; k < L->ndat(); k++) {
				if ((*R)(k) == 0) divide_error(ctx);
				(*P)(k) = MAT_MOD((*P)(k), (*R)(k));
			}
		}
		return api_create_user(ctx, P, MAT_FUNC(opfunc), MAT_FUNC(destroy), MAT_TYPE);
	case TOK_MULEQ:
		if (!R) {
			for (k = 0; k < L->ndat(); k++) (*L)(k) *= rval;
		}
		else {
			if (R->ndat() != L->ndat()) size_error(ctx);
			for (k = 0; k < L->ndat(); k++) (*L)(k) *= (*R)(k);
		}
		return args[0];
	case TOK_DIVEQ:
		if (!R) {
			if (rval == 0) divide_error(ctx);
			for (k = 0; k < L->ndat(); k++) (*L)(k) /= rval;
		}
		else {
			if (R->ndat() != L->ndat()) size_error(ctx);
			for (k = 0; k < L->ndat(); k++) {
				DAT_TYPE d = (*R)(k);
				if (d == 0) divide_error(ctx);
				(*L)(k) /= d;
			}
		}
		return args[0];
	case TOK_MODEQ:
		if (!R) {
			if (rval == 0) divide_error(ctx);
			for (k = 0; k < L->ndat(); k++) (*L)(k) = MAT_MOD((*L)(k), rval);
		}
		else {
			if (R->ndat() != L->ndat()) size_error(ctx);
			for (k = 0; k < L->ndat(); k++) {
				DAT_TYPE d = (*R)(k);
				if (d == 0) divide_error(ctx);
				(*L)(k) = MAT_MOD((*L)(k), d);
			}
		}
		return args[0];
	case TOK_ADDEQ:
		if (!R) {
			for (k = 0; k < L->ndat(); k++) (*L)(k) += rval;
		}
		else {
			if (R->ndat() != L->ndat()) size_error(ctx);
			for (k = 0; k < L->ndat(); k++) (*L)(k) += (*R)(k);
		}
		return args[0];
	case TOK_SUBEQ:
		if (!R) {
			for (k = 0; k < L->ndat(); k++) (*L)(k) -= rval;
		}
		else {
			if (R->ndat() != L->ndat()) size_error(ctx);
			for (k = 0; k < L->ndat(); k++) (*L)(k) -= (*R)(k);
		}
		return args[0];
	}
	api_runtime_error(ctx, "undefined operation for matrix");
	return 0;
}


class MAT_FUNC(zsRegPrimitive)
{
public:
	MAT_FUNC(zsRegPrimitive)()
	{
		api_add_primitive(REG_NAME, 0, MAT_FUNC(create));
		api_add_primitive("csv", MAT_TYPE, MAT_FUNC(csv));
		api_add_primitive("print", MAT_TYPE, MAT_FUNC(print));
		api_add_primitive("fill", MAT_TYPE, MAT_FUNC(fill));
		api_add_primitive("size", MAT_TYPE, MAT_FUNC(size));
		api_add_primitive("ptr", MAT_TYPE, MAT_FUNC(ptr));
		api_add_primitive("reshape", MAT_TYPE, MAT_FUNC(reshape));
		api_add_primitive("resize", MAT_TYPE, MAT_FUNC(resize));
		api_add_primitive("reserve", MAT_TYPE, MAT_FUNC(reserve));
		api_add_primitive("flip", MAT_TYPE, MAT_FUNC(flip));
		api_add_primitive("insert", MAT_TYPE, MAT_FUNC(insert));
		api_add_primitive("delete", MAT_TYPE, MAT_FUNC(delete));
		api_add_primitive("trans", MAT_TYPE, MAT_FUNC(trans));
		api_add_primitive("sort", MAT_TYPE, MAT_FUNC(sort));
		api_add_primitive("unique", MAT_TYPE, MAT_FUNC(unique));
		api_add_primitive("min", MAT_TYPE, MAT_FUNC(min));
		api_add_primitive("max", MAT_TYPE, MAT_FUNC(max));
		api_add_primitive("sum", MAT_TYPE, MAT_FUNC(sum));
		api_add_primitive("mean", MAT_TYPE, MAT_FUNC(mean));
		api_add_primitive("stdev", MAT_TYPE, MAT_FUNC(stdev));
		api_add_primitive("parse", MAT_TYPE, MAT_FUNC(parse));
		api_add_primitive("clone", MAT_TYPE, MAT_FUNC(clone));
		api_add_primitive("import", MAT_TYPE, MAT_FUNC(import));
		api_add_primitive("find", MAT_TYPE, MAT_FUNC(find));
		api_add_primitive("replace", MAT_TYPE, MAT_FUNC(replace));
		api_add_primitive("abs", MAT_TYPE, MAT_FUNC(abs));
#ifdef MAT_REAL
		api_add_primitive("rand", MAT_TYPE, MAT_FUNC(rand));
		api_add_primitive("prod", MAT_TYPE, MAT_FUNC(prod));
#endif
		api_add_primitive("cos", MAT_TYPE, MAT_FUNC(cos));
		api_add_primitive("cosh", MAT_TYPE, MAT_FUNC(cosh));
		api_add_primitive("acos", MAT_TYPE, MAT_FUNC(acos));
		api_add_primitive("sin", MAT_TYPE, MAT_FUNC(sin));
		api_add_primitive("sinh", MAT_TYPE, MAT_FUNC(sinh));
		api_add_primitive("asin", MAT_TYPE, MAT_FUNC(asin));
		api_add_primitive("tan", MAT_TYPE, MAT_FUNC(tan));
		api_add_primitive("tanh", MAT_TYPE, MAT_FUNC(tanh));
		api_add_primitive("atan", MAT_TYPE, MAT_FUNC(atan));
		api_add_primitive("sqrt", MAT_TYPE, MAT_FUNC(sqrt));
		api_add_primitive("exp", MAT_TYPE, MAT_FUNC(exp));
		api_add_primitive("log", MAT_TYPE, MAT_FUNC(log));
		api_add_primitive("log10", MAT_TYPE, MAT_FUNC(log10));
		api_add_primitive("ceil", MAT_TYPE, MAT_FUNC(ceil));
		api_add_primitive("floor", MAT_TYPE, MAT_FUNC(floor));
		api_add_primitive("atan2", MAT_TYPE, MAT_FUNC(atan2));
		api_add_primitive("pow", MAT_TYPE, MAT_FUNC(pow));
		api_add_primitive("cal2jul", MAT_TYPE, MAT_FUNC(cal2jul));
		api_add_primitive("jul2cal", MAT_TYPE, MAT_FUNC(jul2cal));
		api_add_primitive("__get", MAT_TYPE, MAT_FUNC(__get));
		api_add_primitive("__set", MAT_TYPE, MAT_FUNC(__set));
		api_add_primitive("opfunc", MAT_TYPE, MAT_FUNC(opfunc));
	}
};

static MAT_FUNC(zsRegPrimitive) MAT_FUNC(register);

#endif