#include <math.h>
#include <string.h>
#include <vector>

extern char *scan_number(char *p);

void* MAT_FUNC(print)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
#ifdef MAT_INTEGER
	const char* format = "  %d";
#endif
#ifdef MAT_UNSIGNED
	const char* format = "  %u";
#endif
#ifdef MAT_REAL
	const char* format = "  %f";
#endif
	if (nargs > 1) format = api_get_string(ctx, args[1]);
	int len = strlen(format);
	if (len > 30) api_input_error(ctx);
	char s[32];
	bool csv = false;
	strcpy(s, format);
	if (format[len-1] == ',') {
		s[len-1] = 0;
		csv = true;
	}
	FILE *file = stdout;
	if (nargs > 2) {
		const char *fname = api_get_string(ctx, args[2]);
		const char *mode = "a";
		if (nargs > 3) mode = api_get_string(ctx, args[3]);
		file = fopen(fname, mode);
		if (!file) api_runtime_error(ctx, "failed to open file for output");
	}
	me->print(file, s, csv);
	fflush(file);
	if (file != stdout) fclose(file);
	return 0;
}
REG_PRIM(print, 0);


void* MAT_FUNC(fill)(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
#ifdef MAT_REAL
	double v0 = api_get_number(ctx, args[1]);
	double dv = api_get_number(ctx, args[2]);
	me->fill(v0, dv);
#else
	int v0 = api_get_integer(ctx, args[1]);
	int dv = api_get_integer(ctx, args[2]);
	me->fill(v0, dv);
#endif
	return 0;
}
REG_PRIM(fill, 1);


void* MAT_FUNC(size)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	void* arr = api_create_array(ctx, 3);
	api_set_array_object(ctx, arr, "0", api_create_integer(0, me->nrow()));
	api_set_array_object(ctx, arr, "1", api_create_integer(0, me->ncol()));
	api_set_array_object(ctx, arr, "2", api_create_integer(0, me->ndat()));
	return arr;
}
REG_PRIM(size, 2);


void* MAT_FUNC(ptr)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	int offset = 0;
	if (nargs > 1) {
		offset = api_get_integer(ctx, args[1]);
		if (offset < 0 || offset >= me->ndat()) api_input_error(ctx);
	}
	void* arr = api_create_array(ctx, 3);
	api_set_array_object(ctx, arr, "0", api_create_user(0, me->ptr()+offset, 0, 0, 0));
	api_set_array_object(ctx, arr, "1", api_create_integer(0, me->ndat()));
	api_set_array_object(ctx, arr, "2", api_create_integer(0, MAT_SIZE));
	return arr;
}
REG_PRIM(ptr, 3);


void* MAT_FUNC(reshape)(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	int nrow = api_get_integer(ctx, args[1]);
	int ncol = api_get_integer(ctx, args[2]);
	if (!me->reshape(nrow, ncol)) index_error(ctx);
	return 0;
}
REG_PRIM(reshape, 4);


void* MAT_FUNC(resize)(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	int nrow = api_get_integer(ctx, args[1]);
	int ncol = api_get_integer(ctx, args[2]);
	if (!me->resize(nrow, ncol)) index_error(ctx);
	return 0;
}
REG_PRIM(resize, 5);


void* MAT_FUNC(reserve)(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	int n = api_get_integer(ctx, args[1]);
	if (n <= 0) api_input_error(ctx);
	me->reserve(n);
	return 0;
}
REG_PRIM(reserve, 5a);


void* MAT_FUNC(flip)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	const char* c = "c";
	if (nargs > 1) c = api_get_string(ctx, args[1]);
	if (c[0] == 'c' || c[0] == 'C') {
		if (!me->flipcol()) index_error(ctx);
	}
	else if (c[0] == 'r' || c[0] == 'R') {
		if (!me->fliprow()) index_error(ctx);
	}
	else {
		api_input_error(ctx);
	}
	return 0;
}
REG_PRIM(flip, 6);


void* MAT_FUNC(insert)(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	int i = api_get_integer(ctx, args[1]);
	OBJ_TYPE *src = 0;
	double v;
	if (api_get_type(args[2]) == MAT_TYPE) {
		src = reinterpret_cast<OBJ_TYPE*>(api_get_ptr(ctx, args[2]));
	}
	else {
		v = api_get_number(ctx, args[2]);
	}
	const char* c = "c";
	if (nargs > 3) c = api_get_string(ctx, args[3]);
	int status;
	if (c[0] == 'c' || c[0] == 'C') {
		if (src) status = me->insertcol(*src, i);
		else     status = me->insertcol(v, i);
		if (!status) index_error(ctx);
	}
	else if (c[0] == 'r' || c[0] == 'R') {
		if (src) status = me->insertrow(*src, i);
		else     status = me->insertrow(v, i);
		if (!status) index_error(ctx);
	}
	else {
		api_input_error(ctx);
	}
	return 0;
}
REG_PRIM(insert, 7);


void* MAT_FUNC(delete)(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	int i = api_get_integer(ctx, args[1]);
	const char* c = "c";
	if (nargs > 2) c = api_get_string(ctx, args[2]);
	if (c[0] == 'c' || c[0] == 'C') {
		if (!me->delcol(i)) index_error(ctx);
	}
	else if (c[0] == 'r' || c[0] == 'R') {
		if (!me->delrow(i)) index_error(ctx);
	}
	else {
		api_input_error(ctx);
	}
	return 0;
}
REG_PRIM(delete, 8);


void* MAT_FUNC(trans)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	me->transpose();
	return 0;
}
REG_PRIM(trans, 9);


void* MAT_FUNC(sort)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	int i = 0; 
	if (nargs > 1) i = api_get_integer(ctx, args[1]);
	int dir = 1;
	if (nargs > 2) dir = api_get_integer(ctx, args[2]);
	if (!me->sort(i, dir)) index_error(ctx);
	return 0;
}
REG_PRIM(sort, 10);


void* MAT_FUNC(unique)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	int dir = 1;
	if (nargs > 1) dir = api_get_integer(ctx, args[1]);
	me->unique(dir);
	return 0;
}
REG_PRIM(unique, 11);


void* MAT_FUNC(min)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
#ifdef MAT_REAL
	return api_create_real(ctx, me->vmin(MAT_MAX));
#else
	return api_create_integer(ctx, me->vmin(MAT_MAX));
#endif
}
REG_PRIM(min, 12);


void* MAT_FUNC(max)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
#ifdef MAT_REAL
	return api_create_real(ctx, me->vmax(MAT_MIN));
#else
	return api_create_integer(ctx, me->vmax(MAT_MIN));
#endif
}
REG_PRIM(max, 13);


void* MAT_FUNC(sum)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
#ifdef MAT_REAL
	return api_create_real(ctx, me->sum());
#else
	return api_create_integer(ctx, me->sum());
#endif
}
REG_PRIM(sum, 14);


void* MAT_FUNC(mean)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	return api_create_real(ctx, me->mean());
}
REG_PRIM(mean, 14a);


void* MAT_FUNC(stdev)(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	return api_create_real(ctx, me->stdev());
}
REG_PRIM(stdev, 14b);


void* MAT_FUNC(join)(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	OBJ_TYPE *m1 = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	OBJ_TYPE *m2 = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[1], MAT_TYPE));
	double ms = api_get_number(ctx, args[2]);
	OBJ_TYPE *p = new OBJ_TYPE;
	p->join(*m1, *m2, ms);
	return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
}
REG_PRIM(join, 15);


void* MAT_FUNC(replace)(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	zoINT *idx = reinterpret_cast<zoINT*>(api_get_user(ctx, args[1], MAT_INT));
	if (api_is_user(args[2])) {
		OBJ_TYPE *m2 = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[2], MAT_TYPE));
		if (idx->ndat() < m2->ndat()) api_runtime_error(ctx, "Invalid index matrix size.");
		for (int i = 0; i < m2->ndat(); i++) {
			if ((*idx)(i) < 0 || (*idx)(i) >= me->ndat()) api_runtime_error(ctx, "Invalid value in index matrix.");
			(*me)((*idx)(i)) = (*m2)(i);
		}
	}
	else {
#ifdef MAT_INTEGER
		integer_t missing = api_get_integer(ctx, args[2]);
#else
		real_t missing = api_get_integer(ctx, args[2]);
#endif
		for (int i = 0; i < idx->ndat(); i++) {
			if ((*idx)(i) < 0 || (*idx)(i) >= me->ndat()) api_runtime_error(ctx, "Invalid value in index matrix.");
			(*me)((*idx)(i)) = missing;
		}
	}
	return 0;
}
REG_PRIM(replace, 15a);


void* MAT_FUNC(parse)(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	if (nargs == 2) {
		const char *s = api_get_string(ctx, args[1]);
		char *ptr = (char*)malloc(strlen(s)+1);
		strcpy(ptr, s);
		int i = 0;
		while (ptr[i]) {
			if (ptr[i] == ',') ptr[i] = ' ';
			i++;
		}
		char *cur = ptr;
		char *tok = ptr;
#ifdef MAT_REAL
		std::vector<double> tmp;
#else
		std::vector<int> tmp;
#endif
		while (cur = scan_number(cur)) {
			if (!isspace(tok[0])) {
				int n = cur-tok;
				char str[64];
				strncpy(str, tok, n);
				str[n] = 0;
#ifdef MAT_REAL
				tmp.push_back(atof(str));
#else
				tmp.push_back(atoi(str));
#endif
			}
			tok = cur;
		}
		free(ptr);
		me->resize(tmp.size(), 1);
		for (i = 0; i < tmp.size(); i++) (*me)(i) = tmp[i];
		return 0;
	}
	int i = api_get_integer(ctx, args[1]);
	const char *s = api_get_string(ctx, args[2]);
	if (!me->parse(i, s)) index_error(ctx);
	return 0;
}
REG_PRIM(parse, 16);


void* MAT_FUNC(clone)(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	mat_clone(ctx, api_get_user(ctx, args[0], MAT_TYPE), MAT_TYPE, api_get_ptr(ctx, args[1]), api_get_type(args[1]));
	return 0;
}
REG_PRIM(clone, 17);


void* MAT_FUNC(import)(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	void* ptr1 = api_get_user(ctx, args[0], MAT_TYPE);
	void* ptr2 = api_get_ptr(ctx, args[1]);
	mat_import(ctx, ptr1, MAT_TYPE, ptr2);
	return 0;
}
REG_PRIM(import, 17a);


void* MAT_FUNC(find)(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
#ifdef MAT_REAL
	double v = api_get_number(ctx, args[1]);
#else
	int v = api_get_integer(ctx, args[1]);
#endif
	int i0 = 0, j0 = 0;
	if (nargs > 3) {
		i0 = api_get_integer(ctx, args[2]);
		j0 = api_get_integer(ctx, args[3]);
		if (i0 < 0 || j0 < 0) api_input_error(ctx);
	}
	for (int i = i0; i < me->nrow(); i++) {
		for (int j = j0; j < me->ncol(); j++) {
			if ((*me)(i, j) == v) {
				void *arr = api_create_array(ctx, 2);
				api_set_array_object(ctx, arr, "0", api_create_integer(0, i));
				api_set_array_object(ctx, arr, "1", api_create_integer(0, j));
				return arr;
			}
		}
	}
	return 0;
}
REG_PRIM(find, 17b);

extern size_t get_array_index(void *ctx, void *obj, size_t n, size_t idx[]);

void* MAT_FUNC(__get)(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	
	size_t i, j, k, l, m, n, idx[3], jdx[3];
	zoCHAR *cidx, *cjdx;

	if (nargs == 2) {

		if (api_get_type(args[1]) == MAT_CHAR) {
			// A[I]
			cidx = reinterpret_cast<zoCHAR*>(api_get_ptr(ctx, args[1]));
			if (cidx->ndat() != me->ndat()) index_error(ctx);
			n = cidx->countne(0);
			if (n == 0) index_error(ctx);
			OBJ_TYPE *p = new OBJ_TYPE;
			p->resize(1, n);
			for (k = 0, i = 0; i < cidx->ndat(); i++) {
				if ((*cidx)(i) != 0) {
					(*p)(k) = (*me)(i);
					k++;
				}
			}
			return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
		}
		
		n = get_array_index(ctx, args[1], me->ndat(), idx);

		if (n == 1) {
			// A[i]
#ifdef MAT_REAL
			return api_create_real(ctx, (*me)(idx[0]));
#else
			return api_create_integer(ctx, (*me)(idx[0]));
#endif
		}

		if (n == me->ndat()) {
			// A[*]
			return api_create_user(ctx, new OBJ_TYPE(*me), OP_FUNC, MAT_DESTROY, MAT_TYPE);
		}

		// A[a:b:c]
		OBJ_TYPE *p = new OBJ_TYPE;
		p->resize(1, n);
		for (k = 0, i = idx[0]; i <= idx[1]; i += idx[2]) {
			(*p)(k) = (*me)(i);
			k++;
		}
		return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
	}

	if (api_get_type(args[1]) == MAT_CHAR) {
		cidx = reinterpret_cast<zoCHAR*>(api_get_ptr(ctx, args[1]));
		if (cidx->ndat() != me->nrow()) index_error(ctx);
		m = cidx->countne(0);
		if (m == 0) index_error(ctx);
	}
	else {
		m = get_array_index(ctx, args[1], me->nrow(), idx);
		cidx = 0;
	}

	if (api_get_type(args[2]) == MAT_CHAR) {
		cjdx = reinterpret_cast<zoCHAR*>(api_get_ptr(ctx, args[2]));
		if (cjdx->ndat() != me->ncol()) index_error(ctx);
		n = cjdx->countne(0);
		if (n == 0) index_error(ctx);
	}
	else {
		n = get_array_index(ctx, args[2], me->ncol(), jdx);
		cjdx = 0;
	}

	if (n*m == 1) {
		// A[i.j]
#ifdef MAT_REAL
		return api_create_real(ctx, (*me)(idx[0],jdx[0]));
#else
		return api_create_integer(ctx, (*me)(idx[0],jdx[0]));
#endif
	}

	if (n*m == me->ndat()) {
		// A[*, *]
		return api_create_user(ctx, new OBJ_TYPE(*me), OP_FUNC, MAT_DESTROY, MAT_TYPE);
	}

	if (cidx) {		// A[I,?]

		if (cjdx) {
			// A[I,J]
			OBJ_TYPE *p = new OBJ_TYPE;
			p->resize(m, n);
			for (k = 0, i = 0; i < cidx->ndat(); i++) {
				if ((*cidx)(i) != 0) {
					for (l = 0, j = 0; j < cjdx->ndat(); j++) {
						if ((*cjdx)(j) != 0) {
							(*p)(k,l) = (*me)(i,j);
							l++;
						}
					}
					k++;
				}
			}
			return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
		}

		// A[I,j], A[I,*], or A[I,a:b:c]
		OBJ_TYPE *p = new OBJ_TYPE;
		p->resize(m, n);
		for (k = 0, i = 0; i < cidx->ndat(); i++) {
			if ((*cidx)(i) != 0) {
				for (l = 0, j = jdx[0]; j <= jdx[1]; j += jdx[2]) {
					(*p)(k,l) = (*me)(i,j);
					l++;
				}
				k++;
			}
		}
		return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);

	}

	if (cjdx) {
		// A[i,J], A[*,J], or A[a:b:c,J]
		OBJ_TYPE *p = new OBJ_TYPE;
		p->resize(m, n);
		for (k = 0, i = idx[0]; i <= idx[1]; i += idx[2]) {
			for (l = 0, j = 0; j < cjdx->ndat(); j++) {
				if ((*cjdx)(j) != 0) {
					(*p)(k,l) = (*me)(i,j);
					l++;
				}
			}
			k++;
		}
		return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);

	}

	OBJ_TYPE *p = new OBJ_TYPE;
	p->resize(m, n);
	for (k = 0, i = idx[0]; i <= idx[1]; i += idx[2]) {
		for (l = 0, j = jdx[0]; j <= jdx[1]; j += jdx[2]) {
			(*p)(k,l) = (*me)(i,j);
			l++;
		}
		k++;
	}
	return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
}
REG_PRIM(__get, 18);


void* MAT_FUNC(__set)(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	OBJ_TYPE *me = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));

	size_t i, j, k, m, n, idx[3], jdx[3];
	zoCHAR *cidx, *cjdx;
	real_t r;
	OBJ_TYPE *R = 0;

	if (api_get_type(args[nargs-1]) == MAT_TYPE) {
		R = reinterpret_cast<OBJ_TYPE*>(api_get_ptr(ctx, args[nargs-1]));
	}
	else {
		r = api_get_number(ctx, args[nargs-1]);
	}

	if (nargs == 3) {

		if (api_get_type(args[1]) == MAT_CHAR) {
			// A[I] = r
			cidx = reinterpret_cast<zoCHAR*>(api_get_ptr(ctx, args[1]));
			if (cidx->ndat() != me->ndat()) index_error(ctx);
			n = cidx->countne(0);
			if (n == 0) return 0;
			if (R && R->ndat() < n) index_error(ctx);
			for (k = 0, i = 0; i < cidx->ndat(); i++) {
				if ((*cidx)(i) != 0) {
					if (R) {
						(*me)(i) = (*R)(k++);
					}
					else {
						(*me)(i) = r;
					}
				}
			}
			return 0;
		}
		
		n = get_array_index(ctx, args[1], me->ndat(), idx);
		if (R && R->ndat() < n) index_error(ctx);

		if (n == 1) {
			// A[i]
			if (R)
				(*me)(idx[0]) = (*R)(0);
			else
				(*me)(idx[0]) = r;
			return 0;
		}

		if (n == me->ndat()) {
			// A[*]
			for (i = 0; i < me->ndat(); i++) {
				if (R)
					(*me)(i) = (*R)(i);
				else
					(*me)(i) = r;
			}
			return 0;
		}

		// A[a:b:c]
		for (k = 0, i = idx[0]; i <= idx[1]; i += idx[2]) {
			if (R)
				(*me)(i) = (*R)(k++);
			else
				(*me)(i) = r;
		}
		return 0;
	}

	if (api_get_type(args[1]) == MAT_CHAR) {
		cidx = reinterpret_cast<zoCHAR*>(api_get_ptr(ctx, args[1]));
		if (cidx->ndat() != me->nrow()) index_error(ctx);
		m = cidx->countne(0);
		if (m == 0) index_error(ctx);
	}
	else {
		m = get_array_index(ctx, args[1], me->nrow(), idx);
		cidx = 0;
	}

	if (api_get_type(args[2]) == MAT_CHAR) {
		cjdx = reinterpret_cast<zoCHAR*>(api_get_ptr(ctx, args[2]));
		if (cjdx->ndat() != me->ncol()) index_error(ctx);
		n = cjdx->countne(0);
		if (n == 0) index_error(ctx);
	}
	else {
		n = get_array_index(ctx, args[2], me->ncol(), jdx);
		cjdx = 0;
	}

	if (R && R->ndat() < m*n) index_error(ctx);

	if (n*m == 1) {
		// A[i.j]
		if (R)
			(*me)(idx[0],jdx[0]) = (*R)(0);
		else
			(*me)(idx[0],jdx[0]) = r;
		return 0;
	}

	if (cidx) {		// A[I,?]

		if (cjdx) {
			// A[I,J]
			for (k = 0, i = 0; i < cidx->ndat(); i++) {
				if ((*cidx)(i) != 0) {
					for (j = 0; j < cjdx->ndat(); j++) {
						if ((*cjdx)(j) != 0) {
							if (R)
								(*me)(i,j) = (*R)(k++);
							else
								(*me)(i,j) = r;
						}
					}
				}
			}
			return 0;
		}

		// A[I,j], A[I,*], or A[I,a:b:c]
		for (k = 0, i = 0; i < cidx->ndat(); i++) {
			if ((*cidx)(i) != 0) {
				for (j = jdx[0]; j <= jdx[1]; j += jdx[2]) {
					if (R)
						(*me)(i,j) = (*R)(k++);
					else
						(*me)(i,j) = r;
				}
			}
		}
		return 0;

	}

	if (cjdx) {
		// A[i,J], A[*,J], or A[a:b:c,J]
		for (k = 0, i = idx[0]; i <= idx[1]; i += idx[2]) {
			for (j = 0; j < cjdx->ndat(); j++) {
				if ((*cjdx)(j) != 0) {
					if (R)
						(*me)(i,j) = (*R)(k++);
					else
						(*me)(i,j) = r;
				}
			}
		}
		return 0;

	}

	for (k = 0, i = idx[0]; i <= idx[1]; i += idx[2]) {
		for (j = jdx[0]; j <= jdx[1]; j += jdx[2]) {
			if (R)
				(*me)(i,j) = (*R)(k++);
			else
				(*me)(i,j) = r;
		}
	}

	return 0;
}
REG_PRIM(__set, 19);


void* MAT_FUNC(Create)(void *ctx, int nargs, void** args)
{
	OBJ_TYPE *p = new OBJ_TYPE;
	if (nargs > 2) {
		p->resize(1, nargs);
#ifdef MAT_REAL
		for (int i = 0; i < nargs; i++) (*p)(i) = api_get_number(ctx, args[i]);
#else
		for (int i = 0; i < nargs; i++) (*p)(i) = api_get_integer(ctx, args[i]);
#endif
	}
	else {
		if (nargs > 0) {
			if (api_is_user(args[0])) {
				mat_clone(ctx, p, MAT_TYPE, api_get_ptr(ctx, args[0]), api_get_type(args[0]));
			}
			else {
				int nrow = api_get_integer(ctx, args[0]);
				int ncol = 1;
				if (nargs == 2) ncol = api_get_integer(ctx, args[1]);
				p->resize(nrow, ncol);
			}
		}
	}
	return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
}


void* OP_FUNC(void *ctx, int nargs, void** args)
{
	zoCHAR *c;
	OBJ_TYPE *l = reinterpret_cast<OBJ_TYPE*>(api_get_user(ctx, args[0], MAT_TYPE));
	OBJ_TYPE *r = 0, *p, q;
	int i, j, k;
	int op = api_get_integer(ctx, args[1]);
	if (nargs > 2 && api_get_type(args[2]) == MAT_TYPE) r = reinterpret_cast<OBJ_TYPE*>(api_get_ptr(ctx, args[2]));

	switch (op) {

	case TOK_NEGATE:
#ifdef MAT_UNSIGNED
		break;
#endif
		p = new OBJ_TYPE(*l);
		p->negate();
		return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);

	case '~':
		return api_create_user(ctx, l->transpose2(), OP_FUNC, MAT_DESTROY, MAT_TYPE);

	case TOK_INCR:
		l->increment();
		return args[0];

	case TOK_DECR:
		l->decrement();
		return args[0];

	case TOK_DIV2:
		p = new OBJ_TYPE(*l);
		if (!p->invert()) divide_error(ctx);
		p->mul(api_get_number(ctx, args[2]));
		return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);

	case TOK_MOD2:
		p = new OBJ_TYPE(*l);
#ifdef MAT_REAL
		{
		double v = api_get_number(ctx, args[2]);
		int i, n = p->ndat();
		for (i = 0; i < n; i++) {
			if ((*p)(i) == 0) divide_error(ctx);
			(*p)(i) = fmod(v, (*p)(i));
		}
		return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
		}
#else
		{
		int v = api_get_integer(ctx, args[2]);
		int i, n = p->ndat();
		for (i = 0; i < n; i++) {
			if ((*p)(i) == 0) divide_error(ctx);
			(*p)(i) = v % (*p)(i);
		}
		return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
		}
#endif
		break;

	case TOK_SUB2:
#ifdef MAT_UNSIGNED
		break;
#endif
		p = new OBJ_TYPE(*l);
		p->negate();
		p->add(api_get_number(ctx, args[2]));
		return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);

	case TOK_LAND:
		if (r == 0) operand_error(ctx);
		c = new zoCHAR;
		if (!l->and(*c, *r)) index_error(ctx);
		return api_create_user(ctx, c, mat_opfunc_CHAR, mat_destroy_CHAR, MAT_CHAR);

	case TOK_LOR:
		if (r == 0) operand_error(ctx);
		c = new zoCHAR;
		if (!l->or(*c, *r)) index_error(ctx);
		return api_create_user(ctx, c, mat_opfunc_CHAR, mat_destroy_CHAR, MAT_CHAR);

	case '|':		// column append
		if (r == 0 || r->nrow() != l->nrow()) operand_error(ctx);
		p = new OBJ_TYPE;
		p->resize(l->nrow(), l->ncol()+r->ncol());
		k = 0;
		for (j = 0; j < l->ncol(); j++) {
			l->getcol(q, j);
			p->setcol(q, k++);
		}
		for (j = 0; j < r->ncol(); j++) {
			r->getcol(q, j);
			p->setcol(q, k++);
		}
		return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);

	case '&':		// row append
		if (r == 0 || r->ncol() != l->ncol()) operand_error(ctx);
		p = new OBJ_TYPE;
		p->resize(l->nrow()+r->nrow(), l->ncol());
		k = 0;
		for (i = 0; i < l->nrow(); i++) {
			l->getrow(q, i);
			p->setrow(q, k++);
		}
		for (i = 0; i < r->nrow(); i++) {
			r->getrow(q, i);
			p->setrow(q, k++);
		}
		return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);

	case '^':
		p = new OBJ_TYPE(*l);
		if (!p->shiftrow(api_get_integer(ctx, args[2]))) index_error(ctx);
		return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);

	case TOK_LSHIFT:
		p = new OBJ_TYPE(*l);
		if (!p->shiftcol(api_get_integer(ctx, args[2]))) index_error(ctx);
		return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);

	case TOK_RSHIFT:
		p = new OBJ_TYPE(*l);
		if (!p->shiftcol(-api_get_integer(ctx, args[2]))) index_error(ctx);
		return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);

	case TOK_LEQ:
		c = new zoCHAR;
		if (r == 0) {
			l->leq(*c, api_get_number(ctx, args[2]));
			return api_create_user(ctx, c, mat_opfunc_CHAR, mat_destroy_CHAR, MAT_CHAR);
		}
		else {
			if (!l->leq(*c, *r)) index_error(ctx);
			return api_create_user(ctx, c, mat_opfunc_CHAR, mat_destroy_CHAR, MAT_CHAR);
		}

	case TOK_GEQ:
		c = new zoCHAR;
		if (r == 0) {
			l->geq(*c, api_get_number(ctx, args[2]));
			return api_create_user(ctx, c, mat_opfunc_CHAR, mat_destroy_CHAR, MAT_CHAR);
		}
		else {
			if (l->geq(*c, *r)) index_error(ctx);
			return api_create_user(ctx, c, mat_opfunc_CHAR, mat_destroy_CHAR, MAT_CHAR);
		}

	case '<':
		c = new zoCHAR;
		if (r == 0) {
			l->less(*c, api_get_number(ctx, args[2]));
			return api_create_user(ctx, c, mat_opfunc_CHAR, mat_destroy_CHAR, MAT_CHAR);
		}
		else {
			if (l->less(*c, *r)) index_error(ctx);
			return api_create_user(ctx, c, mat_opfunc_CHAR, mat_destroy_CHAR, MAT_CHAR);
		}

	case '>':
		c = new zoCHAR;
		if (r == 0) {
			l->greater(*c, api_get_number(ctx, args[2]));
			return api_create_user(ctx, c, mat_opfunc_CHAR, mat_destroy_CHAR, MAT_CHAR);
		}
		else {
			if (l->greater(*c, *r)) index_error(ctx);
			return api_create_user(ctx, c, mat_opfunc_CHAR, mat_destroy_CHAR, MAT_CHAR);
		}

	case TOK_EQL:
		c = new zoCHAR;
		if (r == 0) {
			l->eq(*c, api_get_number(ctx, args[2]));
			return api_create_user(ctx, c, mat_opfunc_CHAR, mat_destroy_CHAR, MAT_CHAR);
		}
		else {
			if (l->eq(*c, *r)) index_error(ctx);
			return api_create_user(ctx, c, mat_opfunc_CHAR, mat_destroy_CHAR, MAT_CHAR);
		}

	case TOK_NEQ:
		c = new zoCHAR;
		if (r == 0) {
			l->ne(*c, api_get_number(ctx, args[2]));
			return api_create_user(ctx, c, mat_opfunc_CHAR, mat_destroy_CHAR, MAT_CHAR);
		}
		else {
			if (l->ne(*c, *r)) index_error(ctx);
			return api_create_user(ctx, c, mat_opfunc_CHAR, mat_destroy_CHAR, MAT_CHAR);
		}

	case '*':
		p = new OBJ_TYPE(*l);
		if (r == 0) {
			p->mul(api_get_number(ctx, args[2]));
			return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
		}
		else {
			if (!p->mul(*r)) operand_error(ctx);
			return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
		}

	case '/':
		p = new OBJ_TYPE(*l);
		if (r == 0) {
			double v = api_get_number(ctx, args[2]);
			if (v == 0) divide_error(ctx);
			p->div(v);
			return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
		}
		else {
			if (!p->div(*r)) operand_error(ctx);
			return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
		}

	case '+':
		p = new OBJ_TYPE(*l);
		if (r == 0) {
			p->add(api_get_number(ctx, args[2]));
			return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
		}
		else {
			if (!p->add(*r)) operand_error(ctx);
			return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
		}

	case '-':
		p = new OBJ_TYPE(*l);
		if (r == 0) {
			p->sub(api_get_number(ctx, args[2]));
			return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
		}
		else {
			if (!p->sub(*r)) operand_error(ctx);
			return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
		}

	case '%':
		p = new OBJ_TYPE(*l);
#ifdef MAT_REAL
		if (r == 0) {
			double v = api_get_number(ctx, args[2]);
			if (v == 0) divide_error(ctx);
			int i, n = p->ndat();
			for (i = 0; i < n; i++) (*p)(i) = fmod((*p)(i), v);
			return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
		}
		else {
			int i, n = p->ndat();
			if (r->ndat() != n) index_error(ctx);
			for (i = 0; i < n; i++) {
				double v = (*r)(i);
				if (v == 0) divide_error(ctx);
				(*p)(i) = fmod((*p)(i), v);
			}
			return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
		}
#else
		if (r == 0) {
			int v = api_get_integer(ctx, args[2]);
			if (v == 0) divide_error(ctx);
			int i, n = p->ndat();
			for (i = 0; i < n; i++) (*p)(i) %= v;
			return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
		}
		else {
			int i, n = p->ndat();
			if (r->ndat() != n) index_error(ctx);
			for (i = 0; i < n; i++) {
				int v = (*r)(i);
				if (v == 0) divide_error(ctx);
				(*p)(i) %= v;
			}
			return api_create_user(ctx, p, OP_FUNC, MAT_DESTROY, MAT_TYPE);
		}
#endif

	case TOK_MULEQ:
		if (r == 0) {
			l->mul(api_get_number(ctx, args[2]));
		}
		else {
			if (!l->mul(*r)) operand_error(ctx);
		}
		return args[0];

	case TOK_DIVEQ:
		if (r == 0) {
			double v = api_get_number(ctx, args[2]);
			if (v == 0) divide_error(ctx);
			l->div(v);
		}
		else {
			if (!l->div(*r)) operand_error(ctx);
		}
		return args[0];

	case TOK_MODEQ:
#ifdef MAT_REAL
		if (r == 0) {
			double v = api_get_number(ctx, args[2]);
			if (v == 0) divide_error(ctx);
			int i, n = l->ndat();
			for (i = 0; i < n; i++) (*l)(i) = fmod((*l)(i), v);
			return args[0];
		}
		else {
			int i, n = l->ndat();
			if (r->ndat() != n) operand_error(ctx);
			for (i = 0; i < n; i++) {
				double v = (*r)(i);
				if (v == 0) divide_error(ctx);
				(*l)(i) = fmod((*l)(i), v);
			}
			return args[0];
		}
#else
		if (r == 0) {
			int v = api_get_integer(ctx, args[2]);
			if (v == 0) divide_error(ctx);
			int i, n = l->ndat();
			for (i = 0; i < n; i++) (*l)(i) %= v;
			return args[0];
		}
		else {
			int i, n = l->ndat();
			if (r->ndat() != n) operand_error(ctx);
			for (i = 0; i < n; i++) {
				int v = (*r)(i);
				if (v == 0) divide_error(ctx);
				(*l)(i) %= v;
			}
			return args[0];
		}
#endif
	case TOK_ADDEQ:
		if (r == 0) {
			l->add(api_get_number(ctx, args[2]));
		}
		else {
			if (!l->add(*r)) operand_error(ctx);
		}
		return args[0];

	case TOK_SUBEQ:
		if (r == 0) {
			l->sub(api_get_number(ctx, args[2]));
		}
		else {
			if (!l->sub(*r)) operand_error(ctx);
		}
		return args[0];
	}

	api_runtime_error(ctx, "undefined operation for integer matrix");
	return 0;
}
