#include "matrixtmp.h"
#include "matrix.h"
#include "api.h"
#include <math.h>
#include <ctype.h>
#include <string.h>

#pragma warning(disable: 4244)

#define DEG_TO_RAD		0.017453292519943 


zsRegPrimitive::zsRegPrimitive(const char* name, int type, void* func)
{
	api_add_primitive(name, type, func);
}

void index_error(void* ctx)   { api_runtime_error(ctx, "matrix index or dimension error"); }
void divide_error(void* ctx)  { api_runtime_error(ctx, "divided by zero"); }
void operand_error(void* ctx) { api_runtime_error(ctx, "different type/size of opeands"); }


void mat_destroy_CHAR(void* mat)   { delete (zoCHAR*)mat; mat = 0; }
void mat_destroy_UCHAR(void* mat)  { delete (zoUCHAR*)mat; mat = 0; }
void mat_destroy_SHORT(void* mat)  { delete (zoSHORT*)mat; mat = 0; }
void mat_destroy_USHORT(void* mat) { delete (zoUSHORT*)mat; mat = 0; }
void mat_destroy_INT(void* mat)    { delete (zoINT*)mat; mat = 0; }
void mat_destroy_UINT(void* mat)   { delete (zoUINT*)mat; mat = 0; }
void mat_destroy_FLOAT(void* mat)  { delete (zoFLOAT*)mat; mat = 0; }
void mat_destroy_DOUBLE(void* mat) { delete (zoDOUBLE*)mat; mat = 0; }

extern void* matCreateCHAR(void *ctx, int nargs, void** args);
extern void* matCreateUCHAR(void *ctx, int nargs, void** args);
extern void* matCreateSHORT(void *ctx, int nargs, void** args);
extern void* matCreateUSHORT(void *ctx, int nargs, void** args);
extern void* matCreateINT(void *ctx, int nargs, void** args);
extern void* matCreateUINT(void *ctx, int nargs, void** args);
extern void* matCreateFLOAT(void *ctx, int nargs, void** args);
extern void* matCreateDOUBLE(void *ctx, int nargs, void** args);

static zsRegPrimitive mat1("char", 0, matCreateCHAR);
static zsRegPrimitive mat2("uchar", 0, matCreateUCHAR);
static zsRegPrimitive mat3("short", 0, matCreateSHORT);
static zsRegPrimitive mat4("ushort", 0, matCreateUSHORT);
static zsRegPrimitive mat5("int", 0, matCreateINT);
static zsRegPrimitive mat6("uint", 0, matCreateUINT);
static zsRegPrimitive mat7("float", 0, matCreateFLOAT);
static zsRegPrimitive mat8("double", 0, matCreateDOUBLE);

void* mat_create(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);

	if (api_is_number(args[0])) {
		zoDOUBLE *p = new zoDOUBLE;
		p->resize(1, nargs);
		for (int i = 0; i < nargs; i++) (*p)(i) = api_get_number(ctx, args[i]);
		return api_create_user(ctx, p, mat_opfunc_DOUBLE, mat_destroy_DOUBLE, MAT_DOUBLE);
	}

	const char* s = api_get_string(ctx, args[0]);
	int nrow = 0, ncol = 0, type = 0;
	void *ptr = 0;

	if (nargs > 1) {
		type = api_get_type(args[1]);
		if (type < MAT_CHAR || type > MAT_DOUBLE) {
			nrow = api_get_integer(ctx, args[1]);
			if (nrow <= 0) api_input_error(ctx);
			ncol = 1;
		}
		else {
			ptr = api_get_ptr(ctx,args[1]);
		}
	}

	if (!ptr && nargs > 2) {
		ncol = api_get_integer(ctx, args[2]);
		if (ncol <= 0) api_input_error(ctx);
	}

	if (strcmp(s, "double") == 0) {
		zoDOUBLE *p = new zoDOUBLE;
		if (nrow > 0 && ncol > 0) p->resize(nrow, ncol);
		if (ptr) mat_clone(ctx, p, MAT_DOUBLE, ptr, type);
		return api_create_user(ctx, p, mat_opfunc_DOUBLE, mat_destroy_DOUBLE, MAT_DOUBLE);
	}
	if (strcmp(s, "int") == 0) {
		zoINT *p = new zoINT;
		if (nrow > 0 && ncol > 0) p->resize(nrow, ncol);
		if (ptr) mat_clone(ctx, p, MAT_INT, ptr, type);
		return api_create_user(ctx, p, mat_opfunc_INT, mat_destroy_INT, MAT_INT);
	}
	if (strcmp(s, "uint") == 0) {
		zoUINT *p = new zoUINT;
		if (nrow > 0 && ncol > 0) p->resize(nrow, ncol);
		if (ptr) mat_clone(ctx, p, MAT_UINT, ptr, type);
		return api_create_user(ctx, p, mat_opfunc_UINT, mat_destroy_UINT, MAT_UINT);
	}
	if (strcmp(s, "float") == 0) {
		zoFLOAT *p = new zoFLOAT;
		if (nrow > 0 && ncol > 0) p->resize(nrow, ncol);
		if (ptr) mat_clone(ctx, p, MAT_FLOAT, ptr, type);
		return api_create_user(ctx, p, mat_opfunc_FLOAT, mat_destroy_FLOAT, MAT_FLOAT);
	}
	if (strcmp(s, "short") == 0) {
		zoSHORT *p = new zoSHORT;
		if (nrow > 0 && ncol > 0) p->resize(nrow, ncol);
		if (ptr) mat_clone(ctx, p, MAT_SHORT, ptr, type);
		return api_create_user(ctx, p, mat_opfunc_SHORT, mat_destroy_SHORT, MAT_SHORT);
	}
	if (strcmp(s, "ushort") == 0) {
		zoUSHORT *p = new zoUSHORT;
		if (nrow > 0 && ncol > 0) p->resize(nrow, ncol);
		if (ptr) mat_clone(ctx, p, MAT_USHORT, ptr, type);
		return api_create_user(ctx, p, mat_opfunc_USHORT, mat_destroy_USHORT, MAT_USHORT);
	}
	if (strcmp(s, "char") == 0) {
		zoCHAR *p = new zoCHAR;
		if (nrow > 0 && ncol > 0) p->resize(nrow, ncol);
		if (ptr) mat_clone(ctx, p, MAT_CHAR, ptr, type);
		return api_create_user(ctx, p, mat_opfunc_CHAR, mat_destroy_CHAR, MAT_CHAR);
	}
	if (strcmp(s, "uchar") == 0) {
		zoUCHAR *p = new zoUCHAR;
		if (nrow > 0 && ncol > 0) p->resize(nrow, ncol);
		if (ptr) mat_clone(ctx, p, MAT_UCHAR, ptr, type);
		return api_create_user(ctx, p, mat_opfunc_UCHAR, mat_destroy_UCHAR, MAT_UCHAR);
	}
	api_input_error(ctx);
	return 0;
}
static zsRegPrimitive mat0("matrix", 0, mat_create);


/////////////////////////////////////////////////////////////////////


int math_cal2jul(int yy, int mm, int dd)
{
	yy -= mm < 3;
	mm += 12 * (mm < 3);
	int a = 365.25 * (yy + 4716),
		b = 30.6001 * (mm + 1);
	return a + b + dd - 2416556;
}

double math_jul2cal(int *yy, int *mm, int *dd, double jul)
{
	int a = jul + 2415032,
		b = a + 1524,
		c = (double(b) - 122.1) / 365.25,
		d = 365.25 * c,
		e = double(b - d) / 30.6001,
		f = 30.6001 * e;

	*dd = b - d - f;
	*mm = e - 1 - 12 * (e > 13);
	*yy = c - 4715 - (*mm > 2);

	return jul - floor(jul);
}

int math_jul2tim(int *hh, int *mn, int *sc, double jul)
{
	const double sd = 1.0e-10; // this constant may be OS or compiler dependent
	if (jul == 0.0) {
		*hh = 0; *mn = 0; *sc = 0;
		return 0;
	}
	if (jul + sd < 1.0) jul += sd; 
	jul = fabs(jul) * 24.;
	*hh = jul;
	jul = (jul - *hh) * 60.;
	*mn = jul;
	jul = (jul - *mn) * 60.;
	*sc = jul;
	return (jul - *sc) * 1000;
}


/////////////////////////////////////////////////////////////////////


void mat_clone(void *ctx, void* m1, int type1, void* m2, int type2)
{
	switch(type1) {

	case MAT_CHAR:
		{
		zoCHAR *me = reinterpret_cast<zoCHAR*>(m1);

		switch (type2) {
		case MAT_CHAR: {
			zoCHAR* src = reinterpret_cast<zoCHAR*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_UCHAR: {
			zoUCHAR* src = reinterpret_cast<zoUCHAR*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_SHORT: {
			zoSHORT* src = reinterpret_cast<zoSHORT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_USHORT: {
			zoUSHORT* src = reinterpret_cast<zoUSHORT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_INT: {
			zoINT* src = reinterpret_cast<zoINT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_UINT: {
			zoUINT* src = reinterpret_cast<zoUINT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_FLOAT: {
			zoFLOAT* src = reinterpret_cast<zoFLOAT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_DOUBLE: {
			zoDOUBLE* src = reinterpret_cast<zoDOUBLE*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		}
		}
		break;

	case MAT_UCHAR:
		{
		zoUCHAR *me = reinterpret_cast<zoUCHAR*>(m1);

		switch (type2) {
		case MAT_CHAR: {
			zoCHAR* src = reinterpret_cast<zoCHAR*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_UCHAR: {
			zoUCHAR* src = reinterpret_cast<zoUCHAR*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_SHORT: {
			zoSHORT* src = reinterpret_cast<zoSHORT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_USHORT: {
			zoUSHORT* src = reinterpret_cast<zoUSHORT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_INT: {
			zoINT* src = reinterpret_cast<zoINT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_UINT: {
			zoUINT* src = reinterpret_cast<zoUINT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_FLOAT: {
			zoFLOAT* src = reinterpret_cast<zoFLOAT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_DOUBLE: {
			zoDOUBLE* src = reinterpret_cast<zoDOUBLE*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		}
		}
		break;

	case MAT_SHORT:
		{
		zoSHORT *me = reinterpret_cast<zoSHORT*>(m1);

		switch (type2) {
		case MAT_CHAR: {
			zoCHAR* src = reinterpret_cast<zoCHAR*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_UCHAR: {
			zoUCHAR* src = reinterpret_cast<zoUCHAR*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_SHORT: {
			zoSHORT* src = reinterpret_cast<zoSHORT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_USHORT: {
			zoUSHORT* src = reinterpret_cast<zoUSHORT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_INT: {
			zoINT* src = reinterpret_cast<zoINT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_UINT: {
			zoUINT* src = reinterpret_cast<zoUINT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_FLOAT: {
			zoFLOAT* src = reinterpret_cast<zoFLOAT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_DOUBLE: {
			zoDOUBLE* src = reinterpret_cast<zoDOUBLE*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		}
		}
		break;

	case MAT_USHORT:
		{
		zoUSHORT *me = reinterpret_cast<zoUSHORT*>(m1);

		switch (type2) {
		case MAT_CHAR: {
			zoCHAR* src = reinterpret_cast<zoCHAR*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_UCHAR: {
			zoUCHAR* src = reinterpret_cast<zoUCHAR*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_SHORT: {
			zoSHORT* src = reinterpret_cast<zoSHORT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_USHORT: {
			zoUSHORT* src = reinterpret_cast<zoUSHORT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_INT: {
			zoINT* src = reinterpret_cast<zoINT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_UINT: {
			zoUINT* src = reinterpret_cast<zoUINT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_FLOAT: {
			zoFLOAT* src = reinterpret_cast<zoFLOAT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_DOUBLE: {
			zoDOUBLE* src = reinterpret_cast<zoDOUBLE*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		}
		}
		break;

	case MAT_INT:
		{
		zoINT *me = reinterpret_cast<zoINT*>(m1);

		switch (type2) {
		case MAT_CHAR: {
			zoCHAR* src = reinterpret_cast<zoCHAR*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_UCHAR: {
			zoUCHAR* src = reinterpret_cast<zoUCHAR*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_SHORT: {
			zoSHORT* src = reinterpret_cast<zoSHORT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_USHORT: {
			zoUSHORT* src = reinterpret_cast<zoUSHORT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_INT: {
			zoINT* src = reinterpret_cast<zoINT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_UINT: {
			zoUINT* src = reinterpret_cast<zoUINT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_FLOAT: {
			zoFLOAT* src = reinterpret_cast<zoFLOAT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_DOUBLE: {
			zoDOUBLE* src = reinterpret_cast<zoDOUBLE*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		}
		}
		break;

	case MAT_UINT:
		{
		zoUINT *me = reinterpret_cast<zoUINT*>(m1);

		switch (type2) {
		case MAT_CHAR: {
			zoCHAR* src = reinterpret_cast<zoCHAR*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_UCHAR: {
			zoUCHAR* src = reinterpret_cast<zoUCHAR*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_SHORT: {
			zoSHORT* src = reinterpret_cast<zoSHORT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_USHORT: {
			zoUSHORT* src = reinterpret_cast<zoUSHORT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_INT: {
			zoINT* src = reinterpret_cast<zoINT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_UINT: {
			zoUINT* src = reinterpret_cast<zoUINT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_FLOAT: {
			zoFLOAT* src = reinterpret_cast<zoFLOAT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_DOUBLE: {
			zoDOUBLE* src = reinterpret_cast<zoDOUBLE*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		}
		}
		break;

	case MAT_FLOAT:
		{
		zoFLOAT *me = reinterpret_cast<zoFLOAT*>(m1);

		switch (type2) {
		case MAT_CHAR: {
			zoCHAR* src = reinterpret_cast<zoCHAR*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_UCHAR: {
			zoUCHAR* src = reinterpret_cast<zoUCHAR*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_SHORT: {
			zoSHORT* src = reinterpret_cast<zoSHORT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_USHORT: {
			zoUSHORT* src = reinterpret_cast<zoUSHORT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_INT: {
			zoINT* src = reinterpret_cast<zoINT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_UINT: {
			zoUINT* src = reinterpret_cast<zoUINT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_FLOAT: {
			zoFLOAT* src = reinterpret_cast<zoFLOAT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_DOUBLE: {
			zoDOUBLE* src = reinterpret_cast<zoDOUBLE*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		}
		}
		break;

	case MAT_DOUBLE: 
		{
		zoDOUBLE *me = reinterpret_cast<zoDOUBLE*>(m1);

		switch (type2) {
		case MAT_CHAR: {
			zoCHAR* src = reinterpret_cast<zoCHAR*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_UCHAR: {
			zoUCHAR* src = reinterpret_cast<zoUCHAR*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_SHORT: {
			zoSHORT* src = reinterpret_cast<zoSHORT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_USHORT: {
			zoUSHORT* src = reinterpret_cast<zoUSHORT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_INT: {
			zoINT* src = reinterpret_cast<zoINT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_UINT: {
			zoUINT* src = reinterpret_cast<zoUINT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_FLOAT: {
			zoFLOAT* src = reinterpret_cast<zoFLOAT*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		case MAT_DOUBLE: {
			zoDOUBLE* src = reinterpret_cast<zoDOUBLE*>(m2);
			me->resize(src->nrow(), src->ncol());
			for (int i = 0; i < src->ndat(); i++) (*me)(i) = (*src)(i);
			return;
			}
		}
		}
		break;
	}

	api_runtime_error(ctx, "call clone with unexpected type");
}

void mat_import(void *ctx, void* ptr1, int type, void* ptr2)
{
	switch(type) {

	case MAT_CHAR:
		{
		zoCHAR *me = reinterpret_cast<zoCHAR*>(ptr1);
		if (me->ptr()) memcpy(me->ptr(), ptr2, me->ndat());
		}
		break;

	case MAT_UCHAR:
		{
		zoUCHAR *me = reinterpret_cast<zoUCHAR*>(ptr1);
		if (me->ptr()) memcpy(me->ptr(), ptr2, me->ndat());
		}
		break;

	case MAT_SHORT:
		{
		zoSHORT *me = reinterpret_cast<zoSHORT*>(ptr1);
		if (me->ptr()) memcpy(me->ptr(), ptr2, me->ndat()*sizeof(short));
		}
		break;

	case MAT_USHORT:
		{
		zoUSHORT *me = reinterpret_cast<zoUSHORT*>(ptr1);
		if (me->ptr()) memcpy(me->ptr(), ptr2, me->ndat()*sizeof(unsigned short));
		}
		break;

	case MAT_INT:
		{
		zoINT *me = reinterpret_cast<zoINT*>(ptr1);
		if (me->ptr()) memcpy(me->ptr(), ptr2, me->ndat()*sizeof(int));
		}
		break;

	case MAT_UINT:
		{
		zoUINT *me = reinterpret_cast<zoUINT*>(ptr1);
		if (me->ptr()) memcpy(me->ptr(), ptr2, me->ndat()*sizeof(unsigned int));
		}
		break;

	case MAT_FLOAT:
		{
		zoFLOAT *me = reinterpret_cast<zoFLOAT*>(ptr1);
		if (me->ptr()) memcpy(me->ptr(), ptr2, me->ndat()*sizeof(float));
		}
		break;

	case MAT_DOUBLE: 
		{
		zoDOUBLE *me = reinterpret_cast<zoDOUBLE*>(ptr1);
		if (me->ptr()) memcpy(me->ptr(), ptr2, me->ndat()*sizeof(double));
		}
		break;
	default:
		api_runtime_error(ctx, "unexpected type");
	}
}

size_t get_array_index(void *ctx, void *obj, size_t n, size_t idx[3])
{
	if (api_is_integer(obj)) {
		idx[0] = api_get_integer(ctx, obj);
		idx[1] = idx[0];
		idx[2] = 1;
		if (idx[0] < 0 || idx[0] >= n) index_error(ctx);
		return 1;
	}
	if (api_is_null(obj)) {
		idx[0] = 0;
		idx[1] = n - 1;
		idx[2] = 1;
		return n;
	}
	if (!api_is_array(obj)) {
		api_runtime_error(ctx, "matrix index must be integer, null, array (range), or char type matrix"); 
	}
	void *a = api_get_array_object(ctx, obj, "0");
	if (api_is_null(a))
		idx[0] = 0;
	else
		idx[0] = api_get_integer(ctx, a);
	a = api_get_array_object(ctx, obj, "1");
	if (api_is_null(a))
		idx[1] = n - 1;
	else
		idx[1] = api_get_integer(ctx, a);
	if (api_get_array_size(ctx, obj) > 2)
		idx[2] = api_get_integer(ctx, api_get_array_object(ctx, obj, "2"));
	else
		idx[2] = 1;
	if (idx[0] < 0 || idx[1] < idx[0] || idx[2] <= 0 || idx[1] >= n) index_error(ctx);
	return (idx[1] - idx[0]) / idx[2] + 1;
}

