#include "clipping.h"
#include <math.h>
#include <list>

#define EPS 1.0e-20
#define D2R 0.017453292519943    // degree to radius

#define ABS(a) ((a) > 0 ? (a) : -(a))
#define MIN(a, b) ((a) < (b) ? (a) : (b))
#define MAX(a, b) ((a) > (b) ? (a) : (b))

#define DOT2D(x1, y1, x2, y2) ((x1)*(x2) + (y1)*(y2))
#define DOT3D(x1, y1, z1, x2, y2, z2) ((x1)*(x2) + (y1)*(y2) + (z1)*(z2))
#define PERP2D(x1, y1, x2, y2) ((x1)*(y2) - (x2)*(y1))

using namespace std;

enum { BD_LEFT, BD_BOTTOM, BD_RIGHT, BD_TOP };


bool inside_rectangle(const xy_t& p, const rect_t& rect, int edge)
{
	switch (edge)
	{
	case BD_LEFT:
		return p.x >= rect.x;
	case BD_RIGHT:
		return p.x <= rect.x + rect.w;
	case BD_TOP:
		return p.y <= rect.y + rect.h;
	case BD_BOTTOM:
		return p.y >= rect.y;
	default:
		return true;
	}
}

///////////////////////////////////////////
// p should be outside and q inside edge
//
xy_t intersect_edge(const xy_t& p, const xy_t& q, const rect_t& rect, int edge)
{
	real_t t;
	switch (edge)
	{
	case BD_LEFT:
		t = (rect.x - p.x) / (q.x - p.x);
		break;
	case BD_RIGHT:
		t = (p.x - rect.x - rect.w) / (p.x - q.x);
		break;
	case BD_TOP:
		t = (p.y - rect.y - rect.h) / (p.y - q.y);
		break;
	case BD_BOTTOM:
		t = (rect.y - p.y) / (q.y - p.y);
		break;
	}
	xy_t r = { (1-t)*p.x + t*q.x, (1-t)*p.y + t*q.y };
	return r;
}

void clip_edge(vector<xy_t>& out, const vector<xy_t>& in, const rect_t& rect, int edge)
{
	if (in.size() < 2) return;

	xy_t r, p = in.back();
	int i = 0;

	if (p.x == in[0].x && p.y == in[0].y) {
		p = in[0];
		i = 1;
	}

	bool flag = inside_rectangle(p, rect, edge);

	while (i < in.size()) {
		r = in[i];
		if (flag) {
			if (inside_rectangle(in[i], rect, edge)) {
				// p inside, q inside
				out.push_back(p);
				p = in[i++];
				flag = true;
			}
			else {
				// p inside, q outside
				out.push_back(p);
				out.push_back(intersect_edge(in[i], p, rect, edge));
				p = in[i++];
				flag = false;
			}
		}
		else {
			if (inside_rectangle(in[i], rect, edge)) {
				// p outside, q inside
				out.push_back(intersect_edge(p, in[i], rect, edge));
				p = in[i++];
				flag = true;
			}
			else {
				// p outside, q outside
				p = in[i++];
				flag = false;
			}
		}
	}
}

void clip_rectangle(vector<xy_t>& out, const real_t *x, const real_t *y, int n, const rect_t& rect)
{
	vector<xy_t> in(n);

	if (polygon_area(x, y, n) > 0) {
		for (int i = 0; i < n; i++) {
			xy_t p = { x[i], y[i] };
			in[i] = p;
		}
	}
	else {
		for (int i = n-1; i >= 0; i--) {
			xy_t p = { x[i], y[i] };
			in[i] = p;
		}
	}

	clip_edge(out, in, rect, BD_LEFT);
	in.clear();
	clip_edge(in, out, rect, BD_BOTTOM);
	out.clear();
	clip_edge(out, in, rect, BD_RIGHT);
	in.clear();
	clip_edge(in, out, rect, BD_TOP);
	out = in;
}

int line_intersect(real_t out[2], const real_t p1[2], const real_t p2[2], const real_t q1[2], const real_t q2[2])
{
	real_t dpx = p2[0] - p1[0];
	real_t dpy = p2[1] - p1[1];
	real_t dqx = q2[0] - q1[0];
	real_t dqy = q2[1] - q1[1];

	real_t s = dqx*dpy - dqy*dpx;
	if (s == 0) return 0;

	s = (p1[0]*dpy - p1[1]*dpx - q1[0]*dpy + q1[1]*dpx) / s;
	if (s <= 0 || s >= 1) return 0;

	out[0] = q1[0] + s*dqx;
	out[1] = q1[1] + s*dqy;
	return 1;
}

// http://www.ecse.rpi.edu/Homepages/wrf/Research/Short_Notes/pnpoly.html
// W. Randolph Franklin
//
int point_in_polygon(const real_t *xp, const real_t *yp, int n, real_t x, real_t y)
{
	int i, j, c = 0;
	for (i = 0, j = n-1; i < n; j = i++) {
		if ((((yp[i] <= y) && (y < yp[j])) ||
			 ((yp[j] <= y) && (y < yp[i]))) &&
			(x < (xp[j] - xp[i]) * (y - yp[i]) / (yp[j] - yp[i]) + xp[i]))
		c = !c;
	}
	return c;
}

real_t polygon_area(const real_t *x, const real_t *y, int n)
{
	real_t x0 = x[0];
	real_t y0 = y[0];
	real_t x1 = x[1] - x0;
	real_t y1 = y[1] - y0;
	real_t a = 0;
	for (int i = 2; i < n; i++) {
		real_t x2 = x[i] - x0;
		real_t y2 = y[i] - y0;
		a += PERP2D(x1, y1, x2, y2);
		x1 = x2;
		y1 = y2;
	}
	return 0.5*a;
}

list<xyz_t>::iterator ear_cut_erase(list<xyz_t>& in, list<xyz_t>::iterator pos)
{
		pos = in.erase(pos);

		list<xyz_t>::iterator pos1 = pos, pos2 = pos;
		pos1--;
		pos2++;
		if (pos2 == in.end()) pos2 = in.begin();

		real_t z = PERP2D((*pos2).x-(*pos).x, (*pos2).y-(*pos).y, (*pos1).x-(*pos).x, (*pos1).y-(*pos).y);

		if (z != 0) { (*pos).z = z; } else { (*pos).z = 0; }

		if (pos1 != in.begin()) { pos2 = pos1; } else { pos2 = in.end(); }
		pos2--;

		z = PERP2D((*pos).x-(*pos1).x, (*pos).y-(*pos1).y, (*pos2).x-(*pos1).x, (*pos2).y-(*pos1).y);

		if (z != 0) { (*pos1).z = z; } else { (*pos1).z = 0; }

		return pos;
}

list<xyz_t>::iterator ear_cut_triangulate(vector<xy_t>& out, list<xyz_t>& in, list<xyz_t>::iterator pos)
{
	if (in.size() == 3) {
		pos = in.begin();
		pos++;
		if (in.front().z != 0 && (*pos).z != 0 && in.back().z != 0) {
			xy_t p1 = { in.front().x, in.front().y };
			out.push_back(p1);
			in.pop_front();
			xy_t p2 = { in.front().x, in.front().y };
			out.push_back(p2);
			xy_t p3 = { in.back().x, in.back().y };
			out.push_back(p3);
		}
		else {
			in.pop_back();
		}
		return in.begin();
	}

	if (pos == in.end()) pos = in.begin();
	if (pos == in.begin()) pos++;
	if ((*pos).z == 0) return ear_cut_erase(in, pos);
	if ((*pos).z < 0) return ++pos;

	list<xyz_t>::iterator pos1 = pos, pos2 = pos;
	pos1--;
	pos2++;
	if (pos2 == in.end()) pos2 = in.begin();

	real_t x[3] = { (*pos1).x, (*pos).x, (*pos2).x };
	real_t y[3] = { (*pos1).y, (*pos).y, (*pos2).y };

	int inside = 0;

	if (pos2 != in.begin()) {
		while(++pos2 != in.end()) {
			if ((*pos2).z < 0) {
				inside = point_in_polygon(x, y, 3, (*pos2).x, (*pos2).y);
				if (inside > 0) break;
			}
		}
	}

	if (inside <= 0 && pos1 != in.begin()) {
		while(--pos1 != in.begin()) {
			if ((*pos1).z < 0) {
				inside = point_in_polygon(x, y, 3, (*pos1).x, (*pos1).y);
				if (inside > 0) break;
			}
		}
		if ((*pos1).z < 0) inside = point_in_polygon(x, y, 3, (*pos1).x, (*pos1).y);
	}

	if (inside <= 0) {
		pos1 = pos;
		pos1--;
		pos2 = pos;
		pos2++;
		if (pos2 == in.end()) pos2 = in.begin();

		xy_t p1 = { (*pos1).x, (*pos1).y };
		out.push_back(p1);
		xy_t p2 = { (*pos).x, (*pos).y };
		out.push_back(p2);
		xy_t p3 = { (*pos2).x, (*pos2).y };
		out.push_back(p3);

		pos = ear_cut_erase(in, pos);
	}
	else {
		pos++;
	}

	return pos;
}

void refine_triangle(vector<xy_t>& out, const xy_t &p1, const xy_t &p2, const xy_t &p3, real_t factor, bool sphere)
{
	real_t dx, dy, d12, d23, d31;
	
	if (sphere) {
		real_t cosy1 = cos(p1.y*D2R);
		real_t cosy2 = cos(p2.y*D2R);
		real_t cosy3 = cos(p3.y*D2R);
		real_t siny1 = sin(p1.y*D2R);
		real_t siny2 = sin(p2.y*D2R);
		real_t siny3 = sin(p3.y*D2R);

		d12  = acos(cosy1*cosy2*cos((p1.x-p2.x)*D2R) + siny1*siny2);
		d23  = acos(cosy2*cosy3*cos((p2.x-p3.x)*D2R) + siny2*siny3);
		d31  = acos(cosy1*cosy3*cos((p1.x-p3.x)*D2R) + siny1*siny3);
	}
	else {
		dx = p2.x - p1.x;
		dy = p2.y - p1.y;
		d12  = sqrt(dx*dx + dy*dy);

		dx = p3.x - p2.x;
		dy = p3.y - p2.y;
		d23  = sqrt(dx*dx + dy*dy);

		dx = p1.x - p3.x;
		dy = p1.y - p3.y;
		d31  = sqrt(dx*dx + dy*dy);
	}

	if (d12 > d23) {
		if (d12 > d31) {
			// break d12
			if (d12 <= factor) {
				out.push_back(p1);
				out.push_back(p2);
				out.push_back(p3);
			}
			else {
				xy_t p = { .5*(p1.x+p2.x), .5*(p1.y+p2.y) };
				refine_triangle(out, p1, p, p3, factor, sphere);
				refine_triangle(out, p, p2, p3, factor, sphere);
			}
			return;
		}
	}
	else {
		if (d23 > d31) {
			// break d23
			if (d23 <= factor) {
				out.push_back(p1);
				out.push_back(p2);
				out.push_back(p3);
			}
			else {
				xy_t p = { .5*(p2.x+p3.x), .5*(p2.y+p3.y) };
				refine_triangle(out, p1, p2, p, factor, sphere);
				refine_triangle(out, p1, p, p3, factor, sphere);
			}
			return;
		}
	}

	// break d31

	if (d31 <= factor) {
		out.push_back(p1);
		out.push_back(p2);
		out.push_back(p3);
	}
	else {
		xy_t p = { .5*(p1.x+p3.x), .5*(p1.y+p3.y) };
		refine_triangle(out, p1, p2, p, factor, sphere);
		refine_triangle(out, p2, p3, p, factor, sphere);
	}
}

void ear_cut_triangulate(vector<xy_t>& out, const real_t *x, const real_t *y, int n, real_t refine, bool sphere)
{
	if (n <= 3) return;

	if (polygon_area(x, y, n) > 0){
		list<xyz_t> tmp;
		for (int i = 0, i1 = n-1, i2 = 1; i < n; i++, i1++, i2++) {
			if (i1 >= n) i1 = 0;
			if (i2 >= n) i2 = 0;
			real_t z = PERP2D(x[i2]-x[i], y[i2]-y[i], x[i1]-x[i], y[i1]-y[i]);
			if (z != 0) {
				xyz_t p = { x[i], y[i], z };
				tmp.push_back(p);
			}
		}
		list<xyz_t>::iterator pos = tmp.begin();
		while (tmp.size() >= 3) pos = ear_cut_triangulate(out, tmp, pos);
	}
	else {
		list<xyz_t> tmp;
		for (int i = n-1, i1 = 0, i2 = n-2; i >= 0; i--, i1--, i2--) {
			if (i1 < 0) i1 = n-1;
			if (i2 < 0) i2 = n-1;
			real_t z = PERP2D(x[i2]-x[i], y[i2]-y[i], x[i1]-x[i], y[i1]-y[i]);
			if (z != 0) {
				xyz_t p = { x[i], y[i], z };
				tmp.push_back(p);
			}
		}
		list<xyz_t>::iterator pos = tmp.begin();
		while (tmp.size() >= 3) pos = ear_cut_triangulate(out, tmp, pos);
	}

	if (refine > 0) {
		vector<xy_t> tmp;
		for (int i = 2; i < out.size(); i += 3) refine_triangle(tmp, out[i-2], out[i-1], out[i], refine, sphere);
		out = tmp;
	}
}