#include "hdf5.h"
#include "api.h"
#include "buffer.h"
#include <stdio.h>
#include <stdlib.h>
#include <string>

#ifdef _WIN32
#pragma warning(disable: 4018)
#pragma warning(disable: 4244)
#endif

#define HDF_TYPE	'H5'
#define BUFF_SIZE	1024

/////////////////////////////////////////////////////////////////////

enum { HDF_CHAR, HDF_UCHAR, HDF_SHORT, HDF_USHORT, HDF_INT, HDF_UINT, HDF_FLOAT, HDF_DOUBLE, HDF_STRING};

static char g_tname[10][10]={ "char", "uchar", "short", "ushort", "int", "uint", "float", "double", "string", "unknown" };

static const char *type2name(int type)
{
	switch (type) {
	case HDF_CHAR:
		return g_tname[0];
	case HDF_UCHAR:
		return g_tname[1];
	case HDF_SHORT:
		return g_tname[2];
	case HDF_USHORT:
		return g_tname[3];
	case HDF_INT:
		return g_tname[4];
	case HDF_UINT:
		return g_tname[5];
	case HDF_FLOAT:
		return g_tname[6];
	case HDF_DOUBLE:
		return g_tname[7];
	case HDF_STRING:
		return g_tname[8];
	default:
		return g_tname[9];
	}
	return g_tname[9];
}

/////////////////////////////////////////////////////////////////////

class zsAttID
{
public:
	zsAttID(hid_t id, const char *name);

	~zsAttID();

	hid_t   aid,		// attribute ID
			tid,		// type ID
			sid;		// space ID
	int ndim;			// number of dimensions of a attribute
	hsize_t dims[H5S_MAX_RANK];		// dimensions	
};

zsAttID::zsAttID(hid_t id, const char*name) : aid(-1), tid(-1), sid(-1), ndim(0)
{
	if (id >= 0 && name) {
		ndim=0;
		if (sid >= 0) H5Sclose(sid);
		sid=-1;
		if (tid >= 0) H5Tclose(tid);
		tid=-1;
		if (aid >= 0) H5Aclose(aid);
		aid = H5Aopen_name(id, name);
		if (aid >= 0) {
			tid = H5Aget_type(aid);
			if (tid >= 0) {
				sid = H5Aget_space(aid);
				if (sid >= 0) {
					ndim = H5Sget_simple_extent_ndims(sid);
					if (ndim > 0) H5Sget_simple_extent_dims(sid, dims, NULL);
				}
				else {
					H5Tclose(tid);
					tid=-1;
					H5Aclose(aid);
					aid=-1;
				}
			}
			else {
				H5Aclose(aid);
				aid = -1;
			}
		}
	}
}

zsAttID::~zsAttID()
{
	if (sid >= 0) H5Sclose(sid);
	if (tid >= 0) H5Tclose(tid);
	if (aid >= 0) H5Aclose(aid);
}

class zsHDF
{
	size_t size;					// ptr bytes

public:
	zsHDF();

	~zsHDF();

	void checksize();

	void checkindex(void *ctx, int nargs, void** args);

	void checkdataset(void *ctx);

	void hyperslab(void *ctx);

	hid_t	fid,		// file ID
			gid,		// group ID
			did,		// dataset ID
			tid,		// dataset type ID
			sid,		// dataset space ID
			mid;		// memory space ID

	hsize_t	np,						// number of data points
			ns,						// number of delected points
			dims[H5S_MAX_RANK],		// dimensions
			mstart[H5S_MAX_RANK],	// memory start
			mcount[H5S_MAX_RANK],	// memory count
			dstart[H5S_MAX_RANK],	// dataset start
			dcount[H5S_MAX_RANK];	// dataset count

	size_t	ndim,		// number of dimensions
			tsize;		// bytes of datum type

	bool flag;		// true if dataset is the target; otherwise group is the target.
	int type;		// zs type
	int endian;		// -1: little endian; 0: native; 1: bigendian
	int machine;	// machine endian
	void *ptr;		// pointer to extracted data
};

zsHDF::zsHDF() : fid(-1), gid(-1), did(-1), tid(-1), sid(-1), mid(-1), endian(0), flag(false), ndim(0), size(0), ptr(0), np(0), ns(0)
{
	union probe{ 
		unsigned int num;
		unsigned char bytes[sizeof(unsigned int)];
	};
	probe p = { 1U };
	if (p.bytes[0] == 1U) {
		machine = -1;
	}
	else {
		machine = 1;
	}
}

zsHDF::~zsHDF()
{
	if (mid >= 0) H5Sclose(mid);
	if (sid >= 0) H5Sclose(sid);
	if (tid >= 0) H5Tclose(tid);
	if (did >= 0) H5Dclose(did);
	if (gid >= 0) H5Gclose(gid);
	if (fid >= 0) H5Fclose(fid);
	if (ptr) free(ptr);
}

void zsHDF::checksize()
{
	size_t n = ns*tsize;
	if (size < n || size > 2*n) {
		if (ptr) free(ptr);
		ptr = malloc(n);
		size = n;
	}
}

void zsHDF::hyperslab(void *ctx)
{
	if (mid >= 0) H5Sclose(mid);
	mid  = H5Screate_simple(1, mcount, NULL);
	if (sid >= 0) H5Sclose(sid);
	sid = H5Dget_space(did);
	if (H5Sselect_hyperslab(sid, H5S_SELECT_SET, dstart, NULL, dcount, NULL) < 0 ||
		H5Sselect_hyperslab(mid, H5S_SELECT_SET, mstart, NULL, mcount, NULL) > 0)
		api_runtime_error(ctx, "failed in H5Select_hyperslab");
}

void zsHDF::checkdataset(void *ctx)
{
	if (!flag) api_runtime_error(ctx, "dataset not specified");
}

void zsHDF::checkindex(void *ctx, int nargs, void** args)
{
	if (nargs-1 > ndim) api_runtime_error(ctx, "too many dimension specifiers");

	checkdataset(ctx);

	int i, k;

	ns = 1;
	for (k = 0; k < ndim; k++) {
		dstart[k] = 0;
		dcount[k] = dims[k];
		ns *= dims[k];
	}

	for (i = 1; i < nargs; i++) {
		k = i - 1;
		if (k >= ndim) api_runtime_error(ctx, "index number exceeds dataset dimension number");
		if (!api_is_null(args[i])) {
			ns /= dims[k];
			if (api_is_integer(args[i])) {
				// e.g., hfd[3]
				dstart[k] = api_get_integer(ctx, args[i]);
				dcount[k] = 1;
			}
			else {
				// e.g. hfd[3:?]
				dstart[k] = api_get_integer(ctx, api_get_array_object(ctx, args[i], "0"));
				if (dstart[k] < 0 || dstart[k] >= dims[k]) api_runtime_error(ctx, "invalid start index");
				void *a1 = api_get_array_object(ctx, args[i], "1");
				if (api_is_null(a1)) {
					// e.g. hdf[3:*]
					dcount[k] = dims[k] - dstart[k];
				}
				else {
					// e.g. hdf[3:5]
					dcount[k] = api_get_integer(ctx, a1);
					dcount[k] = dcount[k] - dstart[k] + 1;
				}
				if (dcount[k] <= 0 || dcount[k] >= dims[k]) api_runtime_error(ctx, "invalid end index");
				ns *= dcount[k];
			}
		}
	}

	mstart[0] = 0;
	mcount[0] = ns;
}

/////////////////////////////////////////////////////////////////////

static void type_error(void* ctx, const char* msg) { api_runtime_error2(ctx, "unsupported variable/attibute type: ", msg);  }

static void hdf_destroy(void *ptr) { delete (zsHDF*)ptr; ptr = 0; }

static herr_t hdf_error(void *data) { H5Eclear1(); return 1; }

/////////////////////////////////////////////////////////////////////

static void* hdf_open(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	const char *fname = api_get_string(ctx, args[0]);
	zsHDF *o = new zsHDF;
	H5Eset_auto1(hdf_error, o);
	if (nargs > 1) {
		const char *s = api_get_string(ctx, args[1]);
		if (s[0] == 'w' || s[0] == 'W') {
			o->fid = H5Fopen(fname, H5F_ACC_RDWR, H5P_DEFAULT);
			if (o->fid < 0) {
				o->fid = H5Fcreate(fname, H5F_ACC_EXCL, H5P_DEFAULT, H5P_DEFAULT);
			}
		}
		else if (s[0] == 'c' || s[0] == 'C') {
			o->fid = H5Fcreate(fname, H5F_ACC_EXCL, H5P_DEFAULT, H5P_DEFAULT);
		}
		else {
			o->fid = H5Fopen(fname, H5F_ACC_RDONLY, H5P_DEFAULT);
		}
	}
	else {
		o->fid = H5Fopen(fname, H5F_ACC_RDONLY, H5P_DEFAULT);
	}
	if (o->fid < 0) {
		return 0;
	}
	o->gid = H5Gopen(o->fid, "/", H5P_DEFAULT);
	if (o->gid < 0) o->gid = H5Gcreate(o->fid, "/", H5P_DEFAULT, H5P_DEFAULT, H5P_DEFAULT);
	if (o->gid < 0) api_runtime_error(ctx, "failed to open or create group");
	return api_create_user(ctx, o, 0, hdf_destroy, HDF_TYPE);
}

static void* hdf_version(void *ctx, int nargs, void** args)
{
	unsigned v1, v2, v3;
	H5get_libversion(&v1, &v2, &v3);
	static char buf[64];
	sprintf(buf, "version-%d.%d.%d", v1, v2, v3);
	return api_create_string(ctx, buf);
}

void hdf_set_group(void *ctx, zsHDF *o, const char *name, bool flag)
{
	char copy[256];
	if (strlen(name) >= 256) api_runtime_error2(ctx, "bad group/dataset name is too long: ", name);
	hid_t gid;
	int n = 0;
	while (name[n]) {
		if (name[n] == '/') {
			if (n==0) {
				strncpy(copy, name, 1);
				copy[1] = 0;
			}
			else {
				strncpy(copy, name, n);
				copy[n] = 0;
			}
			gid = H5Gopen(o->fid, copy, H5P_DEFAULT);
			if (gid < 0) gid = H5Gcreate(o->fid, copy, H5P_DEFAULT, H5P_DEFAULT, H5P_DEFAULT);
			if (gid < 0) api_runtime_error2(ctx, "failed to open/create group: ", copy);
			if (o->gid >= 0) H5Gclose(o->gid);
			o->gid = gid;
		}
		n++;
	}
	if (flag) {
		gid = H5Gopen(o->fid, name, H5P_DEFAULT);
		if (gid < 0) gid = H5Gcreate(o->fid, name, H5P_DEFAULT, H5P_DEFAULT, H5P_DEFAULT);
		if (gid < 0) api_runtime_error2(ctx, "failed to open/create group: ", name);
		if (o->gid >= 0) H5Gclose(o->gid);
		o->gid = gid;
	}
}


static void* hdf_group(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zsHDF *o = (zsHDF*)api_get_user(ctx, args[0], HDF_TYPE);
	if (nargs == 1) {
		// get sub-group names
		hsize_t k, n;
		zsBuffer buf(BUFF_SIZE);
		H5Gget_num_objs(o->gid, &n);
		if (n <= 0) return 0;

		void *arr = api_create_array(ctx, n);
		int count = 0;
		for (k = 0; k < n; k++) {
			hid_t type = H5Gget_objtype_by_idx(o->gid, k);
			if (type == H5G_GROUP) {
				H5Gget_objname_by_idx(o->gid, k, buf.u.pchar, BUFF_SIZE-1);
				buf.u.pchar[BUFF_SIZE-1] = 0;
				api_set_array_object2(ctx, arr, count++, api_create_string(0, buf.u.pchar));
			}
			H5Tclose(type);
		}
		return arr;
	}
	// get or create group
	const char *name = api_get_string(ctx, args[1]);
	hdf_set_group(ctx, o, name, true);

	return 0;
}

static void* hdf_dataset(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zsHDF *o = (zsHDF*)api_get_user(ctx, args[0], HDF_TYPE);

	if (nargs == 1) {
		// get dataset names
		hsize_t k, n;
		zsBuffer buf(BUFF_SIZE);
		H5Gget_num_objs(o->gid, &n);
		if (n <=0 ) return 0;

		void *arr = api_create_array(ctx, n);
		int count = 0;
		for (k = 0; k < n; k++) {
			hid_t type = H5Gget_objtype_by_idx(o->gid, k);
			if (type == H5G_DATASET) {
				H5Gget_objname_by_idx(o->gid, k, buf.u.pchar, BUFF_SIZE-1);
				buf.u.pchar[BUFF_SIZE-1] = 0;
				api_set_array_object2(ctx, arr, count++, api_create_string(0, buf.u.pchar));
			}
			H5Tclose(type);
		}
		return arr;
	}

	const char *name = api_get_string(ctx, args[1]);

	hdf_set_group(ctx, o, name, false);

	if (nargs == 2) {
		// set dataset target
		if (o->did >= 0) H5Dclose(o->did);
		o->did = H5Dopen1(o->gid, name);
		if (o->did < 0) return api_create_integer(ctx, 0);

		if (o->tid >= 0) H5Tclose(o->tid);
		o->tid = H5Dget_type(o->did);
		o->tsize = H5Tget_size(o->tid);

		switch (H5Tget_class(o->tid)) {
		case H5T_INTEGER:
			if (H5Tequal(o->tid, H5T_NATIVE_SCHAR)) {
				o->type = HDF_CHAR;
				o->endian = 0;
			}
			else if (H5Tequal(o->tid, H5T_STD_I8BE)) {
				o->type = HDF_CHAR;
				o->endian = 0;
			}
			else if (H5Tequal(o->tid, H5T_STD_I8LE)) {
				o->type = HDF_CHAR;
				o->endian = 0;
			}
			else if (H5Tequal(o->tid, H5T_NATIVE_UCHAR)) {
				o->type = HDF_UCHAR;
				o->endian = 0;
			}
			else if (H5Tequal(o->tid, H5T_STD_U8BE)) {
				o->type = HDF_UCHAR;
				o->endian = 0;
			}
			else if (H5Tequal(o->tid, H5T_STD_U8LE)) {
				o->type = HDF_UCHAR;
				o->endian = 0;
			}
			else if (H5Tequal(o->tid, H5T_NATIVE_SHORT)) {
				o->type = HDF_SHORT;
				o->endian = 0;
			}
			else if (H5Tequal(o->tid, H5T_STD_I16BE)) {
				o->type = HDF_SHORT;
				o->endian = 1;
			}
			else if (H5Tequal(o->tid, H5T_STD_I16LE)) {
				o->type = HDF_SHORT;
				o->endian = -1;
			}
			else if (H5Tequal(o->tid, H5T_NATIVE_USHORT)) {
				o->type = HDF_USHORT;
				o->endian = 0;
			}
			else if (H5Tequal(o->tid, H5T_STD_U16BE)) {
				o->type = HDF_USHORT;
				o->endian = 1;
			}
			else if (H5Tequal(o->tid, H5T_STD_U16LE)) {
				o->type = HDF_USHORT;
				o->endian = -1;
			}
			else if (H5Tequal(o->tid, H5T_NATIVE_INT)) {
				o->type = HDF_INT;
				o->endian = 0;
			}
			else if (H5Tequal(o->tid, H5T_STD_I32BE)) {
				o->type = HDF_INT;
				o->endian = 1;
			}
			else if (H5Tequal(o->tid, H5T_STD_I32LE)) {
				o->type = HDF_INT;
				o->endian = -1;
			}
			else if (H5Tequal(o->tid, H5T_NATIVE_UINT)) {
				o->type = HDF_UINT;
				o->endian = 0;
			}
			else if (H5Tequal(o->tid, H5T_STD_U32BE)) {
				o->type = HDF_UINT;
				o->endian = 1;
			}
			else if (H5Tequal(o->tid, H5T_STD_U32LE)) {
				o->type = HDF_UINT;
				o->endian = -1;
			}
			else {
				type_error(ctx, " --H5T_INTEGER_?-- ");
			}
			break;
		case H5T_FLOAT:
			if (H5Tequal(o->tid, H5T_NATIVE_FLOAT)) {
				o->type = HDF_FLOAT;
				o->endian = 0;
			}
			else if (H5Tequal(o->tid, H5T_IEEE_F32BE)) {
				o->type = HDF_FLOAT;
				o->endian = 1;
			}
			else if (H5Tequal(o->tid, H5T_IEEE_F32LE)) {
				o->type = HDF_FLOAT;
				o->endian = -1;
			}
			else if (H5Tequal(o->tid, H5T_NATIVE_DOUBLE)) {
				o->type = HDF_DOUBLE;
				o->endian = 0;
			}
			else if (H5Tequal(o->tid, H5T_IEEE_F64BE)) {
				o->type = HDF_DOUBLE;
				o->endian = 1;
			}
			else if (H5Tequal(o->tid, H5T_IEEE_F64LE)) {
				o->type = HDF_DOUBLE;
				o->endian = -1;
			}
			else {
				type_error(ctx, " --H5T_FLOAT_?-- ");
			}
			break;
		case H5T_STRING:
			o->type = HDF_STRING;
			o->endian = 0;
			break;
		case H5T_TIME:
			type_error(ctx, " --H5T_TIME-- ");
			break;
		case H5T_BITFIELD:
			type_error(ctx, " --H5T_BITFIELD-- ");
			break;
		case H5T_OPAQUE:
			type_error(ctx, " --H5T_OPAQUE-- ");
			break;
		case H5T_COMPOUND:
			type_error(ctx, " --H5T_COMPOUND-- ");
			break;
		case H5T_REFERENCE:
			type_error(ctx, " --H5T_REFERENCE-- ");
			break;
		case H5T_ENUM:
			type_error(ctx, " --H5T_ENUM-- ");
			break;
		case H5T_VLEN:
			type_error(ctx, " --H5T_VLEN-- ");
			break;
		case H5T_ARRAY:
			type_error(ctx, " --H5T_ARRAY-- ");
			break;
		default:
			type_error(ctx, " --H5T_?-- ");
			break;
		}

		if (o->sid >= 0) H5Sclose(o->sid);
		o->sid = H5Dget_space(o->did);
		if (o->sid < 0) api_runtime_error(ctx, "failed to get dataset space");

		if (H5Sis_simple(o->sid) > 0) {
			o->ndim = H5Sget_simple_extent_dims(o->sid, o->dims, 0);
			if (o->ndim < 0) api_runtime_error(ctx, "failed to get dataset space dimensions");
			o->np = H5Sget_simple_extent_npoints(o->sid); 
		}
		else {
			 api_runtime_error(ctx, "complex data space not supported");
		}

		o->flag = true;

		return api_create_integer(ctx, 1);
	}

	// define dataset

	if (nargs < 4) api_input_error(ctx);
	const char *type = api_get_string(ctx, args[2]);
	if (o->tid >= 0) H5Tclose(o->tid);

	if (!strcmp(type, "char")) {
		o->tid = H5Tcopy(H5T_NATIVE_CHAR);
		o->tsize = sizeof(char);
		o->type = HDF_CHAR;
	}
	else if (!strcmp(type, "uchar")) {
		o->tid = H5Tcopy(H5T_NATIVE_UCHAR);
		o->tsize = sizeof(unsigned char);
		o->type = HDF_UCHAR;
	}
	else if (!strcmp(type, "short")) {
		o->tid = H5Tcopy(H5T_NATIVE_SHORT);
		o->tsize = sizeof(short);
		o->type = HDF_SHORT;
	}
	else if (!strcmp(type, "ushort")) {
		o->tid = H5Tcopy(H5T_NATIVE_USHORT);
		o->tsize = sizeof(unsigned short);
		o->type = HDF_USHORT;
	}
	else if (!strcmp(type, "int")) {
		o->tid = H5Tcopy(H5T_NATIVE_INT);
		o->tsize = sizeof(int);
		o->type = HDF_INT;
	}
	else if (!strcmp(type, "uint")) {
		o->tid = H5Tcopy(H5T_NATIVE_UINT);
		o->tsize = sizeof(unsigned int);
		o->type = HDF_UINT;
	}
	else if (!strcmp(type, "float")) {
		o->tid = H5Tcopy(H5T_NATIVE_FLOAT);
		o->tsize = sizeof(float);
		o->type = HDF_FLOAT;
	}
	else if (!strcmp(type, "double")) {
		o->tid = H5Tcopy(H5T_NATIVE_DOUBLE);
		o->tsize = sizeof(double);
		o->type = HDF_DOUBLE;
	}
	else if (!strcmp(type, "string")) {
		o->tid = H5Tcopy(H5T_C_S1);
		o->tsize = 1;
		o->type = HDF_STRING;
	}
	else {
		api_input_error(ctx);
	}

	o->endian = 0;
	o->ndim = 0;
	o->np = 1;
	for (int i = 3; i < nargs && o->ndim < H5S_MAX_RANK; i++) {
		integer_t n = api_get_integer(ctx, args[i]);
		if (n <= 0) api_input_error(ctx);
		o->dims[o->ndim] = n;
		o->np *= n;
		o->ndim++;
	}

	if (o->type == HDF_STRING) {
		if (o->ndim < 2) api_input_error(ctx);
		o->ndim--;
		// for string array, the last number is the string length.
		o->tsize = o->dims[o->ndim];
		H5Tset_size(o->tid, o->tsize);
	}

	if (o->sid >= 0) H5Sclose(o->sid);
	o->sid = H5Screate_simple(o->ndim, o->dims, 0);
	if (o->sid < 0) api_runtime_error(ctx, "failed to create data space");

	if (o->did >= 0) H5Dclose(o->did);
	o->did = H5Dcreate1(o->gid, name, o->tid, o->sid, H5P_DEFAULT);
	if (o->did < 0) api_runtime_error(ctx, "failed to create dataset");
	
	o->flag = true;
	
	return api_create_integer(ctx, 1);
}

static void* hdf_size(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zsHDF *o = (zsHDF*)api_get_user(ctx, args[0], HDF_TYPE);
	o->checkdataset(ctx);
	void *arr = api_create_array(ctx, 3);
	api_set_array_object(ctx, arr, "0", api_create_integer(0, o->np));
	api_set_array_object(ctx, arr, "1", api_create_integer(0, o->tsize));
	api_set_array_object(ctx, arr, "2", api_create_string(0, type2name(o->type)));
	return arr;
}

static void* hdf_dims(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zsHDF *o = (zsHDF*)api_get_user(ctx, args[0], HDF_TYPE);
	o->checkdataset(ctx);
	if (o->ndim == 1) {
		return api_create_integer(ctx, o->dims[0]);
	}
	void *arr = api_create_array(ctx, o->ndim);
	for (int i = 0; i < o->ndim; i++) api_set_array_object2(ctx, arr, i, api_create_integer(0, o->dims[i]));
	return arr;
}

static void* hdf_getatt(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);
	zsHDF *o = (zsHDF*)api_get_user(ctx, args[0], HDF_TYPE);
	const char *name = api_get_string(ctx, args[1]);
	hid_t id = o->did;
	if (!o->flag) id = o->gid;

	zsAttID ids(id, name);
	if (ids.aid < 0) return 0;

	void *ret = 0;

	if (ids.ndim == 0) {
		switch (H5Tget_class(ids.tid)) {
		case H5T_INTEGER:
			{
			int value;
			if (H5Aread(ids.aid, H5T_NATIVE_INT, &value) >= 0) ret = api_create_integer(ctx, value);
			}
			break;
		case H5T_FLOAT:
			{
			double value;
			if (H5Aread(ids.aid, H5T_NATIVE_DOUBLE, &value) >= 0) ret = api_create_real(ctx, value);
			}
			break;
		case H5T_STRING:
			{
			int n = H5Aget_storage_size(ids.aid);
			if (n <= 0) api_runtime_error(ctx, "att storgae <= 0");
			zsBuffer buf(n+1);
			hid_t atype = H5Tcopy(H5T_C_S1);
			H5Tset_size(atype, n);
			if (H5Aread(ids.aid, atype, buf.u.pchar) >= 0) {
				buf.u.pchar[n] = 0;
				ret = api_create_string(ctx, buf.u.pchar);
			}
			H5Tclose(atype);
			}
			break;
		default:
			type_error(ctx, " --NON INTEGER/FLOAT/STRING ATTRIBUTE-- ");
			break;
		}
	}
	else if (ids.ndim == 1) {
		int n = H5Aget_storage_size(ids.aid);
		if (n <= 0) api_runtime_error(ctx, "att storgae <= 0");
		void *ptr = malloc(n);
		if (H5Aread(ids.aid, H5T_NATIVE_UCHAR, ptr) >= 0) {
			ret = api_create_array(ctx, 2);
			api_set_array_object(ctx, ret, "0", api_create_user(0,ptr,0,free,0));
			api_set_array_object(ctx, ret, "1", api_create_integer(0,ids.dims[0]));
		}
		else {
			free(ptr);
		}
	}
	else {
		api_runtime_error2(ctx, name, ": number of dimensions of the attribute > 1");
	}

	return ret;
}

static void* hdf_putatt(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);
	zsHDF *o = (zsHDF*)api_get_user(ctx, args[0], HDF_TYPE);
	const char *name = api_get_string(ctx, args[1]);
	hid_t id = o->did;
	if (!o->flag) id = o->gid;

	H5Adelete(id, name);

	zsAttID ids(-1, 0);

	hid_t status = -1;

	if (api_is_real(args[2])) {
		// e.g., hdf.name = 1.23;
		double value = api_get_real(ctx, args[2]);
		ids.sid = H5Screate(H5S_SCALAR);
		ids.aid = H5Acreate1(id, name, H5T_NATIVE_DOUBLE, ids.sid, H5P_DEFAULT);
		if (ids.aid > 0) status = H5Awrite(ids.aid, H5T_NATIVE_DOUBLE, &value);
	}
	else if (api_is_integer(args[2])) {
		// e.g., hdf.name = 123;
		int value = api_get_integer(ctx, args[2]);
		ids.sid = H5Screate(H5S_SCALAR);
		ids.aid = H5Acreate1(id, name, H5T_NATIVE_INT, ids.sid, H5P_DEFAULT);
		if (ids.aid > 0) status = H5Awrite(ids.aid, H5T_NATIVE_INT, &value);
	}
	else if (api_is_string(args[2])) {
		// e.g., hdf.name = "my property";
		const char *ptr = api_get_string(ctx, args[2]);
		ids.sid = H5Screate(H5S_SCALAR);
		ids.tid = H5Tcopy(H5T_C_S1);
		H5Tset_size(ids.tid, strlen(ptr));
		ids.aid = H5Acreate1(id, name, ids.tid, ids.sid, H5P_DEFAULT);
		if (ids.aid > 0) status = H5Awrite(ids.aid, ids.tid, ptr);
	}
	else if (api_is_array(args[2])) {
		// e.g., hdf.name = [ptr, n];
		void *ptr = api_get_ptr(ctx, api_get_array_object(ctx,args[2],"0"));
		int n = api_get_integer(ctx, api_get_array_object(ctx,args[2],"1"));
		if (n <= 0) api_runtime_error(ctx, "pointer size <= 0");
		ids.dims[0] = n;
		ids.sid = H5Screate_simple(1, ids.dims, NULL);
		ids.aid = H5Acreate2(id, name, H5T_NATIVE_UCHAR, ids.sid, H5P_DEFAULT, H5P_DEFAULT);
		if (ids.aid > 0) status = H5Awrite(ids.aid, H5T_NATIVE_UCHAR, ptr);
	}

	if (status < 0) api_runtime_error(ctx, "failed to write attribute");

	return 0;
}

static void* hdf_attribute(void *ctx, int nargs, void** args)
{
	if (nargs < 4) api_input_error(ctx);
	void *ret = hdf_getatt(ctx, nargs, args);
	if (!ret) api_runtime_error(ctx, "no such an attribute exists");
	if (!api_is_array(ret)) api_runtime_error(ctx, "attribute is not array of numbers");
	int idx = api_get_integer(ctx, args[2]);
	if (idx < 0 || idx >= api_get_array_size(ctx, ret)) api_runtime_error(ctx, "bad array index");
	api_set_array_object2(ctx, ret, idx, args[3]);
	void *p[3] = { args[0], args[1], ret };
	hdf_putatt(ctx, 3, p);
	return 0;
}

static void* hdf_get(void *ctx, int nargs, void** args)
{
	if (nargs < 2) api_input_error(ctx);

	if (api_is_string(args[1])) {
		// hdf.name
		return hdf_getatt(ctx, nargs, args);
	}

	// hdf[...]

	zsHDF *o = (zsHDF*)api_get_user(ctx, args[0], HDF_TYPE);
	
	o->checkindex(ctx, nargs, args);
	o->checksize();
	o->hyperslab(ctx);

	if (H5Dread(o->did, o->tid, o->mid, o->sid, H5P_DEFAULT, o->ptr) < 0) api_runtime_error(ctx, "failed to read data");

	if (o->ns == 1) {
		switch (o->type) {
		case HDF_CHAR:
			return api_create_integer(ctx, ((char*)o->ptr)[0]);
		case HDF_UCHAR:
			return api_create_integer(ctx, ((unsigned char*)o->ptr)[0]);
		case HDF_SHORT:
			return api_create_integer(ctx, ((short*)o->ptr)[0]);
		case HDF_USHORT:
			return api_create_integer(ctx, ((unsigned short*)o->ptr)[0]);
		case HDF_INT:
			return api_create_integer(ctx, ((int*)o->ptr)[0]);
		case HDF_UINT:
			return api_create_integer(ctx, ((unsigned int*)o->ptr)[0]);
		case HDF_FLOAT:
			return api_create_real(ctx, ((float*)o->ptr)[0]);
		case HDF_DOUBLE:
			return api_create_real(ctx, ((double*)o->ptr)[0]);
		case HDF_STRING:
			((char*)o->ptr)[o->tsize] = 0;
			return api_create_string(ctx, (char*)o->ptr);
		default:
			type_error(ctx, " --HDF_?-- ");
		}
	}

	if (o->type == HDF_STRING) {
		void *ret = api_create_array(ctx, o->ns);
		char *ptr = (char*)o->ptr;
		zsBuffer buf(o->tsize+1);
		buf.u.pchar[o->tsize] = 0;
		for (int i=0; i<o->ns; i++) {
			strncpy(buf.u.pchar, ptr, o->tsize);
			api_set_array_object2(ctx, ret, i, api_create_string(0,buf.u.pchar));
			ptr += o->tsize;
		}
		return ret;
	}

	return api_create_user(ctx, o->ptr, 0, 0, 0);
}

static void* hdf_set(void *ctx, int nargs, void** args)
{
	if (nargs < 3) api_input_error(ctx);

	if (api_is_string(args[1])) {
		// hdf.name = attribute
		return hdf_putatt(ctx, nargs, args);
	}

	// hdf[...]

	zsHDF *o = (zsHDF*)api_get_user(ctx, args[0], HDF_TYPE);

	o->checkindex(ctx, nargs-1, args);
	o->hyperslab(ctx);

	hid_t status = -1;

	if (api_is_user(args[nargs-1])) {
		// hdf[...] = ptr;
		void *ptr = api_get_ptr(ctx, args[nargs-1]);
		status = H5Dwrite(o->did, o->tid, o->mid, o->sid, H5P_DEFAULT, ptr);
	}
	else {
		// hdf[...] = value;
		zsBuffer buf(o->ns*o->tsize);
		real_t value;
		const char *str;
		hsize_t k, i, l;
		switch (o->type) {
		case HDF_CHAR:
			value = api_get_number(ctx, args[nargs-1]);
			for (k = 0; k < o->ns; k++) buf.u.pchar[k] = char(value);
			break;
		case HDF_UCHAR:
			value = api_get_number(ctx, args[nargs-1]);
			for (k = 0; k < o->ns; k++) buf.u.puchar[k] = (unsigned char)(value);
			break;
		case HDF_SHORT:
			value = api_get_number(ctx, args[nargs-1]);
			for (k = 0; k < o->ns; k++) buf.u.pshort[k] = short(value);
			break;
		case HDF_USHORT:
			value = api_get_number(ctx, args[nargs-1]);
			for (k = 0; k < o->ns; k++) buf.u.pushort[k] = (unsigned short)(value);
			break;
		case HDF_INT:
			value = api_get_number(ctx, args[nargs-1]);
			for (k = 0; k < o->ns; k++) buf.u.pint[k] = int(value);
			break;
		case HDF_UINT:
			value = api_get_number(ctx, args[nargs-1]);
			for (k = 0; k < o->ns; k++) buf.u.puint[k] = (unsigned int)(value);
			break;
		case HDF_FLOAT:
			value = api_get_number(ctx, args[nargs-1]);
			for (k = 0; k < o->ns; k++) buf.u.pfloat[k] = float(value);
			break;
		case HDF_DOUBLE:
			value = api_get_number(ctx, args[nargs-1]);
			for (k = 0; k < o->ns; k++) buf.u.pdouble[k] = double(value);
			break;
		case HDF_STRING:
			str = api_get_string(ctx, args[nargs-1]);
			l = strlen(str);
			i = 0;
			for (k = 0; k < o->ns; k++) {
				buf.u.pchar[k] = str[i];
				i++;
				i %= l;
			}
			break;
		default:
			type_error(ctx, " --HDF_?-- ");
		}
		status = H5Dwrite(o->did, o->tid, o->mid, o->sid, H5P_DEFAULT, buf.u.ptr);
	}

	if (status < 0) api_runtime_error(ctx, "failed to write to dataset");

	return 0;
}

static void* hdf_cmm(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zsHDF *o = (zsHDF*)api_get_user(ctx, args[0], HDF_TYPE);
	hid_t id = o->did;
	if (!o->flag) id = o->gid;
	if (nargs == 1) {
		zsBuffer buf(BUFF_SIZE);
		if (H5Gget_comment(id, ".", BUFF_SIZE-1, buf.u.pchar) < 0) api_runtime_error(ctx, "failed to read comment");
		return api_create_string(ctx, buf.u.pchar);
	}
	if (H5Gset_comment(id, ".", api_get_string(ctx, args[1])) < 0) api_runtime_error(ctx, "failed to write comment");
	return 0;
}

static void write_space(FILE *fp, int n)
{
	for (int k = 0; k < n; k++) fputc(' ', fp);
}

static void list_attribute(hid_t id, FILE *fp, int nc)
{
	zsBuffer buf(BUFF_SIZE);
	zsBuffer buf2(BUFF_SIZE);
	
	unsigned int k, n = H5Aget_num_attrs(id);

	for (k = 0; k < n; k++) {
		hid_t aid = H5Aopen_idx(id, k);
		
		if (aid < 0) continue;

		H5Aget_name(aid, BUFF_SIZE-1, buf.u.pchar);
		buf.u.pchar[BUFF_SIZE-1] = 0;

		hid_t tid = H5Aget_type(aid);

		if (tid < 0) {
			H5Aclose(aid);
			continue;
		}

		bool flag = false;
		hid_t sid = 0, atype = 0;

		switch (H5Tget_class(tid)) {
		case H5T_INTEGER:
			sid = H5Aget_space(aid);
			if (sid > 0) {
				int ndim = H5Sget_simple_extent_ndims(sid);
				if (ndim == 0) {
					int value;
					if (H5Aread(aid, H5T_NATIVE_INT, &value) >= 0) {
						flag = true;
						write_space(fp, nc);
						fprintf(fp, "%s=%d\n", buf.u.pchar, value);
					}
				}
			}
			H5Sclose(sid);
			break;
		case H5T_FLOAT:
			sid = H5Aget_space(aid);
			if (sid > 0) {
				int ndim = H5Sget_simple_extent_ndims(sid);
				if (ndim == 0) {
					double value;
					if (H5Aread(aid, H5T_NATIVE_DOUBLE, &value) >= 0) 
						flag = true;
						write_space(fp, nc);
						fprintf(fp, "%s=%f\n", buf.u.pchar, value);
					}
			}
			H5Sclose(sid);
			break;
		case H5T_STRING:
			atype = H5Tcopy(H5T_C_S1);
			H5Tset_size(atype, BUFF_SIZE-1);
			if (H5Aread(aid, atype, buf2.u.pchar) >= 0) {
				buf2.u.pchar[BUFF_SIZE-1] = 0;
				flag = true;
				write_space(fp, nc);
				fprintf(fp, "%s=%s\n", buf.u.pchar, buf2.u.pchar);
			}
			H5Tclose(atype);
			break;
		default:
			write_space(fp, nc);
			fprintf(fp, "%s=?(unexpected type)\n", buf.u.pchar);
			break;
		}

		H5Tclose(tid);

		H5Aclose(aid);
	}

	fflush(fp);
}

static void list_dataset(hid_t gid, const char *name, FILE *fp, int nc)
{
	hid_t did = H5Dopen1(gid, name);
	if (did < 0) return;

	hid_t tid = H5Dget_type(did);

	write_space(fp, nc);
	fprintf(fp, "%s(", name);

	hid_t sid = H5Dget_space(did);

	if (sid > 0) {
		hsize_t dims[H5S_MAX_RANK];
		int k, ndim = H5Sget_simple_extent_dims(sid, dims, 0);
		for (k = 0; k < ndim; k++) {
			if (k == 0)
				fprintf(fp,"%d", dims[k]);
			else
				fprintf(fp,",%d", dims[k]);
		}
	}
	fprintf(fp, ") -- ");

	H5Sclose(sid);

	switch (H5Tget_class(tid)) {
	case H5T_INTEGER:
		if (H5Tequal(tid, H5T_NATIVE_SCHAR)) {
			fprintf(fp, "H5T_NATIVE_SCHAR\n");
		}
		else if (H5Tequal(tid, H5T_STD_I8BE)) {
			fprintf(fp, "H5T_STD_I8BE\n");
		}
		else if (H5Tequal(tid, H5T_STD_I8LE)) {
			fprintf(fp, "H5T_STD_I8LE\n");
		}
		else if (H5Tequal(tid, H5T_NATIVE_UCHAR)) {
			fprintf(fp, "H5T_NATIVE_UCHAR\n");
		}
		else if (H5Tequal(tid, H5T_STD_U8BE)) {
			fprintf(fp, "H5T_STD_U8BE\n");
		}
		else if (H5Tequal(tid, H5T_STD_U8LE)) {
			fprintf(fp, "H5T_STD_U8LE\n");
		}
		else if (H5Tequal(tid, H5T_NATIVE_SHORT)) {
			fprintf(fp, "H5T_NATIVE_SHORT\n");
		}
		else if (H5Tequal(tid, H5T_STD_I16BE)) {
			fprintf(fp, "H5T_STD_I16BE\n");
		}
		else if (H5Tequal(tid, H5T_STD_I16LE)) {
			fprintf(fp, "H5T_STD_I16LE\n");
		}
		else if (H5Tequal(tid, H5T_NATIVE_USHORT)) {
			fprintf(fp, "H5T_STD_NATIVE_USHORT\n");
		}
		else if (H5Tequal(tid, H5T_STD_U16BE)) {
			fprintf(fp, "H5T_STD_U16BE\n");
		}
		else if (H5Tequal(tid, H5T_STD_U16LE)) {
			fprintf(fp, "H5T_STD_U16LE\n");
		}
		else if (H5Tequal(tid, H5T_NATIVE_INT)) {
			fprintf(fp, "H5T_NATIVE_INT\n");
		}
		else if (H5Tequal(tid, H5T_STD_I32BE)) {
			fprintf(fp, "H5T_STD_I32BE\n");
		}
		else if (H5Tequal(tid, H5T_STD_I32LE)) {
			fprintf(fp, "H5T_STD_I32LE\n");
		}
		else if (H5Tequal(tid, H5T_NATIVE_UINT)) {
			fprintf(fp, "H5T_NATIVE_UINT\n");
		}
		else if (H5Tequal(tid, H5T_STD_U32BE)) {
			fprintf(fp, "H5T_STD_U32BE\n");
		}
		else if (H5Tequal(tid, H5T_STD_U32LE)) {
			fprintf(fp, "H5T_STD_U32LE\n");
		}
		else {
			fprintf(fp, "H5T_INTEGER_?\n");
		}
		break;
	case H5T_FLOAT:
		if (H5Tequal(tid, H5T_NATIVE_FLOAT)) {
			fprintf(fp, "H5T_NATIVE_FLOAT\n");
		}
		else if (H5Tequal(tid, H5T_IEEE_F32BE)) {
			fprintf(fp, "H5T_IEEE_F32BE\n");
		}
		else if (H5Tequal(tid, H5T_IEEE_F32LE)) {
			fprintf(fp, "H5T_IEEE_F32LE\n");
		}
		else if (H5Tequal(tid, H5T_NATIVE_DOUBLE)) {
			fprintf(fp, "H5T_NATIVE_DOUBLE\n");
		}
		else if (H5Tequal(tid, H5T_IEEE_F64BE)) {
			fprintf(fp, "H5T_IEEE_F64BE\n");
		}
		else if (H5Tequal(tid, H5T_IEEE_F64LE)) {
			fprintf(fp, "H5T_IEEE_F64LE\n");
		}
		else {
			fprintf(fp, "H5T_FLOAT_?\n");
		}
		break;
	case H5T_STRING:
		fprintf(fp, "H5T_STRING\n");
		break;
	case H5T_TIME:
		fprintf(fp, "H5T_TIME\n");
		break;
	case H5T_BITFIELD:
		fprintf(fp, "H5T_BITFIELD\n");
		break;
	case H5T_OPAQUE:
		fprintf(fp, "H5T_OPAQUE\n");
		break;
	case H5T_COMPOUND:
		fprintf(fp, "H5T_COMPOUND\n");
		break;
	case H5T_REFERENCE:
		fprintf(fp, "H5T_REFERENCE\n");
		break;
	case H5T_ENUM:
		fprintf(fp, "H5T_ENUM\n");
		break;
	case H5T_VLEN:
		fprintf(fp, "H5T_VLEN\n");
		break;
	case H5T_ARRAY:
		fprintf(fp, "H5T_ARRAY\n");
		break;
	default:
		fprintf(fp, "H5T_?\n");
		break;
	}

	fflush(fp);

	list_attribute(did, fp, nc+4);

	H5Dclose(tid);
	H5Dclose(did);
}

static void list_group(hid_t gid, const char *name, FILE *fp, int nc)
{
	hsize_t k, n;
	zsBuffer buf(BUFF_SIZE);

	hid_t id = gid;

	if (name) {
		write_space(fp, nc);
		fprintf(fp, "%s\n", name);
		id = H5Gopen(gid, name, H5P_DEFAULT);
		if (id < 0) return;
		nc += 4;
	}
	
	if (H5Gget_comment(id, ".", BUFF_SIZE-1, buf.u.pchar) > 0) {
		write_space(fp, nc);
		fprintf(fp, "comment: %s\n", buf.u.pchar);
	}

	H5Gget_num_objs(id, &n);

	for (k = 0; k < n; k++) {
		H5Gget_objname_by_idx(id, k, buf.u.pchar, BUFF_SIZE-1);
		buf.u.pchar[BUFF_SIZE-1] = 0;
		hid_t type = H5Gget_objtype_by_idx(id, k);
		switch(type) {
		case H5G_LINK:
			{
			zsBuffer buf2(BUFF_SIZE);
			H5Gget_linkval(id, buf.u.pchar, BUFF_SIZE-1, buf2.u.pchar);
			buf2.u.pchar[BUFF_SIZE-1] = 0;
			write_space(fp, nc);
			fprintf(fp, "%s is a link to %s\n", buf.u.pchar, buf2.u.pchar);
			}
			break;
		case H5G_GROUP:
			list_group(id, buf.u.pchar, fp, nc);
			break;
		case H5G_DATASET:
			list_dataset(id, buf.u.pchar, fp, nc);
			break;
		case H5G_TYPE:
			write_space(fp, nc);
			fprintf(fp, "%s is a datatype\n", buf.u.pchar);
			break;
		default:
			write_space(fp, nc);
			fprintf(fp, "%s is ?\n", buf.u.pchar);
		}
		H5Tclose(type);
	}

	list_attribute(id, fp, nc);

	if (name) H5Gclose(id);
}

static void* hdf_list(void *ctx, int nargs, void** args)
{
	if (nargs < 1) api_input_error(ctx);
	zsHDF *o = (zsHDF*)api_get_user(ctx, args[0], HDF_TYPE);
	FILE *fp =stdout;
	if (nargs > 1) {
		const char *fname = api_get_string(ctx, args[1]);
		fp = fopen(fname, "w");
		if (!fp) api_runtime_error2(ctx, "failed to open file: ", fname);
	}
	list_group(o->gid, 0, fp, 0);
	return 0;
}


/////////////////////////////////////////////////////////////////////

class hdfRegPrimitive
{
public:
	hdfRegPrimitive()
	{
		api_add_primitive("hdf",		0,				hdf_open);
		api_add_primitive("version",	HDF_TYPE,		hdf_version);
		api_add_primitive("group",		HDF_TYPE,		hdf_group);
		api_add_primitive("dataset",	HDF_TYPE,		hdf_dataset);
		api_add_primitive("size",		HDF_TYPE,		hdf_size);
		api_add_primitive("dims",		HDF_TYPE,		hdf_dims);
		api_add_primitive("__get",		HDF_TYPE,		hdf_get);
		api_add_primitive("__set",		HDF_TYPE,		hdf_set);
		api_add_primitive("cmm",		HDF_TYPE,		hdf_cmm);
		api_add_primitive("list",		HDF_TYPE,		hdf_list);
		api_add_primitive("attribute",	HDF_TYPE,		hdf_attribute);
	}

	~hdfRegPrimitive() { }
};

static hdfRegPrimitive hdf_register;
