#include "cairo-pdf.h"
#include "cairo-ps.h"
#include "cairo-svg.h"
#include "cairo-win32.h"
#include "cairo.h"
#include "api.h"
#include "gshhs1.33.h"

//!!!pixman must compile with PIXMAN_NO_TLS to avoid crash in XP in calling cairo_show_text

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <vector>

#define CAIRO	'CR'
#define EXLARGE	1.e31
#define D2R		0.017453293
#define PI4		0.785398163
#define CROSSGW 270.0

#pragma warning(disable: 4244)

#ifdef _WIN32
#include <windows.h>
#endif

using namespace std;

/////////////////////////////////////////////////////////////////////

enum { TRANSFORM_NONE=0, TRANSFORM_PLOT, TRANSFORM_PLOT3D, TRANSFORM_SPHERE,
	   TRANSFORM_POLAR, TRANSFORM_LAMBERT, TRANSFORM_HAMMER, TRANSFORM_ECKERT,
	   CAIRO_IMG, CAIRO_PDF, CAIRO_EPS, CAIRO_SVG, CAIRO_WIN};

struct ZExyz {
	double x, y, z;
};

struct ZEcolor {
	double r, g, b;
};

static ZExyz cr_interp2D(const ZExyz &p1, const ZExyz &p2, double isoValue)
{
	ZExyz p;
	if (p1.z==p2.z) {
		p=p1;
	}
	else {
		double f=(isoValue-p1.z)/(p2.z-p1.z);
		p.x=p1.x+f*(p2.x-p1.x);
		p.y=p1.y+f*(p2.y-p1.y);
	}
	p.z=isoValue;
	return p;
}

class zsCairo
{
public:
	zsCairo();

	~zsCairo();

	void checksize(unsigned int length);

	double transform(double &x, double &y, double z) const;

	void line(double x1, double y1, double z1, double x2, double y2, double z2, int dxyflag=2);

	void line(double *x, double *y, double *z, int n);

	void polygon(double *x, double *y, double *z, int n);

	void circle(double x, double y, double z, double r, bool fill=true);

	void arc(double x, double y, double z, double r, double a1, double a2, bool fill=true);

	void arrow(double x, double y, double z, double u, double v, double size);

	void text(const char *str, double x, double y, double z, double angle, int xalign, int yalign, double xoff=0, double yoff=0);

	void contour4p(double isoValue, const ZExyz &p0, const ZExyz &p1, const ZExyz &p2, const ZExyz &p3, double z);

	double xangle() const;
	double yangle() const;

	void dxdy(const char *str, double angle, double &dx, double &dy) const;

	ZEcolor rgb(double v) const;

	void clip(int flag, double z);

public:
	// plot parameters
	double _width, _height;
	double _xoffset, _yoffset, _zoffset;
	double _xscale, _yscale, _zscale, _scale;
	double _xrotate, _zrotate;
	double _xmin, _xmax, _ymin, _ymax, _zmin, _zmax;
	double _mr1[3], _mr2[3], _mr3[3];
	double _lat, _lon;
	double _fontsize;
	int _transform, _clip;

	// color pallete
	ZEcolor _color, _bgcolor;
	vector<ZEcolor> _colors;
	vector<double> _values;
	bool _gradient;

	// Cairo object
	cairo_t *_CR;
	cairo_surface_t *_SFC;
	cairo_pattern_t *_PAT;
	unsigned char *_stream;
	unsigned int _length, _size;
	int _type;
};

static cairo_status_t write_stream(void *closure, const unsigned char *data, unsigned int length)
{
	zsCairo *o=(zsCairo*)closure;
	o->checksize(length);
	memcpy(o->_stream+o->_length,data,length);
	o->_length+=length;
	return CAIRO_STATUS_SUCCESS;
}

zsCairo::zsCairo()
{
	_CR=0;
	_SFC=0;
	_PAT=0;
	_stream=0;
	_length=_size=0;
	_xoffset=_yoffset=_zoffset=0.0;
	_xscale=_yscale=_zscale=0.7;
	_scale=1.0;
	_xmin=_ymin=_zmin=0.0;
	_xmax=_ymax=_zmax=1.0;
	_xrotate=_zrotate=0.0;
	_transform=TRANSFORM_NONE;
	_gradient=true;
	_clip=0;
	_bgcolor.r=_bgcolor.g=_bgcolor.b=1.0;
	_color.r=_color.g=_color.b=0.0;
	_fontsize=16;
}

zsCairo::~zsCairo()
{
	if (_stream) free(_stream);
	if (_CR) cairo_destroy(_CR);
	if (_SFC) cairo_surface_destroy(_SFC);
	if (_PAT) cairo_pattern_destroy(_PAT);
}

void zsCairo::checksize(unsigned int length)
{
	if (_size-_length<length) {
		_size=1048576+_length+length;
		unsigned char *ptr=_stream;
		_stream=(unsigned char*)malloc(_size);
		if (ptr) {
			memcpy(_stream,ptr,_length);
			free(ptr);
		}
	}
}

double zsCairo::transform(double &x, double &y, double z) const
{
	double ret=1;
	if (_clip) {
		if (x<_xmin || x>_xmax || y<_ymin || y>_ymax || z<_zmin || z>_zmax) ret=-1;
	}
	switch (_transform)
	{
	case TRANSFORM_PLOT:
		x=_width*(1.0-_xscale)/2.0+(x-_xmin)*_xscale*_width/(_xmax-_xmin)+_xoffset;
		y=_height*(1.0-_yscale)/2.0+(y-_ymin)*_yscale*_height/(_ymax-_ymin)-_yoffset;
		y=_height-y;
		return ret;
	case TRANSFORM_PLOT3D:
		{
		double u=((x-_xmin)/(_xmax-_xmin))*_xscale*_width;
		double v=((y-_ymin)/(_ymax-_ymin))*_yscale*_height;
		double w=((z-_zmin)/(_zmax-_zmin))*_zscale*_height;
		x=_mr1[0]*u+_mr1[1]*v+_mr1[2]*w;
		y=_mr2[0]*u+_mr2[1]*v+_mr2[2]*w;
		z=_mr3[0]*u+_mr3[1]*v+_mr3[2]*w;
		u=_xscale*_width;
		v=_yscale*_height;
		double dx=u-_mr1[0]*u-_mr1[1]*v;
		double dy=v-_mr2[0]*u-_mr2[1]*v;
		x+=dx/2.0+_xoffset+_width* (1.0-_xscale)/2.0;
		y+=dy/2.0-_yoffset+_height*(1.0-_yscale)/2.0;
		y=_height-y;
		return ret;
		}
	case TRANSFORM_SPHERE:
		{
		double lon=x*D2R;
		double lat=y*D2R;
		double scale=_scale*0.5;
		double z=scale*sin(lat);
		double r=scale*cos(lat);
		double x2=r*cos(lon);
		double y2=r*sin(lon);
		x=_mr1[0]*x2+_mr1[1]*y2+_mr1[2]*z+_width/2.0+_xoffset;
		y=_mr2[0]*x2+_mr2[1]*y2+_mr2[2]*z+_height/2.0-_yoffset;
		y=_height-y;
		ret=_mr3[0]*x2+_mr3[1]*y2+_mr3[2]*z;
		return ret;
		}
	case TRANSFORM_POLAR:
		{
		// true latitude at north/south pole
		if (_ymin<0) {
			if (y>0) ret=-1;
		}
		else {
			if (y<0) ret=-1;
		}
		double r=fabs(cos(y*D2R))/(1.0+fabs(sin(y*D2R)));
		double alpha=(x-0.5*(_xmin+_xmax))*D2R;
		double px=r*sin(alpha);
		double py=-r*cos(alpha);
		if (_ymin<0) px=-px;
		x=0.5*(_width+px*_scale)+_xoffset;
		y=0.5*(_height+py*_scale)-_yoffset;
		y=_height-y;
		return ret;
		}
	case TRANSFORM_LAMBERT:
		{
		// http://en.wikipedia.org/wiki/Lambert_conformal_conic_projection
		if (_ymin<0) {
			if (y>0) ret=-1;
		}
		else {
			if (y<0) ret=-1;
		}
		double lat=y*D2R;
		double lon=x*D2R;
		double n=_mr1[0];
		double F=_mr1[1];
		double ro0=_mr1[2];
		double ro=F*pow(cos(PI4+0.5*fabs(lat)*0.99999),n);
		double lon0=0.5*(_xmin+_xmax)*D2R;
		x=ro*sin(n*(lon-lon0));
		y=ro0-ro*cos(n*(lon-lon0));
		x/=2.311;
		if (_ymin<0) x=-x;
		y=(y+1.176)/2.27;
		y-=0.5;
		x=0.5*_width+_scale*x+_xoffset;
		y=0.5*_height+_scale*y-_yoffset;
		y=_height-y;
		return ret;
		}
	case TRANSFORM_HAMMER:
		{
		// http://en.wikipedia.org/wiki/Hammer_projection
		double lon=(x-0.5*(_xmin+_xmax))*D2R/2.0;double lat=y*D2R;
		double d=sqrt(1.0+cos(lat)*cos(lon))/sqrt(2.0);
		double px=cos(lat)*sin(lon)/d;
		double py=0.5*sin(lat)/d;
		x=0.5*(_width+px*_scale)+_xoffset;
		y=0.5*(_height+py*_scale)-_yoffset;
		y=_height-y;
		return ret;
		}
	case TRANSFORM_ECKERT:
		{
		// http://mathworld.wolfram.com/EckertVIProjection.html
		double lat=y*D2R;
		double lat0=2.57079633*sin(lat); //(1+pi/2)sin(lat)
		for (int k=0; k<3; k++) {
			double dlat=-(lat-lat0+sin(lat))/(1+cos(lat));
			lat+=dlat;
		}
		y=90*lat/1.57079633;	// max(lat)=1.57079633
		double xscale=_xscale*0.5*(1+cos(lat));	// max(1+cos(lat))=2;
		x=_width*(1.0-xscale)/2.0+(x-_xmin)*xscale*_width/(_xmax-_xmin)+_xoffset;
		y=_height*(1.0-_yscale)/2.0+(y-_ymin)*_yscale*_height/(_ymax-_ymin)-_yoffset;
		y=_height-y;
		return ret;
		}
	default:
		return ret;
	}
	return 1;
}

void zsCairo::line(double x1, double y1, double z1, double x2, double y2, double z2, int dxyflag)
{
	cairo_new_path(_CR);
	double d1=transform(x1,y1,z1);
	if (dxyflag>1 || dxyflag<-1) {
		// regular line
		double d2=transform(x2,y2,z2);
		if (d1>=0 && d2>=0) {
			cairo_move_to(_CR,x1,y1);
			cairo_line_to(_CR,x2,y2);
		}
	}
	else if (dxyflag==0) {
		// cross line
		cairo_move_to(_CR,x1-x2,y1-y2);
		cairo_line_to(_CR,x1+x2,y1+y2);
	}
	else {
		// tick line
		cairo_move_to(_CR,x1,y1);
		cairo_line_to(_CR,x1+dxyflag*x2,y1-dxyflag*y2);
	}
	cairo_stroke(_CR);
}


void zsCairo::line(double *x, double *y, double *z, int n)
{
	cairo_new_path(_CR);
	double xs=x[0];
	double ys=y[0];
	double d1=transform(xs,ys,z?z[0]:_zmin);
	if (d1>=0) cairo_move_to(_CR,xs,ys);
	for (int i=1; i<n; i++) {
		xs=x[i];
		ys=y[i];
		double d2=transform(xs,ys,z?z[i]:_zmin);
		if (d2>=0) {
			if (d1>=0) {
				cairo_line_to(_CR,xs,ys);
			}
			else {
				cairo_stroke(_CR);
				cairo_move_to(_CR,xs,ys);
			}
		}
		d1=d2;
	}
	cairo_stroke(_CR);
}

void zsCairo::polygon(double *x, double *y, double *z, int n)
{
	cairo_new_path(_CR);
	double xs=x[0];
	double ys=y[0];
	double d1=transform(xs,ys,z?z[0]:_zmin);
	if (d1>=0) cairo_move_to(_CR,xs,ys);
	double lw=cairo_get_line_width(_CR);
	cairo_set_line_width(_CR,1);
	for (int i=1; i<n; i++) {
		xs=x[i];
		ys=y[i];
		double d2=transform(xs,ys,z?z[i]:_zmin);
		if (d2>=0) {
			if (d1>=0) {
				cairo_line_to(_CR,xs,ys);
			}
			else {
				cairo_fill_preserve(_CR);
				cairo_stroke(_CR);
				cairo_move_to(_CR,xs,ys);
			}
		}
	}
	cairo_fill_preserve(_CR);
	cairo_stroke(_CR);
	cairo_set_line_width(_CR,lw);
}

void zsCairo::circle(double x, double y, double z, double r, bool fill) {
	cairo_new_path(_CR);
	double d=transform(x,y,z);
	if (d<0) return;
	cairo_arc(_CR,x,y,r,0.0,360*D2R);
	if (fill) {
		double lw=cairo_get_line_width(_CR);
		cairo_set_line_width(_CR,1);
		cairo_fill_preserve(_CR);
		cairo_stroke(_CR);
		cairo_set_line_width(_CR,lw);
	}
	else {
		cairo_stroke(_CR);
	}
}

void zsCairo::arc(double x, double y, double z, double r, double a1, double a2, bool fill)
{
	cairo_new_path(_CR);
	double d=transform(x,y,z);
	if (d<0) return;
	cairo_arc(_CR,x,y,r,a1*D2R,a2*D2R);
	if (fill) {
		double lw=cairo_get_line_width(_CR);
		cairo_set_line_width(_CR,1);
		cairo_fill_preserve(_CR);
		cairo_stroke(_CR);
		cairo_set_line_width(_CR,lw);
	}
	else {
		cairo_stroke(_CR);
	}
}

void zsCairo::arrow(double x, double y, double z, double u, double v, double size)
{
	cairo_new_path(_CR);
	double r=sqrt(u*u+v*v)+1.e-12;
	double x1, y1, x2, y2;
	// arrow
	x1=x;
	y1=y;
	double d=transform(x1,y1,z);
	if (d<0) return;
	x2=x1+u;
	y2=y1+v;
	cairo_move_to(_CR,x1,y1);
	cairo_line_to(_CR,x2,y2);
	cairo_stroke(_CR);
	// arrow head
	double ax=-size;
	double ay=size/3.0;
	double cosphi=u/r;
	double sinphi=v/r;
	x1=x2+cosphi*ax-sinphi*ay;
	y1=y2+sinphi*ax+cosphi*ay;
	cairo_move_to(_CR,x1,y1);
	cairo_line_to(_CR,x2,y2);
	ay=-ay;
	x1=x2+cosphi*ax-sinphi*ay;
	y1=y2+sinphi*ax+cosphi*ay;
	cairo_line_to(_CR,x1,y1);
	cairo_stroke(_CR);
}

void zsCairo::text(const char *str, double x, double y, double z, double angle, int xalign, int yalign, double xoff, double yoff)
{
	cairo_new_path(_CR);
	transform(x,y,z);
	cairo_text_extents_t extents;
	cairo_text_extents(_CR,str,&extents);
	if (angle>0) {
		double angl2=D2R*(90-angle);
		angle*=D2R;
		if (xalign==0) {
			x-=extents.width*cos(angle)/2;
			x+=extents.height*cos(angl2)/2;
		}
		else if (xalign>0) {
			x-=extents.width*cos(angle);
		}
		else {
			x+=extents.height*cos(angl2);
		}
		if (yalign==0) {
			y+=extents.width*sin(angle)/2;
			y+=extents.height*sin(angl2)/2;
		}
		else if (yalign>0) {
			y+=extents.width*sin(angle);
			y+=extents.height*sin(angl2);
		}
	}
	else if (angle<0) {
		double angl2=D2R*(90+angle);
		angle*=D2R;
		if (xalign==0) {
			x-=extents.width*cos(angle)/2;
			x-=extents.height*cos(angl2)/2;
		}
		else if (xalign>0) {
			x-=extents.width*cos(angle);
			x-=extents.height*cos(angl2);
		}
		if (yalign==0) {
			y+=extents.width*sin(angle)/2;
			y+=extents.height*sin(angl2)/2;
		}
		else if (yalign>0) {
			y+=extents.height*sin(angl2);
		}
		else {
			y+=extents.width*sin(angle);
		}
	}
	else {
		if (xalign==0) {
			x-=extents.width/2;
		}
		else if (xalign>0) {
			x-=extents.width;
		}
		if (yalign==0) {
			y+=extents.height/2;
		}
		else if (yalign>0) {
			y+=extents.height;
		}
	}
	cairo_move_to(_CR,x+xoff,y+yoff);
	if (angle!=0) {
		cairo_save(_CR);
		cairo_rotate(_CR,-angle);
	}
	cairo_show_text(_CR,str);
	if (angle!=0) {
		cairo_restore(_CR);
	}
}

ZEcolor zsCairo::rgb(double v) const
{
	if (_colors.size()==0) return _color;
	ZEcolor c1=_colors[0];
	ZEcolor c2=c1;
	double v1=_values[0];
	if (v<=v1) return c1;
	if (v>=_values.back()) return _colors.back();
	for (size_t i=1; i<_colors.size(); i++) {
		ZEcolor c2=_colors[i];
		double v2=_values[i];
		if (v<v2) {
			if (!_gradient) return c1;
			double f1=(v2-v)/(v2-v1);
			double f2=(v-v1)/(v2-v1);
			ZEcolor c={f1*c1.r+f2*c2.r,f1*c1.g+f2*c2.g,f1*c1.b+f2*c2.b};
			return c;
		}
		c1=c2;
		v1=v2;
	}
	return c2;
}

void zsCairo::contour4p(double isoValue, const ZExyz &p0, const ZExyz &p1, const ZExyz &p2, const ZExyz &p3, double z)
{
	int edges=0;
	if (p0.z>=isoValue) edges |= 1;
	if (p1.z>=isoValue) edges |= 2;
	if (p2.z>=isoValue) edges |= 4;
	if (p3.z>=isoValue) edges |= 8;

	ZExyz e1, e2;

	switch (edges) {
	case 1:
	case 14:
		e1=cr_interp2D(p0,p1,isoValue);
		e2=cr_interp2D(p0,p3,isoValue);
		line(e1.x,e1.y,z,e2.x,e2.y,z);
		break;
	case 2:
	case 13:
		e1=cr_interp2D(p1,p0,isoValue);
		e2=cr_interp2D(p1,p2,isoValue);
		line(e1.x,e1.y,z,e2.x,e2.y,z);
		break;
	case 3:
	case 12:
		e1=cr_interp2D(p1,p2,isoValue);
		e2=cr_interp2D(p0,p3,isoValue);
		line(e1.x,e1.y,z,e2.x,e2.y,z);
		break;
	case 4:
	case 11:
		e1=cr_interp2D(p2,p1,isoValue);
		e2=cr_interp2D(p2,p3,isoValue);
		line(e1.x,e1.y,z,e2.x,e2.y,z);
		break;
	case 5:
	case 10:
		e1=cr_interp2D(p0,p1,isoValue);
		e2=cr_interp2D(p1,p2,isoValue);
		line(e1.x,e1.y,z,e2.x,e2.y,z);

		e1=cr_interp2D(p2,p3,isoValue);
		e2=cr_interp2D(p3,p0,isoValue);
		line(e1.x,e1.y,z,e2.x,e2.y,z);
		break;
	case 6:
	case 9:
		e1=cr_interp2D(p0,p1,isoValue);
		e2=cr_interp2D(p2,p3,isoValue);
		line(e1.x,e1.y,z,e2.x,e2.y,z);
		break;
	case 7:
	case 8:
		e1=cr_interp2D(p3,p2,isoValue);
		e2=cr_interp2D(p3,p0,isoValue);
		line(e1.x,e1.y,z,e2.x,e2.y,z);
		break;
	}
}

double zsCairo::xangle() const
{
	double sx1=_xmin, sy1=_ymin, sx2=_xmax, sy2=_ymin;
	transform(sx1,sy1,0.5*(_zmin+_zmax));
	transform(sx2,sy2,0.5*(_zmin+_zmax));
	double dx=sx2-sx1, dy=sy1-sy2;
	double angle=asin(dy/sqrt(dx*dx+dy*dy));
	angle/=D2R;
	return angle;
}

double zsCairo::yangle() const
{
	double sx1=_xmin, sy1=_ymin, sx2=_xmin, sy2=_ymax;
	transform(sx1,sy1,0.5*(_zmin+_zmax));
	transform(sx2,sy2,0.5*(_zmin+_zmax));
	double dx=sx2-sx1, dy=sy1-sy2;
	double angle=acos(dx/sqrt(dx*dx+dy*dy));
	angle/=D2R;
	if (_zrotate>0) angle-=180;
	return angle;
}

void zsCairo::dxdy(const char *str, double angle, double &dx, double &dy) const
{
	cairo_text_extents_t extents;
	cairo_text_extents(_CR,str,&extents);
	dx=extents.width;
	dy=extents.height;
	if (angle!=0.0) {
		double dx1=dy*cos(D2R*(90+angle));
		double dy1=dy*sin(D2R*(90+angle));
		double dx2=dx*cos(D2R*angle);
		double dy2=dx*sin(D2R*angle);
		dx=fabs(dx2-dx1);
		dy=fabs(dy2-dy1);
	}
}

void zsCairo::clip(int flag, double z)
{
	_clip=flag;
	if (!flag) {
		cairo_reset_clip(_CR);
		return;
	}
	if (_transform==TRANSFORM_PLOT || _transform==TRANSFORM_PLOT3D) {
		double x=_xmin;
		double y=_ymin;
		transform(x,y,z);
		cairo_move_to(_CR,x,y);
		x=_xmax;
		y=_ymin;
		transform(x,y,z);
		cairo_line_to(_CR,x,y);
		x=_xmax;
		y=_ymax;
		transform(x,y,z);
		cairo_line_to(_CR,x,y);
		x=_xmin;
		y=_ymax;
		transform(x,y,z);
		cairo_line_to(_CR,x,y);
	}
	else if (_transform==TRANSFORM_SPHERE) {
		double r=0.5*_scale;
		double x=_lon;
		double y=_lat;
		transform(x,y,_zmin);
		cairo_arc(_CR,x,y,r,0.0,360*D2R);
	}
	else if (_transform==TRANSFORM_HAMMER || _transform==TRANSFORM_ECKERT) {
		double x=_xmin;
		double y=_ymin;
		transform(x,y,z);
		cairo_move_to(_CR,x,y);
		x=_xmax;
		y=_ymin;
		transform(x,y,z);
		cairo_line_to(_CR,x,y);
		double lat;
		for (lat=_ymin+1; lat<=_ymax; lat++) {
			x=_xmax;
			y=lat;
			transform(x,y,z);
			cairo_line_to(_CR,x,y);
		}
		x=_xmin;
		y=_ymax;
		transform(x,y,z);
		cairo_line_to(_CR,x,y);
		for (lat=_ymax-1; lat>=_ymin; lat--) {
			x=_xmin;
			y=lat;
			transform(x,y,z);
			cairo_line_to(_CR,x,y);
		}
	}
	else if (_transform==TRANSFORM_POLAR || _transform==TRANSFORM_LAMBERT) {
		double x, y, lon;
		if (_ymin>=0) {
			x=_xmin;
			y=_ymin;
			transform(x,y,z);
			cairo_move_to(_CR,x,y);
			for (lon=_xmin+1; lon<=_xmax; lon++) {
				x=lon;
				y=_ymin;
				transform(x,y,z);
				cairo_line_to(_CR,x,y);
			}
			if (_transform==TRANSFORM_LAMBERT || (_xmax-_xmin)<360 || _ymax<90) {
				x=_xmax;
				y=_ymax;
				transform(x,y,z);
				cairo_line_to(_CR,x,y);
				for (lon=_xmax-1; lon>=_xmin; lon--) {
					x=lon;
					y=_ymax;
					transform(x,y,z);
					cairo_line_to(_CR,x,y);
				}
				x=_xmin;
				y=_ymin;
				transform(x,y,z);
				cairo_line_to(_CR,x,y);
			}
		}
		else {
			x=_xmin;
			y=_ymax;
			transform(x,y,z);
			cairo_move_to(_CR,x,y);
			for (lon=_xmin+1; lon<=_xmax; lon++) {
				x=lon;
				y=_ymax;
				transform(x,y,z);
				cairo_line_to(_CR,x,y);
			}
			if (_transform==TRANSFORM_LAMBERT || (_xmax-_xmin)<360 || _ymax>-90) {
				x=_xmax;
				y=_ymin;
				transform(x,y,z);
				cairo_line_to(_CR,x,y);
				for (lon=_xmax-1; lon>=_xmin; lon--) {
					x=lon;
					y=_ymin;
					transform(x,y,z);
					cairo_line_to(_CR,x,y);
				}
				x=_xmin;
				y=_ymax;
				transform(x,y,z);
				cairo_line_to(_CR,x,y);
			}
		}
	}
	cairo_clip(_CR);
}

/////////////////////////////***///////////////////////////////////////////////

void delete_cairo(void *ptr) { delete (zsCairo*)ptr; }

static
void* cr_create(void *ctx, int nargs, void** args)
{
	if (nargs<1) api_input_error(ctx);
	zsCairo *o=new zsCairo;
	int w, h;
	if (nargs==1) {
		const char *fname=api_get_string(ctx,args[0]);
		o->_SFC=cairo_image_surface_create_from_png(fname);
		if (!o->_SFC) api_runtime_error2(ctx,"Cannot load image from ",fname);
		o->_type=CAIRO_IMG;
		w=cairo_image_surface_get_width(o->_SFC);
		h=cairo_image_surface_get_height(o->_SFC);
	}
	else {
		w=api_get_integer(ctx,args[0]);
		h=api_get_integer(ctx,args[1]);
		const char *type="img";
		if (nargs>2) type=api_get_string(ctx,args[2]);
		if (strcmp(type,"pdf")==0) {
			o->_SFC=cairo_pdf_surface_create_for_stream(write_stream,o,w,h);
			o->_type=CAIRO_PDF;
		}
		else if (strcmp(type,"eps")==0) {
			o->_SFC=cairo_ps_surface_create_for_stream(write_stream,o,w,h);
			cairo_ps_surface_set_eps(o->_SFC,true);
			o->_type=CAIRO_EPS;
		}
		else if (strcmp(type,"svg")==0) {
			o->_SFC=cairo_svg_surface_create_for_stream(write_stream,o,w,h);
			o->_type=CAIRO_SVG;
		}
#ifdef _WIN32
		else if (strcmp(type,"win")==0) {
			o->_SFC=cairo_win32_surface_create_with_dib(CAIRO_FORMAT_RGB24,w,h);
			o->_type=CAIRO_WIN;
		}
#endif
		else {
			o->_SFC=cairo_image_surface_create(CAIRO_FORMAT_ARGB32,w,h);
			o->_type=CAIRO_IMG;
		}
	}
	o->_width=w;
	o->_height=h;
	o->_xmin=0;
	o->_xmax=w;
	o->_ymin=0;
	o->_ymax=h;
	o->_CR=cairo_create(o->_SFC);
    cairo_select_font_face(o->_CR,"Arial",CAIRO_FONT_SLANT_NORMAL,CAIRO_FONT_WEIGHT_BOLD);
	cairo_set_font_size(o->_CR,o->_fontsize);
	cairo_set_source_rgb(o->_CR,0.0,0.0,0.0);
	cairo_set_antialias(o->_CR,CAIRO_ANTIALIAS_BEST);
	cairo_set_line_width(o->_CR,1.5);
	return api_create_user(ctx,o,0,delete_cairo,CAIRO);
}

static
void* cr_version(void *ctx, int nargs, void** args)
{
	if (nargs<1) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	return api_create_string(ctx,cairo_version_string());
}

#ifdef _WIN32
LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	if (message==WM_PAINT) {
		zsCairo *o=(zsCairo*)GetWindowLong(hWnd,GWL_USERDATA);
		PAINTSTRUCT ps;
		HDC hDC=BeginPaint(hWnd,&ps);
		BitBlt(hDC,0,0,o->_width,o->_height,cairo_win32_surface_get_dc(o->_SFC),0,0,SRCCOPY);
		EndPaint(hWnd,&ps);
	}
	else if (message == WM_DESTROY) {
		PostQuitMessage(0);
		return 0;
	}
	return DefWindowProc(hWnd, message, wParam, lParam);
}

static
void* cr_show(void *ctx, int nargs, void** args)
{
	if (nargs<1) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	if (o->_type!=CAIRO_WIN) api_runtime_error(ctx,"Image type must be \"win\" for display");
	cairo_surface_flush(o->_SFC);
	WNDCLASS wndClass;
   	wndClass.style=CS_HREDRAW|CS_VREDRAW|CS_DBLCLKS; 
	wndClass.lpfnWndProc=WndProc;
	wndClass.cbClsExtra=0;
	wndClass.cbWndExtra=0;
	wndClass.hInstance=0;
	wndClass.hIcon=LoadIcon(NULL,IDI_WINLOGO);
	wndClass.hCursor=LoadCursor(NULL,IDC_ARROW);
	wndClass.hbrBackground=(HBRUSH)(COLOR_WINDOW);
	wndClass.lpszMenuName=0;
	wndClass.lpszClassName=TEXT("ZeCairo");
	RegisterClass(&wndClass);
	RECT rect={0,0,o->_width,o->_height};
	AdjustWindowRect(&rect,WS_SYSMENU|WS_THICKFRAME,1);
	HWND hWnd=CreateWindow(
		TEXT("ZeCairo"),TEXT("ZeGraph (http://www.zegraph.com/)"),
		WS_SYSMENU|WS_MINIMIZEBOX,
		CW_USEDEFAULT,CW_USEDEFAULT,
		rect.right-rect.left,rect.bottom-rect.top,
		0,0,0,0);
	SetWindowLong(hWnd,GWL_USERDATA,(LONG)o);
	ShowWindow(hWnd,SW_SHOW);
	UpdateWindow(hWnd);
	MSG msg;
	while(GetMessage(&msg,NULL,0,0)) {
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}
	return 0;
}
#endif

static
void* cr_save(void *ctx, int nargs, void** args)
{
	if (nargs<1) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	if (o->_type==CAIRO_WIN) api_runtime_error(ctx,"cannot save the image in a window's surface");
	cairo_surface_flush(o->_SFC);
	if (o->_type==CAIRO_IMG) cairo_surface_write_to_png_stream(o->_SFC,write_stream,o);
	cairo_surface_finish(o->_SFC);
	if (nargs>1) {
		const char *fname=api_get_string(ctx,args[1]);
		int ret=CAIRO_STATUS_SUCCESS+1;
		FILE *fp=fopen(fname,"wb");
		if (fp) {
			fwrite(o->_stream,1,o->_length,fp);
			fclose(fp);
			ret=CAIRO_STATUS_SUCCESS;
		}
		return api_create_integer(ctx,ret==CAIRO_STATUS_SUCCESS);
	}
	if (!o->_stream) return 0;
	void *arr=api_create_array(ctx,2);
	api_set_array_object(ctx,arr,"0",api_create_user(0,o->_stream,0,0,0));
	api_set_array_object(ctx,arr,"1",api_create_integer(0,o->_length));
	return arr;
}

static
void* cr_size(void *ctx, int nargs, void** args)
{
	if (nargs<2) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	void *arr = api_create_array(ctx,2);
	api_set_array_object(ctx,arr,"0",api_create_integer(0,o->_width));
	api_set_array_object(ctx,arr,"1",api_create_integer(0,o->_height));
	return arr;
}

static
void* cr_color(void *ctx, int nargs, void** args)
{
	if (nargs<1) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	if (nargs>3) {
		o->_color.r=api_get_number(ctx,args[1]);
		o->_color.g=api_get_number(ctx,args[2]);
		o->_color.b=api_get_number(ctx,args[3]);
	}
	else {
		o->_color=o->rgb(api_get_number(ctx,args[1]));
	}
	return 0;
}

static
void* cr_bgcolor(void *ctx, int nargs, void** args)
{
	if (nargs<4) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	o->_bgcolor.r=api_get_number(ctx,args[1]);
	o->_bgcolor.g=api_get_number(ctx,args[2]);
	o->_bgcolor.b=api_get_number(ctx,args[3]);
	cairo_move_to(o->_CR,0,0);
	cairo_set_source_rgb(o->_CR,o->_bgcolor.r,o->_bgcolor.g,o->_bgcolor.b);
	cairo_paint(o->_CR);
	cairo_new_path(o->_CR);
	cairo_set_source_rgb(o->_CR,o->_color.r,o->_color.g,o->_color.b);
	return 0;
}

static
void *cr_linear(void *ctx, int nargs, void **args)
{
	if (nargs<11) api_input_error(ctx);
	zsCairo* o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	double x1=api_get_number(ctx,args[1]);
	double y1=api_get_number(ctx,args[2]);
	double r1=api_get_number(ctx,args[3]);
	double g1=api_get_number(ctx,args[4]);
	double b1=api_get_number(ctx,args[5]);
	double x2=api_get_number(ctx,args[6]);
	double y2=api_get_number(ctx,args[7]);
	double r2=api_get_number(ctx,args[8]);
	double g2=api_get_number(ctx,args[9]);
	double b2=api_get_number(ctx,args[10]);
	double z=o->_zmin;
	if (nargs>11) api_get_number(ctx,args[11]);
	o->transform(x1,y1,z);
	o->transform(x2,y2,z);
	if (o->_PAT) cairo_pattern_destroy(o->_PAT);
	o->_PAT=cairo_pattern_create_linear(x1,y1,x2,y2);
	cairo_pattern_add_color_stop_rgb(o->_PAT,0,r1,g1,b1);
	cairo_pattern_add_color_stop_rgb(o->_PAT,1,r2,g2,b2);
	cairo_set_source(o->_CR,o->_PAT);
	return 0;
}

static
void *cr_radial(void *ctx, int nargs, void **args)
{
	if (nargs<13) api_input_error(ctx);
	zsCairo* o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	double cx1=api_get_number(ctx,args[1]);
	double cy1=api_get_number(ctx,args[2]);
	double cr1=api_get_number(ctx,args[3]);
	double r1=api_get_number(ctx,args[4]);
	double g1=api_get_number(ctx,args[5]);
	double b1=api_get_number(ctx,args[6]);
	double cx2=api_get_number(ctx,args[7]);
	double cy2=api_get_number(ctx,args[8]);
	double cr2=api_get_number(ctx,args[9]);
	double r2=api_get_number(ctx,args[10]);
	double g2=api_get_number(ctx,args[11]);
	double b2=api_get_number(ctx,args[12]);
	double z=o->_zmin;
	if (nargs>13) z=api_get_number(ctx,args[13]);
	o->transform(cx1,cy1,z);
	o->transform(cx2,cy2,z);
	if (o->_PAT) cairo_pattern_destroy(o->_PAT);
	o->_PAT=cairo_pattern_create_radial(cx1,cy1,cr1,cx2,cy2,cr2);
	cairo_pattern_add_color_stop_rgb(o->_PAT,0,r1,g1,b1);
	cairo_pattern_add_color_stop_rgb(o->_PAT,1,r2,g2,b2);
	cairo_set_source(o->_CR,o->_PAT);
	return 0;
}

static
void *cr_vertex(void *ctx, int nargs, void **args)
{
	if (nargs<16) api_input_error(ctx);
	zsCairo* o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	double x1=api_get_number(ctx,args[1]);
	double y1=api_get_number(ctx,args[2]);
	double r1=api_get_number(ctx,args[3]);
	double g1=api_get_number(ctx,args[4]);
	double b1=api_get_number(ctx,args[5]);
	double x2=api_get_number(ctx,args[6]);
	double y2=api_get_number(ctx,args[7]);
	double r2=api_get_number(ctx,args[8]);
	double g2=api_get_number(ctx,args[9]);
	double b2=api_get_number(ctx,args[10]);
	double x3=api_get_number(ctx,args[11]);
	double y3=api_get_number(ctx,args[12]);
	double r3=api_get_number(ctx,args[13]);
	double g3=api_get_number(ctx,args[14]);
	double b3=api_get_number(ctx,args[15]);
	double x4, y4, r4, g4, b4;
	double z=o->_zmin;
	if (nargs<21) {
		if (nargs>16) z=api_get_number(ctx,args[16]);
	}
	else {
		x4=api_get_number(ctx,args[16]);
		y4=api_get_number(ctx,args[17]);
		r4=api_get_number(ctx,args[18]);
		g4=api_get_number(ctx,args[19]);
		b4=api_get_number(ctx,args[20]);
		if (nargs>21) z=api_get_number(ctx,args[21]);
		o->transform(x4,y4,z);
	}
	o->transform(x1,y1,z);
	o->transform(x2,y2,z);
	o->transform(x3,y3,z);
	if (o->_PAT) cairo_pattern_destroy(o->_PAT);
	o->_PAT=cairo_pattern_create_mesh();
	cairo_mesh_pattern_begin_patch(o->_PAT);
	cairo_mesh_pattern_move_to(o->_PAT,x1,y1);
	cairo_mesh_pattern_line_to(o->_PAT,x2,y2);
	cairo_mesh_pattern_line_to(o->_PAT,x3,y3);
	if (nargs>=21) cairo_mesh_pattern_line_to(o->_PAT,x4,y4);
	cairo_mesh_pattern_set_corner_color_rgb(o->_PAT,0,r1,g1,b1);
	cairo_mesh_pattern_set_corner_color_rgb(o->_PAT,1,r2,g2,b2);
	cairo_mesh_pattern_set_corner_color_rgb(o->_PAT,2,r3,g3,b3);
	if (nargs>=21) cairo_mesh_pattern_set_corner_color_rgb(o->_PAT,3,r4,g4,b4);
	cairo_mesh_pattern_end_patch(o->_PAT);
	cairo_set_source(o->_CR,o->_PAT);
	return 0;
}

static void* cr_font(void *ctx, int nargs, void** args)
{
	if (nargs<2) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	if (api_is_string(args[1])) {
	   cairo_select_font_face(o->_CR,api_get_string(ctx,args[1]),CAIRO_FONT_SLANT_NORMAL,CAIRO_FONT_WEIGHT_BOLD);
	}
	else {
		o->_fontsize=api_get_number(ctx,args[1]);
		cairo_set_font_size(o->_CR,o->_fontsize);
	}
	return 0;
}

static
void* cr_clip(void *ctx, int nargs, void** args)
{
	if (nargs<1) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	double z=o->_zmin;
	if (nargs>1) z=api_get_number(ctx,args[1]);
	o->clip(1,z);
	return 0;
}

static
void* cr_unclip(void *ctx, int nargs, void** args)
{
	if (nargs<1) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	o->clip(0,o->_zmin);
	return 0;
}

static void* cr_lwidth(void *ctx, int nargs, void** args)
{
	if (nargs<2) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	cairo_set_line_width(o->_CR,api_get_number(ctx,args[1]));
	return 0;
}

static void* cr_lstyle(void *ctx, int nargs, void** args)
{
	if (nargs<3) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	double *d=(double*)api_get_ptr(ctx,args[1]);
	int n=api_get_integer(ctx,args[2]);
	cairo_set_dash(o->_CR,d,n,0);
	return 0;
}

static void* cr_state(void *ctx, int nargs, void** args)
{
	if (nargs<2) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	int flag=api_get_integer(ctx,args[1]);
	if (flag<0) {
		cairo_save(o->_CR);
	}
	else {
		cairo_restore(o->_CR);
	}
	return 0;
}

static
void* cr_image(void *ctx, int nargs, void** args)
{
	if (nargs<2) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	const char *fname=api_get_string(ctx,args[1]);
	cairo_surface_t *image=cairo_image_surface_create_from_png(fname);
	if (!image) {
		api_runtime_error2(ctx,"failed to load image: ",fname);
	}
	int f=cairo_image_surface_get_format(image);
	if (f!=CAIRO_FORMAT_ARGB32 && f!=CAIRO_FORMAT_RGB24) {
		api_runtime_error(ctx,"image format is neither RGB24 nor ARGB32");
	}
	int w=cairo_image_surface_get_width(image);
	int h=cairo_image_surface_get_height(image);
	int s=cairo_image_surface_get_stride(image);
	if (o->_transform==TRANSFORM_NONE) {
		double x=0, y=0;
		if (nargs>2) x=api_get_number(ctx,args[2]);
		if (nargs>3) y=api_get_number(ctx,args[3]);
		cairo_set_source_surface(o->_CR,image,x,y);
		cairo_paint(o->_CR);
		cairo_identity_matrix(o->_CR); 
	}
	else if (o->_transform==TRANSFORM_PLOT) {
		double x1=o->_xmin, y1=o->_ymin, x2=o->_xmax, y2=o->_ymax;
		o->transform(x1,y1,o->_zmin);
		o->transform(x2,y2,o->_zmin);
		if (fabs(x2-x1)!=w || fabs(y2-y1)!=h) {
			double xs=fabs(x2-x1)/w;
			double ys=fabs(y2-y1)/h;
			cairo_scale(o->_CR,xs,ys);
			x1/=xs;
			y2/=ys;
		}
		cairo_set_source_surface(o->_CR,image,x1,y2);
		cairo_paint(o->_CR);
		cairo_identity_matrix(o->_CR); 
	}
	else {
		unsigned char *ptr=cairo_image_surface_get_data(image);
		double dx=(o->_xmax-o->_xmin)/w;
		double dy=(o->_ymax-o->_ymin)/h;
		double lw=cairo_get_line_width(o->_CR);
		cairo_set_line_width(o->_CR,1);
		for (int i=0; i<h; i++) {
			for (int j=0; j<w; j++) {
				int k=i*s+j*4;
				double b=ptr[k+0]/255.0;
				double g=ptr[k+1]/255.0;
				double r=ptr[k+2]/255.0;
				double a=ptr[k+3]/255.0;
				double x1=o->_xmin+j*dx;
				double y1=o->_ymax-i*dy;
				double x2=x1+dx;
				double y2=y1+dy;
				o->transform(x1,y1,o->_zmin);
				o->transform(x2,y2,o->_zmin);
				cairo_move_to(o->_CR,x1,y1);
				cairo_line_to(o->_CR,x2,y1);
				cairo_line_to(o->_CR,x2,y2);
				cairo_line_to(o->_CR,x1,y2);
				cairo_set_source_rgba(o->_CR,r,g,b,a);
				cairo_fill_preserve(o->_CR);
				cairo_stroke(o->_CR);
			}
		}
		cairo_set_line_width(o->_CR,lw);
	}
	cairo_surface_destroy(image);
	cairo_set_source_rgb(o->_CR,o->_color.r,o->_color.g,o->_color.b);
	void *arr=api_create_array(ctx,2);
	api_set_array_object(ctx,arr,"0",api_create_integer(0,w));
	api_set_array_object(ctx,arr,"1",api_create_integer(0,h));
	return arr;
}

static
void* cr_text(void *ctx, int nargs, void** args)
{
	if (nargs<4) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	const char *str=api_get_string(ctx,args[1]);
	double x=api_get_number(ctx,args[2]);
	double y=api_get_number(ctx,args[3]);
	double angle=0;
	if (nargs>4) angle=api_get_number(ctx,args[4]);
	int xalign=-1;
	if (nargs>5) xalign=api_get_integer(ctx,args[5]);
	int yalign=-1;
	if (nargs>6) yalign=api_get_integer(ctx,args[6]);
	double xoff=0;
	if (nargs>7) xoff=api_get_number(ctx,args[7]);
	double yoff=0;
	if (nargs>8) yoff=api_get_number(ctx,args[8]);
	o->text(str,x,y,o->_zmin,angle,xalign,yalign,xoff,yoff);
	return 0;	
}

static
void* cr_text3D(void *ctx, int nargs, void** args)
{
	if (nargs<5) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	const char *str=api_get_string(ctx,args[1]);
	double x=api_get_number(ctx,args[2]);
	double y=api_get_number(ctx,args[3]);
	double z=api_get_number(ctx,args[4]);
	double angle=0;
	if (nargs>5) angle=api_get_number(ctx,args[5]);
	int xalign=-1;
	if (nargs>6) xalign=api_get_integer(ctx,args[6]);
	int yalign=-1;
	if (nargs>7) yalign=api_get_integer(ctx,args[7]);
	o->text(str,x,y,z,angle,xalign,yalign);
	return 0;	
}

static
void* cr_line(void *ctx, int nargs, void** args)
{
	if (nargs<4) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	if (nargs==4) {
		double *x=(double*)api_get_ptr(ctx,args[1]);
		double *y=(double*)api_get_ptr(ctx,args[2]);
		int n=api_get_integer(ctx,args[3]);
		o->line(x,y,0,n);
	}
	else {
		if (nargs<5) api_input_error(ctx);
		o->line(
			api_get_number(ctx,args[1]),
			api_get_number(ctx,args[2]),
			o->_zmin,
			api_get_number(ctx,args[3]),
			api_get_number(ctx,args[4]),
			o->_zmin);
	}
	return 0;
}

static
void* cr_line3D(void *ctx, int nargs, void** args)
{
	if (nargs<5) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	if (nargs==5) {
		double *x=(double*)api_get_ptr(ctx,args[1]);
		double *y=(double*)api_get_ptr(ctx,args[2]);
		double *z=(double*)api_get_ptr(ctx,args[3]);
		int n=api_get_integer(ctx,args[4]);
		o->line(x,y,z,n);
	}
	else {
		if (nargs<7) api_input_error(ctx);
		o->line(
			api_get_number(ctx,args[1]),
			api_get_number(ctx,args[2]),
			api_get_number(ctx,args[3]),
			api_get_number(ctx,args[4]),
			api_get_number(ctx,args[5]),
			api_get_number(ctx,args[6])
			);
	}
	return 0;
}

static
void* cr_curve(void *ctx, int nargs, void** args)
{
	if (nargs<9) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	double x0=api_get_number(ctx,args[1]);
	double y0=api_get_number(ctx,args[2]);
	o->transform(x0,y0,o->_zmin);
	double x1=api_get_number(ctx,args[3]);
	double y1=api_get_number(ctx,args[4]);
	o->transform(x1,y1,o->_zmin);
	double x2=api_get_number(ctx,args[5]);
	double y2=api_get_number(ctx,args[6]);
	o->transform(x2,y2,o->_zmin);
	double x3=api_get_number(ctx,args[7]);
	double y3=api_get_number(ctx,args[8]);
	o->transform(x3,y3,o->_zmin);
	cairo_move_to(o->_CR,x0,y0);
	cairo_curve_to(o->_CR,x1,y1,x2,y2,x3,y3);
	return 0;
}

static
void* cr_curve3D(void *ctx, int nargs, void** args)
{
	if (nargs<13) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	double x0=api_get_number(ctx,args[1]);
	double y0=api_get_number(ctx,args[2]);
	double z0=api_get_number(ctx,args[3]);
	o->transform(x0,y0,z0);
	double x1=api_get_number(ctx,args[4]);
	double y1=api_get_number(ctx,args[5]);
	double z1=api_get_number(ctx,args[6]);
	o->transform(x1,y1,z1);
	double x2=api_get_number(ctx,args[7]);
	double y2=api_get_number(ctx,args[8]);
	double z2=api_get_number(ctx,args[9]);
	o->transform(x2,y2,z2);
	double x3=api_get_number(ctx,args[10]);
	double y3=api_get_number(ctx,args[11]);
	double z3=api_get_number(ctx,args[12]);
	o->transform(x3,y3,z3);
	cairo_move_to(o->_CR,x0,y0);
	cairo_curve_to(o->_CR,x1,y1,x2,y2,x3,y3);
	return 0;
}

static
void* cr_polygon(void *ctx, int nargs, void** args)
{
	if (nargs<4) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	if (api_is_user(args[1])) {
		double *x=(double*)api_get_ptr(ctx,args[1]);
		double *y=(double*)api_get_ptr(ctx,args[2]);
		int n=api_get_integer(ctx,args[3]);
		o->polygon(x,y,0,n);
	}
	else {
		vector<double> x;
		vector<double> y;
		for (int i=2; i<nargs; i+=2) {
			x.push_back(api_get_number(ctx,args[i-1]));
			y.push_back(api_get_number(ctx,args[i]));
		}
		if (x.size()<3) api_input_error(ctx);
		o->polygon(&x[0],&y[0],0,x.size());
	}
	return 0;
}

static
void* cr_polygon3D(void *ctx, int nargs, void** args)
{
	if (nargs<5) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	if (api_is_user(args[1])) {
		double *x=(double*)api_get_ptr(ctx,args[1]);
		double *y=(double*)api_get_ptr(ctx,args[2]);
		double *z=(double*)api_get_ptr(ctx,args[3]);
		int n=api_get_integer(ctx,args[4]);
		o->polygon(x,y,z,n);
	}
	else {
		vector<double> x;
		vector<double> y;
		vector<double> z;
		for (int i=3; i<nargs; i+=3) {
			x.push_back(api_get_number(ctx,args[i-2]));
			y.push_back(api_get_number(ctx,args[i-1]));
			z.push_back(api_get_number(ctx,args[i]));
		}
		if (x.size()<3) api_input_error(ctx);
		o->polygon(&x[0],&y[0],&z[0],x.size());
	}
	return 0;
}

static
void* cr_circle(void *ctx, int nargs, void** args)
{
	if (nargs<4) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	double x=api_get_number(ctx,args[1]);
	double y=api_get_number(ctx,args[2]);
	double r=api_get_number(ctx,args[3]);
	bool fill=true;
	if (nargs>4) fill=api_get_integer(ctx,args[4])!=0;
	o->circle(x,y,o->_zmin,r,fill);
	return 0;
}

static
void* cr_circle3D(void *ctx, int nargs, void** args)
{
	if (nargs<5) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	double x=api_get_number(ctx,args[1]);
	double y=api_get_number(ctx,args[2]);
	double z=api_get_number(ctx,args[3]);
	double r=api_get_number(ctx,args[4]);
	o->circle(x,y,z,r);
	return 0;
}

static
void* cr_arc(void *ctx, int nargs, void** args)
{
	if (nargs<6) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	double x=api_get_number(ctx,args[1]);
	double y=api_get_number(ctx,args[2]);
	double r=api_get_number(ctx,args[3]);
	double a1=api_get_number(ctx,args[4]);
	double a2=api_get_number(ctx,args[5]);
	bool fill=true;
	if (nargs>6) fill=api_get_integer(ctx,args[6])!=0;
	o->arc(x,y,o->_zmin,r,a1,a2,fill);
	return 0;
}

void* cr_arc3D(void *ctx, int nargs, void** args)
{
	if (nargs<7) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	double x=api_get_number(ctx,args[1]);
	double y=api_get_number(ctx,args[2]);
	double z=api_get_number(ctx,args[3]);
	double r=api_get_number(ctx,args[4]);
	double a1=api_get_number(ctx,args[5]);
	double a2=api_get_number(ctx,args[6]);
	bool fill=true;
	if (nargs>7) fill=api_get_integer(ctx,args[7])!=0;
	o->arc(x,y,z,r,a1,a2,fill);
	return 0;
}

static
void* cr_arrow(void *ctx, int nargs, void** args)
{
	if (nargs<5) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	double size=7;
	if (api_is_user(args[1])) {
		if (nargs<6) api_input_error(ctx);
		double *x=(double*)api_get_ptr(ctx,args[1]);
		double *y=(double*)api_get_ptr(ctx,args[2]);
		double *u=(double*)api_get_ptr(ctx,args[3]);
		double *v=(double*)api_get_ptr(ctx,args[4]);
		int n=api_get_integer(ctx,args[5]);
		if (nargs>6) size=api_get_number(ctx,args[6]);
		for (int i=0; i<n; i++) {
			o->arrow(x[i],y[i],o->_zmin,u[i],v[i],size);
		}
	}
	else {
		double x=api_get_number(ctx,args[1]);
		double y=api_get_number(ctx,args[2]);
		double u=api_get_number(ctx,args[3]);
		double v=api_get_number(ctx,args[4]);
		if (nargs>5) size=api_get_number(ctx,args[5]);
		o->arrow(x,y,o->_zmin,u,v,size);
	}
	return 0;
}

static
void* cr_arrow3D(void *ctx, int nargs, void** args)
{
	if (nargs<6) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	double size=7;
	if (api_is_user(args[1])) {
		if (nargs<7) api_input_error(ctx);
		double *x=(double*)api_get_ptr(ctx,args[1]);
		double *y=(double*)api_get_ptr(ctx,args[2]);
		double *z=(double*)api_get_ptr(ctx,args[3]);
		double *u=(double*)api_get_ptr(ctx,args[4]);
		double *v=(double*)api_get_ptr(ctx,args[5]);
		int n=api_get_integer(ctx,args[6]);
		if (nargs>7) size=api_get_number(ctx,args[7]);
		for (int i=0; i<n; i++) {
			o->arrow(x[i],y[i],z[i],u[i],v[i],size);
		}
	}
	else {
		double x=api_get_number(ctx,args[1]);
		double y=api_get_number(ctx,args[2]);
		double z=api_get_number(ctx,args[3]);
		double u=api_get_number(ctx,args[4]);
		double v=api_get_number(ctx,args[5]);
		if (nargs>6) size=api_get_number(ctx,args[6]);
		o->arrow(x,y,z,u,v,size);
	}
	return 0;
}

bool
cr_symbol_xyz(zsCairo *o, const char *name, double x, double y, double z, double size)
{
	double lw=cairo_get_line_width(o->_CR);
	size/=2;
	if (name[0]=='o') {
		o->circle(x,y,z,size,false);
		return true;
	}
	if (name[0]=='O') {
		o->circle(x,y,z,size,true);
		return true;
	}
	if (name[0]=='+') {
		double d=o->transform(x,y,z);
		if (d>=0) {
			cairo_new_path(o->_CR);
			cairo_move_to(o->_CR,x-size,y);
			cairo_line_to(o->_CR,x+size,y);
			cairo_move_to(o->_CR,x,y-size);
			cairo_line_to(o->_CR,x,y+size);
			cairo_stroke(o->_CR);
		}
		return true;
	}
	if (name[0]=='x' || name[0]=='X') {
		size*=0.71; //cos(pi/4)
		double d=o->transform(x,y,z);
		if (d>=0) {
			cairo_new_path(o->_CR);
			cairo_move_to(o->_CR,x-size,y-size);
			cairo_line_to(o->_CR,x+size,y+size);
			cairo_move_to(o->_CR,x-size,y+size);
			cairo_line_to(o->_CR,x+size,y-size);
			cairo_stroke(o->_CR);
		}
		return true;
	}
	if (name[0]=='s' || name[0]=='S') {
		size*=0.9;
		double d=o->transform(x,y,z);
		if (d>=0) {
			cairo_new_path(o->_CR);
			cairo_move_to(o->_CR,x-size,y-size);
			cairo_line_to(o->_CR,x+size,y-size);
			cairo_line_to(o->_CR,x+size,y+size);
			cairo_line_to(o->_CR,x-size,y+size);
			cairo_line_to(o->_CR,x-size,y-size);
			if (name[0]=='S') {
				cairo_set_line_width(o->_CR,1);
				cairo_fill_preserve(o->_CR);
			}
			cairo_stroke(o->_CR);
			if (name[0]=='S') {
				cairo_set_line_width(o->_CR,lw);
			}
		}
		return true;
	}
	if (name[0]=='t' || name[0]=='T') {
		double d=o->transform(x,y,z);
		if (d>=0) {
			cairo_new_path(o->_CR);
			cairo_move_to(o->_CR,x-size,y+size);
			cairo_line_to(o->_CR,x+size,y+size);
			cairo_line_to(o->_CR,x,y-size);
			cairo_line_to(o->_CR,x-size,y+size);
			if (name[0]=='T') {
				cairo_set_line_width(o->_CR,1);
				cairo_fill_preserve(o->_CR);
			}
			cairo_stroke(o->_CR);
			if (name[0]=='T') {
				cairo_set_line_width(o->_CR,lw);
			}
		}
		return true;
	}
	if (name[0]=='d' || name[0]=='D') {
		size*=0.9;
		double d=o->transform(x,y,z);
		if (d>=0) {
			cairo_new_path(o->_CR);
			cairo_move_to(o->_CR,x,y-size);
			cairo_line_to(o->_CR,x+size,y);
			cairo_line_to(o->_CR,x,y+size);
			cairo_line_to(o->_CR,x-size,y);
			cairo_line_to(o->_CR,x,y-size);
			if (name[0]=='D') {
				cairo_set_line_width(o->_CR,1);
				cairo_fill_preserve(o->_CR);
			}
			cairo_stroke(o->_CR);
			if (name[0]=='D') {
				cairo_set_line_width(o->_CR,lw);
			}
		}
		return true;
	}
	return false;
}

static
void* cr_symbol(void *ctx, int nargs, void** args)
{
	if (nargs<2) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	const char *name=api_get_string(ctx,args[1]);
	if (api_is_user(args[2])) {
		if (nargs<5) api_input_error(ctx);
		double *x=(double*)api_get_ptr(ctx,args[2]);
		double *y=(double*)api_get_ptr(ctx,args[3]);
		int n=api_get_integer(ctx,args[4]);
		double w=o->_fontsize;
		if (nargs>5) w=api_get_number(ctx,args[5]);
		for (int i=0; i<n; i++) {
			if (!cr_symbol_xyz(o,name,x[i],y[i],o->_zmin,w)) {
				api_runtime_error2(ctx,"bad symbol name: ",name);
			}
		}
	}
	else {
		if (nargs<4) api_input_error(ctx);
		double x=api_get_number(ctx,args[2]);
		double y=api_get_number(ctx,args[3]);
		double w=o->_fontsize;
		if (nargs>4) w=api_get_number(ctx,args[4]);
		if (!cr_symbol_xyz(o,name,x,y,o->_zmin,w)) {
			api_runtime_error2(ctx,"bad symbol name: ",name);
		}
	}
	return 0;
}

static
void* cr_symbol3D(void *ctx, int nargs, void** args)
{
	if (nargs<2) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	const char *name=api_get_string(ctx,args[1]);
	if (api_is_user(args[2])) {
		if (nargs<6) api_input_error(ctx);
		double *x=(double*)api_get_ptr(ctx,args[2]);
		double *y=(double*)api_get_ptr(ctx,args[3]);
		double *z=(double*)api_get_ptr(ctx,args[4]);
		int n=api_get_integer(ctx,args[5]);
		double w=o->_fontsize;
		if (nargs>6) w=api_get_number(ctx,args[6]);
		for (int i=0; i<n; i++) {
			if (!cr_symbol_xyz(o,name,x[i],y[i],z[i],w)) {
				api_runtime_error2(ctx,"bad symbol name: ",name);
			}
		}
	}
	else {
		if (nargs<5) api_input_error(ctx);
		double x=api_get_number(ctx,args[2]);
		double y=api_get_number(ctx,args[3]);
		double z=api_get_number(ctx,args[4]);
		double w=o->_fontsize;
		if (nargs>5) w=api_get_number(ctx,args[5]);
		if (!cr_symbol_xyz(o,name,x,y,z,w)) {
			api_runtime_error2(ctx,"bad symbol name: ",name);
		}
	}
	return 0;
}

static
void* cr_plot(void *ctx, int nargs, void** args)
{
	if (nargs<1) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	if (nargs==1) {
		void *arr=api_create_array(ctx,13);
		api_set_array_object(ctx,arr,"xscale",api_create_real(0,o->_xscale));
		api_set_array_object(ctx,arr,"yscale",api_create_real(0,o->_yscale));
		api_set_array_object(ctx,arr,"zscale",api_create_real(0,o->_zscale));
		api_set_array_object(ctx,arr,"xmin",api_create_real(0,o->_xmin));
		api_set_array_object(ctx,arr,"ymin",api_create_real(0,o->_ymin));
		api_set_array_object(ctx,arr,"zmin",api_create_real(0,o->_zmin));
		api_set_array_object(ctx,arr,"xmax",api_create_real(0,o->_xmax));
		api_set_array_object(ctx,arr,"ymax",api_create_real(0,o->_ymax));
		api_set_array_object(ctx,arr,"zmax",api_create_real(0,o->_zmax));
		api_set_array_object(ctx,arr,"xoffset",api_create_real(0,o->_xoffset));
		api_set_array_object(ctx,arr,"yoffset",api_create_real(0,o->_yoffset));
		api_set_array_object(ctx,arr,"zoffset",api_create_real(0,o->_zoffset));
		api_set_array_object(ctx,arr,"xrotate",api_create_real(0,o->_xrotate));
		api_set_array_object(ctx,arr,"zrotate",api_create_real(0,o->_zrotate));
		return arr;
	}
	if (nargs<3) api_input_error(ctx);
	o->_transform=TRANSFORM_PLOT;
	for (int i=2; i<nargs; i+=2) {
		const char *name=api_get_string(ctx,args[i-1]);
		if (!strcmp(name,"xscale")) {
			o->_xscale=api_get_number(ctx,args[i]);
			if (o->_xscale<=0) api_runtime_error(ctx, "xscale <= 0");
		}
		else if (!strcmp(name,"yscale")) {
			o->_yscale=api_get_number(ctx,args[i]);
			if (o->_yscale<=0) api_runtime_error(ctx, "yscale <= 0");
		}
		else if (!strcmp(name,"zscale")) {
			o->_zscale=api_get_number(ctx,args[i]);
			if (o->_zscale<=0) api_runtime_error(ctx, "zscale <= 0");
		}
		else if (!strcmp(name,"xoffset")) {
			o->_xoffset=api_get_number(ctx,args[i]);
		}
		else if (!strcmp(name,"yoffset")) {
			o->_yoffset=api_get_number(ctx,args[i]);
		}
		else if (!strcmp(name,"zoffset")) {
			o->_zoffset=api_get_number(ctx,args[i]);
		}
		else if (!strcmp(name,"xmin")) {
			o->_xmin=api_get_number(ctx,args[i]);
		}
		else if (!strcmp(name,"ymin")) {
			o->_ymin=api_get_number(ctx,args[i]);
		}
		else if (!strcmp(name,"zmin")) {
			o->_zmin=api_get_number(ctx,args[i]);
		}
		else if (!strcmp(name,"xmax")) {
			o->_xmax=api_get_number(ctx,args[i]);
		}
		else if (!strcmp(name,"ymax")) {
			o->_ymax=api_get_number(ctx,args[i]);
		}
		else if (!strcmp(name,"zmax")) {
			o->_zmax=api_get_number(ctx,args[i]);
		}
		else if (!strcmp(name,"zrotate")) {
			o->_zrotate=api_get_number(ctx,args[i]);
			if (o->_xrotate==0) o->_xrotate=-45;
			if (o->_zrotate!=0) o->_transform=TRANSFORM_PLOT3D;
		}
		else if (!strcmp(name,"xrotate")) {
			o->_xrotate=api_get_number(ctx,args[i]);
			if (o->_xrotate>0) o->_xrotate=-o->_xrotate;
			if (o->_zrotate==0) o->_zrotate=30;
			if (o->_xrotate!=0) o->_transform=TRANSFORM_PLOT3D;
		}
		else {
			api_runtime_error2(ctx, "unknown plot parameter name: ", name);
		}
	}
	
	if (o->_xmax<=o->_xmin) api_runtime_error(ctx, "xmax < xmin");
	if (o->_ymax<=o->_ymin) api_runtime_error(ctx, "ymax < ymin");
	if (o->_zmax<=o->_zmin) api_runtime_error(ctx, "zmax < zmin");

	o->_scale=o->_yscale*o->_height;
	if (o->_xscale*o->_width<o->_scale) o->_scale=o->_xscale*o->_width;

	if (o->_transform==TRANSFORM_PLOT3D) {
		double rotz=o->_zrotate*D2R;
		double rotx=o->_xrotate*D2R;
		o->_mr1[0]=cos(rotz);
		o->_mr1[1]=-sin(rotz);
		o->_mr1[2]=0;
		o->_mr2[0]=cos(rotx)*sin(rotz);
		o->_mr2[1]=cos(rotx)*cos(rotz);
		o->_mr2[2]=-sin(rotx);
		o->_mr3[0]=sin(rotx)*sin(rotz);
		o->_mr3[1]=sin(rotx)*cos(rotz);
		o->_mr3[2]=cos(rotx);
	}
	
	return 0;
}

static
void* cr_axis(void *ctx, int nargs, void** args, char type)
{
	if (nargs < 3) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);

	if (o->_transform!=TRANSFORM_PLOT && o->_transform!=TRANSFORM_PLOT3D) return 0;

	int i, j, k;
	const char *title=0;
	int side=-1;
	double start=2*EXLARGE;
	double end=2*EXLARGE;
	double step=2*EXLARGE;
	double angle=0;
	void *labels=0;
	int nl=0;
	int minor=0;
	int between=0;
	char fmt[24];
	char str[256];

	strcpy(fmt,"%0.1f");

	for (i=2; i<nargs; i+=2) {
		const char *name=api_get_string(ctx,args[i-1]);
		if (!strcmp(name,"title")) {
			title=api_get_string(ctx,args[i]);
		}
		else if (!strcmp(name,"side")) {
			side=api_get_integer(ctx,args[i]);
		}
		else if (!strcmp(name,"digit")) {
			int digit=api_get_integer(ctx,args[i]);
			if (digit<0 || digit>10) api_runtime_error(ctx,"digit < 0 || digit > 10");
			sprintf(fmt,"%%0.%df",digit);
		}
		else if (!strcmp(name,"start")) {
			start=api_get_number(ctx, args[i]);
		}
		else if (!strcmp(name, "end")) {
			end=api_get_number(ctx,args[i]);
		}
		else if (!strcmp(name,"step")) {
			step=api_get_number(ctx,args[i]);
		}
		else if (!strcmp(name,"minor")) {
			minor=api_get_integer(ctx, args[i]);
		}
		else if (!strcmp(name,"angle")) {
			angle=api_get_number(ctx,args[i]);
			if (angle<0 || angle>90) api_runtime_error(ctx,"angle must be between 0 and 90 degrees");
		}
		else if (!strcmp(name,"between")) {
			between=api_get_integer(ctx,args[i]);
		}
		else if (!strcmp(name,"labels")) {
			labels=args[i];
			if (!api_is_array(labels)) api_runtime_error(ctx,"labels must be an array of strings");
			nl=api_get_array_size(ctx,labels);
		}
		else {
			api_runtime_error2(ctx,"unknown axis parameter name: ",name);
		}
	}

	double x, y, z, x0, y0, z0, x1, y1, z1, sx, sy;
	double dx, dy, dz, tickLen, labelXOff, labelYOff, titleXOff, titleYOff;
	double v, dv, dmax;
	int xalign, yalign;

	x0=o->_xmin;
	x1=o->_xmax;
	dx=x1-x0;
	y0=o->_ymin;
	y1=o->_ymax;
	dy=y1-y0;
	z0=o->_zmin;
	z1=o->_zmax;
	dz=z1-z0;

	tickLen=0.015*(o->_width*o->_xscale+o->_height*o->_yscale)/2.0;

	if (type == 'x') {
		if (start>EXLARGE) start=x0;
		if (end>EXLARGE) end=x1;
		if (step>EXLARGE) step=dx/4.0;
		// axis line
		z=o->_zmin;
		y=y0;
		xalign=0;
		yalign=0;
		if (o->_transform==TRANSFORM_PLOT3D) {
			side=-1;
			between=0;
			angle=o->xangle();
			tickLen=1.1*tickLen;
			labelYOff=tickLen+0.5*o->_fontsize;
			labelXOff=tickLen*cos(angle*D2R);
			titleYOff=labelYOff+1.5*o->_fontsize;
			titleXOff=labelYOff*cos(angle*D2R);
			if (angle<0) {
				labelXOff=-labelXOff;
				titleXOff=-titleXOff;
			}
		}
		else {
			labelXOff=0;
			labelYOff=1.5*tickLen;
			titleXOff=0;
			if (side>0) {
				y=y1;
				yalign=-1;
				titleYOff=labelYOff+1.5*o->_fontsize;
				labelYOff=-labelYOff;
				titleYOff=-titleYOff;
			}
			else {
				yalign=1;
				titleYOff=labelYOff+1.4*o->_fontsize;
				if (side==0) y=(y0+y1)/2;
			}
			if (angle!=0) {
				if (side>0)
					xalign=-1;
				else
					xalign=1;
			}
		}
		// axis line
		o->line(x0,y,z,x1,y,z);

		// ticks
		k=0;
		step-=1.e-12*step;
		dmax=0;
		for (x=start; x<=end; x+=step) {
			if (x<x0 || x>x1) continue;
			o->line(x,y,z,0,tickLen,0,side);
			//minor ticks
			if (minor>0 && x<end) {
				dv=step/(minor+1);
				for (j=1; j<=minor; j++) {
					v=x+j*dv;
					if (v<x0 || v>x1) continue;
					o->line(v,y,z,0,tickLen/2,0,side);
				}
			}
			// tick labels
			v=x;
			if (between) {
				v+=step/2;
				if (v>end) continue;
				labelXOff=0.5*o->_fontsize*cos(D2R*(90+angle));
				if (side<=0) labelXOff=-labelXOff;
			}
			if (labels && k<nl) {
				strcpy(str,api_get_string(ctx,api_get_array_object2(ctx,labels,k++)));
			}
			else {
				if (fabs(x)<1.e-11) {
					sprintf(str,fmt,0.0);
				}
				else {
					sprintf(str,fmt,x);
				}
			}
			o->text(str,v,y,z,angle,xalign,yalign,labelXOff,labelYOff);
			// max y-offset of tick labels
			if (o->_transform!=TRANSFORM_PLOT3D) {
				sx=sy=0;
				o->dxdy(str,angle,sx,sy);
				dmax=max(dmax,fabs(sy));
			}
		}
		// minor tick on both ends
		if (minor>0) {
			dv=step/(minor+1);
			if (o->_xmin<start) {
				for (x=start-dv; x>=o->_xmin; x-=dv) {
					o->line(x,y,z,0,tickLen/2,0,side);
				}
			}
			if (o->_xmax>end) {
				for (x=end+dv; x<=o->_xmax; x+=dv) {
					o->line(x,y,z,0,tickLen/2,0,side);
				}
			}
		}
		// title ***
		if (title) {
			if (o->_transform!=TRANSFORM_PLOT3D) angle=0;
			cairo_set_font_size(o->_CR,1.2*o->_fontsize);
			o->text(title,(x0+x1)/2,y,z,angle,0,yalign,titleXOff,titleYOff-side*dmax);
			cairo_set_font_size(o->_CR,o->_fontsize);
		}
	}
	else if (type=='y') {
		if (start>EXLARGE) start=y0;
		if (end>EXLARGE) end=y1;
		if (step>EXLARGE) step=dy/4.0;
		// axis line
		z=o->_zmin;
		x=x1;
		xalign=-1;
		yalign=1;
		double yangle=90;
		if (o->_transform==TRANSFORM_PLOT3D) {
			side=1;
			xalign=0;
			yalign=0;
			angle=o->yangle();
			yangle=angle;
			tickLen=1.1*tickLen;
			labelYOff=tickLen+0.6*o->_fontsize;
			labelXOff=tickLen*cos(angle*D2R);
			titleYOff=labelYOff+1.5*o->_fontsize;
			titleXOff=labelYOff*cos(angle*D2R);
			if (o->_zrotate>0) {
				side=-1;
				x=x0;
				labelXOff=-labelXOff;
				titleXOff=-titleXOff;
			}
		}
		else {
			angle=0;
			xalign=1;
			yalign=0;
			labelXOff=1.5*tickLen;
			labelYOff=0;
			titleXOff=labelXOff+o->_fontsize;
			titleYOff=0;
			if (side>0) {
				x=x1;
				xalign=-1;
			}
			else {
				x=x0;
				if (side==0) x=(x0+x1)/2;
				labelXOff=-labelXOff;
				titleXOff=-titleXOff;
			}
		}
		// axis line
		o->line(x,y0,z,x,y1,z);
		// ticks
		k=0;
		step-=1.e-12*step;
		dmax=0;
		for (y=start; y<=end; y+=step) {
			if (y<y0 || y>y1) continue;
			if (o->_transform==TRANSFORM_PLOT3D) {
				o->line(x,y,z,0,tickLen,0,-1);
			}
			else {
				o->line(x,y,z,tickLen,0,0,side);
			}
			//minor ticks
			if (minor>0 && y<end) {
				dv=step/(minor+1);
				for (j=1; j<=minor; j++) {
					v=y+j*dv;
					if (v<y0 || v>y1) continue;
					if (o->_transform==TRANSFORM_PLOT3D) {
						o->line(x,v,z,0,tickLen/2,0,-1);
					}
					else {
						o->line(x,v,z,tickLen/2,0,0,side);
					}
				}
			}
			// tick labels
			if (labels && k<nl) {
				strcpy(str,api_get_string(ctx,api_get_array_object2(ctx,labels,k++)));
			}
			else {
				if (fabs(y)<1.e-11) {
					sprintf(str,fmt,0.0);
				}
				else {
					sprintf(str,fmt,y);
				}
			}
			o->text(str,x,y,z,angle,xalign,yalign,labelXOff,labelYOff);
			// max y-offset of tick labels
			sx=sy=0;
			o->dxdy(str,angle,sx,sy);
			dmax=max(dmax,fabs(sx)*cos(angle*D2R));
		}
		if (minor>0) {
			dv=step/(minor+1);
			if (o->_ymin<start) {
				for (y=start-dv; y>=o->_ymin; y-=dv) {
					if (o->_transform==TRANSFORM_PLOT3D) {
						o->line(x,y,z,0,tickLen/2,0,-1);
					}
					else {
						o->line(x,y,z,tickLen/2,0,0,side);
					}
				}
			}
			if (o->_ymax>end) {
				for (y=end+dv; y<=o->_ymax; y+=dv) {
					if (o->_transform==TRANSFORM_PLOT3D) {
						o->line(x,y,z,0,tickLen/2,0,-1);
					}
					else {
						o->line(x,y,z,tickLen/2,0,0,side);
					}
				}
			}
		}
		// title
		if (title) {
			cairo_set_font_size(o->_CR,1.2*o->_fontsize);
			o->text(title,x,(y0+y1)/2,z,yangle,xalign,yalign,titleXOff+side*dmax,titleYOff);
			cairo_set_font_size(o->_CR,o->_fontsize);
		}
	}
	else {
		if (o->_transform!=TRANSFORM_PLOT3D) return 0;
		// axis line
		if (start>EXLARGE) start=z0;
		if (end>EXLARGE) end=z1;
		if (step>EXLARGE) step=dz/4.0;
		// axis line
		yalign=0;
		y=y1;
		if (o->_zrotate>0) {
			side=-1;
			xalign=1;
			x=x0;
			labelXOff=-tickLen-0.4*o->_fontsize;
			titleXOff=labelXOff-o->_fontsize;
		}
		else {
			side=1;
			xalign=-1;
			x=x1;
			labelXOff=tickLen+0.4*o->_fontsize;
			titleXOff=labelXOff+o->_fontsize;
		}
		// axis line
		o->line(x,y,z0,x,y,z1);
		// ticks
		k=0;
		step-=1.e-12*step;
		dmax=0;
		for (z=start; z<=end; z+=step) {
			if (z<z0 || z>z1) continue;
			o->line(x,y,z,tickLen,0,0,side);
			//minor ticks
			if (minor>0 && y<end) {
				dv=step/(minor+1);
				for (j=1; j<=minor; j++) {
					v=z+j*dv;
					if (v<z0 || v>z1) continue;
					o->line(x,y,v,tickLen/2,0,0,side);
				}
			}
			// tick labels
			if (labels && k<nl) {
				strcpy(str,api_get_string(ctx,api_get_array_object2(ctx,labels,k++)));
			}
			else {
				if (fabs(z)<1.e-11) {
					sprintf(str,fmt,0.0);
				}
				else {
					sprintf(str,fmt,z);
				}
			}
			o->text(str,x,y,z,0,xalign,0,labelXOff,0);
			// max y-offset of tick labels
			sx=sy=0;
			o->dxdy(str,angle,sx,sy);
			dmax=max(dmax,fabs(sx));
		}
		if (minor>0) {
			dv=step/(minor+1);
			if (o->_zmin<start) {
				for (v=start-dv; v>=o->_zmin; v-=dv) {
					o->line(x,y,v,tickLen/2,0,0,side);
				}
			}
			if (o->_zmax>end) {
				for (v=end+dv; v<=o->_zmax; v+=dv) {
					o->line(x,y,v,tickLen/2,0,0,side);
				}
			}
		}
		// title
		if (title) {
			cairo_set_font_size(o->_CR,1.2*o->_fontsize);
			o->text(title,x,y,(z0+z1)/2,90,0,0,titleXOff+side*dmax,0);
			cairo_set_font_size(o->_CR,o->_fontsize);
		}
	}
	return 0;
}

static
void* cr_xaxis(void *ctx, int nargs, void** args)
{
	return cr_axis(ctx,nargs,args,'x');
}

static
void* cr_yaxis(void *ctx, int nargs, void** args)
{
	return cr_axis(ctx,nargs,args,'y');
}

static
void* cr_zaxis(void *ctx, int nargs, void** args)
{
	return cr_axis(ctx,nargs,args,'z');
}

static
void *cr_globe(void *ctx, int nargs, void **args)
{
	if (nargs<3) api_input_error(ctx);
	zsCairo* o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	o->_lat=api_get_number(ctx,args[1]);
	o->_lon=api_get_number(ctx,args[2]);
    double rotx=-(90-o->_lat)*D2R;
    double rotz=-(90+o->_lon)*D2R;
	o->_mr1[0]=cos(rotz);
	o->_mr1[1]=-sin(rotz);
	o->_mr1[2]=0;
	o->_mr2[0]=cos(rotx)*sin(rotz);
	o->_mr2[1]=cos(rotx)*cos(rotz);
	o->_mr2[2]=-sin(rotx);
	o->_mr3[0]=sin(rotx)*sin(rotz);
	o->_mr3[1]=sin(rotx)*cos(rotz);
	o->_mr3[2]=cos(rotx);
	o->_transform=TRANSFORM_SPHERE;
	o->_xmin=0;
	o->_xmax=360;
	o->_ymin=-90;
	o->_ymax=90;
	return 0;
}

static
void *cr_polar(void *ctx, int nargs, void **args)
{
	if (nargs<3) api_input_error(ctx);
	zsCairo* o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	o->_transform=TRANSFORM_POLAR;
	o->_xmin=api_get_number(ctx,args[1]);
	o->_xmax=api_get_number(ctx,args[2]);
	if (o->_xmin>=o->_xmax) api_input_error(ctx);
	o->_ymin=0;
	o->_ymax=90;
	if (nargs>4) {
		o->_ymin=api_get_number(ctx,args[3]);
		o->_ymax=api_get_number(ctx,args[4]);
		if (o->_ymin>=o->_ymax) api_input_error(ctx);
		if (o->_ymin<0 && o->_ymax>0) api_input_error(ctx);
		if (o->_ymax>0 && o->_ymin<0) api_input_error(ctx);
	}
	return 0;
}

static
void *cr_lambert(void *ctx, int nargs, void **args)
{
	if (nargs<3) api_input_error(ctx);
	zsCairo* o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	o->_transform=TRANSFORM_LAMBERT;
	o->_xmin=api_get_number(ctx,args[1]);
	o->_xmax=api_get_number(ctx,args[2]);
	if (o->_xmin>=o->_xmax) api_input_error(ctx);
	o->_ymin=0;
	o->_ymax=90;
	if (nargs>4) {
		o->_ymin=api_get_number(ctx,args[3]);
		o->_ymax=api_get_number(ctx,args[4]);
		if (o->_ymin>=o->_ymax) api_input_error(ctx);
		if (o->_ymin<0 && o->_ymax>0) api_input_error(ctx);
		if (o->_ymax>0 && o->_ymin<0) api_input_error(ctx);
	}

	double lat0=90*D2R, lat1=30*D2R, lat2=60*D2R;
	double lon0=0.5*(o->_xmin+o->_xmax)*D2R;
	o->_mr1[0]=log(cos(lat1)/cos(lat2))/log(tan(PI4+0.5*lat2)*cos(PI4+0.5*lat1));
	o->_mr1[1]=cos(lat1)*pow(tan(PI4+0.5*lat1),o->_mr1[0])/o->_mr1[0];
	o->_mr1[2]=o->_mr1[1]*pow(cos(PI4+0.5*lat0*0.99999),o->_mr1[0]);

	return 0;
}

static
void *cr_hammer(void *ctx, int nargs, void **args)
{
	if (nargs<2) api_input_error(ctx);
	zsCairo* o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	o->_transform=TRANSFORM_HAMMER;
	o->_ymin=-90;
	o->_ymax=90;
	if (api_get_integer(ctx,args[1])!=0) {
		// Pacific in the center
		o->_xmin=0;
		o->_xmax=360;
	}
	else {
		o->_xmin=-180;
		o->_xmax=180;
	}
	return 0;
}

static
void *cr_eckert(void *ctx, int nargs, void **args)
{
	if (nargs<2) api_input_error(ctx);
	zsCairo* o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	o->_transform=TRANSFORM_ECKERT;
	o->_ymin=-90;
	o->_ymax=90;
	if (api_get_integer(ctx,args[1])!=0) {
		// Pacific in the center
		o->_xmin=0;
		o->_xmax=360;
	}
	else {
		o->_xmin=-180;
		o->_xmax=180;
	}
	return 0;
}

static
void* cr_palette(void *ctx, int nargs, void** args)
{
	if (nargs<3) api_input_error(ctx);
	zsCairo* o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);;
	int i, j, interp=0;
	for (i=2; i<nargs; i+=2) {
		const char *name=api_get_string(ctx,args[i-1]);
		if (i==2) {
			o->_colors.clear();
			o->_values.clear();
		}
		if (!strcmp(name,"color")) {
			double r=api_get_number(ctx,api_get_array_object(ctx,args[i],"0"));
			double g=api_get_number(ctx,api_get_array_object(ctx,args[i],"1"));
			double b=api_get_number(ctx,api_get_array_object(ctx,args[i],"2"));
			double a=api_get_number(ctx,api_get_array_object(ctx,args[i],"3"));
			ZEcolor rgb={r,g,b};
			o->_colors.push_back(rgb);
			o->_values.push_back(a);
		}
		else if (!strcmp(name,"gradient")) {
			if (api_get_integer(ctx,args[i])!=0)
				o->_gradient=true;
			else
				o->_gradient=false;
		}
		else if (!strcmp(name, "interpolate")) {
			interp=api_get_integer(ctx,args[i]);
		}
		else {
			api_runtime_error2(ctx, "unknown palette parameter: ", name);
		}
	}

	if (interp>0) {
		vector<ZEcolor> colors;
		vector<double> values;
		ZEcolor c1, c2;
		double v1, v2;
		interp++;
		for (size_t i=1; i<o->_colors.size(); i++) {
			c1=o->_colors[i-1];
			c2=o->_colors[i];
			v1=o->_values[i-1];
			v2=o->_values[i];
			for (j=0; j<interp; j++) {
				double f1=double(interp-j)/interp;
				double f2=double(j)/interp;
				ZEcolor c={f1*c1.r+f2*c2.r,f1*c1.g+f2*c2.g,f1*c1.b+f2*c2.b};
				colors.push_back(c);
				values.push_back(f1*v1+f2*v2);
			}
		}
		colors.push_back(c2);
		values.push_back(v2);
		o->_colors=colors;
		o->_values=values;
	}

	return 0;
}

static
void* cr_colorbar(void *ctx, int nargs, void** args)
{
	if (nargs<5) api_input_error(ctx);
	zsCairo* o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	double x=api_get_number(ctx,args[1]);
	double y=api_get_number(ctx,args[2]);
	double w=api_get_number(ctx,args[3]);
	double h=api_get_number(ctx,args[4]);
	int digit=1;
	
	if (nargs>5) digit=api_get_integer(ctx,args[5]);
	if (w<0 || h<0) api_runtime_error(ctx,"w<0 || h<0");
	if (o->_colors.size()<2) api_runtime_error(ctx,"colors < 2");
	
	char fmt[24], str[24];
	sprintf(fmt,"%%0.%df",digit);

	vector<ZEcolor> colors=o->_colors;
	vector<double> values=o->_values;
	if (!o->_gradient) {
		colors.push_back(o->_colors.back());
		values.push_back(o->_values.back());
	}

	if (w>h) {
		// horizontal bar
		double y1=y, y2=y-h;
		double x1, x2;
		double dx=double(w)/(colors.size()-1);
		ZEcolor c1=colors[0];
		for (size_t i=1; i<colors.size(); i++) {
			ZEcolor c2=colors[i];
			x2=x+i*dx;
			x1=x2-dx;
			if (o->_gradient) {
				cairo_pattern_t *pat=cairo_pattern_create_linear(x1,y,x2,y);
				cairo_pattern_add_color_stop_rgb(pat,0,c1.r,c1.g,c1.b);
				cairo_pattern_add_color_stop_rgb(pat,1,c2.r,c2.g,c2.b);
				cairo_move_to(o->_CR,x1,y1);
				cairo_line_to(o->_CR,x2,y1);
				cairo_line_to(o->_CR,x2,y2);
				cairo_line_to(o->_CR,x1,y2);
				cairo_set_source(o->_CR,pat);
				cairo_fill(o->_CR);
				cairo_pattern_destroy(pat);
			}
			else {
				cairo_set_source_rgb(o->_CR,c1.r,c1.g,c1.b);
				cairo_move_to(o->_CR,x1,y1);
				cairo_line_to(o->_CR,x2,y1);
				cairo_line_to(o->_CR,x2,y2);
				cairo_line_to(o->_CR,x1,y2);
				cairo_fill(o->_CR);
			}
			cairo_set_source_rgb(o->_CR,o->_color.r,o->_color.g,o->_color.b);
			cairo_move_to(o->_CR,x1,y1);
			cairo_line_to(o->_CR,x1,y2);
			cairo_stroke(o->_CR);
			sprintf(str,fmt,values[i-1]);
			cairo_text_extents_t extents;
			cairo_text_extents(o->_CR,str,&extents);
			cairo_move_to(o->_CR,x1-extents.width/2,y2-0.3*o->_fontsize);
			cairo_show_text(o->_CR,str);
			c1=c2;
		}
		if (o->_gradient) {
			cairo_move_to(o->_CR,x2,y1);
			cairo_line_to(o->_CR,x2,y2);
			cairo_stroke(o->_CR);
			sprintf(str,fmt,values.back());
			cairo_text_extents_t extents;
			cairo_text_extents(o->_CR,str,&extents);
			cairo_move_to(o->_CR,x2-extents.width/2,y2-0.3*o->_fontsize);
			cairo_show_text(o->_CR,str);
		}
	}
	else {
		// vertical bar
		double x1=x, x2=x+w;
		double dy=double(h)/(colors.size()-1);
		double y1, y2;
		ZEcolor c1=colors[0];
		for (size_t i=1; i<colors.size(); i++) {
			ZEcolor c2=colors[i];
			y2=y-i*dy;
			y1=y2+dy;
			if (o->_gradient) {
				cairo_pattern_t *pat=cairo_pattern_create_linear(x,y1,x,y2);
				cairo_pattern_add_color_stop_rgb(pat,0,c1.r,c1.g,c1.b);
				cairo_pattern_add_color_stop_rgb(pat,1,c2.r,c2.g,c2.b);
				cairo_move_to(o->_CR,x1,y1);
				cairo_line_to(o->_CR,x2,y1);
				cairo_line_to(o->_CR,x2,y2);
				cairo_line_to(o->_CR,x1,y2);
				cairo_set_source(o->_CR,pat);
				cairo_fill(o->_CR);
				cairo_pattern_destroy(pat);
			}
			else {
				cairo_set_source_rgb(o->_CR,c1.r,c1.g,c1.b);
				cairo_move_to(o->_CR,x1,y1);
				cairo_line_to(o->_CR,x2,y1);
				cairo_line_to(o->_CR,x2,y2);
				cairo_line_to(o->_CR,x1,y2);
				cairo_fill(o->_CR);
			}
			cairo_set_source_rgb(o->_CR,o->_color.r,o->_color.g,o->_color.b);
			cairo_move_to(o->_CR,x1,y1);
			cairo_line_to(o->_CR,x2,y1);
			cairo_stroke(o->_CR);
			sprintf(str,fmt,values[i-1]);
			cairo_text_extents_t extents;
			cairo_text_extents(o->_CR,str,&extents);
			cairo_move_to(o->_CR,x2+0.3*o->_fontsize,y1+extents.height/2);
			cairo_show_text(o->_CR,str);
			c1=c2;
		}
		if (o->_gradient) {
			cairo_move_to(o->_CR,x1,y2);
			cairo_line_to(o->_CR,x2,y2);
			cairo_stroke(o->_CR);
			sprintf(str,fmt,values.back());
			cairo_text_extents_t extents;
			cairo_text_extents(o->_CR,str,&extents);
			cairo_move_to(o->_CR,x2+0.3*o->_fontsize,y2+extents.height/2);
			cairo_show_text(o->_CR,str);
		}
	}

	return 0;
}

static
void *cr_grid(void *ctx, int nargs, void **args)
{
	if (nargs<3) api_input_error(ctx);
	zsCairo* o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);;
	double dx=api_get_number(ctx,args[1]);
	double dy=api_get_number(ctx,args[2]);
	if (dx<=0 || dy<=0) api_runtime_error(ctx,"dx<=0 || dy<=0");
	double x, y;
	if (o->_transform==TRANSFORM_SPHERE) {
        for (y=-90+dy; y<90; y+=dy) {
            for (x=1; x<=360; x++) o->line(x-1,y,o->_zmin,x,y,o->_zmin);
        }
        for (x=0; x<360; x+=dx) {
            for (y=-89; y<=90; y++) o->line(x,y-1,o->_zmin,x,y,o->_zmin);
        }
	}
	else if (o->_transform==TRANSFORM_HAMMER) {
        for (y=-90+dy; y<90; y+=dy) {
            for (x=o->_xmin+1; x<=o->_xmax; x++) o->line(x-1,y,o->_zmin,x,y,o->_zmin);
        }
        for (x=o->_xmin; x<=o->_xmax; x+=dx) {
            for (y=-89; y<=90; y++) o->line(x,y-1,o->_zmin,x,y,o->_zmin);
        }
	}
	else if (o->_transform==TRANSFORM_ECKERT) {
        for (y=-90; y<=90; y+=dy) {
            o->line(o->_xmin,y,o->_zmin,o->_xmax,y,o->_zmin);
        }
        for (x=o->_xmin; x<=o->_xmax; x+=dx) {
            for (y=-89; y<=90; y++) o->line(x,y-1,o->_zmin,x,y,o->_zmin);
        }
	}
	else if (o->_transform==TRANSFORM_POLAR || o->_transform==TRANSFORM_LAMBERT) {
        for (y=o->_ymin; y<=o->_ymax; y+=dy) {
            for (x=o->_xmin; x<o->_xmax; x++) o->line(x,y,o->_zmin,x+1,y,o->_zmin);
        }
        for (x=o->_xmin; x<=o->_xmax; x+=dx) {
            o->line(x,o->_ymin,o->_zmin,x,o->_ymax,o->_zmin);
        }
	}
	else {
        for (y=o->_ymin+dy; y<o->_ymax; y+=dy) {
            o->line(o->_xmin,y,o->_zmin,o->_xmax,y,o->_zmin);
        }
        for (x=o->_xmin+dx; x<o->_xmax; x+=dx) {
            o->line(x,o->_ymin,o->_zmin,x,o->_ymax,o->_zmin);
        }
	}

	return 0;
}

static
void *cr_frame(void *ctx, int nargs, void **args)
{
	if (nargs<1) api_input_error(ctx);
	zsCairo* o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	int fill=0;
	if (nargs>1) fill=api_get_integer(ctx,args[1]);
	double x, y, lon, lat;
	if (o->_transform==TRANSFORM_SPHERE) {
		double r=0.5*o->_scale;
		double x=o->_lon;
		double y=o->_lat;
		o->transform(x,y,o->_zmin);
		cairo_arc(o->_CR,x,y,r,0.0,360*D2R);
	}
	else if (o->_transform==TRANSFORM_HAMMER || o->_transform==TRANSFORM_ECKERT) {
		x=o->_xmin;
		y=o->_ymin;
		o->transform(x,y,o->_zmin);
		cairo_move_to(o->_CR,x,y);
		for (lat=o->_ymin+1; lat<=o->_ymax; lat++) {
			x=o->_xmin;
			y=lat;
			o->transform(x,y,o->_zmin);
			cairo_line_to(o->_CR,x,y);
		}
		x=o->_xmax;
		y=o->_ymax;
		o->transform(x,y,o->_zmin);
		cairo_line_to(o->_CR,x,y);
		for (lat=o->_ymax-1; lat>=o->_ymin; lat--) {
			x=o->_xmax;
			y=lat;
			o->transform(x,y,o->_zmin);
			cairo_line_to(o->_CR,x,y);
		}
		x=o->_xmin;
		y=o->_ymin;
		o->transform(x,y,o->_zmin);
		cairo_line_to(o->_CR,x,y);
	}
	else if (o->_transform==TRANSFORM_POLAR || o->_transform==TRANSFORM_LAMBERT) {
		if (o->_ymin>-90) {
			// northern hemiersphere
			x=o->_xmin;
			y=o->_ymin;
			o->transform(x,y,o->_zmin);
			cairo_move_to(o->_CR,x,y);
			for (lon=o->_xmin+1; lon<=o->_xmax; lon++) {
				x=lon;
				y=o->_ymin;
				o->transform(x,y,o->_zmin);
				cairo_line_to(o->_CR,x,y);
			}
			if (o->_xmax-o->_xmin<360) {
				x=o->_xmax;
				y=o->_ymax;
				o->transform(x,y,o->_zmin);
				cairo_line_to(o->_CR,x,y);
				x=o->_xmin;
				y=o->_ymin;
				o->transform(x,y,o->_zmin);
				cairo_line_to(o->_CR,x,y);
			}
        }
		else {
			// southern hemiersphere
			x=o->_xmin;
			y=o->_ymax;
			o->transform(x,y,o->_zmin);
			cairo_move_to(o->_CR,x,y);
			for (lon=o->_xmin+1; lon<=o->_xmax; lon++) {
				x=lon;
				y=o->_ymax;
				o->transform(x,y,o->_zmin);
				cairo_line_to(o->_CR,x,y);
			}
			if (o->_xmax-o->_xmin<360) {
				x=o->_xmax;
				y=o->_ymin;
				o->transform(x,y,o->_zmin);
				cairo_line_to(o->_CR,x,y);
				x=o->_xmin;
				y=o->_ymax;
				o->transform(x,y,o->_zmin);
				cairo_line_to(o->_CR,x,y);
			}
		}
	}
	else {
		x=o->_xmin;
		y=o->_ymin;
		o->transform(x,y,o->_zmin);
		cairo_move_to(o->_CR,x,y);
		x=o->_xmin;
		y=o->_ymax;
		o->transform(x,y,o->_zmin);
		cairo_line_to(o->_CR,x,y);
		x=o->_xmax;
		y=o->_ymax;
		o->transform(x,y,o->_zmin);
		cairo_line_to(o->_CR,x,y);
		x=o->_xmax;
		y=o->_ymin;
		o->transform(x,y,o->_zmin);
		cairo_line_to(o->_CR,x,y);
		x=o->_xmin;
		y=o->_ymin;
		o->transform(x,y,o->_zmin);
		cairo_line_to(o->_CR,x,y);
	}
	if (fill) {
		cairo_fill(o->_CR);
	}
	else {
		cairo_stroke(o->_CR);
	}
	return 0;
}

static void
fill_gshhs(zsCairo* o, const vector<double> &x, const vector<double> &y, double offset)
{
	if (x.size()<3) return;
	double sx=x[0]+offset;
	double sy=y[0];
	o->transform(sx,sy,o->_zmin);
	cairo_new_path(o->_CR);
	cairo_move_to(o->_CR,sx,sy);
	for (size_t i=1; i<x.size(); i++) {
		sx=x[i]+offset;
		sy=y[i];
		o->transform(sx,sy,o->_zmin);
		cairo_line_to(o->_CR,sx,sy);
	}
	cairo_fill(o->_CR);
}


static void
*cr_gshhs(void *ctx, int nargs, void **args)
{
	if (nargs<2) api_input_error(ctx);
	zsCairo* o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);;
	FILE *fp=fopen(api_get_string(ctx,args[1]),"rb");
	if (!fp) api_runtime_error(ctx,"failed to open GSHHS file");
	int fill=0;
	if (nargs>2) fill=api_get_integer(ctx,args[2]);
	int type=1;
	if (nargs>3) {
		type=api_get_integer(ctx,args[3]);
	}

	int clip=o->_clip;

	GSHHS h;
	fread(&h,sizeof(GSHHS),1,fp);
	int version=(h.flag>>8)&255;
	int flip=(version!=GSHHS_DATA_RELEASE);		// Take as sign that byte-swabbing is needed
	fseek(fp,0,SEEK_SET);

	o->clip(1,o->_zmin);

	while (!feof(fp)) {
		if (fread(&h,sizeof(GSHHS),1,fp)!=1) break;
		if (flip) {
			h.id=swabi4((unsigned int)h.id);
			h.n=swabi4((unsigned int)h.n);
			h.west=swabi4((unsigned int)h.west);
			h.east=swabi4((unsigned int)h.east);
			h.south=swabi4((unsigned int)h.south);
			h.north=swabi4((unsigned int)h.north);
			h.area=swabi4((unsigned int)h.area);
			h.area_full=swabi4((unsigned int)h.area_full);
			h.flag=swabi4((unsigned int)h.flag);
			h.container=swabi4((unsigned int)h.container);
			h.ancestor=swabi4((unsigned int)h.ancestor);
		}

		int greenwich=(h.flag>>16)&1;
		int level=h.flag&255;
		int river=(h.flag>>25)&1;

		if (river==1) {
			fseek(fp,sizeof(GSHHS_XY)*h.n,SEEK_CUR);
			continue;
		}

		if (level!=type) {
			fseek(fp,sizeof(GSHHS_XY)*h.n,SEEK_CUR);
			continue;
		}

		if (GSHHS_SCL*h.south>=o->_ymax || GSHHS_SCL*h.north<=o->_ymin) {
			// out of latitude boundary
			fseek(fp,sizeof(GSHHS_XY)*h.n,SEEK_CUR);
			continue;
		}

		int i;
		vector<double> x, y;

		for (i=0; i<h.n; i++) {
			GSHHS_XY xy;
			fread(&xy,sizeof(GSHHS_XY),1,fp);
			if (flip) {
				xy.x=swabi4((unsigned int)xy.x);
				xy.y=swabi4((unsigned int)xy.y);
			}
			double lat=GSHHS_SCL*xy.y;
			double lon=GSHHS_SCL*xy.x;
			if (greenwich==1 && lon>CROSSGW) {
				// negative lon has been transformed as lon=lon+360;
				lon-=360.0;
			}
			y.push_back(lat);
			x.push_back(lon);
		}

		if (fill) {
			if (GSHHS_SCL*h.south<-60.0 && fabs(x.back()-x[0])>30.0) {
				// south pole
				x.push_back(x.back());
				y.push_back(-90.0);
				x.push_back(x[0]);
				y.push_back(-90.0);
			}
			fill_gshhs(o,x,y,0.0);
			if (o->_xmin<0.0) {
				fill_gshhs(o,x,y,-360.0);
			}
			else {
				fill_gshhs(o,x,y,360.0);
			}
		}
		else {
			for (i=1; i<h.n; i++) {
				double sx1=x[i-1];
				double sy1=y[i-1];
				double sx2=x[i];
				double sy2=y[i];
				o->line(sx1,sy1,o->_zmin,sx2,sy2,o->_zmin);
				if (o->_xmin<0.0) {
					o->line(sx1-360.0,sy1,o->_zmin,sx2-360.0,sy2,o->_zmin);
				}
				else {
					o->line(sx1+360.0,sy1,o->_zmin,sx2+360.0,sy2,o->_zmin);
				}
			}
		}
	}

	fclose(fp);
	o->clip(clip,o->_zmin);

	if (fill) {
		cairo_set_source_rgb(o->_CR,o->_color.r,o->_color.g,o->_color.b);
	}

	return 0;
}

static
void* cr_contour(void *ctx, int nargs, void** args)
{
	if (nargs<8) api_input_error(ctx);
	zsCairo* o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	double x0=api_get_number(ctx,args[1]);
	double dx=api_get_number(ctx,args[2]);
	int nx=api_get_integer(ctx,args[3]);
	double y0=api_get_number(ctx,args[4]);
	double dy=api_get_number(ctx,args[5]);
	int ny=api_get_integer(ctx,args[6]);
	double *zptr=(double*)api_get_ptr(ctx,args[7]);
	if (nargs==8) {
		// contour map:	nz=ny*nx
		double lw=cairo_get_line_width(o->_CR);
		cairo_set_line_width(o->_CR,0.25);
		for (int i=1; i<ny; i++) {
			double y1=y0+(i-1)*dy;
			double y2=y1+dy;
			for (int j=1; j<nx; j++) {
				double x1=x0+(j-1)*dx;
				double x2=x1+dx;
				int k=i*nx+j; 
				
				ZExyz p0={x1,y1,zptr[k-nx-1]};
				ZExyz p3={x2,y1,zptr[k-nx]};
				ZExyz p2={x2,y2,zptr[k]};
				ZExyz p1={x1,y2,zptr[k-1]};
				
				double d1=o->transform(p0.x,p0.y,o->_zmin);
				double d2=o->transform(p1.x,p1.y,o->_zmin);
				double d3=o->transform(p2.x,p2.y,o->_zmin);
				double d4=o->transform(p3.x,p3.y,o->_zmin);
				if (d1<0 || d2<0 || d3<0 || d4<0) continue;

				ZEcolor c=o->rgb(0.25*(p0.z+p1.z+p2.z+p3.z));
				cairo_move_to(o->_CR,p0.x,p0.y);
				cairo_line_to(o->_CR,p1.x,p1.y);
				cairo_line_to(o->_CR,p2.x,p2.y);
				cairo_line_to(o->_CR,p3.x,p3.y);
				cairo_set_source_rgb(o->_CR,c.r,c.g,c.b);
				cairo_fill_preserve(o->_CR);
				cairo_stroke(o->_CR);
			}
		}
		cairo_set_line_width(o->_CR,lw);
	}
	else {
		// contour line: nz=nx*ny
		double iso=api_get_number(ctx,args[8]);
		if (o->_values.size()>0) {
			ZEcolor c=o->rgb(iso);
			cairo_set_source_rgb(o->_CR,c.r,c.g,c.b);
		}
		for (int i=1; i<ny; i++) {
			double y1=y0+(i-1)*dy;
			double y2=y1+dy;
			for (int j=1; j<nx; j++) {
				double x1=x0+(j-1)*dx;
				double x2=x1+dx;
				int k=i*nx+j; 
				ZExyz p1={x1,y1,zptr[k-nx-1]};
				ZExyz p2={x2,y1,zptr[k-nx]};
				ZExyz p3={x2,y2,zptr[k]};
				ZExyz p4={x1,y2,zptr[k-1]};
				o->contour4p(iso,p1,p2,p3,p4,o->_zmin);
			}
		}
	}
	cairo_set_source_rgb(o->_CR,o->_color.r,o->_color.g,o->_color.b);
	return 0;
}

static
void* cr_field(void *ctx, int nargs, void** args)
{
	if (nargs<9) api_input_error(ctx);
	zsCairo* o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	double x0=api_get_number(ctx,args[1]);
	double dx=api_get_number(ctx,args[2]);
	int nx=api_get_integer(ctx,args[3]);
	double y0=api_get_number(ctx,args[4]);
	double dy=api_get_number(ctx,args[5]);
	int ny=api_get_integer(ctx,args[6]);
	double *u=(double*)api_get_ptr(ctx,args[7]);
	double *v=(double*)api_get_ptr(ctx,args[8]);
	double size=7;
	if (nargs>9) size=api_get_number(ctx,args[9]);
	for (int i=0; i<ny; i++) {
		double y=y0+i*dy;
		for (int j=0; j<nx; j++) {
			double x=x0+j*dx;
			int k=i*nx+j;
			if (o->_values.size()>0) {
				ZEcolor c=o->rgb(sqrt(u[k]*u[k]+v[k]*v[k]));
				cairo_set_source_rgb(o->_CR,c.r,c.g,c.b);
			}
			o->arrow(x,y,o->_zmin,u[k],v[k],size);
		}
	}
	cairo_set_source_rgb(o->_CR,o->_color.r,o->_color.g,o->_color.b);
	return 0;
}

static
void* cr_get(void *ctx, int nargs, void** args)
{
	if (nargs<3) api_input_error(ctx);
	zsCairo *o=(zsCairo*)api_get_user(ctx,args[0],CAIRO);
	int w=cairo_image_surface_get_width(o->_SFC);
	int h=cairo_image_surface_get_height(o->_SFC);
	int s=cairo_image_surface_get_stride(o->_SFC);
	int rgba=s/w;
	double x=api_get_number(ctx,args[1]);
	double y=api_get_number(ctx,args[2]);
	double z=o->_zmin;
	if (nargs>3) z=api_get_number(ctx,args[2]);
	o->transform(x,y,z);
	x=floor(x);
	y=floor(y);
	if (x<0 || x>=w || y<0 || y>=h) api_runtime_error(ctx,"x or y is out side the plot area");
	cairo_surface_flush(o->_SFC); 
	unsigned char *ptr=cairo_image_surface_get_data(o->_SFC);
	int k=int(y)*s+int(x)*rgba;
	void *arr=api_create_array(ctx,3);
	api_set_array_object(ctx,arr,"0",api_create_integer(ctx,ptr[k+2]));
	api_set_array_object(ctx,arr,"1",api_create_integer(ctx,ptr[k+1]));
	api_set_array_object(ctx,arr,"2",api_create_integer(ctx,ptr[k+0]));
	return arr;
}


/////////////////////////////////////////////////////////////////////

class cairoRegPrimitive
{
public:
	cairoRegPrimitive()
	{
		api_add_primitive("cairo",		0,			cr_create);
#ifdef _WIN32
		api_add_primitive("show",		CAIRO,		cr_show);
#endif
		api_add_primitive("save",		CAIRO,		cr_save);
		api_add_primitive("size",		CAIRO,		cr_size);
		api_add_primitive("version",	CAIRO,		cr_version);

		api_add_primitive("color",		CAIRO,		cr_color);
		api_add_primitive("bgcolor",	CAIRO,		cr_bgcolor);
		api_add_primitive("linear",		CAIRO,		cr_linear);
		api_add_primitive("radial",		CAIRO,		cr_radial);
		api_add_primitive("vertex",		CAIRO,		cr_vertex);
		api_add_primitive("font",		CAIRO,		cr_font);
		api_add_primitive("clip",		CAIRO,		cr_clip);
		api_add_primitive("unclip",		CAIRO,		cr_unclip);
		api_add_primitive("lwidth",		CAIRO,		cr_lwidth);
		api_add_primitive("lstyle",		CAIRO,		cr_lstyle);
		api_add_primitive("state",		CAIRO,		cr_state);

		api_add_primitive("image",		CAIRO,		cr_image);
		api_add_primitive("text",		CAIRO,		cr_text);
		api_add_primitive("text3D",		CAIRO,		cr_text3D);
		api_add_primitive("line",		CAIRO,		cr_line);
		api_add_primitive("line3D",		CAIRO,		cr_line3D);
		api_add_primitive("curve",		CAIRO,		cr_curve);
		api_add_primitive("curve3D",	CAIRO,		cr_curve3D);
		api_add_primitive("polygon",	CAIRO,		cr_polygon);
		api_add_primitive("polygon3D",	CAIRO,		cr_polygon3D);
		api_add_primitive("circle",		CAIRO,		cr_circle);
		api_add_primitive("circle3D",	CAIRO,		cr_circle3D);
		api_add_primitive("arc",		CAIRO,		cr_arc);
		api_add_primitive("arc3D",		CAIRO,		cr_arc3D);
		api_add_primitive("arrow",		CAIRO,		cr_arrow);
		api_add_primitive("arrow3D",	CAIRO,		cr_arrow3D);
		api_add_primitive("symbol",		CAIRO,		cr_symbol);
		api_add_primitive("symbol3D",	CAIRO,		cr_symbol3D);

		api_add_primitive("plot",		CAIRO,		cr_plot);
		api_add_primitive("xaxis",		CAIRO,		cr_xaxis);
		api_add_primitive("yaxis",		CAIRO,		cr_yaxis);
		api_add_primitive("zaxis",		CAIRO,		cr_zaxis);
		api_add_primitive("globe",		CAIRO,		cr_globe);
		api_add_primitive("polar",		CAIRO,		cr_polar);
		api_add_primitive("lambert",	CAIRO,		cr_lambert);
		api_add_primitive("hammer",		CAIRO,		cr_hammer);
		api_add_primitive("eckert",		CAIRO,		cr_eckert);

		api_add_primitive("palette",	CAIRO,		cr_palette);
		api_add_primitive("colorbar",	CAIRO,		cr_colorbar);
		api_add_primitive("grid",		CAIRO,		cr_grid);
		api_add_primitive("frame",		CAIRO,		cr_frame);
		api_add_primitive("gshhs",		CAIRO,		cr_gshhs);
		api_add_primitive("contour",	CAIRO,		cr_contour);
		api_add_primitive("field",		CAIRO,		cr_field);
		api_add_primitive("get",		CAIRO,		cr_get);
	}
};

static cairoRegPrimitive cairo_register;
